﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/elasticloadbalancingv2/ElasticLoadBalancingv2Request.h>
#include <aws/elasticloadbalancingv2/ElasticLoadBalancingv2_EXPORTS.h>

#include <utility>

namespace Aws {
namespace ElasticLoadBalancingv2 {
namespace Model {

/**
 */
class DescribeTrustStoreRevocationsRequest : public ElasticLoadBalancingv2Request {
 public:
  AWS_ELASTICLOADBALANCINGV2_API DescribeTrustStoreRevocationsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeTrustStoreRevocations"; }

  AWS_ELASTICLOADBALANCINGV2_API Aws::String SerializePayload() const override;

 protected:
  AWS_ELASTICLOADBALANCINGV2_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the trust store.</p>
   */
  inline const Aws::String& GetTrustStoreArn() const { return m_trustStoreArn; }
  inline bool TrustStoreArnHasBeenSet() const { return m_trustStoreArnHasBeenSet; }
  template <typename TrustStoreArnT = Aws::String>
  void SetTrustStoreArn(TrustStoreArnT&& value) {
    m_trustStoreArnHasBeenSet = true;
    m_trustStoreArn = std::forward<TrustStoreArnT>(value);
  }
  template <typename TrustStoreArnT = Aws::String>
  DescribeTrustStoreRevocationsRequest& WithTrustStoreArn(TrustStoreArnT&& value) {
    SetTrustStoreArn(std::forward<TrustStoreArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The revocation IDs of the revocation files you want to describe.</p>
   */
  inline const Aws::Vector<long long>& GetRevocationIds() const { return m_revocationIds; }
  inline bool RevocationIdsHasBeenSet() const { return m_revocationIdsHasBeenSet; }
  template <typename RevocationIdsT = Aws::Vector<long long>>
  void SetRevocationIds(RevocationIdsT&& value) {
    m_revocationIdsHasBeenSet = true;
    m_revocationIds = std::forward<RevocationIdsT>(value);
  }
  template <typename RevocationIdsT = Aws::Vector<long long>>
  DescribeTrustStoreRevocationsRequest& WithRevocationIds(RevocationIdsT&& value) {
    SetRevocationIds(std::forward<RevocationIdsT>(value));
    return *this;
  }
  inline DescribeTrustStoreRevocationsRequest& AddRevocationIds(long long value) {
    m_revocationIdsHasBeenSet = true;
    m_revocationIds.push_back(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The marker for the next set of results. (You received this marker from a
   * previous call.)</p>
   */
  inline const Aws::String& GetMarker() const { return m_marker; }
  inline bool MarkerHasBeenSet() const { return m_markerHasBeenSet; }
  template <typename MarkerT = Aws::String>
  void SetMarker(MarkerT&& value) {
    m_markerHasBeenSet = true;
    m_marker = std::forward<MarkerT>(value);
  }
  template <typename MarkerT = Aws::String>
  DescribeTrustStoreRevocationsRequest& WithMarker(MarkerT&& value) {
    SetMarker(std::forward<MarkerT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results to return with this call.</p>
   */
  inline int GetPageSize() const { return m_pageSize; }
  inline bool PageSizeHasBeenSet() const { return m_pageSizeHasBeenSet; }
  inline void SetPageSize(int value) {
    m_pageSizeHasBeenSet = true;
    m_pageSize = value;
  }
  inline DescribeTrustStoreRevocationsRequest& WithPageSize(int value) {
    SetPageSize(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_trustStoreArn;

  Aws::Vector<long long> m_revocationIds;

  Aws::String m_marker;

  int m_pageSize{0};
  bool m_trustStoreArnHasBeenSet = false;
  bool m_revocationIdsHasBeenSet = false;
  bool m_markerHasBeenSet = false;
  bool m_pageSizeHasBeenSet = false;
};

}  // namespace Model
}  // namespace ElasticLoadBalancingv2
}  // namespace Aws
