/*
 * Decompiled with CFR 0.152.
 */
package org.apache.storm.localizer;

import com.codahale.metrics.Histogram;
import com.codahale.metrics.Meter;
import com.codahale.metrics.Timer;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Stream;
import org.apache.storm.blobstore.ClientBlobStore;
import org.apache.storm.blobstore.InputStreamWithMeta;
import org.apache.storm.generated.AuthorizationException;
import org.apache.storm.generated.KeyNotFoundException;
import org.apache.storm.localizer.BlobChangingCallback;
import org.apache.storm.localizer.GoodToGo;
import org.apache.storm.localizer.IOFunction;
import org.apache.storm.localizer.PortAndAssignment;
import org.apache.storm.metric.StormMetricsRegistry;
import org.apache.storm.utils.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class LocallyCachedBlob {
    public static final long NOT_DOWNLOADED_VERSION = -1L;
    private static final Logger LOG = LoggerFactory.getLogger(LocallyCachedBlob.class);
    private static final BlobChangingCallback NOOP_CB = (assignment, port, resource, go) -> {};
    private final ConcurrentHashMap<PortAndAssignment, BlobChangingCallback> references = new ConcurrentHashMap();
    private final String blobDescription;
    private final String blobKey;
    private AtomicLong lastUsed = new AtomicLong(Time.currentTimeMillis());
    private final Histogram fetchingRate;
    private final Meter numBlobUpdateVersionChanged;
    private final Timer singleBlobLocalizationDuration;
    protected long localUpdateTime = -1L;

    protected LocallyCachedBlob(String blobDescription, String blobKey, StormMetricsRegistry metricsRegistry) {
        this.blobDescription = blobDescription;
        this.blobKey = blobKey;
        this.fetchingRate = metricsRegistry.registerHistogram("supervisor:blob-fetching-rate-MB/s");
        this.numBlobUpdateVersionChanged = metricsRegistry.registerMeter("supervisor:num-blob-update-version-changed");
        this.singleBlobLocalizationDuration = metricsRegistry.registerTimer("supervisor:single-blob-localization-duration");
    }

    protected DownloadMeta fetch(ClientBlobStore store, String key, IOFunction<Long, Path> pathSupplier, IOFunction<File, OutputStream> outStreamSupplier) throws KeyNotFoundException, AuthorizationException, IOException {
        try (InputStreamWithMeta in = store.getBlob(key);){
            long duration;
            long newVersion = in.getVersion();
            long currentVersion = this.getLocalVersion();
            if (newVersion == currentVersion) {
                LOG.warn("The version did not change, but going to download again {} {}", (Object)currentVersion, (Object)key);
            }
            Path downloadPath = pathSupplier.apply(newVersion);
            LOG.debug("Downloading {} to {}", (Object)key, (Object)downloadPath);
            long totalRead = 0L;
            try (OutputStream out = outStreamSupplier.apply(downloadPath.toFile());){
                int read;
                long startTime = Time.nanoTime();
                byte[] buffer = new byte[4096];
                while ((read = in.read(buffer)) >= 0) {
                    out.write(buffer, 0, read);
                    totalRead += (long)read;
                }
                duration = Time.nanoTime() - startTime;
            }
            long expectedSize = in.getFileLength();
            if (totalRead != expectedSize) {
                throw new IOException("We expected to download " + expectedSize + " bytes but found we got " + totalRead);
            }
            double downloadRate = (double)totalRead * 1000.0 / (double)duration;
            this.fetchingRate.update(Math.round(downloadRate));
            DownloadMeta downloadMeta = new DownloadMeta(downloadPath, newVersion);
            return downloadMeta;
        }
    }

    public abstract long getLocalVersion();

    public abstract long getRemoteVersion(ClientBlobStore var1) throws KeyNotFoundException, AuthorizationException;

    public abstract long fetchUnzipToTemp(ClientBlobStore var1) throws IOException, KeyNotFoundException, AuthorizationException;

    protected abstract void commitNewVersion(long var1) throws IOException;

    public abstract void cleanupOrphanedData() throws IOException;

    public abstract void completelyRemove() throws IOException;

    public abstract long getSizeOnDisk();

    protected static long getSizeOnDisk(Path p) throws IOException {
        if (!Files.exists(p, new LinkOption[0])) {
            return 0L;
        }
        if (Files.isRegularFile(p, new LinkOption[0])) {
            return Files.size(p);
        }
        try (Stream<Path> stream = Files.walk(p, new FileVisitOption[0]);){
            long l = stream.filter(subp -> Files.isRegularFile(subp, LinkOption.NOFOLLOW_LINKS)).mapToLong(subp -> {
                try {
                    return Files.size(subp);
                }
                catch (IOException e) {
                    LOG.warn("Could not get the size of {}", subp);
                    return 0L;
                }
            }).sum();
            return l;
        }
    }

    protected void touch() {
        this.lastUsed.set(Time.currentTimeMillis());
        LOG.debug("Setting {} ts to {}", (Object)this.blobKey, (Object)this.lastUsed.get());
    }

    public long getLastUsed() {
        return this.lastUsed.get();
    }

    public boolean isUsed() {
        return !this.references.isEmpty();
    }

    public void addReference(PortAndAssignment pna, BlobChangingCallback cb) {
        this.touch();
        LOG.info("Adding reference {} with timestamp {} to {}", new Object[]{pna, this.getLastUsed(), this.blobDescription});
        if (cb == null) {
            cb = NOOP_CB;
        }
        if (this.references.put(pna, cb) != null) {
            LOG.warn("{} already has a reservation for {}", (Object)pna, (Object)this.blobDescription);
        }
    }

    public boolean removeReference(PortAndAssignment pna) {
        LOG.info("Removing reference {} from {}", (Object)pna, (Object)this.blobDescription);
        PortAndAssignment reservedReference = null;
        for (Map.Entry<PortAndAssignment, BlobChangingCallback> entry : this.references.entrySet()) {
            if (!entry.getKey().isEquivalentTo(pna)) continue;
            reservedReference = entry.getKey();
            break;
        }
        if (reservedReference != null) {
            this.references.remove(reservedReference);
            this.touch();
            return true;
        }
        LOG.warn("{} had no reservation for {}, current references are {} with last update at {}", new Object[]{pna, this.blobDescription, this.getDependencies(), this.getLastUsed()});
        return false;
    }

    public synchronized void informReferencesAndCommitNewVersion(long newVersion) throws IOException {
        CompletableFuture<Void> doneUpdating = this.informAllOfChangeAndWaitForConsensus();
        this.commitNewVersion(newVersion);
        doneUpdating.complete(null);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private CompletableFuture<Void> informAllOfChangeAndWaitForConsensus() {
        HashMap<PortAndAssignment, BlobChangingCallback> refsCopy = new HashMap<PortAndAssignment, BlobChangingCallback>(this.references);
        CountDownLatch cdl = new CountDownLatch(refsCopy.size());
        CompletableFuture<Void> doneUpdating = new CompletableFuture<Void>();
        for (Map.Entry<PortAndAssignment, BlobChangingCallback> entry : refsCopy.entrySet()) {
            GoodToGo gtg = new GoodToGo(cdl, doneUpdating);
            try {
                PortAndAssignment pna = entry.getKey();
                BlobChangingCallback cb = entry.getValue();
                cb.blobChanging(pna.getAssignment(), pna.getPort(), this, gtg);
            }
            finally {
                gtg.countDownIfLatchWasNotGotten();
            }
        }
        try {
            cdl.await(3L, TimeUnit.MINUTES);
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
        return doneUpdating;
    }

    public String getKey() {
        return this.blobKey;
    }

    public Collection<PortAndAssignment> getDependencies() {
        return this.references.keySet();
    }

    public abstract boolean isFullyDownloaded();

    boolean requiresUpdate(ClientBlobStore blobStore, long remoteBlobstoreUpdateTime) throws KeyNotFoundException, AuthorizationException {
        long remoteVersion;
        if (!this.isUsed()) {
            return false;
        }
        if (!this.isFullyDownloaded()) {
            return true;
        }
        if (remoteBlobstoreUpdateTime > 0L && this.localUpdateTime == remoteBlobstoreUpdateTime) {
            LOG.debug("{} is up to date, blob localUpdateTime matches remote timestamp {}", (Object)this, (Object)remoteBlobstoreUpdateTime);
            return false;
        }
        long localVersion = this.getLocalVersion();
        if (localVersion != (remoteVersion = this.getRemoteVersion(blobStore))) {
            return true;
        }
        this.localUpdateTime = remoteBlobstoreUpdateTime;
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void download(ClientBlobStore blobStore, long remoteBlobstoreUpdateTime) throws AuthorizationException, IOException, KeyNotFoundException {
        if (this.isFullyDownloaded()) {
            this.numBlobUpdateVersionChanged.mark();
        }
        Timer.Context timer = this.singleBlobLocalizationDuration.time();
        try {
            long newVersion = this.fetchUnzipToTemp(blobStore);
            this.informReferencesAndCommitNewVersion(newVersion);
            this.localUpdateTime = remoteBlobstoreUpdateTime;
            LOG.debug("local blob {} downloaded, in sync with remote blobstore to time {}", (Object)this, (Object)remoteBlobstoreUpdateTime);
        }
        finally {
            timer.stop();
            this.cleanupOrphanedData();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void update(ClientBlobStore blobStore, long remoteBlobstoreUpdateTime) throws KeyNotFoundException, AuthorizationException, IOException {
        LocallyCachedBlob locallyCachedBlob = this;
        synchronized (locallyCachedBlob) {
            if (this.requiresUpdate(blobStore, remoteBlobstoreUpdateTime)) {
                this.download(blobStore, remoteBlobstoreUpdateTime);
            }
        }
    }

    static class DownloadMeta {
        private final Path downloadPath;
        private final long version;

        DownloadMeta(Path downloadPath, long version) {
            this.downloadPath = downloadPath;
            this.version = version;
        }

        public Path getDownloadPath() {
            return this.downloadPath;
        }

        public long getVersion() {
            return this.version;
        }
    }
}

