/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.vfs2.provider.ram;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import org.apache.commons.vfs2.AllFileSelector;
import org.apache.commons.vfs2.FileContent;
import org.apache.commons.vfs2.FileObject;
import org.apache.commons.vfs2.FileSelector;
import org.apache.commons.vfs2.FileSystemException;
import org.apache.commons.vfs2.FileSystemOptions;
import org.apache.commons.vfs2.FileType;
import org.apache.commons.vfs2.impl.DefaultFileSystemManager;
import org.apache.commons.vfs2.provider.FileProvider;
import org.apache.commons.vfs2.provider.UriParser;
import org.apache.commons.vfs2.provider.ram.RamFileProvider;
import org.apache.commons.vfs2.provider.ram.RamFileSystemConfigBuilder;
import org.junit.Assert;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

public class CustomRamProviderTest {
    private static final byte[] NON_EMPTY_FILE_CONTENT = new byte[]{1, 2, 3};
    final char[] ENC = new char[]{'!', '?'};
    private final List<Closeable> closeables = new ArrayList<Closeable>();
    FileSystemOptions defaultRamFso = new FileSystemOptions();
    DefaultFileSystemManager manager;
    FileSystemOptions smallSizedFso = new FileSystemOptions();
    FileSystemOptions zeroSizedFso = new FileSystemOptions();

    private <C extends Closeable> C closeOnTearDown(C closeable) {
        this.closeables.add(closeable);
        return closeable;
    }

    private InputStream createEmptyFile() throws FileSystemException, IOException {
        FileObject root = this.manager.resolveFile("ram://file");
        root.createFile();
        return this.closeOnTearDown(root.getContent().getInputStream());
    }

    private InputStream createNonEmptyFile() throws FileSystemException, IOException {
        FileObject root = this.manager.resolveFile("ram://file");
        root.createFile();
        FileContent content = root.getContent();
        OutputStream output = this.closeOnTearDown(content.getOutputStream());
        output.write(1);
        output.write(2);
        output.write(3);
        output.flush();
        output.close();
        return this.closeOnTearDown(content.getInputStream());
    }

    private FileObject prepareSpecialFile(String dirName, String testFileName) throws FileSystemException {
        FileObject dir = this.manager.resolveFile("ram:" + dirName);
        dir.createFolder();
        String filePath = dir.getName().getPath() + "/" + UriParser.encode((String)testFileName, (char[])this.ENC);
        FileObject specialFile = dir.resolveFile(filePath);
        specialFile.createFile();
        return dir;
    }

    @BeforeEach
    public void setUp() throws Exception {
        this.manager = new DefaultFileSystemManager();
        this.manager.addProvider("ram", (FileProvider)new RamFileProvider());
        this.manager.init();
        RamFileSystemConfigBuilder.getInstance().setMaxSize(this.zeroSizedFso, 0L);
        RamFileSystemConfigBuilder.getInstance().setMaxSize(this.smallSizedFso, 10L);
    }

    @AfterEach
    public void tearDown() {
        for (Closeable closeable : this.closeables) {
            try {
                closeable.close();
            }
            catch (Exception exception) {}
        }
        this.manager.close();
    }

    @Test
    public void testFSOptions() throws Exception {
        FileObject fo1 = this.manager.resolveFile("ram:/");
        FileObject fo2 = this.manager.resolveFile("ram:/");
        Assert.assertSame((String)"Both files should exist in the same fs instance.", (Object)fo1.getFileSystem(), (Object)fo2.getFileSystem());
        FileSystemOptions fsOptions = fo1.getFileSystem().getFileSystemOptions();
        long maxFilesystemSize = RamFileSystemConfigBuilder.getInstance().getLongMaxSize(fsOptions);
        Assertions.assertEquals((long)Long.MAX_VALUE, (long)maxFilesystemSize, (String)"Filesystem option maxSize must be unlimited");
        FileObject fo3 = this.manager.resolveFile("ram:/fo3", this.smallSizedFso);
        FileObject fo4 = this.manager.resolveFile("ram:/", this.smallSizedFso);
        Assert.assertSame((String)"Both files should exist in the same FileSystem instance.", (Object)fo3.getFileSystem(), (Object)fo4.getFileSystem());
        Assert.assertNotSame((String)"Both files should exist in different FileSystem instance.", (Object)fo1.getFileSystem(), (Object)fo3.getFileSystem());
        fsOptions = fo3.getFileSystem().getFileSystemOptions();
        maxFilesystemSize = RamFileSystemConfigBuilder.getInstance().getLongMaxSize(fsOptions);
        Assertions.assertEquals((long)10L, (long)maxFilesystemSize, (String)"Filesystem option maxSize must be set");
    }

    @Test
    public void testMoveFile() throws FileSystemException {
        FileObject fileSource = this.manager.resolveFile("ram://virtual/source");
        fileSource.createFile();
        FileObject fileDest = this.manager.resolveFile("ram://virtual/dest");
        Assertions.assertTrue((boolean)fileSource.canRenameTo(fileDest));
        fileSource.moveTo(fileDest);
    }

    @Test
    public void testReadEmptyFileByteByByte() throws FileSystemException, IOException {
        InputStream input = this.createEmptyFile();
        Assertions.assertEquals((int)-1, (int)input.read(), (String)"Empty file didn't return EOF -1");
    }

    @Test
    public void testReadEmptyFileIntoBuffer() throws FileSystemException, IOException {
        InputStream input = this.createEmptyFile();
        byte[] buffer = new byte[100];
        Assertions.assertEquals((int)-1, (int)input.read(buffer), (String)"Empty file didn't return when filling buffer");
        Assertions.assertArrayEquals((byte[])new byte[100], (byte[])buffer, (String)"Buffer was written too");
    }

    @Test
    public void testReadEmptyFileIntoBufferWithOffsetAndLength() throws FileSystemException, IOException {
        InputStream input = this.createEmptyFile();
        byte[] buffer = new byte[100];
        Assertions.assertEquals((int)-1, (int)input.read(buffer, 10, 90), (String)"Empty file didn't return when filling buffer");
        Assertions.assertArrayEquals((byte[])new byte[100], (byte[])buffer, (String)"Buffer was written too");
    }

    @Test
    public void testReadNonEmptyFileByteByByte() throws FileSystemException, IOException {
        InputStream input = this.createNonEmptyFile();
        Assertions.assertEquals((int)1, (int)input.read(), (String)"Read 1st byte failed");
        Assertions.assertEquals((int)2, (int)input.read(), (String)"Read 2st byte failed");
        Assertions.assertEquals((int)3, (int)input.read(), (String)"Read 3st byte failed");
        Assertions.assertEquals((int)-1, (int)input.read(), (String)"File should be empty");
    }

    @Test
    public void testReadNonEmptyFileIntoBuffer() throws FileSystemException, IOException {
        InputStream input = this.createNonEmptyFile();
        byte[] buffer = new byte[100];
        Assertions.assertEquals((int)NON_EMPTY_FILE_CONTENT.length, (int)input.read(buffer), (String)"Filling buffer failed when file is not empty");
        byte[] expectedBuffer = Arrays.copyOf(NON_EMPTY_FILE_CONTENT, 100);
        Assertions.assertArrayEquals((byte[])expectedBuffer, (byte[])buffer, (String)"Buffer not filled");
        Arrays.fill(buffer, (byte)0);
        Arrays.fill(expectedBuffer, (byte)0);
        Assertions.assertEquals((int)-1, (int)input.read(buffer), (String)"File should be empty after filling buffer");
        Assertions.assertArrayEquals((byte[])expectedBuffer, (byte[])buffer, (String)"Buffer was written when empty");
    }

    @Test
    public void testReadNonEmptyFileIntoBufferWithOffsetAndLength() throws FileSystemException, IOException {
        InputStream input = this.createNonEmptyFile();
        byte[] buffer = new byte[100];
        int offset = 10;
        Assertions.assertEquals((int)NON_EMPTY_FILE_CONTENT.length, (int)input.read(buffer, 10, 90), (String)"Filling buffer failed when file is not empty");
        byte[] expectedBuffer = new byte[100];
        System.arraycopy(NON_EMPTY_FILE_CONTENT, 0, expectedBuffer, 10, NON_EMPTY_FILE_CONTENT.length);
        Assertions.assertArrayEquals((byte[])expectedBuffer, (byte[])buffer, (String)"Buffer not filled");
        Arrays.fill(buffer, (byte)0);
        Arrays.fill(expectedBuffer, (byte)0);
        Assertions.assertEquals((int)-1, (int)input.read(buffer, 10, 90), (String)"File should be empty after filling buffer");
        Assertions.assertArrayEquals((byte[])expectedBuffer, (byte[])buffer, (String)"Buffer was written when empty");
    }

    @Test
    public void testRootFolderExists() throws FileSystemException {
        FileObject root = this.manager.resolveFile("ram:///", this.defaultRamFso);
        Assertions.assertTrue((boolean)root.getType().hasChildren());
        try {
            root.delete();
            Assertions.fail();
        }
        catch (FileSystemException fileSystemException) {
            // empty catch block
        }
    }

    @Test
    public void testSchemePrefix() throws FileSystemException {
        String KNOWN_SCHEME = this.manager.getSchemes()[0];
        String testDir = "/prefixtest/";
        String testFileName = KNOWN_SCHEME + ":test:txt";
        String expectedName = "/prefixtest/" + testFileName;
        FileObject dir = this.prepareSpecialFile("/prefixtest/", testFileName);
        FileObject[] findFilesResult = dir.findFiles((FileSelector)new AllFileSelector());
        FileObject[] getChildrenResult = dir.getChildren();
        FileObject getChildResult = dir.getChild(testFileName);
        Assertions.assertEquals((int)2, (int)findFilesResult.length, () -> "Unexpected result findFiles: " + Arrays.toString(findFilesResult));
        String resultName = findFilesResult[0].getName().getPathDecoded();
        Assertions.assertEquals((Object)expectedName, (Object)resultName, (String)"findFiles Child name does not match");
        Assertions.assertEquals((Object)FileType.FILE, (Object)findFilesResult[0].getType(), (String)"Did findFiles but child was no file");
        Assertions.assertEquals((int)1, (int)getChildrenResult.length, () -> "Unexpected result getChildren: " + Arrays.toString(getChildrenResult));
        resultName = getChildrenResult[0].getName().getPathDecoded();
        Assertions.assertEquals((Object)expectedName, (Object)resultName, (String)"getChildren Child name does not match");
        Assertions.assertEquals((Object)FileType.FILE, (Object)getChildrenResult[0].getType(), (String)"Did getChildren but child was no file");
        Assertions.assertNotNull((Object)getChildResult, (String)"Did not find direct child");
        resultName = getChildResult.getName().getPathDecoded();
        Assertions.assertEquals((Object)expectedName, (Object)resultName, (String)"getChild name does not match");
        Assertions.assertEquals((Object)FileType.FILE, (Object)getChildResult.getType(), (String)"getChild was no file");
    }

    @Test
    public void testSmallFS() throws Exception {
        OutputStream os;
        FileObject fo3 = this.manager.resolveFile("ram:/fo3", this.smallSizedFso);
        fo3.createFile();
        try {
            os = fo3.getContent().getOutputStream();
            os.write(new byte[10]);
            os.close();
        }
        catch (FileSystemException e) {
            Assertions.fail((String)"Test should be able to save such a small file");
        }
        try {
            os = fo3.getContent().getOutputStream();
            os.write(new byte[11]);
            os.close();
            Assertions.fail((String)"It shouldn't save such a big file");
        }
        catch (FileSystemException fileSystemException) {
            // empty catch block
        }
    }

    @Test
    public void testSpecialName() throws FileSystemException {
        String testDir = "/specialtest/";
        String testFileName = "test:+-_ \"()<>%#.txt";
        String expectedName = "/specialtest/test:+-_ \"()<>%#.txt";
        FileObject dir = this.prepareSpecialFile("/specialtest/", "test:+-_ \"()<>%#.txt");
        FileObject[] findFilesResult = dir.findFiles((FileSelector)new AllFileSelector());
        FileObject[] getChildrenResult = dir.getChildren();
        FileObject getChildResult = dir.getChild(UriParser.encode((String)"test:+-_ \"()<>%#.txt", (char[])this.ENC));
        Assertions.assertEquals((int)2, (int)findFilesResult.length, () -> "Unexpected result findFiles: " + Arrays.toString(findFilesResult));
        String resultName = findFilesResult[0].getName().getPathDecoded();
        Assertions.assertEquals((Object)"/specialtest/test:+-_ \"()<>%#.txt", (Object)resultName, (String)"findFiles Child name does not match");
        Assertions.assertEquals((Object)FileType.FILE, (Object)findFilesResult[0].getType(), (String)"Did findFiles but child was no file");
        Assertions.assertEquals((int)1, (int)getChildrenResult.length, () -> "Unexpected result getChildren: " + Arrays.toString(getChildrenResult));
        resultName = getChildrenResult[0].getName().getPathDecoded();
        Assertions.assertEquals((Object)"/specialtest/test:+-_ \"()<>%#.txt", (Object)resultName, (String)"getChildren Child name does not match");
        Assertions.assertEquals((Object)FileType.FILE, (Object)getChildrenResult[0].getType(), (String)"Did getChildren but child was no file");
        Assertions.assertNotNull((Object)getChildResult, (String)"Did not find direct child");
        resultName = getChildResult.getName().getPathDecoded();
        Assertions.assertEquals((Object)"/specialtest/test:+-_ \"()<>%#.txt", (Object)resultName, (String)"getChild name does not match");
        Assertions.assertEquals((Object)FileType.FILE, (Object)getChildResult.getType(), (String)"getChild was no file");
    }
}

