/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.repositories.azure;

import com.azure.core.http.HttpMethod;
import com.azure.core.http.HttpRequest;
import com.azure.core.http.HttpResponse;
import com.azure.core.http.rest.Response;
import com.azure.core.util.Context;
import com.azure.storage.blob.BlobClient;
import com.azure.storage.blob.BlobContainerClient;
import com.azure.storage.blob.BlobServiceClient;
import com.azure.storage.blob.models.BlobErrorCode;
import com.azure.storage.blob.models.BlobItem;
import com.azure.storage.blob.models.BlobItemProperties;
import com.azure.storage.blob.models.BlobListDetails;
import com.azure.storage.blob.models.BlobRange;
import com.azure.storage.blob.models.BlobRequestConditions;
import com.azure.storage.blob.models.BlobStorageException;
import com.azure.storage.blob.models.ListBlobsOptions;
import com.azure.storage.blob.options.BlobParallelUploadOptions;
import java.io.IOException;
import java.io.InputStream;
import java.net.URISyntaxException;
import java.nio.file.FileAlreadyExistsException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.BiConsumer;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.core.util.Throwables;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.opensearch.action.support.PlainActionFuture;
import org.opensearch.cluster.metadata.RepositoryMetadata;
import org.opensearch.common.Nullable;
import org.opensearch.common.blobstore.BlobContainer;
import org.opensearch.common.blobstore.BlobMetadata;
import org.opensearch.common.blobstore.BlobPath;
import org.opensearch.common.blobstore.BlobStore;
import org.opensearch.common.blobstore.DeleteResult;
import org.opensearch.common.blobstore.support.PlainBlobMetadata;
import org.opensearch.common.collect.MapBuilder;
import org.opensearch.common.collect.Tuple;
import org.opensearch.common.util.concurrent.AbstractRunnable;
import org.opensearch.repositories.azure.AzureBlobContainer;
import org.opensearch.repositories.azure.AzureRepository;
import org.opensearch.repositories.azure.AzureStorageService;
import org.opensearch.repositories.azure.AzureStorageSettings;
import org.opensearch.repositories.azure.LocationMode;
import org.opensearch.repositories.azure.SocketAccess;
import org.opensearch.threadpool.ThreadPool;

public class AzureBlobStore
implements BlobStore {
    private static final Logger logger = LogManager.getLogger(AzureBlobStore.class);
    private final AzureStorageService service;
    private final ThreadPool threadPool;
    private final String clientName;
    private final String container;
    private final LocationMode locationMode;
    private final Stats stats = new Stats();
    private final BiConsumer<HttpRequest, HttpResponse> metricsCollector;

    public AzureBlobStore(RepositoryMetadata metadata, AzureStorageService service, ThreadPool threadPool) {
        this.container = (String)AzureRepository.Repository.CONTAINER_SETTING.get(metadata.settings());
        this.clientName = (String)AzureRepository.Repository.CLIENT_NAME.get(metadata.settings());
        this.service = service;
        this.threadPool = threadPool;
        this.locationMode = (LocationMode)((Object)AzureRepository.Repository.LOCATION_MODE_SETTING.get(metadata.settings()));
        Map<String, AzureStorageSettings> prevSettings = this.service.refreshAndClearCache(Collections.emptyMap());
        Map<String, AzureStorageSettings> newSettings = AzureStorageSettings.overrideLocationMode(prevSettings, this.locationMode);
        this.service.refreshAndClearCache(newSettings);
        this.metricsCollector = (request, response) -> {
            if (response.getStatusCode() >= 300) {
                return;
            }
            HttpMethod method = request.getHttpMethod();
            if (method.equals((Object)HttpMethod.HEAD)) {
                this.stats.headOperations.incrementAndGet();
                return;
            }
            if (method.equals((Object)HttpMethod.GET)) {
                String queryParams;
                String query = request.getUrl().getQuery();
                String string = queryParams = query == null ? "" : query;
                if (queryParams.contains("comp=list")) {
                    this.stats.listOperations.incrementAndGet();
                } else {
                    this.stats.getOperations.incrementAndGet();
                }
            } else if (method.equals((Object)HttpMethod.PUT)) {
                String queryParams;
                String query = request.getUrl().getQuery();
                String string = queryParams = query == null ? "" : query;
                if (queryParams.contains("comp=block") && queryParams.contains("blockid=")) {
                    this.stats.putBlockOperations.incrementAndGet();
                } else if (queryParams.contains("comp=blocklist")) {
                    this.stats.putBlockListOperations.incrementAndGet();
                } else {
                    this.stats.putOperations.incrementAndGet();
                }
            }
        };
    }

    public String toString() {
        return this.container;
    }

    public AzureStorageService getService() {
        return this.service;
    }

    public LocationMode getLocationMode() {
        return this.locationMode;
    }

    public BlobContainer blobContainer(BlobPath path) {
        return new AzureBlobContainer(path, this, this.threadPool);
    }

    public void close() throws IOException {
        this.service.close();
    }

    public boolean blobExists(String blob) throws URISyntaxException, BlobStorageException {
        Tuple<BlobServiceClient, Supplier<Context>> client = this.client();
        BlobContainerClient blobContainer = ((BlobServiceClient)client.v1()).getBlobContainerClient(this.container);
        return SocketAccess.doPrivilegedException(() -> {
            BlobClient azureBlob = blobContainer.getBlobClient(blob);
            Response response = azureBlob.existsWithResponse(this.timeout(), (Context)((Supplier)client.v2()).get());
            return (Boolean)response.getValue();
        });
    }

    public void deleteBlob(String blob) throws URISyntaxException, BlobStorageException {
        Tuple<BlobServiceClient, Supplier<Context>> client = this.client();
        BlobContainerClient blobContainer = ((BlobServiceClient)client.v1()).getBlobContainerClient(this.container);
        logger.trace(() -> new ParameterizedMessage("delete blob for container [{}], blob [{}]", (Object)this.container, (Object)blob));
        SocketAccess.doPrivilegedVoidException(() -> {
            BlobClient azureBlob = blobContainer.getBlobClient(blob);
            logger.trace(() -> new ParameterizedMessage("container [{}]: blob [{}] found. removing.", (Object)this.container, (Object)blob));
            Response response = azureBlob.deleteWithResponse(null, null, this.timeout(), (Context)((Supplier)client.v2()).get());
            logger.trace(() -> new ParameterizedMessage("container [{}]: blob [{}] deleted status [{}].", new Object[]{this.container, blob, response.getStatusCode()}));
        });
    }

    public DeleteResult deleteBlobDirectory(String path, Executor executor) throws URISyntaxException, BlobStorageException, IOException {
        final Tuple<BlobServiceClient, Supplier<Context>> client = this.client();
        final BlobContainerClient blobContainer = ((BlobServiceClient)client.v1()).getBlobContainerClient(this.container);
        final List exceptions = Collections.synchronizedList(new ArrayList());
        final AtomicLong outstanding = new AtomicLong(1L);
        final PlainActionFuture result = PlainActionFuture.newFuture();
        final AtomicLong blobsDeleted = new AtomicLong();
        final AtomicLong bytesDeleted = new AtomicLong();
        ListBlobsOptions listBlobsOptions = new ListBlobsOptions().setPrefix(path);
        SocketAccess.doPrivilegedVoidException(() -> {
            for (final BlobItem blobItem : blobContainer.listBlobs(listBlobsOptions, this.timeout())) {
                assert (blobItem.isPrefix() == null || !blobItem.isPrefix().booleanValue()) : "Only blobs (not prefixes) are expected";
                outstanding.incrementAndGet();
                executor.execute((Runnable)new AbstractRunnable(){

                    protected void doRun() throws Exception {
                        long len = blobItem.getProperties().getContentLength();
                        BlobClient azureBlob = blobContainer.getBlobClient(blobItem.getName());
                        logger.trace(() -> new ParameterizedMessage("container [{}]: blob [{}] found. removing.", (Object)AzureBlobStore.this.container, (Object)blobItem.getName()));
                        Response response = azureBlob.deleteWithResponse(null, null, AzureBlobStore.this.timeout(), (Context)((Supplier)client.v2()).get());
                        logger.trace(() -> new ParameterizedMessage("container [{}]: blob [{}] deleted status [{}].", new Object[]{AzureBlobStore.this.container, blobItem.getName(), response.getStatusCode()}));
                        blobsDeleted.incrementAndGet();
                        if (len >= 0L) {
                            bytesDeleted.addAndGet(len);
                        }
                    }

                    public void onFailure(Exception e) {
                        exceptions.add(e);
                    }

                    public void onAfter() {
                        if (outstanding.decrementAndGet() == 0L) {
                            result.onResponse(null);
                        }
                    }
                });
            }
        });
        if (outstanding.decrementAndGet() == 0L) {
            result.onResponse(null);
        }
        result.actionGet();
        if (!exceptions.isEmpty()) {
            IOException ex = new IOException("Deleting directory [" + path + "] failed");
            exceptions.forEach(ex::addSuppressed);
            throw ex;
        }
        return new DeleteResult(blobsDeleted.get(), bytesDeleted.get());
    }

    public InputStream getInputStream(String blob, long position, @Nullable Long length) throws URISyntaxException, BlobStorageException {
        Tuple<BlobServiceClient, Supplier<Context>> client = this.client();
        BlobContainerClient blobContainer = ((BlobServiceClient)client.v1()).getBlobContainerClient(this.container);
        BlobClient azureBlob = blobContainer.getBlobClient(blob);
        logger.trace(() -> new ParameterizedMessage("reading container [{}], blob [{}]", (Object)this.container, (Object)blob));
        return (InputStream)SocketAccess.doPrivilegedException(() -> {
            if (length == null) {
                return azureBlob.openInputStream(new BlobRange(position), null);
            }
            return azureBlob.openInputStream(new BlobRange(position, length), null);
        });
    }

    public Map<String, BlobMetadata> listBlobsByPrefix(String keyPath, String prefix) throws URISyntaxException, BlobStorageException {
        HashMap blobsBuilder = new HashMap();
        Tuple<BlobServiceClient, Supplier<Context>> client = this.client();
        BlobContainerClient blobContainer = ((BlobServiceClient)client.v1()).getBlobContainerClient(this.container);
        logger.trace(() -> new ParameterizedMessage("listing container [{}], keyPath [{}], prefix [{}]", new Object[]{this.container, keyPath, prefix}));
        ListBlobsOptions listBlobsOptions = new ListBlobsOptions().setDetails(new BlobListDetails().setRetrieveMetadata(true)).setPrefix(keyPath + (prefix == null ? "" : prefix));
        SocketAccess.doPrivilegedVoidException(() -> {
            for (BlobItem blobItem : blobContainer.listBlobsByHierarchy("/", listBlobsOptions, this.timeout())) {
                if (blobItem.isPrefix() != null && blobItem.isPrefix().booleanValue()) continue;
                String name = this.getBlobName(blobItem.getName(), this.container, keyPath);
                logger.trace(() -> new ParameterizedMessage("blob name [{}]", (Object)name));
                BlobItemProperties properties = blobItem.getProperties();
                logger.trace(() -> new ParameterizedMessage("blob name [{}], size [{}]", (Object)name, (Object)properties.getContentLength()));
                blobsBuilder.put(name, new PlainBlobMetadata(name, properties.getContentLength().longValue()));
            }
        });
        return MapBuilder.newMapBuilder(blobsBuilder).immutableMap();
    }

    public Map<String, BlobContainer> children(BlobPath path) throws URISyntaxException, BlobStorageException {
        HashSet blobsBuilder = new HashSet();
        Tuple<BlobServiceClient, Supplier<Context>> client = this.client();
        BlobContainerClient blobContainer = ((BlobServiceClient)client.v1()).getBlobContainerClient(this.container);
        String keyPath = path.buildAsString();
        ListBlobsOptions listBlobsOptions = new ListBlobsOptions().setDetails(new BlobListDetails().setRetrieveMetadata(true)).setPrefix(keyPath);
        SocketAccess.doPrivilegedVoidException(() -> {
            for (BlobItem blobItem : blobContainer.listBlobsByHierarchy("/", listBlobsOptions, this.timeout())) {
                if (blobItem.isPrefix() == null || !blobItem.isPrefix().booleanValue()) continue;
                String name = this.getBlobName(blobItem.getName(), this.container, keyPath).replaceAll("/$", "");
                logger.trace(() -> new ParameterizedMessage("blob name [{}]", (Object)name));
                blobsBuilder.add(name);
            }
        });
        return Collections.unmodifiableMap(blobsBuilder.stream().collect(Collectors.toMap(Function.identity(), name -> new AzureBlobContainer(path.add(name), this, this.threadPool))));
    }

    public void writeBlob(String blobName, InputStream inputStream, long blobSize, boolean failIfAlreadyExists) throws URISyntaxException, BlobStorageException, IOException {
        assert (inputStream.markSupported()) : "Should not be used with non-mark supporting streams as their retry handling in the SDK is broken";
        logger.trace(() -> new ParameterizedMessage("writeBlob({}, stream, {})", (Object)blobName, (Object)blobSize));
        Tuple<BlobServiceClient, Supplier<Context>> client = this.client();
        BlobContainerClient blobContainer = ((BlobServiceClient)client.v1()).getBlobContainerClient(this.container);
        BlobClient blob = blobContainer.getBlobClient(blobName);
        try {
            BlobRequestConditions blobRequestConditions = new BlobRequestConditions();
            if (failIfAlreadyExists) {
                blobRequestConditions.setIfNoneMatch("*");
            }
            SocketAccess.doPrivilegedVoidException(() -> {
                Response response = blob.uploadWithResponse(new BlobParallelUploadOptions(inputStream, blobSize).setRequestConditions(blobRequestConditions).setParallelTransferOptions(this.service.getBlobRequestOptionsForWriteBlob()), this.timeout(), (Context)((Supplier)client.v2()).get());
                logger.trace(() -> new ParameterizedMessage("upload({}, stream, {}) - status [{}]", new Object[]{blobName, blobSize, response.getStatusCode()}));
            });
        }
        catch (BlobStorageException se) {
            if (failIfAlreadyExists && se.getStatusCode() == 409 && BlobErrorCode.BLOB_ALREADY_EXISTS.equals((Object)se.getErrorCode())) {
                throw new FileAlreadyExistsException(blobName, null, se.getMessage());
            }
            throw se;
        }
        catch (RuntimeException ex) {
            if (ex.getCause() != null) {
                Throwables.rethrow((Throwable)ex.getCause());
            }
            throw ex;
        }
        logger.trace(() -> new ParameterizedMessage("writeBlob({}, stream, {}) - done", (Object)blobName, (Object)blobSize));
    }

    private Tuple<BlobServiceClient, Supplier<Context>> client() {
        return this.service.client(this.clientName, this.metricsCollector);
    }

    private Duration timeout() {
        return this.service.getBlobRequestTimeout(this.clientName);
    }

    public Map<String, Long> stats() {
        return this.stats.toMap();
    }

    private String getBlobName(String pathOrName, String container, String keyPath) {
        String name = pathOrName;
        if (name.matches("." + container + ".")) {
            name = name.substring(1 + container.length() + 1);
        }
        if (name.startsWith(keyPath)) {
            name = name.substring(keyPath.length());
        }
        return name;
    }

    private static class Stats {
        private final AtomicLong getOperations = new AtomicLong();
        private final AtomicLong listOperations = new AtomicLong();
        private final AtomicLong headOperations = new AtomicLong();
        private final AtomicLong putOperations = new AtomicLong();
        private final AtomicLong putBlockOperations = new AtomicLong();
        private final AtomicLong putBlockListOperations = new AtomicLong();

        private Stats() {
        }

        private Map<String, Long> toMap() {
            return Map.of("GetBlob", this.getOperations.get(), "ListBlobs", this.listOperations.get(), "GetBlobProperties", this.headOperations.get(), "PutBlob", this.putOperations.get(), "PutBlock", this.putBlockOperations.get(), "PutBlockList", this.putBlockListOperations.get());
        }
    }
}

