/*
 * Copyright (C) 2021 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1.  Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 * 2.  Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "WebPageProxy.h"

#include "ArgumentCoders.h"
#include "CallbackID.h"
#if ENABLE(CONTEXT_MENUS)
#include "ContextMenuContextData.h"
#endif
#if ENABLE(DATA_DETECTION)
#include "DataDetectionResult.h"
#endif
#include "DataReference.h"
#include "Decoder.h"
#include "EditorState.h"
#if PLATFORM(IOS_FAMILY)
#include "FocusedElementInformation.h"
#endif
#include "FormDataReference.h"
#include "FrameInfoData.h"
#include "GeolocationIdentifier.h"
#include "HandleMessage.h"
#include "IdentifierTypes.h"
#if PLATFORM(GTK) || PLATFORM(WPE)
#include "InputMethodState.h"
#endif
#if PLATFORM(IOS_FAMILY)
#include "InteractionInformationAtPosition.h"
#endif
#if HAVE(VISIBILITY_PROPAGATION_VIEW)
#include "LayerHostingContext.h"
#endif
#if ENABLE(ARKIT_INLINE_PREVIEW) || ENABLE(ARKIT_INLINE_PREVIEW_IOS)
#include "ModelIdentifier.h"
#endif
#include "NavigationActionData.h"
#if ENABLE(PDFKIT_PLUGIN)
#include "PDFContextMenu.h"
#endif
#if ENABLE(PDFKIT_PLUGIN) || ENABLE(UI_PROCESS_PDF_HUD)
#include "PDFPluginIdentifier.h"
#endif
#include "PlatformPopupMenuData.h"
#include "PolicyDecision.h"
#include "SessionState.h"
#include "ShareableBitmap.h"
#if USE(QUICK_LOOK)
#include "ShareableResource.h"
#endif
#if ENABLE(ATTACHMENT_ELEMENT)
#include "SharedBufferCopy.h"
#endif
#if PLATFORM(COCOA) && ENABLE(DRAG_SUPPORT) || PLATFORM(IOS_FAMILY)
#include "SharedMemory.h"
#endif
#if HAVE(TOUCH_BAR)
#include "TouchBarMenuData.h"
#endif
#if HAVE(TOUCH_BAR)
#include "TouchBarMenuItemData.h"
#endif
#include "URLSchemeTaskParameters.h"
#include "UndoOrRedo.h"
#include "UserData.h"
#if PLATFORM(GTK) || PLATFORM(WPE)
#include "UserMessage.h"
#endif
#if PLATFORM(IOS_FAMILY)
#include "WebAutocorrectionContext.h"
#endif
#include "WebBackForwardListCounts.h"
#include "WebCoreArgumentCoders.h"
#include "WebHitTestResultData.h"
#include "WebNavigationDataStore.h"
#include "WebPageCreationParameters.h"
#include "WebPageProxyIdentifier.h"
#include "WebPageProxyMessages.h"
#include "WebPopupItem.h"
#if ENABLE(SPEECH_SYNTHESIS)
#include "WebSpeechSynthesisVoice.h"
#endif
#include "WebURLSchemeHandlerIdentifier.h"
#include <JavaScriptCore/InspectorTarget.h>
#if PLATFORM(IOS_FAMILY)
#include <WebCore/ActivityState.h>
#endif
#if ENABLE(APP_HIGHLIGHTS)
#include <WebCore/AppHighlight.h>
#endif
#if ENABLE(APPLE_PAY_AMS_UI)
#include <WebCore/ApplePayAMSUIRequest.h>
#endif
#if PLATFORM(IOS_FAMILY)
#include <WebCore/AudioSession.h>
#endif
#include <WebCore/AutoplayEvent.h>
#include <WebCore/BackForwardItemIdentifier.h>
#if ENABLE(MEDIA_STREAM)
#include <WebCore/CaptureDevice.h>
#endif
#include <WebCore/CertificateInfo.h>
#include <WebCore/ClientOrigin.h>
#include <WebCore/Color.h>
#include <WebCore/ContactInfo.h>
#include <WebCore/ContactsRequestData.h>
#if ENABLE(CONTENT_FILTERING)
#include <WebCore/ContentFilterUnblockHandler.h>
#endif
#if ENABLE(CONTENT_EXTENSIONS)
#include <WebCore/ContentRuleListResults.h>
#endif
#include <WebCore/CookieConsentDecisionResult.h>
#include <WebCore/Cursor.h>
#include <WebCore/DOMPasteAccess.h>
#if ENABLE(DATA_DETECTION)
#include <WebCore/DataDetectorElementInfo.h>
#endif
#if ENABLE(DATALIST_ELEMENT)
#include <WebCore/DataListSuggestionInformation.h>
#endif
#if ENABLE(DATE_AND_TIME_INPUT_TYPES)
#include <WebCore/DateTimeChooserParameters.h>
#endif
#if ENABLE(DEVICE_ORIENTATION)
#include <WebCore/DeviceOrientationOrMotionPermissionState.h>
#endif
#include <WebCore/DiagnosticLoggingClient.h>
#include <WebCore/DiagnosticLoggingDomain.h>
#if USE(DICTATION_ALTERNATIVES)
#include <WebCore/DictationContext.h>
#endif
#if PLATFORM(COCOA)
#include <WebCore/DictionaryPopupInfo.h>
#endif
#include <WebCore/DocumentLoader.h>
#if ENABLE(DRAG_SUPPORT) || PLATFORM(GTK) && ENABLE(DRAG_SUPPORT)
#include <WebCore/DragActions.h>
#endif
#if PLATFORM(COCOA) && ENABLE(DRAG_SUPPORT)
#include <WebCore/DragItem.h>
#endif
#include <WebCore/FileChooser.h>
#include <WebCore/FloatPoint.h>
#if PLATFORM(IOS_FAMILY)
#include <WebCore/FloatQuad.h>
#endif
#include <WebCore/FloatRect.h>
#include <WebCore/FloatSize.h>
#include <WebCore/FrameIdentifier.h>
#include <WebCore/FrameLoaderTypes.h>
#include <WebCore/GlobalWindowIdentifier.h>
#if ENABLE(ARKIT_INLINE_PREVIEW)
#include <WebCore/HTMLModelElementCamera.h>
#endif
#if PLATFORM(IOS_FAMILY)
#include <WebCore/InputMode.h>
#endif
#if PLATFORM(IOS_FAMILY)
#include <WebCore/InspectorOverlay.h>
#endif
#include <WebCore/IntPoint.h>
#include <WebCore/IntRect.h>
#include <WebCore/IntSize.h>
#include <WebCore/LayoutMilestone.h>
#if ENABLE(ARKIT_INLINE_PREVIEW_MAC)
#include <WebCore/LayoutPoint.h>
#endif
#include <WebCore/LinkIcon.h>
#if ENABLE(MEDIA_CONTROLS_CONTEXT_MENUS) && USE(UICONTEXTMENU)
#include <WebCore/MediaControlsContextMenuItem.h>
#endif
#if ENABLE(ENCRYPTED_MEDIA)
#include <WebCore/MediaKeySystemRequestIdentifier.h>
#endif
#if ENABLE(WIRELESS_PLAYBACK_TARGET) && !PLATFORM(IOS_FAMILY)
#include <WebCore/MediaPlaybackTargetContext.h>
#endif
#include <WebCore/MediaProducer.h>
#if ENABLE(MEDIA_USAGE)
#include <WebCore/MediaSessionIdentifier.h>
#endif
#if ENABLE(MEDIA_STREAM)
#include <WebCore/MediaStreamRequest.h>
#endif
#if ENABLE(MEDIA_USAGE)
#include <WebCore/MediaUsageInfo.h>
#endif
#if ENABLE(WEB_AUTHN)
#include <WebCore/MockWebAuthenticationConfiguration.h>
#endif
#include <WebCore/ModalContainerTypes.h>
#include <WebCore/PageIdentifier.h>
#include <WebCore/PermissionDescriptor.h>
#include <WebCore/PermissionState.h>
#if ENABLE(WIRELESS_PLAYBACK_TARGET) && !PLATFORM(IOS_FAMILY)
#include <WebCore/PlaybackTargetClientContextIdentifier.h>
#endif
#if ENABLE(ATTACHMENT_ELEMENT)
#include <WebCore/PromisedAttachmentInfo.h>
#endif
#include <WebCore/RectEdges.h>
#include <WebCore/ResourceError.h>
#include <WebCore/ResourceLoaderIdentifier.h>
#include <WebCore/ResourceRequest.h>
#include <WebCore/ResourceResponse.h>
#include <WebCore/SearchPopupMenu.h>
#if ENABLE(ENCRYPTED_MEDIA) || ENABLE(MEDIA_STREAM)
#include <WebCore/SecurityOriginData.h>
#endif
#if PLATFORM(GTK) && ENABLE(DRAG_SUPPORT)
#include <WebCore/SelectionData.h>
#endif
#if ENABLE(ATTACHMENT_ELEMENT)
#include <WebCore/SerializedAttachmentData.h>
#endif
#include <WebCore/ShareData.h>
#include <WebCore/TextCheckerClient.h>
#include <WebCore/TextChecking.h>
#include <WebCore/TextIndicator.h>
#include <WebCore/TextManipulationController.h>
#if ENABLE(IMAGE_ANALYSIS)
#include <WebCore/TextRecognitionResult.h>
#endif
#if HAVE(TRANSLATION_UI_SERVICES) && ENABLE(CONTEXT_MENUS)
#include <WebCore/TranslationContextMenuInfo.h>
#endif
#if ENABLE(MEDIA_STREAM)
#include <WebCore/UserMediaRequestIdentifier.h>
#endif
#include <WebCore/ViewportArguments.h>
#include <WebCore/WindowFeatures.h>
#include <optional>
#include <utility>
#if ENABLE(ARKIT_INLINE_PREVIEW) || ENABLE(ARKIT_INLINE_PREVIEW_MAC)
#include <wtf/Expected.h>
#endif
#if ENABLE(ARKIT_INLINE_PREVIEW_MAC)
#include <wtf/MachSendRight.h>
#endif
#include <wtf/MonotonicTime.h>
#include <wtf/OptionSet.h>
#if ENABLE(ARKIT_INLINE_PREVIEW)
#include <wtf/Seconds.h>
#endif
#include <wtf/URLHash.h>
#include <wtf/Vector.h>
#include <wtf/text/WTFString.h>

namespace Messages {

namespace WebPageProxy {

void CreateNewPage::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const std::optional<WebCore::PageIdentifier>& newPageID, const std::optional<WebKit::WebPageCreationParameters>& newPageParameters)
{
    encoder.get() << newPageID;
    encoder.get() << newPageParameters;
    connection.sendSyncReply(WTFMove(encoder));
}

void RunJavaScriptAlert::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

void RunJavaScriptConfirm::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

void RunJavaScriptPrompt::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const String& result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

#if ENABLE(WEBGL)

void WebGLPolicyForURL::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, WebCore::WebGLLoadPolicy loadPolicy)
{
    encoder.get() << loadPolicy;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(WEBGL)

void ResolveWebGLPolicyForURL::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, WebCore::WebGLLoadPolicy loadPolicy)
{
    encoder.get() << loadPolicy;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

void GetToolbarsAreVisible::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool toolbarsAreVisible)
{
    encoder.get() << toolbarsAreVisible;
    connection.sendSyncReply(WTFMove(encoder));
}

void GetMenuBarIsVisible::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool menuBarIsVisible)
{
    encoder.get() << menuBarIsVisible;
    connection.sendSyncReply(WTFMove(encoder));
}

void GetStatusBarIsVisible::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool statusBarIsVisible)
{
    encoder.get() << statusBarIsVisible;
    connection.sendSyncReply(WTFMove(encoder));
}

void GetWindowFrame::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const WebCore::FloatRect& windowFrame)
{
    encoder.get() << windowFrame;
    connection.sendSyncReply(WTFMove(encoder));
}

void ScreenToRootView::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const WebCore::IntPoint& windowPoint)
{
    encoder.get() << windowPoint;
    connection.sendSyncReply(WTFMove(encoder));
}

void RootViewToScreen::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const WebCore::IntRect& screenFrame)
{
    encoder.get() << screenFrame;
    connection.sendSyncReply(WTFMove(encoder));
}

void AccessibilityScreenToRootView::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const WebCore::IntPoint& windowPoint)
{
    encoder.get() << windowPoint;
    connection.sendSyncReply(WTFMove(encoder));
}

void RootViewToAccessibilityScreen::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const WebCore::IntRect& screenFrame)
{
    encoder.get() << screenFrame;
    connection.sendSyncReply(WTFMove(encoder));
}

void RunBeforeUnloadConfirmPanel::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool shouldClose)
{
    encoder.get() << shouldClose;
    connection.sendSyncReply(WTFMove(encoder));
}

void ShowShareSheet::callReply(IPC::Decoder& decoder, CompletionHandler<void(bool&&)>&& completionHandler)
{
    std::optional<bool> granted;
    decoder >> granted;
    if (!granted) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*granted));
}

void ShowShareSheet::cancelReply(CompletionHandler<void(bool&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<bool>::create());
}

void ShowShareSheet::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool granted)
{
    encoder.get() << granted;
    connection.sendSyncReply(WTFMove(encoder));
}

void ShowContactPicker::callReply(IPC::Decoder& decoder, CompletionHandler<void(std::optional<Vector<WebCore::ContactInfo>>&&)>&& completionHandler)
{
    std::optional<std::optional<Vector<WebCore::ContactInfo>>> info;
    decoder >> info;
    if (!info) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*info));
}

void ShowContactPicker::cancelReply(CompletionHandler<void(std::optional<Vector<WebCore::ContactInfo>>&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<std::optional<Vector<WebCore::ContactInfo>>>::create());
}

void ShowContactPicker::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const std::optional<Vector<WebCore::ContactInfo>>& info)
{
    encoder.get() << info;
    connection.sendSyncReply(WTFMove(encoder));
}

void PrintFrame::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

void DecidePolicyForNavigationActionSync::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const WebKit::PolicyDecision& PolicyDecision)
{
    encoder.get() << PolicyDecision;
    connection.sendSyncReply(WTFMove(encoder));
}

#if PLATFORM(IOS_FAMILY)

void InterpretKeyEvent::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool handled)
{
    encoder.get() << handled;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

void BackForwardGoToItem::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const WebKit::WebBackForwardListCounts& counts)
{
    encoder.get() << counts;
    connection.sendSyncReply(WTFMove(encoder));
}

void BackForwardItemAtIndex::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const std::optional<WebCore::BackForwardItemIdentifier>& itemID)
{
    encoder.get() << itemID;
    connection.sendSyncReply(WTFMove(encoder));
}

void BackForwardListCounts::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const WebKit::WebBackForwardListCounts& counts)
{
    encoder.get() << counts;
    connection.sendSyncReply(WTFMove(encoder));
}

void CanUndoRedo::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

void ExecuteUndoRedo::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#if ENABLE(IMAGE_ANALYSIS)

void RequestTextRecognition::callReply(IPC::Decoder& decoder, CompletionHandler<void(WebCore::TextRecognitionResult&&)>&& completionHandler)
{
    std::optional<WebCore::TextRecognitionResult> result;
    decoder >> result;
    if (!result) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*result));
}

void RequestTextRecognition::cancelReply(CompletionHandler<void(WebCore::TextRecognitionResult&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<WebCore::TextRecognitionResult>::create());
}

void RequestTextRecognition::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const WebCore::TextRecognitionResult& result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(MEDIA_CONTROLS_CONTEXT_MENUS) && USE(UICONTEXTMENU)

void ShowMediaControlsContextMenu::callReply(IPC::Decoder& decoder, CompletionHandler<void(WebCore::MediaControlsContextMenuItem::ID&&)>&& completionHandler)
{
    std::optional<WebCore::MediaControlsContextMenuItem::ID> selectedItemID;
    decoder >> selectedItemID;
    if (!selectedItemID) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*selectedItemID));
}

void ShowMediaControlsContextMenu::cancelReply(CompletionHandler<void(WebCore::MediaControlsContextMenuItem::ID&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<WebCore::MediaControlsContextMenuItem::ID>::create());
}

void ShowMediaControlsContextMenu::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const WebCore::MediaControlsContextMenuItem::ID& selectedItemID)
{
    encoder.get() << selectedItemID;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

void RequestDOMPasteAccess::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, WebCore::DOMPasteAccessResponse response)
{
    encoder.get() << response;
    connection.sendSyncReply(WTFMove(encoder));
}

void ExceededDatabaseQuota::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, uint64_t newQuota)
{
    encoder.get() << newQuota;
    connection.sendSyncReply(WTFMove(encoder));
}

void ReachedApplicationCacheOriginQuota::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, uint64_t newQuota)
{
    encoder.get() << newQuota;
    connection.sendSyncReply(WTFMove(encoder));
}

#if ENABLE(MEDIA_STREAM)

void EnumerateMediaDevicesForFrame::callReply(IPC::Decoder& decoder, CompletionHandler<void(Vector<WebCore::CaptureDevice>&&, String&&)>&& completionHandler)
{
    std::optional<Vector<WebCore::CaptureDevice>> devices;
    decoder >> devices;
    if (!devices) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    std::optional<String> mediaDeviceIdentifierHashSalt;
    decoder >> mediaDeviceIdentifierHashSalt;
    if (!mediaDeviceIdentifierHashSalt) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*devices), WTFMove(*mediaDeviceIdentifierHashSalt));
}

void EnumerateMediaDevicesForFrame::cancelReply(CompletionHandler<void(Vector<WebCore::CaptureDevice>&&, String&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<Vector<WebCore::CaptureDevice>>::create(), IPC::AsyncReplyError<String>::create());
}

void EnumerateMediaDevicesForFrame::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Vector<WebCore::CaptureDevice>& devices, const String& mediaDeviceIdentifierHashSalt)
{
    encoder.get() << devices;
    encoder.get() << mediaDeviceIdentifierHashSalt;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

void RequestNotificationPermission::callReply(IPC::Decoder& decoder, CompletionHandler<void(bool&&)>&& completionHandler)
{
    std::optional<bool> allowed;
    decoder >> allowed;
    if (!allowed) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*allowed));
}

void RequestNotificationPermission::cancelReply(CompletionHandler<void(bool&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<bool>::create());
}

void RequestNotificationPermission::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool allowed)
{
    encoder.get() << allowed;
    connection.sendSyncReply(WTFMove(encoder));
}

#if USE(UNIFIED_TEXT_CHECKING)

void CheckTextOfParagraph::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Vector<WebCore::TextCheckingResult>& results)
{
    encoder.get() << results;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

void CheckSpellingOfString::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, int32_t misspellingLocation, int32_t misspellingLength)
{
    encoder.get() << misspellingLocation;
    encoder.get() << misspellingLength;
    connection.sendSyncReply(WTFMove(encoder));
}

void CheckGrammarOfString::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Vector<WebCore::GrammarDetail>& results, int32_t badGrammarLocation, int32_t badGrammarLength)
{
    encoder.get() << results;
    encoder.get() << badGrammarLocation;
    encoder.get() << badGrammarLength;
    connection.sendSyncReply(WTFMove(encoder));
}

void SpellingUIIsShowing::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool isShowing)
{
    encoder.get() << isShowing;
    connection.sendSyncReply(WTFMove(encoder));
}

void GetGuessesForWord::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Vector<String>& guesses)
{
    encoder.get() << guesses;
    connection.sendSyncReply(WTFMove(encoder));
}

#if PLATFORM(COCOA)

void ExecuteSavedCommandBySelector::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool handled)
{
    encoder.get() << handled;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if PLATFORM(COCOA)

void GetIsSpeaking::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool isSpeaking)
{
    encoder.get() << isSpeaking;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if USE(APPKIT)

void SubstitutionsPanelIsShowing::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool isShowing)
{
    encoder.get() << isShowing;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if PLATFORM(MAC)

void DismissCorrectionPanelSoon::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const String& result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if USE(DICTATION_ALTERNATIVES)

void DictationAlternatives::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Vector<String>& alternatives)
{
    encoder.get() << alternatives;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

void LoadRecentSearches::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Vector<WebCore::RecentSearch>& result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

#if ENABLE(PDFKIT_PLUGIN)

void ShowPDFContextMenu::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const std::optional<int32_t>& selectedItem)
{
    encoder.get() << selectedItem;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(WEB_CRYPTO)

void WrapCryptoKey::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool succeeded, const Vector<uint8_t>& wrappedKey)
{
    encoder.get() << succeeded;
    encoder.get() << wrappedKey;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(WEB_CRYPTO)

void UnwrapCryptoKey::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool succeeded, const Vector<uint8_t>& key)
{
    encoder.get() << succeeded;
    encoder.get() << key;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if USE(QUICK_LOOK)

void RequestPasswordForQuickLookDocumentInMainFrame::callReply(IPC::Decoder& decoder, CompletionHandler<void(String&&)>&& completionHandler)
{
    std::optional<String> password;
    decoder >> password;
    if (!password) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*password));
}

void RequestPasswordForQuickLookDocumentInMainFrame::cancelReply(CompletionHandler<void(String&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<String>::create());
}

void RequestPasswordForQuickLookDocumentInMainFrame::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const String& password)
{
    encoder.get() << password;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

void HandleSynchronousMessage::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const WebKit::UserData& returnData)
{
    encoder.get() << returnData;
    connection.sendSyncReply(WTFMove(encoder));
}

void LoadSynchronousURLSchemeTask::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const WebCore::ResourceResponse& response, const WebCore::ResourceError& error, const Vector<uint8_t>& data)
{
    encoder.get() << response;
    encoder.get() << error;
    encoder.get() << data;
    connection.sendSyncReply(WTFMove(encoder));
}

#if ENABLE(DEVICE_ORIENTATION)

void ShouldAllowDeviceOrientationAndMotionAccess::callReply(IPC::Decoder& decoder, CompletionHandler<void(WebCore::DeviceOrientationOrMotionPermissionState&&)>&& completionHandler)
{
    std::optional<WebCore::DeviceOrientationOrMotionPermissionState> permissionState;
    decoder >> permissionState;
    if (!permissionState) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*permissionState));
}

void ShouldAllowDeviceOrientationAndMotionAccess::cancelReply(CompletionHandler<void(WebCore::DeviceOrientationOrMotionPermissionState&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<WebCore::DeviceOrientationOrMotionPermissionState>::create());
}

void ShouldAllowDeviceOrientationAndMotionAccess::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, WebCore::DeviceOrientationOrMotionPermissionState permissionState)
{
    encoder.get() << permissionState;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(ATTACHMENT_ELEMENT)

void SerializedAttachmentDataForIdentifiers::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Vector<WebCore::SerializedAttachmentData>& seralizedData)
{
    encoder.get() << seralizedData;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

void SignedPublicKeyAndChallengeString::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const String& result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

#if ENABLE(SPEECH_SYNTHESIS)

void SpeechSynthesisVoiceList::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Vector<WebKit::WebSpeechSynthesisVoice>& voiceList)
{
    encoder.get() << voiceList;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(SPEECH_SYNTHESIS)

void SpeechSynthesisSpeak::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SpeechSynthesisSpeak::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SpeechSynthesisSpeak::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(SPEECH_SYNTHESIS)

void SpeechSynthesisSetFinishedCallback::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SpeechSynthesisSetFinishedCallback::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SpeechSynthesisSetFinishedCallback::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(SPEECH_SYNTHESIS)

void SpeechSynthesisPause::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SpeechSynthesisPause::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SpeechSynthesisPause::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(SPEECH_SYNTHESIS)

void SpeechSynthesisResume::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SpeechSynthesisResume::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SpeechSynthesisResume::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if PLATFORM(GTK)

void ShowEmojiPicker::callReply(IPC::Decoder& decoder, CompletionHandler<void(String&&)>&& completionHandler)
{
    std::optional<String> result;
    decoder >> result;
    if (!result) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*result));
}

void ShowEmojiPicker::cancelReply(CompletionHandler<void(String&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<String>::create());
}

void ShowEmojiPicker::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const String& result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if PLATFORM(GTK) || PLATFORM(WPE)

void SendMessageToWebViewWithReply::callReply(IPC::Decoder& decoder, CompletionHandler<void(WebKit::UserMessage&&)>&& completionHandler)
{
    std::optional<WebKit::UserMessage> replyMessage;
    decoder >> replyMessage;
    if (!replyMessage) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*replyMessage));
}

void SendMessageToWebViewWithReply::cancelReply(CompletionHandler<void(WebKit::UserMessage&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<WebKit::UserMessage>::create());
}

void SendMessageToWebViewWithReply::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const WebKit::UserMessage& replyMessage)
{
    encoder.get() << replyMessage;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(ARKIT_INLINE_PREVIEW_MAC)

void ModelElementDidCreatePreview::callReply(IPC::Decoder& decoder, CompletionHandler<void(Expected<std::pair<String, uint32_t>, WebCore::ResourceError>&&)>&& completionHandler)
{
    std::optional<Expected<std::pair<String, uint32_t>, WebCore::ResourceError>> result;
    decoder >> result;
    if (!result) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*result));
}

void ModelElementDidCreatePreview::cancelReply(CompletionHandler<void(Expected<std::pair<String, uint32_t>, WebCore::ResourceError>&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<Expected<std::pair<String, uint32_t>, WebCore::ResourceError>>::create());
}

void ModelElementDidCreatePreview::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Expected<std::pair<String, uint32_t>, WebCore::ResourceError>& result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(ARKIT_INLINE_PREVIEW_MAC)

void ModelElementSizeDidChange::callReply(IPC::Decoder& decoder, CompletionHandler<void(Expected<MachSendRight, WebCore::ResourceError>&&)>&& completionHandler)
{
    std::optional<Expected<MachSendRight, WebCore::ResourceError>> result;
    decoder >> result;
    if (!result) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*result));
}

void ModelElementSizeDidChange::cancelReply(CompletionHandler<void(Expected<MachSendRight, WebCore::ResourceError>&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<Expected<MachSendRight, WebCore::ResourceError>>::create());
}

void ModelElementSizeDidChange::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Expected<MachSendRight, WebCore::ResourceError>& result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(ARKIT_INLINE_PREVIEW)

void ModelElementGetCamera::callReply(IPC::Decoder& decoder, CompletionHandler<void(Expected<WebCore::HTMLModelElementCamera, WebCore::ResourceError>&&)>&& completionHandler)
{
    std::optional<Expected<WebCore::HTMLModelElementCamera, WebCore::ResourceError>> result;
    decoder >> result;
    if (!result) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*result));
}

void ModelElementGetCamera::cancelReply(CompletionHandler<void(Expected<WebCore::HTMLModelElementCamera, WebCore::ResourceError>&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<Expected<WebCore::HTMLModelElementCamera, WebCore::ResourceError>>::create());
}

void ModelElementGetCamera::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Expected<WebCore::HTMLModelElementCamera, WebCore::ResourceError>& result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(ARKIT_INLINE_PREVIEW)

void ModelElementSetCamera::callReply(IPC::Decoder& decoder, CompletionHandler<void(bool&&)>&& completionHandler)
{
    std::optional<bool> success;
    decoder >> success;
    if (!success) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*success));
}

void ModelElementSetCamera::cancelReply(CompletionHandler<void(bool&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<bool>::create());
}

void ModelElementSetCamera::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool success)
{
    encoder.get() << success;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(ARKIT_INLINE_PREVIEW)

void ModelElementIsPlayingAnimation::callReply(IPC::Decoder& decoder, CompletionHandler<void(Expected<bool, WebCore::ResourceError>&&)>&& completionHandler)
{
    std::optional<Expected<bool, WebCore::ResourceError>> result;
    decoder >> result;
    if (!result) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*result));
}

void ModelElementIsPlayingAnimation::cancelReply(CompletionHandler<void(Expected<bool, WebCore::ResourceError>&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<Expected<bool, WebCore::ResourceError>>::create());
}

void ModelElementIsPlayingAnimation::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Expected<bool, WebCore::ResourceError>& result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(ARKIT_INLINE_PREVIEW)

void ModelElementSetAnimationIsPlaying::callReply(IPC::Decoder& decoder, CompletionHandler<void(bool&&)>&& completionHandler)
{
    std::optional<bool> success;
    decoder >> success;
    if (!success) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*success));
}

void ModelElementSetAnimationIsPlaying::cancelReply(CompletionHandler<void(bool&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<bool>::create());
}

void ModelElementSetAnimationIsPlaying::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool success)
{
    encoder.get() << success;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(ARKIT_INLINE_PREVIEW)

void ModelElementIsLoopingAnimation::callReply(IPC::Decoder& decoder, CompletionHandler<void(Expected<bool, WebCore::ResourceError>&&)>&& completionHandler)
{
    std::optional<Expected<bool, WebCore::ResourceError>> result;
    decoder >> result;
    if (!result) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*result));
}

void ModelElementIsLoopingAnimation::cancelReply(CompletionHandler<void(Expected<bool, WebCore::ResourceError>&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<Expected<bool, WebCore::ResourceError>>::create());
}

void ModelElementIsLoopingAnimation::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Expected<bool, WebCore::ResourceError>& result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(ARKIT_INLINE_PREVIEW)

void ModelElementSetIsLoopingAnimation::callReply(IPC::Decoder& decoder, CompletionHandler<void(bool&&)>&& completionHandler)
{
    std::optional<bool> success;
    decoder >> success;
    if (!success) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*success));
}

void ModelElementSetIsLoopingAnimation::cancelReply(CompletionHandler<void(bool&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<bool>::create());
}

void ModelElementSetIsLoopingAnimation::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool success)
{
    encoder.get() << success;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(ARKIT_INLINE_PREVIEW)

void ModelElementAnimationDuration::callReply(IPC::Decoder& decoder, CompletionHandler<void(Expected<Seconds, WebCore::ResourceError>&&)>&& completionHandler)
{
    std::optional<Expected<Seconds, WebCore::ResourceError>> result;
    decoder >> result;
    if (!result) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*result));
}

void ModelElementAnimationDuration::cancelReply(CompletionHandler<void(Expected<Seconds, WebCore::ResourceError>&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<Expected<Seconds, WebCore::ResourceError>>::create());
}

void ModelElementAnimationDuration::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Expected<Seconds, WebCore::ResourceError>& result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(ARKIT_INLINE_PREVIEW)

void ModelElementAnimationCurrentTime::callReply(IPC::Decoder& decoder, CompletionHandler<void(Expected<Seconds, WebCore::ResourceError>&&)>&& completionHandler)
{
    std::optional<Expected<Seconds, WebCore::ResourceError>> result;
    decoder >> result;
    if (!result) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*result));
}

void ModelElementAnimationCurrentTime::cancelReply(CompletionHandler<void(Expected<Seconds, WebCore::ResourceError>&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<Expected<Seconds, WebCore::ResourceError>>::create());
}

void ModelElementAnimationCurrentTime::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Expected<Seconds, WebCore::ResourceError>& result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(ARKIT_INLINE_PREVIEW)

void ModelElementSetAnimationCurrentTime::callReply(IPC::Decoder& decoder, CompletionHandler<void(bool&&)>&& completionHandler)
{
    std::optional<bool> success;
    decoder >> success;
    if (!success) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*success));
}

void ModelElementSetAnimationCurrentTime::cancelReply(CompletionHandler<void(bool&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<bool>::create());
}

void ModelElementSetAnimationCurrentTime::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool success)
{
    encoder.get() << success;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(ARKIT_INLINE_PREVIEW)

void ModelElementHasAudio::callReply(IPC::Decoder& decoder, CompletionHandler<void(Expected<bool, WebCore::ResourceError>&&)>&& completionHandler)
{
    std::optional<Expected<bool, WebCore::ResourceError>> result;
    decoder >> result;
    if (!result) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*result));
}

void ModelElementHasAudio::cancelReply(CompletionHandler<void(Expected<bool, WebCore::ResourceError>&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<Expected<bool, WebCore::ResourceError>>::create());
}

void ModelElementHasAudio::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Expected<bool, WebCore::ResourceError>& result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(ARKIT_INLINE_PREVIEW)

void ModelElementIsMuted::callReply(IPC::Decoder& decoder, CompletionHandler<void(Expected<bool, WebCore::ResourceError>&&)>&& completionHandler)
{
    std::optional<Expected<bool, WebCore::ResourceError>> result;
    decoder >> result;
    if (!result) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*result));
}

void ModelElementIsMuted::cancelReply(CompletionHandler<void(Expected<bool, WebCore::ResourceError>&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<Expected<bool, WebCore::ResourceError>>::create());
}

void ModelElementIsMuted::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Expected<bool, WebCore::ResourceError>& result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(ARKIT_INLINE_PREVIEW)

void ModelElementSetIsMuted::callReply(IPC::Decoder& decoder, CompletionHandler<void(bool&&)>&& completionHandler)
{
    std::optional<bool> success;
    decoder >> success;
    if (!success) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*success));
}

void ModelElementSetIsMuted::cancelReply(CompletionHandler<void(bool&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<bool>::create());
}

void ModelElementSetIsMuted::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool success)
{
    encoder.get() << success;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

void QueryPermission::callReply(IPC::Decoder& decoder, CompletionHandler<void(std::optional<WebCore::PermissionState>&&, bool&&)>&& completionHandler)
{
    std::optional<std::optional<WebCore::PermissionState>> state;
    decoder >> state;
    if (!state) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    std::optional<bool> shouldCache;
    decoder >> shouldCache;
    if (!shouldCache) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*state), WTFMove(*shouldCache));
}

void QueryPermission::cancelReply(CompletionHandler<void(std::optional<WebCore::PermissionState>&&, bool&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<std::optional<WebCore::PermissionState>>::create(), IPC::AsyncReplyError<bool>::create());
}

void QueryPermission::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const std::optional<WebCore::PermissionState>& state, bool shouldCache)
{
    encoder.get() << state;
    encoder.get() << shouldCache;
    connection.sendSyncReply(WTFMove(encoder));
}

#if ENABLE(APPLE_PAY_AMS_UI)

void StartApplePayAMSUISession::callReply(IPC::Decoder& decoder, CompletionHandler<void(std::optional<bool>&&)>&& completionHandler)
{
    std::optional<std::optional<bool>> result;
    decoder >> result;
    if (!result) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*result));
}

void StartApplePayAMSUISession::cancelReply(CompletionHandler<void(std::optional<bool>&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<std::optional<bool>>::create());
}

void StartApplePayAMSUISession::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const std::optional<bool>& result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

void RequestCookieConsent::callReply(IPC::Decoder& decoder, CompletionHandler<void(WebCore::CookieConsentDecisionResult&&)>&& completionHandler)
{
    std::optional<WebCore::CookieConsentDecisionResult> result;
    decoder >> result;
    if (!result) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*result));
}

void RequestCookieConsent::cancelReply(CompletionHandler<void(WebCore::CookieConsentDecisionResult&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<WebCore::CookieConsentDecisionResult>::create());
}

void RequestCookieConsent::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, WebCore::CookieConsentDecisionResult result)
{
    encoder.get() << result;
    connection.sendSyncReply(WTFMove(encoder));
}

void ClassifyModalContainerControls::callReply(IPC::Decoder& decoder, CompletionHandler<void(Vector<WebCore::ModalContainerControlType>&&)>&& completionHandler)
{
    std::optional<Vector<WebCore::ModalContainerControlType>> types;
    decoder >> types;
    if (!types) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*types));
}

void ClassifyModalContainerControls::cancelReply(CompletionHandler<void(Vector<WebCore::ModalContainerControlType>&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<Vector<WebCore::ModalContainerControlType>>::create());
}

void ClassifyModalContainerControls::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const Vector<WebCore::ModalContainerControlType>& types)
{
    encoder.get() << types;
    connection.sendSyncReply(WTFMove(encoder));
}

void DecidePolicyForModalContainer::callReply(IPC::Decoder& decoder, CompletionHandler<void(WebCore::ModalContainerDecision&&)>&& completionHandler)
{
    std::optional<WebCore::ModalContainerDecision> decision;
    decoder >> decision;
    if (!decision) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*decision));
}

void DecidePolicyForModalContainer::cancelReply(CompletionHandler<void(WebCore::ModalContainerDecision&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<WebCore::ModalContainerDecision>::create());
}

void DecidePolicyForModalContainer::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, WebCore::ModalContainerDecision decision)
{
    encoder.get() << decision;
    connection.sendSyncReply(WTFMove(encoder));
}

} // namespace WebPageProxy

} // namespace Messages

namespace WebKit {

void WebPageProxy::didReceiveMessage(IPC::Connection& connection, IPC::Decoder& decoder)
{
    Ref protectedThis { *this };
    if (decoder.messageName() == Messages::WebPageProxy::ShowPage::name())
        return IPC::handleMessage<Messages::WebPageProxy::ShowPage>(connection, decoder, this, &WebPageProxy::showPage);
    if (decoder.messageName() == Messages::WebPageProxy::ClosePage::name())
        return IPC::handleMessage<Messages::WebPageProxy::ClosePage>(connection, decoder, this, &WebPageProxy::closePage);
    if (decoder.messageName() == Messages::WebPageProxy::MouseDidMoveOverElement::name())
        return IPC::handleMessage<Messages::WebPageProxy::MouseDidMoveOverElement>(connection, decoder, this, &WebPageProxy::mouseDidMoveOverElement);
    if (decoder.messageName() == Messages::WebPageProxy::DidChangeViewportProperties::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidChangeViewportProperties>(connection, decoder, this, &WebPageProxy::didChangeViewportProperties);
    if (decoder.messageName() == Messages::WebPageProxy::DidReceiveEvent::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidReceiveEvent>(connection, decoder, this, &WebPageProxy::didReceiveEvent);
    if (decoder.messageName() == Messages::WebPageProxy::SetCursor::name())
        return IPC::handleMessage<Messages::WebPageProxy::SetCursor>(connection, decoder, this, &WebPageProxy::setCursor);
    if (decoder.messageName() == Messages::WebPageProxy::SetCursorHiddenUntilMouseMoves::name())
        return IPC::handleMessage<Messages::WebPageProxy::SetCursorHiddenUntilMouseMoves>(connection, decoder, this, &WebPageProxy::setCursorHiddenUntilMouseMoves);
    if (decoder.messageName() == Messages::WebPageProxy::SetStatusText::name())
        return IPC::handleMessage<Messages::WebPageProxy::SetStatusText>(connection, decoder, this, &WebPageProxy::setStatusText);
    if (decoder.messageName() == Messages::WebPageProxy::SetFocus::name())
        return IPC::handleMessage<Messages::WebPageProxy::SetFocus>(connection, decoder, this, &WebPageProxy::setFocus);
    if (decoder.messageName() == Messages::WebPageProxy::TakeFocus::name())
        return IPC::handleMessage<Messages::WebPageProxy::TakeFocus>(connection, decoder, this, &WebPageProxy::takeFocus);
    if (decoder.messageName() == Messages::WebPageProxy::FocusedFrameChanged::name())
        return IPC::handleMessage<Messages::WebPageProxy::FocusedFrameChanged>(connection, decoder, this, &WebPageProxy::focusedFrameChanged);
    if (decoder.messageName() == Messages::WebPageProxy::SetRenderTreeSize::name())
        return IPC::handleMessage<Messages::WebPageProxy::SetRenderTreeSize>(connection, decoder, this, &WebPageProxy::setRenderTreeSize);
    if (decoder.messageName() == Messages::WebPageProxy::SetToolbarsAreVisible::name())
        return IPC::handleMessage<Messages::WebPageProxy::SetToolbarsAreVisible>(connection, decoder, this, &WebPageProxy::setToolbarsAreVisible);
    if (decoder.messageName() == Messages::WebPageProxy::SetMenuBarIsVisible::name())
        return IPC::handleMessage<Messages::WebPageProxy::SetMenuBarIsVisible>(connection, decoder, this, &WebPageProxy::setMenuBarIsVisible);
    if (decoder.messageName() == Messages::WebPageProxy::SetStatusBarIsVisible::name())
        return IPC::handleMessage<Messages::WebPageProxy::SetStatusBarIsVisible>(connection, decoder, this, &WebPageProxy::setStatusBarIsVisible);
    if (decoder.messageName() == Messages::WebPageProxy::SetIsResizable::name())
        return IPC::handleMessage<Messages::WebPageProxy::SetIsResizable>(connection, decoder, this, &WebPageProxy::setIsResizable);
    if (decoder.messageName() == Messages::WebPageProxy::SetWindowFrame::name())
        return IPC::handleMessage<Messages::WebPageProxy::SetWindowFrame>(connection, decoder, this, &WebPageProxy::setWindowFrame);
#if PLATFORM(COCOA) || PLATFORM(GTK)
    if (decoder.messageName() == Messages::WebPageProxy::ShowValidationMessage::name())
        return IPC::handleMessage<Messages::WebPageProxy::ShowValidationMessage>(connection, decoder, this, &WebPageProxy::showValidationMessage);
#endif
#if PLATFORM(COCOA) || PLATFORM(GTK)
    if (decoder.messageName() == Messages::WebPageProxy::HideValidationMessage::name())
        return IPC::handleMessage<Messages::WebPageProxy::HideValidationMessage>(connection, decoder, this, &WebPageProxy::hideValidationMessage);
#endif
    if (decoder.messageName() == Messages::WebPageProxy::PageDidScroll::name())
        return IPC::handleMessage<Messages::WebPageProxy::PageDidScroll>(connection, decoder, this, &WebPageProxy::pageDidScroll);
    if (decoder.messageName() == Messages::WebPageProxy::SetHasActiveAnimatedScrolls::name())
        return IPC::handleMessage<Messages::WebPageProxy::SetHasActiveAnimatedScrolls>(connection, decoder, this, &WebPageProxy::setHasActiveAnimatedScrolls);
    if (decoder.messageName() == Messages::WebPageProxy::RunOpenPanel::name())
        return IPC::handleMessage<Messages::WebPageProxy::RunOpenPanel>(connection, decoder, this, &WebPageProxy::runOpenPanel);
    if (decoder.messageName() == Messages::WebPageProxy::ShowShareSheet::name())
        return IPC::handleMessageAsync<Messages::WebPageProxy::ShowShareSheet>(connection, decoder, this, &WebPageProxy::showShareSheet);
    if (decoder.messageName() == Messages::WebPageProxy::ShowContactPicker::name())
        return IPC::handleMessageAsync<Messages::WebPageProxy::ShowContactPicker>(connection, decoder, this, &WebPageProxy::showContactPicker);
    if (decoder.messageName() == Messages::WebPageProxy::RunModal::name())
        return IPC::handleMessage<Messages::WebPageProxy::RunModal>(connection, decoder, this, &WebPageProxy::runModal);
    if (decoder.messageName() == Messages::WebPageProxy::NotifyScrollerThumbIsVisibleInRect::name())
        return IPC::handleMessage<Messages::WebPageProxy::NotifyScrollerThumbIsVisibleInRect>(connection, decoder, this, &WebPageProxy::notifyScrollerThumbIsVisibleInRect);
    if (decoder.messageName() == Messages::WebPageProxy::RecommendedScrollbarStyleDidChange::name())
        return IPC::handleMessage<Messages::WebPageProxy::RecommendedScrollbarStyleDidChange>(connection, decoder, this, &WebPageProxy::recommendedScrollbarStyleDidChange);
    if (decoder.messageName() == Messages::WebPageProxy::DidChangeScrollbarsForMainFrame::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidChangeScrollbarsForMainFrame>(connection, decoder, this, &WebPageProxy::didChangeScrollbarsForMainFrame);
    if (decoder.messageName() == Messages::WebPageProxy::DidChangeScrollOffsetPinningForMainFrame::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidChangeScrollOffsetPinningForMainFrame>(connection, decoder, this, &WebPageProxy::didChangeScrollOffsetPinningForMainFrame);
    if (decoder.messageName() == Messages::WebPageProxy::DidChangePageCount::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidChangePageCount>(connection, decoder, this, &WebPageProxy::didChangePageCount);
    if (decoder.messageName() == Messages::WebPageProxy::ThemeColorChanged::name())
        return IPC::handleMessage<Messages::WebPageProxy::ThemeColorChanged>(connection, decoder, this, &WebPageProxy::themeColorChanged);
    if (decoder.messageName() == Messages::WebPageProxy::PageExtendedBackgroundColorDidChange::name())
        return IPC::handleMessage<Messages::WebPageProxy::PageExtendedBackgroundColorDidChange>(connection, decoder, this, &WebPageProxy::pageExtendedBackgroundColorDidChange);
    if (decoder.messageName() == Messages::WebPageProxy::SampledPageTopColorChanged::name())
        return IPC::handleMessage<Messages::WebPageProxy::SampledPageTopColorChanged>(connection, decoder, this, &WebPageProxy::sampledPageTopColorChanged);
    if (decoder.messageName() == Messages::WebPageProxy::SetCanShortCircuitHorizontalWheelEvents::name())
        return IPC::handleMessage<Messages::WebPageProxy::SetCanShortCircuitHorizontalWheelEvents>(connection, decoder, this, &WebPageProxy::setCanShortCircuitHorizontalWheelEvents);
    if (decoder.messageName() == Messages::WebPageProxy::DidChangeContentSize::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidChangeContentSize>(connection, decoder, this, &WebPageProxy::didChangeContentSize);
    if (decoder.messageName() == Messages::WebPageProxy::DidChangeIntrinsicContentSize::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidChangeIntrinsicContentSize>(connection, decoder, this, &WebPageProxy::didChangeIntrinsicContentSize);
#if ENABLE(INPUT_TYPE_COLOR)
    if (decoder.messageName() == Messages::WebPageProxy::ShowColorPicker::name())
        return IPC::handleMessage<Messages::WebPageProxy::ShowColorPicker>(connection, decoder, this, &WebPageProxy::showColorPicker);
#endif
#if ENABLE(INPUT_TYPE_COLOR)
    if (decoder.messageName() == Messages::WebPageProxy::SetColorPickerColor::name())
        return IPC::handleMessage<Messages::WebPageProxy::SetColorPickerColor>(connection, decoder, this, &WebPageProxy::setColorPickerColor);
#endif
#if ENABLE(INPUT_TYPE_COLOR)
    if (decoder.messageName() == Messages::WebPageProxy::EndColorPicker::name())
        return IPC::handleMessage<Messages::WebPageProxy::EndColorPicker>(connection, decoder, this, &WebPageProxy::endColorPicker);
#endif
#if ENABLE(DATALIST_ELEMENT)
    if (decoder.messageName() == Messages::WebPageProxy::ShowDataListSuggestions::name())
        return IPC::handleMessage<Messages::WebPageProxy::ShowDataListSuggestions>(connection, decoder, this, &WebPageProxy::showDataListSuggestions);
#endif
#if ENABLE(DATALIST_ELEMENT)
    if (decoder.messageName() == Messages::WebPageProxy::HandleKeydownInDataList::name())
        return IPC::handleMessage<Messages::WebPageProxy::HandleKeydownInDataList>(connection, decoder, this, &WebPageProxy::handleKeydownInDataList);
#endif
#if ENABLE(DATALIST_ELEMENT)
    if (decoder.messageName() == Messages::WebPageProxy::EndDataListSuggestions::name())
        return IPC::handleMessage<Messages::WebPageProxy::EndDataListSuggestions>(connection, decoder, this, &WebPageProxy::endDataListSuggestions);
#endif
#if ENABLE(DATE_AND_TIME_INPUT_TYPES)
    if (decoder.messageName() == Messages::WebPageProxy::ShowDateTimePicker::name())
        return IPC::handleMessage<Messages::WebPageProxy::ShowDateTimePicker>(connection, decoder, this, &WebPageProxy::showDateTimePicker);
#endif
#if ENABLE(DATE_AND_TIME_INPUT_TYPES)
    if (decoder.messageName() == Messages::WebPageProxy::EndDateTimePicker::name())
        return IPC::handleMessage<Messages::WebPageProxy::EndDateTimePicker>(connection, decoder, this, &WebPageProxy::endDateTimePicker);
#endif
    if (decoder.messageName() == Messages::WebPageProxy::DecidePolicyForResponse::name())
        return IPC::handleMessage<Messages::WebPageProxy::DecidePolicyForResponse>(connection, decoder, this, &WebPageProxy::decidePolicyForResponse);
    if (decoder.messageName() == Messages::WebPageProxy::DecidePolicyForNavigationActionAsync::name())
        return IPC::handleMessage<Messages::WebPageProxy::DecidePolicyForNavigationActionAsync>(connection, decoder, this, &WebPageProxy::decidePolicyForNavigationActionAsync);
    if (decoder.messageName() == Messages::WebPageProxy::DecidePolicyForNewWindowAction::name())
        return IPC::handleMessage<Messages::WebPageProxy::DecidePolicyForNewWindowAction>(connection, decoder, this, &WebPageProxy::decidePolicyForNewWindowAction);
    if (decoder.messageName() == Messages::WebPageProxy::UnableToImplementPolicy::name())
        return IPC::handleMessage<Messages::WebPageProxy::UnableToImplementPolicy>(connection, decoder, this, &WebPageProxy::unableToImplementPolicy);
    if (decoder.messageName() == Messages::WebPageProxy::DidChangeProgress::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidChangeProgress>(connection, decoder, this, &WebPageProxy::didChangeProgress);
    if (decoder.messageName() == Messages::WebPageProxy::DidFinishProgress::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidFinishProgress>(connection, decoder, this, &WebPageProxy::didFinishProgress);
    if (decoder.messageName() == Messages::WebPageProxy::DidStartProgress::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidStartProgress>(connection, decoder, this, &WebPageProxy::didStartProgress);
    if (decoder.messageName() == Messages::WebPageProxy::SetNetworkRequestsInProgress::name())
        return IPC::handleMessage<Messages::WebPageProxy::SetNetworkRequestsInProgress>(connection, decoder, this, &WebPageProxy::setNetworkRequestsInProgress);
    if (decoder.messageName() == Messages::WebPageProxy::DidCreateMainFrame::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidCreateMainFrame>(connection, decoder, this, &WebPageProxy::didCreateMainFrame);
    if (decoder.messageName() == Messages::WebPageProxy::DidCreateSubframe::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidCreateSubframe>(connection, decoder, this, &WebPageProxy::didCreateSubframe);
    if (decoder.messageName() == Messages::WebPageProxy::DidCreateWindow::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidCreateWindow>(connection, decoder, this, &WebPageProxy::didCreateWindow);
    if (decoder.messageName() == Messages::WebPageProxy::DidStartProvisionalLoadForFrame::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidStartProvisionalLoadForFrame>(connection, decoder, this, &WebPageProxy::didStartProvisionalLoadForFrame);
    if (decoder.messageName() == Messages::WebPageProxy::DidReceiveServerRedirectForProvisionalLoadForFrame::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidReceiveServerRedirectForProvisionalLoadForFrame>(connection, decoder, this, &WebPageProxy::didReceiveServerRedirectForProvisionalLoadForFrame);
    if (decoder.messageName() == Messages::WebPageProxy::WillPerformClientRedirectForFrame::name())
        return IPC::handleMessage<Messages::WebPageProxy::WillPerformClientRedirectForFrame>(connection, decoder, this, &WebPageProxy::willPerformClientRedirectForFrame);
    if (decoder.messageName() == Messages::WebPageProxy::DidCancelClientRedirectForFrame::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidCancelClientRedirectForFrame>(connection, decoder, this, &WebPageProxy::didCancelClientRedirectForFrame);
    if (decoder.messageName() == Messages::WebPageProxy::DidChangeProvisionalURLForFrame::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidChangeProvisionalURLForFrame>(connection, decoder, this, &WebPageProxy::didChangeProvisionalURLForFrame);
    if (decoder.messageName() == Messages::WebPageProxy::DidFailProvisionalLoadForFrame::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidFailProvisionalLoadForFrame>(connection, decoder, this, &WebPageProxy::didFailProvisionalLoadForFrame);
    if (decoder.messageName() == Messages::WebPageProxy::DidCommitLoadForFrame::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidCommitLoadForFrame>(connection, decoder, this, &WebPageProxy::didCommitLoadForFrame);
    if (decoder.messageName() == Messages::WebPageProxy::DidFailLoadForFrame::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidFailLoadForFrame>(connection, decoder, this, &WebPageProxy::didFailLoadForFrame);
    if (decoder.messageName() == Messages::WebPageProxy::DidFinishDocumentLoadForFrame::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidFinishDocumentLoadForFrame>(connection, decoder, this, &WebPageProxy::didFinishDocumentLoadForFrame);
    if (decoder.messageName() == Messages::WebPageProxy::DidFinishLoadForFrame::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidFinishLoadForFrame>(connection, decoder, this, &WebPageProxy::didFinishLoadForFrame);
    if (decoder.messageName() == Messages::WebPageProxy::DidFirstLayoutForFrame::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidFirstLayoutForFrame>(connection, decoder, this, &WebPageProxy::didFirstLayoutForFrame);
    if (decoder.messageName() == Messages::WebPageProxy::DidFirstVisuallyNonEmptyLayoutForFrame::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidFirstVisuallyNonEmptyLayoutForFrame>(connection, decoder, this, &WebPageProxy::didFirstVisuallyNonEmptyLayoutForFrame);
    if (decoder.messageName() == Messages::WebPageProxy::DidReachLayoutMilestone::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidReachLayoutMilestone>(connection, decoder, this, &WebPageProxy::didReachLayoutMilestone);
    if (decoder.messageName() == Messages::WebPageProxy::DidReceiveTitleForFrame::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidReceiveTitleForFrame>(connection, decoder, this, &WebPageProxy::didReceiveTitleForFrame);
    if (decoder.messageName() == Messages::WebPageProxy::DidDisplayInsecureContentForFrame::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidDisplayInsecureContentForFrame>(connection, decoder, this, &WebPageProxy::didDisplayInsecureContentForFrame);
    if (decoder.messageName() == Messages::WebPageProxy::DidRunInsecureContentForFrame::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidRunInsecureContentForFrame>(connection, decoder, this, &WebPageProxy::didRunInsecureContentForFrame);
    if (decoder.messageName() == Messages::WebPageProxy::DidDetectXSSForFrame::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidDetectXSSForFrame>(connection, decoder, this, &WebPageProxy::didDetectXSSForFrame);
    if (decoder.messageName() == Messages::WebPageProxy::DidSameDocumentNavigationForFrame::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidSameDocumentNavigationForFrame>(connection, decoder, this, &WebPageProxy::didSameDocumentNavigationForFrame);
    if (decoder.messageName() == Messages::WebPageProxy::DidChangeMainDocument::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidChangeMainDocument>(connection, decoder, this, &WebPageProxy::didChangeMainDocument);
    if (decoder.messageName() == Messages::WebPageProxy::DidExplicitOpenForFrame::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidExplicitOpenForFrame>(connection, decoder, this, &WebPageProxy::didExplicitOpenForFrame);
    if (decoder.messageName() == Messages::WebPageProxy::DidDestroyNavigation::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidDestroyNavigation>(connection, decoder, this, &WebPageProxy::didDestroyNavigation);
    if (decoder.messageName() == Messages::WebPageProxy::MainFramePluginHandlesPageScaleGestureDidChange::name())
        return IPC::handleMessage<Messages::WebPageProxy::MainFramePluginHandlesPageScaleGestureDidChange>(connection, decoder, this, &WebPageProxy::mainFramePluginHandlesPageScaleGestureDidChange);
    if (decoder.messageName() == Messages::WebPageProxy::DidNavigateWithNavigationData::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidNavigateWithNavigationData>(connection, decoder, this, &WebPageProxy::didNavigateWithNavigationData);
    if (decoder.messageName() == Messages::WebPageProxy::DidPerformClientRedirect::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidPerformClientRedirect>(connection, decoder, this, &WebPageProxy::didPerformClientRedirect);
    if (decoder.messageName() == Messages::WebPageProxy::DidPerformServerRedirect::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidPerformServerRedirect>(connection, decoder, this, &WebPageProxy::didPerformServerRedirect);
    if (decoder.messageName() == Messages::WebPageProxy::DidUpdateHistoryTitle::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidUpdateHistoryTitle>(connection, decoder, this, &WebPageProxy::didUpdateHistoryTitle);
    if (decoder.messageName() == Messages::WebPageProxy::DidFinishLoadingDataForCustomContentProvider::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidFinishLoadingDataForCustomContentProvider>(connection, decoder, this, &WebPageProxy::didFinishLoadingDataForCustomContentProvider);
    if (decoder.messageName() == Messages::WebPageProxy::WillSubmitForm::name())
        return IPC::handleMessage<Messages::WebPageProxy::WillSubmitForm>(connection, decoder, this, &WebPageProxy::willSubmitForm);
#if PLATFORM(IOS_FAMILY)
    if (decoder.messageName() == Messages::WebPageProxy::DidReceivePositionInformation::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidReceivePositionInformation>(connection, decoder, this, &WebPageProxy::didReceivePositionInformation);
#endif
#if PLATFORM(IOS_FAMILY)
    if (decoder.messageName() == Messages::WebPageProxy::SaveImageToLibrary::name())
        return IPC::handleMessage<Messages::WebPageProxy::SaveImageToLibrary>(connection, decoder, this, &WebPageProxy::saveImageToLibrary);
#endif
#if PLATFORM(IOS_FAMILY)
    if (decoder.messageName() == Messages::WebPageProxy::ShowPlaybackTargetPicker::name())
        return IPC::handleMessage<Messages::WebPageProxy::ShowPlaybackTargetPicker>(connection, decoder, this, &WebPageProxy::showPlaybackTargetPicker);
#endif
#if PLATFORM(IOS_FAMILY)
    if (decoder.messageName() == Messages::WebPageProxy::CommitPotentialTapFailed::name())
        return IPC::handleMessage<Messages::WebPageProxy::CommitPotentialTapFailed>(connection, decoder, this, &WebPageProxy::commitPotentialTapFailed);
#endif
#if PLATFORM(IOS_FAMILY)
    if (decoder.messageName() == Messages::WebPageProxy::DidNotHandleTapAsClick::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidNotHandleTapAsClick>(connection, decoder, this, &WebPageProxy::didNotHandleTapAsClick);
#endif
#if PLATFORM(IOS_FAMILY)
    if (decoder.messageName() == Messages::WebPageProxy::DidCompleteSyntheticClick::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidCompleteSyntheticClick>(connection, decoder, this, &WebPageProxy::didCompleteSyntheticClick);
#endif
#if PLATFORM(IOS_FAMILY)
    if (decoder.messageName() == Messages::WebPageProxy::DisableDoubleTapGesturesDuringTapIfNecessary::name())
        return IPC::handleMessage<Messages::WebPageProxy::DisableDoubleTapGesturesDuringTapIfNecessary>(connection, decoder, this, &WebPageProxy::disableDoubleTapGesturesDuringTapIfNecessary);
#endif
#if PLATFORM(IOS_FAMILY)
    if (decoder.messageName() == Messages::WebPageProxy::HandleSmartMagnificationInformationForPotentialTap::name())
        return IPC::handleMessage<Messages::WebPageProxy::HandleSmartMagnificationInformationForPotentialTap>(connection, decoder, this, &WebPageProxy::handleSmartMagnificationInformationForPotentialTap);
#endif
#if ENABLE(DATA_DETECTION)
    if (decoder.messageName() == Messages::WebPageProxy::SetDataDetectionResult::name())
        return IPC::handleMessage<Messages::WebPageProxy::SetDataDetectionResult>(connection, decoder, this, &WebPageProxy::setDataDetectionResult);
#endif
#if ENABLE(DATA_DETECTION)
    if (decoder.messageName() == Messages::WebPageProxy::HandleClickForDataDetectionResult::name())
        return IPC::handleMessage<Messages::WebPageProxy::HandleClickForDataDetectionResult>(connection, decoder, this, &WebPageProxy::handleClickForDataDetectionResult);
#endif
    if (decoder.messageName() == Messages::WebPageProxy::PageScaleFactorDidChange::name())
        return IPC::handleMessage<Messages::WebPageProxy::PageScaleFactorDidChange>(connection, decoder, this, &WebPageProxy::pageScaleFactorDidChange);
    if (decoder.messageName() == Messages::WebPageProxy::PluginScaleFactorDidChange::name())
        return IPC::handleMessage<Messages::WebPageProxy::PluginScaleFactorDidChange>(connection, decoder, this, &WebPageProxy::pluginScaleFactorDidChange);
    if (decoder.messageName() == Messages::WebPageProxy::PluginZoomFactorDidChange::name())
        return IPC::handleMessage<Messages::WebPageProxy::PluginZoomFactorDidChange>(connection, decoder, this, &WebPageProxy::pluginZoomFactorDidChange);
#if PLATFORM(GTK) || PLATFORM(WPE)
    if (decoder.messageName() == Messages::WebPageProxy::BindAccessibilityTree::name())
        return IPC::handleMessage<Messages::WebPageProxy::BindAccessibilityTree>(connection, decoder, this, &WebPageProxy::bindAccessibilityTree);
#endif
#if PLATFORM(GTK) || PLATFORM(WPE)
    if (decoder.messageName() == Messages::WebPageProxy::SetInputMethodState::name())
        return IPC::handleMessage<Messages::WebPageProxy::SetInputMethodState>(connection, decoder, this, &WebPageProxy::setInputMethodState);
#endif
    if (decoder.messageName() == Messages::WebPageProxy::BackForwardAddItem::name())
        return IPC::handleMessage<Messages::WebPageProxy::BackForwardAddItem>(connection, decoder, this, &WebPageProxy::backForwardAddItem);
    if (decoder.messageName() == Messages::WebPageProxy::BackForwardClear::name())
        return IPC::handleMessage<Messages::WebPageProxy::BackForwardClear>(connection, decoder, this, &WebPageProxy::backForwardClear);
    if (decoder.messageName() == Messages::WebPageProxy::WillGoToBackForwardListItem::name())
        return IPC::handleMessage<Messages::WebPageProxy::WillGoToBackForwardListItem>(connection, decoder, this, &WebPageProxy::willGoToBackForwardListItem);
    if (decoder.messageName() == Messages::WebPageProxy::RegisterEditCommandForUndo::name())
        return IPC::handleMessage<Messages::WebPageProxy::RegisterEditCommandForUndo>(connection, decoder, this, &WebPageProxy::registerEditCommandForUndo);
    if (decoder.messageName() == Messages::WebPageProxy::ClearAllEditCommands::name())
        return IPC::handleMessage<Messages::WebPageProxy::ClearAllEditCommands>(connection, decoder, this, &WebPageProxy::clearAllEditCommands);
    if (decoder.messageName() == Messages::WebPageProxy::RegisterInsertionUndoGrouping::name())
        return IPC::handleMessage<Messages::WebPageProxy::RegisterInsertionUndoGrouping>(connection, decoder, this, &WebPageProxy::registerInsertionUndoGrouping);
    if (decoder.messageName() == Messages::WebPageProxy::LogDiagnosticMessageFromWebProcess::name())
        return IPC::handleMessage<Messages::WebPageProxy::LogDiagnosticMessageFromWebProcess>(connection, decoder, this, &WebPageProxy::logDiagnosticMessageFromWebProcess);
    if (decoder.messageName() == Messages::WebPageProxy::LogDiagnosticMessageWithResultFromWebProcess::name())
        return IPC::handleMessage<Messages::WebPageProxy::LogDiagnosticMessageWithResultFromWebProcess>(connection, decoder, this, &WebPageProxy::logDiagnosticMessageWithResultFromWebProcess);
    if (decoder.messageName() == Messages::WebPageProxy::LogDiagnosticMessageWithValueFromWebProcess::name())
        return IPC::handleMessage<Messages::WebPageProxy::LogDiagnosticMessageWithValueFromWebProcess>(connection, decoder, this, &WebPageProxy::logDiagnosticMessageWithValueFromWebProcess);
    if (decoder.messageName() == Messages::WebPageProxy::LogDiagnosticMessageWithEnhancedPrivacyFromWebProcess::name())
        return IPC::handleMessage<Messages::WebPageProxy::LogDiagnosticMessageWithEnhancedPrivacyFromWebProcess>(connection, decoder, this, &WebPageProxy::logDiagnosticMessageWithEnhancedPrivacyFromWebProcess);
    if (decoder.messageName() == Messages::WebPageProxy::LogDiagnosticMessageWithValueDictionaryFromWebProcess::name())
        return IPC::handleMessage<Messages::WebPageProxy::LogDiagnosticMessageWithValueDictionaryFromWebProcess>(connection, decoder, this, &WebPageProxy::logDiagnosticMessageWithValueDictionaryFromWebProcess);
    if (decoder.messageName() == Messages::WebPageProxy::LogDiagnosticMessageWithDomainFromWebProcess::name())
        return IPC::handleMessage<Messages::WebPageProxy::LogDiagnosticMessageWithDomainFromWebProcess>(connection, decoder, this, &WebPageProxy::logDiagnosticMessageWithDomainFromWebProcess);
    if (decoder.messageName() == Messages::WebPageProxy::LogScrollingEvent::name())
        return IPC::handleMessage<Messages::WebPageProxy::LogScrollingEvent>(connection, decoder, this, &WebPageProxy::logScrollingEvent);
    if (decoder.messageName() == Messages::WebPageProxy::EditorStateChanged::name())
        return IPC::handleMessage<Messages::WebPageProxy::EditorStateChanged>(connection, decoder, this, &WebPageProxy::editorStateChanged);
    if (decoder.messageName() == Messages::WebPageProxy::CompositionWasCanceled::name())
        return IPC::handleMessage<Messages::WebPageProxy::CompositionWasCanceled>(connection, decoder, this, &WebPageProxy::compositionWasCanceled);
    if (decoder.messageName() == Messages::WebPageProxy::SetHasHadSelectionChangesFromUserInteraction::name())
        return IPC::handleMessage<Messages::WebPageProxy::SetHasHadSelectionChangesFromUserInteraction>(connection, decoder, this, &WebPageProxy::setHasHadSelectionChangesFromUserInteraction);
#if ENABLE(IMAGE_ANALYSIS)
    if (decoder.messageName() == Messages::WebPageProxy::RequestTextRecognition::name())
        return IPC::handleMessageAsync<Messages::WebPageProxy::RequestTextRecognition>(connection, decoder, this, &WebPageProxy::requestTextRecognition);
#endif
#if HAVE(TRANSLATION_UI_SERVICES) && ENABLE(CONTEXT_MENUS)
    if (decoder.messageName() == Messages::WebPageProxy::HandleContextMenuTranslation::name())
        return IPC::handleMessage<Messages::WebPageProxy::HandleContextMenuTranslation>(connection, decoder, this, &WebPageProxy::handleContextMenuTranslation);
#endif
#if ENABLE(MEDIA_CONTROLS_CONTEXT_MENUS) && USE(UICONTEXTMENU)
    if (decoder.messageName() == Messages::WebPageProxy::ShowMediaControlsContextMenu::name())
        return IPC::handleMessageAsync<Messages::WebPageProxy::ShowMediaControlsContextMenu>(connection, decoder, this, &WebPageProxy::showMediaControlsContextMenu);
#endif
#if HAVE(TOUCH_BAR)
    if (decoder.messageName() == Messages::WebPageProxy::SetIsTouchBarUpdateSupressedForHiddenContentEditable::name())
        return IPC::handleMessage<Messages::WebPageProxy::SetIsTouchBarUpdateSupressedForHiddenContentEditable>(connection, decoder, this, &WebPageProxy::setIsTouchBarUpdateSupressedForHiddenContentEditable);
#endif
#if HAVE(TOUCH_BAR)
    if (decoder.messageName() == Messages::WebPageProxy::SetIsNeverRichlyEditableForTouchBar::name())
        return IPC::handleMessage<Messages::WebPageProxy::SetIsNeverRichlyEditableForTouchBar>(connection, decoder, this, &WebPageProxy::setIsNeverRichlyEditableForTouchBar);
#endif
    if (decoder.messageName() == Messages::WebPageProxy::DidCountStringMatches::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidCountStringMatches>(connection, decoder, this, &WebPageProxy::didCountStringMatches);
    if (decoder.messageName() == Messages::WebPageProxy::SetTextIndicator::name())
        return IPC::handleMessage<Messages::WebPageProxy::SetTextIndicator>(connection, decoder, this, &WebPageProxy::setTextIndicator);
    if (decoder.messageName() == Messages::WebPageProxy::ClearTextIndicator::name())
        return IPC::handleMessage<Messages::WebPageProxy::ClearTextIndicator>(connection, decoder, this, &WebPageProxy::clearTextIndicator);
    if (decoder.messageName() == Messages::WebPageProxy::DidFindString::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidFindString>(connection, decoder, this, &WebPageProxy::didFindString);
    if (decoder.messageName() == Messages::WebPageProxy::DidFailToFindString::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidFailToFindString>(connection, decoder, this, &WebPageProxy::didFailToFindString);
    if (decoder.messageName() == Messages::WebPageProxy::DidFindStringMatches::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidFindStringMatches>(connection, decoder, this, &WebPageProxy::didFindStringMatches);
    if (decoder.messageName() == Messages::WebPageProxy::DidGetImageForFindMatch::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidGetImageForFindMatch>(connection, decoder, this, &WebPageProxy::didGetImageForFindMatch);
    if (decoder.messageName() == Messages::WebPageProxy::ShowPopupMenu::name())
        return IPC::handleMessage<Messages::WebPageProxy::ShowPopupMenu>(connection, decoder, this, &WebPageProxy::showPopupMenu);
    if (decoder.messageName() == Messages::WebPageProxy::HidePopupMenu::name())
        return IPC::handleMessage<Messages::WebPageProxy::HidePopupMenu>(connection, decoder, this, &WebPageProxy::hidePopupMenu);
#if ENABLE(CONTEXT_MENUS)
    if (decoder.messageName() == Messages::WebPageProxy::ShowContextMenu::name())
        return IPC::handleMessage<Messages::WebPageProxy::ShowContextMenu>(connection, decoder, this, &WebPageProxy::showContextMenu);
#endif
#if ENABLE(SERVICE_WORKER)
    if (decoder.messageName() == Messages::WebPageProxy::DidFinishServiceWorkerPageRegistration::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidFinishServiceWorkerPageRegistration>(connection, decoder, this, &WebPageProxy::didFinishServiceWorkerPageRegistration);
#endif
    if (decoder.messageName() == Messages::WebPageProxy::RequestGeolocationPermissionForFrame::name())
        return IPC::handleMessage<Messages::WebPageProxy::RequestGeolocationPermissionForFrame>(connection, decoder, this, &WebPageProxy::requestGeolocationPermissionForFrame);
    if (decoder.messageName() == Messages::WebPageProxy::RevokeGeolocationAuthorizationToken::name())
        return IPC::handleMessage<Messages::WebPageProxy::RevokeGeolocationAuthorizationToken>(connection, decoder, this, &WebPageProxy::revokeGeolocationAuthorizationToken);
#if ENABLE(MEDIA_STREAM)
    if (decoder.messageName() == Messages::WebPageProxy::RequestUserMediaPermissionForFrame::name())
        return IPC::handleMessage<Messages::WebPageProxy::RequestUserMediaPermissionForFrame>(connection, decoder, this, &WebPageProxy::requestUserMediaPermissionForFrame);
#endif
#if ENABLE(MEDIA_STREAM)
    if (decoder.messageName() == Messages::WebPageProxy::EnumerateMediaDevicesForFrame::name())
        return IPC::handleMessageAsync<Messages::WebPageProxy::EnumerateMediaDevicesForFrame>(connection, decoder, this, &WebPageProxy::enumerateMediaDevicesForFrame);
#endif
#if ENABLE(MEDIA_STREAM)
    if (decoder.messageName() == Messages::WebPageProxy::BeginMonitoringCaptureDevices::name())
        return IPC::handleMessage<Messages::WebPageProxy::BeginMonitoringCaptureDevices>(connection, decoder, this, &WebPageProxy::beginMonitoringCaptureDevices);
#endif
#if ENABLE(ENCRYPTED_MEDIA)
    if (decoder.messageName() == Messages::WebPageProxy::RequestMediaKeySystemPermissionForFrame::name())
        return IPC::handleMessage<Messages::WebPageProxy::RequestMediaKeySystemPermissionForFrame>(connection, decoder, this, &WebPageProxy::requestMediaKeySystemPermissionForFrame);
#endif
    if (decoder.messageName() == Messages::WebPageProxy::RequestNotificationPermission::name())
        return IPC::handleMessageAsync<Messages::WebPageProxy::RequestNotificationPermission>(connection, decoder, this, &WebPageProxy::requestNotificationPermission);
    if (decoder.messageName() == Messages::WebPageProxy::UpdateSpellingUIWithMisspelledWord::name())
        return IPC::handleMessage<Messages::WebPageProxy::UpdateSpellingUIWithMisspelledWord>(connection, decoder, this, &WebPageProxy::updateSpellingUIWithMisspelledWord);
    if (decoder.messageName() == Messages::WebPageProxy::UpdateSpellingUIWithGrammarString::name())
        return IPC::handleMessage<Messages::WebPageProxy::UpdateSpellingUIWithGrammarString>(connection, decoder, this, &WebPageProxy::updateSpellingUIWithGrammarString);
    if (decoder.messageName() == Messages::WebPageProxy::LearnWord::name())
        return IPC::handleMessage<Messages::WebPageProxy::LearnWord>(connection, decoder, this, &WebPageProxy::learnWord);
    if (decoder.messageName() == Messages::WebPageProxy::IgnoreWord::name())
        return IPC::handleMessage<Messages::WebPageProxy::IgnoreWord>(connection, decoder, this, &WebPageProxy::ignoreWord);
    if (decoder.messageName() == Messages::WebPageProxy::RequestCheckingOfString::name())
        return IPC::handleMessage<Messages::WebPageProxy::RequestCheckingOfString>(connection, decoder, this, &WebPageProxy::requestCheckingOfString);
#if ENABLE(DRAG_SUPPORT)
    if (decoder.messageName() == Messages::WebPageProxy::DidPerformDragControllerAction::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidPerformDragControllerAction>(connection, decoder, this, &WebPageProxy::didPerformDragControllerAction);
#endif
#if ENABLE(DRAG_SUPPORT)
    if (decoder.messageName() == Messages::WebPageProxy::DidEndDragging::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidEndDragging>(connection, decoder, this, &WebPageProxy::didEndDragging);
#endif
#if PLATFORM(COCOA) && ENABLE(DRAG_SUPPORT)
    if (decoder.messageName() == Messages::WebPageProxy::StartDrag::name())
        return IPC::handleMessage<Messages::WebPageProxy::StartDrag>(connection, decoder, this, &WebPageProxy::startDrag);
#endif
#if PLATFORM(COCOA) && ENABLE(DRAG_SUPPORT)
    if (decoder.messageName() == Messages::WebPageProxy::SetPromisedDataForImage::name())
        return IPC::handleMessage<Messages::WebPageProxy::SetPromisedDataForImage>(connection, decoder, this, &WebPageProxy::setPromisedDataForImage);
#endif
#if PLATFORM(GTK) && ENABLE(DRAG_SUPPORT)
    if (decoder.messageName() == Messages::WebPageProxy::StartDrag::name())
        return IPC::handleMessage<Messages::WebPageProxy::StartDrag>(connection, decoder, this, &WebPageProxy::startDrag);
#endif
#if ENABLE(DRAG_SUPPORT)
    if (decoder.messageName() == Messages::WebPageProxy::DidPerformDragOperation::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidPerformDragOperation>(connection, decoder, this, &WebPageProxy::didPerformDragOperation);
#endif
#if PLATFORM(IOS_FAMILY) && ENABLE(DRAG_SUPPORT)
    if (decoder.messageName() == Messages::WebPageProxy::DidHandleDragStartRequest::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidHandleDragStartRequest>(connection, decoder, this, &WebPageProxy::didHandleDragStartRequest);
#endif
#if PLATFORM(IOS_FAMILY) && ENABLE(DRAG_SUPPORT)
    if (decoder.messageName() == Messages::WebPageProxy::DidHandleAdditionalDragItemsRequest::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidHandleAdditionalDragItemsRequest>(connection, decoder, this, &WebPageProxy::didHandleAdditionalDragItemsRequest);
#endif
#if PLATFORM(IOS_FAMILY) && ENABLE(DRAG_SUPPORT)
    if (decoder.messageName() == Messages::WebPageProxy::WillReceiveEditDragSnapshot::name())
        return IPC::handleMessage<Messages::WebPageProxy::WillReceiveEditDragSnapshot>(connection, decoder, this, &WebPageProxy::willReceiveEditDragSnapshot);
#endif
#if PLATFORM(IOS_FAMILY) && ENABLE(DRAG_SUPPORT)
    if (decoder.messageName() == Messages::WebPageProxy::DidReceiveEditDragSnapshot::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidReceiveEditDragSnapshot>(connection, decoder, this, &WebPageProxy::didReceiveEditDragSnapshot);
#endif
#if PLATFORM(COCOA)
    if (decoder.messageName() == Messages::WebPageProxy::DidPerformDictionaryLookup::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidPerformDictionaryLookup>(connection, decoder, this, &WebPageProxy::didPerformDictionaryLookup);
#endif
#if PLATFORM(COCOA)
    if (decoder.messageName() == Messages::WebPageProxy::RegisterWebProcessAccessibilityToken::name())
        return IPC::handleMessage<Messages::WebPageProxy::RegisterWebProcessAccessibilityToken>(connection, decoder, this, &WebPageProxy::registerWebProcessAccessibilityToken);
#endif
#if PLATFORM(COCOA)
    if (decoder.messageName() == Messages::WebPageProxy::Speak::name())
        return IPC::handleMessage<Messages::WebPageProxy::Speak>(connection, decoder, this, &WebPageProxy::speak);
#endif
#if PLATFORM(COCOA)
    if (decoder.messageName() == Messages::WebPageProxy::StopSpeaking::name())
        return IPC::handleMessage<Messages::WebPageProxy::StopSpeaking>(connection, decoder, this, &WebPageProxy::stopSpeaking);
#endif
#if PLATFORM(COCOA)
    if (decoder.messageName() == Messages::WebPageProxy::MakeFirstResponder::name())
        return IPC::handleMessage<Messages::WebPageProxy::MakeFirstResponder>(connection, decoder, this, &WebPageProxy::makeFirstResponder);
#endif
#if PLATFORM(COCOA)
    if (decoder.messageName() == Messages::WebPageProxy::AssistiveTechnologyMakeFirstResponder::name())
        return IPC::handleMessage<Messages::WebPageProxy::AssistiveTechnologyMakeFirstResponder>(connection, decoder, this, &WebPageProxy::assistiveTechnologyMakeFirstResponder);
#endif
#if PLATFORM(COCOA)
    if (decoder.messageName() == Messages::WebPageProxy::SearchWithSpotlight::name())
        return IPC::handleMessage<Messages::WebPageProxy::SearchWithSpotlight>(connection, decoder, this, &WebPageProxy::searchWithSpotlight);
#endif
#if PLATFORM(COCOA)
    if (decoder.messageName() == Messages::WebPageProxy::SearchTheWeb::name())
        return IPC::handleMessage<Messages::WebPageProxy::SearchTheWeb>(connection, decoder, this, &WebPageProxy::searchTheWeb);
#endif
#if HAVE(TOUCH_BAR)
    if (decoder.messageName() == Messages::WebPageProxy::TouchBarMenuDataChanged::name())
        return IPC::handleMessage<Messages::WebPageProxy::TouchBarMenuDataChanged>(connection, decoder, this, &WebPageProxy::touchBarMenuDataChanged);
#endif
#if HAVE(TOUCH_BAR)
    if (decoder.messageName() == Messages::WebPageProxy::TouchBarMenuItemDataAdded::name())
        return IPC::handleMessage<Messages::WebPageProxy::TouchBarMenuItemDataAdded>(connection, decoder, this, &WebPageProxy::touchBarMenuItemDataAdded);
#endif
#if HAVE(TOUCH_BAR)
    if (decoder.messageName() == Messages::WebPageProxy::TouchBarMenuItemDataRemoved::name())
        return IPC::handleMessage<Messages::WebPageProxy::TouchBarMenuItemDataRemoved>(connection, decoder, this, &WebPageProxy::touchBarMenuItemDataRemoved);
#endif
#if USE(AUTOMATIC_TEXT_REPLACEMENT)
    if (decoder.messageName() == Messages::WebPageProxy::toggleSmartInsertDelete::name())
        return IPC::handleMessage<Messages::WebPageProxy::toggleSmartInsertDelete>(connection, decoder, this, &WebPageProxy::toggleSmartInsertDelete);
#endif
#if USE(AUTOMATIC_TEXT_REPLACEMENT)
    if (decoder.messageName() == Messages::WebPageProxy::toggleAutomaticQuoteSubstitution::name())
        return IPC::handleMessage<Messages::WebPageProxy::toggleAutomaticQuoteSubstitution>(connection, decoder, this, &WebPageProxy::toggleAutomaticQuoteSubstitution);
#endif
#if USE(AUTOMATIC_TEXT_REPLACEMENT)
    if (decoder.messageName() == Messages::WebPageProxy::toggleAutomaticLinkDetection::name())
        return IPC::handleMessage<Messages::WebPageProxy::toggleAutomaticLinkDetection>(connection, decoder, this, &WebPageProxy::toggleAutomaticLinkDetection);
#endif
#if USE(AUTOMATIC_TEXT_REPLACEMENT)
    if (decoder.messageName() == Messages::WebPageProxy::toggleAutomaticDashSubstitution::name())
        return IPC::handleMessage<Messages::WebPageProxy::toggleAutomaticDashSubstitution>(connection, decoder, this, &WebPageProxy::toggleAutomaticDashSubstitution);
#endif
#if USE(AUTOMATIC_TEXT_REPLACEMENT)
    if (decoder.messageName() == Messages::WebPageProxy::toggleAutomaticTextReplacement::name())
        return IPC::handleMessage<Messages::WebPageProxy::toggleAutomaticTextReplacement>(connection, decoder, this, &WebPageProxy::toggleAutomaticTextReplacement);
#endif
#if PLATFORM(MAC)
    if (decoder.messageName() == Messages::WebPageProxy::ShowCorrectionPanel::name())
        return IPC::handleMessage<Messages::WebPageProxy::ShowCorrectionPanel>(connection, decoder, this, &WebPageProxy::showCorrectionPanel);
#endif
#if PLATFORM(MAC)
    if (decoder.messageName() == Messages::WebPageProxy::DismissCorrectionPanel::name())
        return IPC::handleMessage<Messages::WebPageProxy::DismissCorrectionPanel>(connection, decoder, this, &WebPageProxy::dismissCorrectionPanel);
#endif
#if PLATFORM(MAC)
    if (decoder.messageName() == Messages::WebPageProxy::RecordAutocorrectionResponse::name())
        return IPC::handleMessage<Messages::WebPageProxy::RecordAutocorrectionResponse>(connection, decoder, this, &WebPageProxy::recordAutocorrectionResponse);
#endif
#if PLATFORM(MAC)
    if (decoder.messageName() == Messages::WebPageProxy::SetEditableElementIsFocused::name())
        return IPC::handleMessage<Messages::WebPageProxy::SetEditableElementIsFocused>(connection, decoder, this, &WebPageProxy::setEditableElementIsFocused);
#endif
#if PLATFORM(MAC)
    if (decoder.messageName() == Messages::WebPageProxy::HandleAcceptsFirstMouse::name())
        return IPC::handleMessage<Messages::WebPageProxy::HandleAcceptsFirstMouse>(connection, decoder, this, &WebPageProxy::handleAcceptsFirstMouse);
#endif
    if (decoder.messageName() == Messages::WebPageProxy::DidUpdateRenderingAfterCommittingLoad::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidUpdateRenderingAfterCommittingLoad>(connection, decoder, this, &WebPageProxy::didUpdateRenderingAfterCommittingLoad);
#if USE(DICTATION_ALTERNATIVES)
    if (decoder.messageName() == Messages::WebPageProxy::ShowDictationAlternativeUI::name())
        return IPC::handleMessage<Messages::WebPageProxy::ShowDictationAlternativeUI>(connection, decoder, this, &WebPageProxy::showDictationAlternativeUI);
#endif
#if USE(DICTATION_ALTERNATIVES)
    if (decoder.messageName() == Messages::WebPageProxy::RemoveDictationAlternatives::name())
        return IPC::handleMessage<Messages::WebPageProxy::RemoveDictationAlternatives>(connection, decoder, this, &WebPageProxy::removeDictationAlternatives);
#endif
#if PLATFORM(IOS_FAMILY)
    if (decoder.messageName() == Messages::WebPageProxy::CouldNotRestorePageState::name())
        return IPC::handleMessage<Messages::WebPageProxy::CouldNotRestorePageState>(connection, decoder, this, &WebPageProxy::couldNotRestorePageState);
#endif
#if PLATFORM(IOS_FAMILY)
    if (decoder.messageName() == Messages::WebPageProxy::RestorePageState::name())
        return IPC::handleMessage<Messages::WebPageProxy::RestorePageState>(connection, decoder, this, &WebPageProxy::restorePageState);
#endif
#if PLATFORM(IOS_FAMILY)
    if (decoder.messageName() == Messages::WebPageProxy::RestorePageCenterAndScale::name())
        return IPC::handleMessage<Messages::WebPageProxy::RestorePageCenterAndScale>(connection, decoder, this, &WebPageProxy::restorePageCenterAndScale);
#endif
#if PLATFORM(IOS_FAMILY)
    if (decoder.messageName() == Messages::WebPageProxy::DidGetTapHighlightGeometries::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidGetTapHighlightGeometries>(connection, decoder, this, &WebPageProxy::didGetTapHighlightGeometries);
#endif
#if PLATFORM(IOS_FAMILY)
    if (decoder.messageName() == Messages::WebPageProxy::ElementDidFocus::name())
        return IPC::handleMessage<Messages::WebPageProxy::ElementDidFocus>(connection, decoder, this, &WebPageProxy::elementDidFocus);
#endif
#if PLATFORM(IOS_FAMILY)
    if (decoder.messageName() == Messages::WebPageProxy::ElementDidBlur::name())
        return IPC::handleMessage<Messages::WebPageProxy::ElementDidBlur>(connection, decoder, this, &WebPageProxy::elementDidBlur);
#endif
#if PLATFORM(IOS_FAMILY)
    if (decoder.messageName() == Messages::WebPageProxy::UpdateInputContextAfterBlurringAndRefocusingElement::name())
        return IPC::handleMessage<Messages::WebPageProxy::UpdateInputContextAfterBlurringAndRefocusingElement>(connection, decoder, this, &WebPageProxy::updateInputContextAfterBlurringAndRefocusingElement);
#endif
#if PLATFORM(IOS_FAMILY)
    if (decoder.messageName() == Messages::WebPageProxy::FocusedElementDidChangeInputMode::name())
        return IPC::handleMessage<Messages::WebPageProxy::FocusedElementDidChangeInputMode>(connection, decoder, this, &WebPageProxy::focusedElementDidChangeInputMode);
#endif
#if PLATFORM(IOS_FAMILY)
    if (decoder.messageName() == Messages::WebPageProxy::ScrollingNodeScrollWillStartScroll::name())
        return IPC::handleMessage<Messages::WebPageProxy::ScrollingNodeScrollWillStartScroll>(connection, decoder, this, &WebPageProxy::scrollingNodeScrollWillStartScroll);
#endif
#if PLATFORM(IOS_FAMILY)
    if (decoder.messageName() == Messages::WebPageProxy::ScrollingNodeScrollDidEndScroll::name())
        return IPC::handleMessage<Messages::WebPageProxy::ScrollingNodeScrollDidEndScroll>(connection, decoder, this, &WebPageProxy::scrollingNodeScrollDidEndScroll);
#endif
#if PLATFORM(IOS_FAMILY)
    if (decoder.messageName() == Messages::WebPageProxy::ShowInspectorHighlight::name())
        return IPC::handleMessage<Messages::WebPageProxy::ShowInspectorHighlight>(connection, decoder, this, &WebPageProxy::showInspectorHighlight);
#endif
#if PLATFORM(IOS_FAMILY)
    if (decoder.messageName() == Messages::WebPageProxy::HideInspectorHighlight::name())
        return IPC::handleMessage<Messages::WebPageProxy::HideInspectorHighlight>(connection, decoder, this, &WebPageProxy::hideInspectorHighlight);
#endif
#if PLATFORM(IOS_FAMILY)
    if (decoder.messageName() == Messages::WebPageProxy::ShowInspectorIndication::name())
        return IPC::handleMessage<Messages::WebPageProxy::ShowInspectorIndication>(connection, decoder, this, &WebPageProxy::showInspectorIndication);
#endif
#if PLATFORM(IOS_FAMILY)
    if (decoder.messageName() == Messages::WebPageProxy::HideInspectorIndication::name())
        return IPC::handleMessage<Messages::WebPageProxy::HideInspectorIndication>(connection, decoder, this, &WebPageProxy::hideInspectorIndication);
#endif
#if PLATFORM(IOS_FAMILY)
    if (decoder.messageName() == Messages::WebPageProxy::EnableInspectorNodeSearch::name())
        return IPC::handleMessage<Messages::WebPageProxy::EnableInspectorNodeSearch>(connection, decoder, this, &WebPageProxy::enableInspectorNodeSearch);
#endif
#if PLATFORM(IOS_FAMILY)
    if (decoder.messageName() == Messages::WebPageProxy::DisableInspectorNodeSearch::name())
        return IPC::handleMessage<Messages::WebPageProxy::DisableInspectorNodeSearch>(connection, decoder, this, &WebPageProxy::disableInspectorNodeSearch);
#endif
#if PLATFORM(IOS_FAMILY)
    if (decoder.messageName() == Messages::WebPageProxy::UpdateStringForFind::name())
        return IPC::handleMessage<Messages::WebPageProxy::UpdateStringForFind>(connection, decoder, this, &WebPageProxy::updateStringForFind);
#endif
#if PLATFORM(IOS_FAMILY)
    if (decoder.messageName() == Messages::WebPageProxy::HandleAutocorrectionContext::name())
        return IPC::handleMessage<Messages::WebPageProxy::HandleAutocorrectionContext>(connection, decoder, this, &WebPageProxy::handleAutocorrectionContext);
#endif
#if PLATFORM(IOS_FAMILY)
    if (decoder.messageName() == Messages::WebPageProxy::ShowDataDetectorsUIForPositionInformation::name())
        return IPC::handleMessage<Messages::WebPageProxy::ShowDataDetectorsUIForPositionInformation>(connection, decoder, this, &WebPageProxy::showDataDetectorsUIForPositionInformation);
#endif
    if (decoder.messageName() == Messages::WebPageProxy::DidChangeInspectorFrontendCount::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidChangeInspectorFrontendCount>(connection, decoder, this, &WebPageProxy::didChangeInspectorFrontendCount);
    if (decoder.messageName() == Messages::WebPageProxy::CreateInspectorTarget::name())
        return IPC::handleMessage<Messages::WebPageProxy::CreateInspectorTarget>(connection, decoder, this, &WebPageProxy::createInspectorTarget);
    if (decoder.messageName() == Messages::WebPageProxy::DestroyInspectorTarget::name())
        return IPC::handleMessage<Messages::WebPageProxy::DestroyInspectorTarget>(connection, decoder, this, &WebPageProxy::destroyInspectorTarget);
    if (decoder.messageName() == Messages::WebPageProxy::SendMessageToInspectorFrontend::name())
        return IPC::handleMessage<Messages::WebPageProxy::SendMessageToInspectorFrontend>(connection, decoder, this, &WebPageProxy::sendMessageToInspectorFrontend);
    if (decoder.messageName() == Messages::WebPageProxy::SaveRecentSearches::name())
        return IPC::handleMessage<Messages::WebPageProxy::SaveRecentSearches>(connection, decoder, this, &WebPageProxy::saveRecentSearches);
#if ENABLE(PDFKIT_PLUGIN) && !ENABLE(UI_PROCESS_PDF_HUD)
    if (decoder.messageName() == Messages::WebPageProxy::SavePDFToFileInDownloadsFolder::name())
        return IPC::handleMessage<Messages::WebPageProxy::SavePDFToFileInDownloadsFolder>(connection, decoder, this, &WebPageProxy::savePDFToFileInDownloadsFolder);
#endif
#if ENABLE(PDFKIT_PLUGIN) && !ENABLE(UI_PROCESS_PDF_HUD)
    if (decoder.messageName() == Messages::WebPageProxy::SavePDFToTemporaryFolderAndOpenWithNativeApplication::name())
        return IPC::handleMessage<Messages::WebPageProxy::SavePDFToTemporaryFolderAndOpenWithNativeApplication>(connection, decoder, this, &WebPageProxy::savePDFToTemporaryFolderAndOpenWithNativeApplication);
#endif
#if ENABLE(PDFKIT_PLUGIN) && !ENABLE(UI_PROCESS_PDF_HUD)
    if (decoder.messageName() == Messages::WebPageProxy::OpenPDFFromTemporaryFolderWithNativeApplication::name())
        return IPC::handleMessage<Messages::WebPageProxy::OpenPDFFromTemporaryFolderWithNativeApplication>(connection, decoder, this, &WebPageProxy::openPDFFromTemporaryFolderWithNativeApplication);
#endif
    if (decoder.messageName() == Messages::WebPageProxy::DidUpdateActivityState::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidUpdateActivityState>(connection, decoder, this, &WebPageProxy::didUpdateActivityState);
#if (ENABLE(TELEPHONE_NUMBER_DETECTION) && PLATFORM(MAC))
    if (decoder.messageName() == Messages::WebPageProxy::ShowTelephoneNumberMenu::name())
        return IPC::handleMessage<Messages::WebPageProxy::ShowTelephoneNumberMenu>(connection, decoder, this, &WebPageProxy::showTelephoneNumberMenu);
#endif
#if USE(QUICK_LOOK)
    if (decoder.messageName() == Messages::WebPageProxy::DidStartLoadForQuickLookDocumentInMainFrame::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidStartLoadForQuickLookDocumentInMainFrame>(connection, decoder, this, &WebPageProxy::didStartLoadForQuickLookDocumentInMainFrame);
#endif
#if USE(QUICK_LOOK)
    if (decoder.messageName() == Messages::WebPageProxy::DidFinishLoadForQuickLookDocumentInMainFrame::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidFinishLoadForQuickLookDocumentInMainFrame>(connection, decoder, this, &WebPageProxy::didFinishLoadForQuickLookDocumentInMainFrame);
#endif
#if USE(QUICK_LOOK)
    if (decoder.messageName() == Messages::WebPageProxy::RequestPasswordForQuickLookDocumentInMainFrame::name())
        return IPC::handleMessageAsync<Messages::WebPageProxy::RequestPasswordForQuickLookDocumentInMainFrame>(connection, decoder, this, &WebPageProxy::requestPasswordForQuickLookDocumentInMainFrame);
#endif
#if ENABLE(CONTENT_FILTERING)
    if (decoder.messageName() == Messages::WebPageProxy::ContentFilterDidBlockLoadForFrame::name())
        return IPC::handleMessage<Messages::WebPageProxy::ContentFilterDidBlockLoadForFrame>(connection, decoder, this, &WebPageProxy::contentFilterDidBlockLoadForFrame);
#endif
    if (decoder.messageName() == Messages::WebPageProxy::IsPlayingMediaDidChange::name())
        return IPC::handleMessage<Messages::WebPageProxy::IsPlayingMediaDidChange>(connection, decoder, this, &WebPageProxy::isPlayingMediaDidChange);
    if (decoder.messageName() == Messages::WebPageProxy::HandleAutoplayEvent::name())
        return IPC::handleMessage<Messages::WebPageProxy::HandleAutoplayEvent>(connection, decoder, this, &WebPageProxy::handleAutoplayEvent);
#if PLATFORM(MAC)
    if (decoder.messageName() == Messages::WebPageProxy::DidPerformImmediateActionHitTest::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidPerformImmediateActionHitTest>(connection, decoder, this, &WebPageProxy::didPerformImmediateActionHitTest);
#endif
    if (decoder.messageName() == Messages::WebPageProxy::HandleMessage::name())
        return IPC::handleMessageWantsConnection<Messages::WebPageProxy::HandleMessage>(connection, decoder, this, &WebPageProxy::handleMessage);
    if (decoder.messageName() == Messages::WebPageProxy::HandleAutoFillButtonClick::name())
        return IPC::handleMessage<Messages::WebPageProxy::HandleAutoFillButtonClick>(connection, decoder, this, &WebPageProxy::handleAutoFillButtonClick);
    if (decoder.messageName() == Messages::WebPageProxy::DidResignInputElementStrongPasswordAppearance::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidResignInputElementStrongPasswordAppearance>(connection, decoder, this, &WebPageProxy::didResignInputElementStrongPasswordAppearance);
#if ENABLE(CONTENT_EXTENSIONS)
    if (decoder.messageName() == Messages::WebPageProxy::ContentRuleListNotification::name())
        return IPC::handleMessage<Messages::WebPageProxy::ContentRuleListNotification>(connection, decoder, this, &WebPageProxy::contentRuleListNotification);
#endif
#if ENABLE(WIRELESS_PLAYBACK_TARGET) && !PLATFORM(IOS_FAMILY)
    if (decoder.messageName() == Messages::WebPageProxy::AddPlaybackTargetPickerClient::name())
        return IPC::handleMessage<Messages::WebPageProxy::AddPlaybackTargetPickerClient>(connection, decoder, this, &WebPageProxy::addPlaybackTargetPickerClient);
#endif
#if ENABLE(WIRELESS_PLAYBACK_TARGET) && !PLATFORM(IOS_FAMILY)
    if (decoder.messageName() == Messages::WebPageProxy::RemovePlaybackTargetPickerClient::name())
        return IPC::handleMessage<Messages::WebPageProxy::RemovePlaybackTargetPickerClient>(connection, decoder, this, &WebPageProxy::removePlaybackTargetPickerClient);
#endif
#if ENABLE(WIRELESS_PLAYBACK_TARGET) && !PLATFORM(IOS_FAMILY)
    if (decoder.messageName() == Messages::WebPageProxy::ShowPlaybackTargetPicker::name())
        return IPC::handleMessage<Messages::WebPageProxy::ShowPlaybackTargetPicker>(connection, decoder, this, &WebPageProxy::showPlaybackTargetPicker);
#endif
#if ENABLE(WIRELESS_PLAYBACK_TARGET) && !PLATFORM(IOS_FAMILY)
    if (decoder.messageName() == Messages::WebPageProxy::PlaybackTargetPickerClientStateDidChange::name())
        return IPC::handleMessage<Messages::WebPageProxy::PlaybackTargetPickerClientStateDidChange>(connection, decoder, this, &WebPageProxy::playbackTargetPickerClientStateDidChange);
#endif
#if ENABLE(WIRELESS_PLAYBACK_TARGET) && !PLATFORM(IOS_FAMILY)
    if (decoder.messageName() == Messages::WebPageProxy::SetMockMediaPlaybackTargetPickerEnabled::name())
        return IPC::handleMessage<Messages::WebPageProxy::SetMockMediaPlaybackTargetPickerEnabled>(connection, decoder, this, &WebPageProxy::setMockMediaPlaybackTargetPickerEnabled);
#endif
#if ENABLE(WIRELESS_PLAYBACK_TARGET) && !PLATFORM(IOS_FAMILY)
    if (decoder.messageName() == Messages::WebPageProxy::SetMockMediaPlaybackTargetPickerState::name())
        return IPC::handleMessage<Messages::WebPageProxy::SetMockMediaPlaybackTargetPickerState>(connection, decoder, this, &WebPageProxy::setMockMediaPlaybackTargetPickerState);
#endif
#if ENABLE(WIRELESS_PLAYBACK_TARGET) && !PLATFORM(IOS_FAMILY)
    if (decoder.messageName() == Messages::WebPageProxy::MockMediaPlaybackTargetPickerDismissPopup::name())
        return IPC::handleMessage<Messages::WebPageProxy::MockMediaPlaybackTargetPickerDismissPopup>(connection, decoder, this, &WebPageProxy::mockMediaPlaybackTargetPickerDismissPopup);
#endif
#if ENABLE(VIDEO_PRESENTATION_MODE)
    if (decoder.messageName() == Messages::WebPageProxy::SetMockVideoPresentationModeEnabled::name())
        return IPC::handleMessage<Messages::WebPageProxy::SetMockVideoPresentationModeEnabled>(connection, decoder, this, &WebPageProxy::setMockVideoPresentationModeEnabled);
#endif
#if ENABLE(POINTER_LOCK)
    if (decoder.messageName() == Messages::WebPageProxy::RequestPointerLock::name())
        return IPC::handleMessage<Messages::WebPageProxy::RequestPointerLock>(connection, decoder, this, &WebPageProxy::requestPointerLock);
#endif
#if ENABLE(POINTER_LOCK)
    if (decoder.messageName() == Messages::WebPageProxy::RequestPointerUnlock::name())
        return IPC::handleMessage<Messages::WebPageProxy::RequestPointerUnlock>(connection, decoder, this, &WebPageProxy::requestPointerUnlock);
#endif
    if (decoder.messageName() == Messages::WebPageProxy::DidFailToSuspendAfterProcessSwap::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidFailToSuspendAfterProcessSwap>(connection, decoder, this, &WebPageProxy::didFailToSuspendAfterProcessSwap);
    if (decoder.messageName() == Messages::WebPageProxy::DidSuspendAfterProcessSwap::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidSuspendAfterProcessSwap>(connection, decoder, this, &WebPageProxy::didSuspendAfterProcessSwap);
    if (decoder.messageName() == Messages::WebPageProxy::ImageOrMediaDocumentSizeChanged::name())
        return IPC::handleMessage<Messages::WebPageProxy::ImageOrMediaDocumentSizeChanged>(connection, decoder, this, &WebPageProxy::imageOrMediaDocumentSizeChanged);
    if (decoder.messageName() == Messages::WebPageProxy::UseFixedLayoutDidChange::name())
        return IPC::handleMessage<Messages::WebPageProxy::UseFixedLayoutDidChange>(connection, decoder, this, &WebPageProxy::useFixedLayoutDidChange);
    if (decoder.messageName() == Messages::WebPageProxy::FixedLayoutSizeDidChange::name())
        return IPC::handleMessage<Messages::WebPageProxy::FixedLayoutSizeDidChange>(connection, decoder, this, &WebPageProxy::fixedLayoutSizeDidChange);
#if ENABLE(VIDEO) && USE(GSTREAMER)
    if (decoder.messageName() == Messages::WebPageProxy::RequestInstallMissingMediaPlugins::name())
        return IPC::handleMessage<Messages::WebPageProxy::RequestInstallMissingMediaPlugins>(connection, decoder, this, &WebPageProxy::requestInstallMissingMediaPlugins);
#endif
    if (decoder.messageName() == Messages::WebPageProxy::DidRestoreScrollPosition::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidRestoreScrollPosition>(connection, decoder, this, &WebPageProxy::didRestoreScrollPosition);
    if (decoder.messageName() == Messages::WebPageProxy::RequestScrollToRect::name())
        return IPC::handleMessage<Messages::WebPageProxy::RequestScrollToRect>(connection, decoder, this, &WebPageProxy::requestScrollToRect);
    if (decoder.messageName() == Messages::WebPageProxy::GetLoadDecisionForIcon::name())
        return IPC::handleMessage<Messages::WebPageProxy::GetLoadDecisionForIcon>(connection, decoder, this, &WebPageProxy::getLoadDecisionForIcon);
#if PLATFORM(MAC)
    if (decoder.messageName() == Messages::WebPageProxy::DidHandleAcceptedCandidate::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidHandleAcceptedCandidate>(connection, decoder, this, &WebPageProxy::didHandleAcceptedCandidate);
#endif
    if (decoder.messageName() == Messages::WebPageProxy::SetIsUsingHighPerformanceWebGL::name())
        return IPC::handleMessage<Messages::WebPageProxy::SetIsUsingHighPerformanceWebGL>(connection, decoder, this, &WebPageProxy::setIsUsingHighPerformanceWebGL);
    if (decoder.messageName() == Messages::WebPageProxy::StartURLSchemeTask::name())
        return IPC::handleMessage<Messages::WebPageProxy::StartURLSchemeTask>(connection, decoder, this, &WebPageProxy::startURLSchemeTask);
    if (decoder.messageName() == Messages::WebPageProxy::StopURLSchemeTask::name())
        return IPC::handleMessage<Messages::WebPageProxy::StopURLSchemeTask>(connection, decoder, this, &WebPageProxy::stopURLSchemeTask);
#if ENABLE(DEVICE_ORIENTATION)
    if (decoder.messageName() == Messages::WebPageProxy::ShouldAllowDeviceOrientationAndMotionAccess::name())
        return IPC::handleMessageAsync<Messages::WebPageProxy::ShouldAllowDeviceOrientationAndMotionAccess>(connection, decoder, this, &WebPageProxy::shouldAllowDeviceOrientationAndMotionAccess);
#endif
#if ENABLE(ATTACHMENT_ELEMENT)
    if (decoder.messageName() == Messages::WebPageProxy::RegisterAttachmentIdentifierFromData::name())
        return IPC::handleMessage<Messages::WebPageProxy::RegisterAttachmentIdentifierFromData>(connection, decoder, this, &WebPageProxy::registerAttachmentIdentifierFromData);
#endif
#if ENABLE(ATTACHMENT_ELEMENT)
    if (decoder.messageName() == Messages::WebPageProxy::RegisterAttachmentIdentifierFromFilePath::name())
        return IPC::handleMessage<Messages::WebPageProxy::RegisterAttachmentIdentifierFromFilePath>(connection, decoder, this, &WebPageProxy::registerAttachmentIdentifierFromFilePath);
#endif
#if ENABLE(ATTACHMENT_ELEMENT)
    if (decoder.messageName() == Messages::WebPageProxy::RegisterAttachmentIdentifier::name())
        return IPC::handleMessage<Messages::WebPageProxy::RegisterAttachmentIdentifier>(connection, decoder, this, &WebPageProxy::registerAttachmentIdentifier);
#endif
#if ENABLE(ATTACHMENT_ELEMENT)
    if (decoder.messageName() == Messages::WebPageProxy::RegisterAttachmentsFromSerializedData::name())
        return IPC::handleMessage<Messages::WebPageProxy::RegisterAttachmentsFromSerializedData>(connection, decoder, this, &WebPageProxy::registerAttachmentsFromSerializedData);
#endif
#if ENABLE(ATTACHMENT_ELEMENT)
    if (decoder.messageName() == Messages::WebPageProxy::CloneAttachmentData::name())
        return IPC::handleMessage<Messages::WebPageProxy::CloneAttachmentData>(connection, decoder, this, &WebPageProxy::cloneAttachmentData);
#endif
#if ENABLE(ATTACHMENT_ELEMENT)
    if (decoder.messageName() == Messages::WebPageProxy::DidInsertAttachmentWithIdentifier::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidInsertAttachmentWithIdentifier>(connection, decoder, this, &WebPageProxy::didInsertAttachmentWithIdentifier);
#endif
#if ENABLE(ATTACHMENT_ELEMENT)
    if (decoder.messageName() == Messages::WebPageProxy::DidRemoveAttachmentWithIdentifier::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidRemoveAttachmentWithIdentifier>(connection, decoder, this, &WebPageProxy::didRemoveAttachmentWithIdentifier);
#endif
#if ENABLE(ATTACHMENT_ELEMENT)
    if (decoder.messageName() == Messages::WebPageProxy::WritePromisedAttachmentToPasteboard::name())
        return IPC::handleMessage<Messages::WebPageProxy::WritePromisedAttachmentToPasteboard>(connection, decoder, this, &WebPageProxy::writePromisedAttachmentToPasteboard);
#endif
#if ENABLE(ATTACHMENT_ELEMENT)
    if (decoder.messageName() == Messages::WebPageProxy::RequestAttachmentIcon::name())
        return IPC::handleMessage<Messages::WebPageProxy::RequestAttachmentIcon>(connection, decoder, this, &WebPageProxy::requestAttachmentIcon);
#endif
#if ENABLE(APP_HIGHLIGHTS)
    if (decoder.messageName() == Messages::WebPageProxy::StoreAppHighlight::name())
        return IPC::handleMessage<Messages::WebPageProxy::StoreAppHighlight>(connection, decoder, this, &WebPageProxy::storeAppHighlight);
#endif
#if ENABLE(SPEECH_SYNTHESIS)
    if (decoder.messageName() == Messages::WebPageProxy::SpeechSynthesisSpeak::name())
        return IPC::handleMessageAsync<Messages::WebPageProxy::SpeechSynthesisSpeak>(connection, decoder, this, &WebPageProxy::speechSynthesisSpeak);
#endif
#if ENABLE(SPEECH_SYNTHESIS)
    if (decoder.messageName() == Messages::WebPageProxy::SpeechSynthesisSetFinishedCallback::name())
        return IPC::handleMessageAsync<Messages::WebPageProxy::SpeechSynthesisSetFinishedCallback>(connection, decoder, this, &WebPageProxy::speechSynthesisSetFinishedCallback);
#endif
#if ENABLE(SPEECH_SYNTHESIS)
    if (decoder.messageName() == Messages::WebPageProxy::SpeechSynthesisCancel::name())
        return IPC::handleMessage<Messages::WebPageProxy::SpeechSynthesisCancel>(connection, decoder, this, &WebPageProxy::speechSynthesisCancel);
#endif
#if ENABLE(SPEECH_SYNTHESIS)
    if (decoder.messageName() == Messages::WebPageProxy::SpeechSynthesisPause::name())
        return IPC::handleMessageAsync<Messages::WebPageProxy::SpeechSynthesisPause>(connection, decoder, this, &WebPageProxy::speechSynthesisPause);
#endif
#if ENABLE(SPEECH_SYNTHESIS)
    if (decoder.messageName() == Messages::WebPageProxy::SpeechSynthesisResume::name())
        return IPC::handleMessageAsync<Messages::WebPageProxy::SpeechSynthesisResume>(connection, decoder, this, &WebPageProxy::speechSynthesisResume);
#endif
#if ENABLE(SPEECH_SYNTHESIS)
    if (decoder.messageName() == Messages::WebPageProxy::SpeechSynthesisResetState::name())
        return IPC::handleMessage<Messages::WebPageProxy::SpeechSynthesisResetState>(connection, decoder, this, &WebPageProxy::speechSynthesisResetState);
#endif
#if ENABLE(UI_PROCESS_PDF_HUD)
    if (decoder.messageName() == Messages::WebPageProxy::CreatePDFHUD::name())
        return IPC::handleMessage<Messages::WebPageProxy::CreatePDFHUD>(connection, decoder, this, &WebPageProxy::createPDFHUD);
#endif
#if ENABLE(UI_PROCESS_PDF_HUD)
    if (decoder.messageName() == Messages::WebPageProxy::UpdatePDFHUDLocation::name())
        return IPC::handleMessage<Messages::WebPageProxy::UpdatePDFHUDLocation>(connection, decoder, this, &WebPageProxy::updatePDFHUDLocation);
#endif
#if ENABLE(UI_PROCESS_PDF_HUD)
    if (decoder.messageName() == Messages::WebPageProxy::RemovePDFHUD::name())
        return IPC::handleMessage<Messages::WebPageProxy::RemovePDFHUD>(connection, decoder, this, &WebPageProxy::removePDFHUD);
#endif
    if (decoder.messageName() == Messages::WebPageProxy::ConfigureLoggingChannel::name())
        return IPC::handleMessage<Messages::WebPageProxy::ConfigureLoggingChannel>(connection, decoder, this, &WebPageProxy::configureLoggingChannel);
#if PLATFORM(GTK)
    if (decoder.messageName() == Messages::WebPageProxy::ShowEmojiPicker::name())
        return IPC::handleMessageAsync<Messages::WebPageProxy::ShowEmojiPicker>(connection, decoder, this, &WebPageProxy::showEmojiPicker);
#endif
#if HAVE(VISIBILITY_PROPAGATION_VIEW)
    if (decoder.messageName() == Messages::WebPageProxy::DidCreateContextInWebProcessForVisibilityPropagation::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidCreateContextInWebProcessForVisibilityPropagation>(connection, decoder, this, &WebPageProxy::didCreateContextInWebProcessForVisibilityPropagation);
#endif
#if ENABLE(WEB_AUTHN)
    if (decoder.messageName() == Messages::WebPageProxy::SetMockWebAuthenticationConfiguration::name())
        return IPC::handleMessage<Messages::WebPageProxy::SetMockWebAuthenticationConfiguration>(connection, decoder, this, &WebPageProxy::setMockWebAuthenticationConfiguration);
#endif
#if PLATFORM(GTK) || PLATFORM(WPE)
    if (decoder.messageName() == Messages::WebPageProxy::SendMessageToWebView::name())
        return IPC::handleMessage<Messages::WebPageProxy::SendMessageToWebView>(connection, decoder, this, &WebPageProxy::sendMessageToWebView);
#endif
#if PLATFORM(GTK) || PLATFORM(WPE)
    if (decoder.messageName() == Messages::WebPageProxy::SendMessageToWebViewWithReply::name())
        return IPC::handleMessageAsync<Messages::WebPageProxy::SendMessageToWebViewWithReply>(connection, decoder, this, &WebPageProxy::sendMessageToWebViewWithReply);
#endif
    if (decoder.messageName() == Messages::WebPageProxy::DidFindTextManipulationItems::name())
        return IPC::handleMessage<Messages::WebPageProxy::DidFindTextManipulationItems>(connection, decoder, this, &WebPageProxy::didFindTextManipulationItems);
#if ENABLE(MEDIA_USAGE)
    if (decoder.messageName() == Messages::WebPageProxy::AddMediaUsageManagerSession::name())
        return IPC::handleMessage<Messages::WebPageProxy::AddMediaUsageManagerSession>(connection, decoder, this, &WebPageProxy::addMediaUsageManagerSession);
#endif
#if ENABLE(MEDIA_USAGE)
    if (decoder.messageName() == Messages::WebPageProxy::UpdateMediaUsageManagerSessionState::name())
        return IPC::handleMessage<Messages::WebPageProxy::UpdateMediaUsageManagerSessionState>(connection, decoder, this, &WebPageProxy::updateMediaUsageManagerSessionState);
#endif
#if ENABLE(MEDIA_USAGE)
    if (decoder.messageName() == Messages::WebPageProxy::RemoveMediaUsageManagerSession::name())
        return IPC::handleMessage<Messages::WebPageProxy::RemoveMediaUsageManagerSession>(connection, decoder, this, &WebPageProxy::removeMediaUsageManagerSession);
#endif
    if (decoder.messageName() == Messages::WebPageProxy::SetHasExecutedAppBoundBehaviorBeforeNavigation::name())
        return IPC::handleMessage<Messages::WebPageProxy::SetHasExecutedAppBoundBehaviorBeforeNavigation>(connection, decoder, this, &WebPageProxy::setHasExecutedAppBoundBehaviorBeforeNavigation);
#if PLATFORM(MAC)
    if (decoder.messageName() == Messages::WebPageProxy::ChangeUniversalAccessZoomFocus::name())
        return IPC::handleMessage<Messages::WebPageProxy::ChangeUniversalAccessZoomFocus>(connection, decoder, this, &WebPageProxy::changeUniversalAccessZoomFocus);
#endif
#if ENABLE(ARKIT_INLINE_PREVIEW_IOS)
    if (decoder.messageName() == Messages::WebPageProxy::TakeModelElementFullscreen::name())
        return IPC::handleMessage<Messages::WebPageProxy::TakeModelElementFullscreen>(connection, decoder, this, &WebPageProxy::takeModelElementFullscreen);
#endif
#if ENABLE(ARKIT_INLINE_PREVIEW_IOS)
    if (decoder.messageName() == Messages::WebPageProxy::ModelElementSetInteractionEnabled::name())
        return IPC::handleMessage<Messages::WebPageProxy::ModelElementSetInteractionEnabled>(connection, decoder, this, &WebPageProxy::modelElementSetInteractionEnabled);
#endif
#if ENABLE(ARKIT_INLINE_PREVIEW_MAC)
    if (decoder.messageName() == Messages::WebPageProxy::ModelElementDidCreatePreview::name())
        return IPC::handleMessageAsync<Messages::WebPageProxy::ModelElementDidCreatePreview>(connection, decoder, this, &WebPageProxy::modelElementDidCreatePreview);
#endif
#if ENABLE(ARKIT_INLINE_PREVIEW_MAC)
    if (decoder.messageName() == Messages::WebPageProxy::ModelElementSizeDidChange::name())
        return IPC::handleMessageAsync<Messages::WebPageProxy::ModelElementSizeDidChange>(connection, decoder, this, &WebPageProxy::modelElementSizeDidChange);
#endif
#if ENABLE(ARKIT_INLINE_PREVIEW_MAC)
    if (decoder.messageName() == Messages::WebPageProxy::HandleMouseDownForModelElement::name())
        return IPC::handleMessage<Messages::WebPageProxy::HandleMouseDownForModelElement>(connection, decoder, this, &WebPageProxy::handleMouseDownForModelElement);
#endif
#if ENABLE(ARKIT_INLINE_PREVIEW_MAC)
    if (decoder.messageName() == Messages::WebPageProxy::HandleMouseMoveForModelElement::name())
        return IPC::handleMessage<Messages::WebPageProxy::HandleMouseMoveForModelElement>(connection, decoder, this, &WebPageProxy::handleMouseMoveForModelElement);
#endif
#if ENABLE(ARKIT_INLINE_PREVIEW_MAC)
    if (decoder.messageName() == Messages::WebPageProxy::HandleMouseUpForModelElement::name())
        return IPC::handleMessage<Messages::WebPageProxy::HandleMouseUpForModelElement>(connection, decoder, this, &WebPageProxy::handleMouseUpForModelElement);
#endif
#if ENABLE(ARKIT_INLINE_PREVIEW)
    if (decoder.messageName() == Messages::WebPageProxy::ModelElementGetCamera::name())
        return IPC::handleMessageAsync<Messages::WebPageProxy::ModelElementGetCamera>(connection, decoder, this, &WebPageProxy::modelElementGetCamera);
#endif
#if ENABLE(ARKIT_INLINE_PREVIEW)
    if (decoder.messageName() == Messages::WebPageProxy::ModelElementSetCamera::name())
        return IPC::handleMessageAsync<Messages::WebPageProxy::ModelElementSetCamera>(connection, decoder, this, &WebPageProxy::modelElementSetCamera);
#endif
#if ENABLE(ARKIT_INLINE_PREVIEW)
    if (decoder.messageName() == Messages::WebPageProxy::ModelElementIsPlayingAnimation::name())
        return IPC::handleMessageAsync<Messages::WebPageProxy::ModelElementIsPlayingAnimation>(connection, decoder, this, &WebPageProxy::modelElementIsPlayingAnimation);
#endif
#if ENABLE(ARKIT_INLINE_PREVIEW)
    if (decoder.messageName() == Messages::WebPageProxy::ModelElementSetAnimationIsPlaying::name())
        return IPC::handleMessageAsync<Messages::WebPageProxy::ModelElementSetAnimationIsPlaying>(connection, decoder, this, &WebPageProxy::modelElementSetAnimationIsPlaying);
#endif
#if ENABLE(ARKIT_INLINE_PREVIEW)
    if (decoder.messageName() == Messages::WebPageProxy::ModelElementIsLoopingAnimation::name())
        return IPC::handleMessageAsync<Messages::WebPageProxy::ModelElementIsLoopingAnimation>(connection, decoder, this, &WebPageProxy::modelElementIsLoopingAnimation);
#endif
#if ENABLE(ARKIT_INLINE_PREVIEW)
    if (decoder.messageName() == Messages::WebPageProxy::ModelElementSetIsLoopingAnimation::name())
        return IPC::handleMessageAsync<Messages::WebPageProxy::ModelElementSetIsLoopingAnimation>(connection, decoder, this, &WebPageProxy::modelElementSetIsLoopingAnimation);
#endif
#if ENABLE(ARKIT_INLINE_PREVIEW)
    if (decoder.messageName() == Messages::WebPageProxy::ModelElementAnimationDuration::name())
        return IPC::handleMessageAsync<Messages::WebPageProxy::ModelElementAnimationDuration>(connection, decoder, this, &WebPageProxy::modelElementAnimationDuration);
#endif
#if ENABLE(ARKIT_INLINE_PREVIEW)
    if (decoder.messageName() == Messages::WebPageProxy::ModelElementAnimationCurrentTime::name())
        return IPC::handleMessageAsync<Messages::WebPageProxy::ModelElementAnimationCurrentTime>(connection, decoder, this, &WebPageProxy::modelElementAnimationCurrentTime);
#endif
#if ENABLE(ARKIT_INLINE_PREVIEW)
    if (decoder.messageName() == Messages::WebPageProxy::ModelElementSetAnimationCurrentTime::name())
        return IPC::handleMessageAsync<Messages::WebPageProxy::ModelElementSetAnimationCurrentTime>(connection, decoder, this, &WebPageProxy::modelElementSetAnimationCurrentTime);
#endif
#if ENABLE(ARKIT_INLINE_PREVIEW)
    if (decoder.messageName() == Messages::WebPageProxy::ModelElementHasAudio::name())
        return IPC::handleMessageAsync<Messages::WebPageProxy::ModelElementHasAudio>(connection, decoder, this, &WebPageProxy::modelElementHasAudio);
#endif
#if ENABLE(ARKIT_INLINE_PREVIEW)
    if (decoder.messageName() == Messages::WebPageProxy::ModelElementIsMuted::name())
        return IPC::handleMessageAsync<Messages::WebPageProxy::ModelElementIsMuted>(connection, decoder, this, &WebPageProxy::modelElementIsMuted);
#endif
#if ENABLE(ARKIT_INLINE_PREVIEW)
    if (decoder.messageName() == Messages::WebPageProxy::ModelElementSetIsMuted::name())
        return IPC::handleMessageAsync<Messages::WebPageProxy::ModelElementSetIsMuted>(connection, decoder, this, &WebPageProxy::modelElementSetIsMuted);
#endif
    if (decoder.messageName() == Messages::WebPageProxy::QueryPermission::name())
        return IPC::handleMessageAsync<Messages::WebPageProxy::QueryPermission>(connection, decoder, this, &WebPageProxy::queryPermission);
#if ENABLE(APPLE_PAY_AMS_UI)
    if (decoder.messageName() == Messages::WebPageProxy::StartApplePayAMSUISession::name())
        return IPC::handleMessageAsync<Messages::WebPageProxy::StartApplePayAMSUISession>(connection, decoder, this, &WebPageProxy::startApplePayAMSUISession);
#endif
#if ENABLE(APPLE_PAY_AMS_UI)
    if (decoder.messageName() == Messages::WebPageProxy::AbortApplePayAMSUISession::name())
        return IPC::handleMessage<Messages::WebPageProxy::AbortApplePayAMSUISession>(connection, decoder, this, &WebPageProxy::abortApplePayAMSUISession);
#endif
    if (decoder.messageName() == Messages::WebPageProxy::RequestCookieConsent::name())
        return IPC::handleMessageAsync<Messages::WebPageProxy::RequestCookieConsent>(connection, decoder, this, &WebPageProxy::requestCookieConsent);
    if (decoder.messageName() == Messages::WebPageProxy::ClassifyModalContainerControls::name())
        return IPC::handleMessageAsync<Messages::WebPageProxy::ClassifyModalContainerControls>(connection, decoder, this, &WebPageProxy::classifyModalContainerControls);
    if (decoder.messageName() == Messages::WebPageProxy::DecidePolicyForModalContainer::name())
        return IPC::handleMessageAsync<Messages::WebPageProxy::DecidePolicyForModalContainer>(connection, decoder, this, &WebPageProxy::decidePolicyForModalContainer);
    UNUSED_PARAM(connection);
    UNUSED_PARAM(decoder);
#if ENABLE(IPC_TESTING_API)
    if (connection.ignoreInvalidMessageForTesting())
        return;
#endif // ENABLE(IPC_TESTING_API)
    ASSERT_NOT_REACHED_WITH_MESSAGE("Unhandled message %s to %" PRIu64, IPC::description(decoder.messageName()), decoder.destinationID());
}

bool WebPageProxy::didReceiveSyncMessage(IPC::Connection& connection, IPC::Decoder& decoder, UniqueRef<IPC::Encoder>& replyEncoder)
{
    Ref protectedThis { *this };
    if (decoder.messageName() == Messages::WebPageProxy::CreateNewPage::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::CreateNewPage>(connection, decoder, replyEncoder, this, &WebPageProxy::createNewPage);
    if (decoder.messageName() == Messages::WebPageProxy::RunJavaScriptAlert::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::RunJavaScriptAlert>(connection, decoder, replyEncoder, this, &WebPageProxy::runJavaScriptAlert);
    if (decoder.messageName() == Messages::WebPageProxy::RunJavaScriptConfirm::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::RunJavaScriptConfirm>(connection, decoder, replyEncoder, this, &WebPageProxy::runJavaScriptConfirm);
    if (decoder.messageName() == Messages::WebPageProxy::RunJavaScriptPrompt::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::RunJavaScriptPrompt>(connection, decoder, replyEncoder, this, &WebPageProxy::runJavaScriptPrompt);
#if ENABLE(WEBGL)
    if (decoder.messageName() == Messages::WebPageProxy::WebGLPolicyForURL::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::WebGLPolicyForURL>(connection, decoder, replyEncoder, this, &WebPageProxy::webGLPolicyForURL);
#endif
#if ENABLE(WEBGL)
    if (decoder.messageName() == Messages::WebPageProxy::ResolveWebGLPolicyForURL::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::ResolveWebGLPolicyForURL>(connection, decoder, replyEncoder, this, &WebPageProxy::resolveWebGLPolicyForURL);
#endif
    if (decoder.messageName() == Messages::WebPageProxy::GetToolbarsAreVisible::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::GetToolbarsAreVisible>(connection, decoder, replyEncoder, this, &WebPageProxy::getToolbarsAreVisible);
    if (decoder.messageName() == Messages::WebPageProxy::GetMenuBarIsVisible::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::GetMenuBarIsVisible>(connection, decoder, replyEncoder, this, &WebPageProxy::getMenuBarIsVisible);
    if (decoder.messageName() == Messages::WebPageProxy::GetStatusBarIsVisible::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::GetStatusBarIsVisible>(connection, decoder, replyEncoder, this, &WebPageProxy::getStatusBarIsVisible);
    if (decoder.messageName() == Messages::WebPageProxy::GetWindowFrame::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::GetWindowFrame>(connection, decoder, replyEncoder, this, &WebPageProxy::getWindowFrame);
    if (decoder.messageName() == Messages::WebPageProxy::ScreenToRootView::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::ScreenToRootView>(connection, decoder, replyEncoder, this, &WebPageProxy::screenToRootView);
    if (decoder.messageName() == Messages::WebPageProxy::RootViewToScreen::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::RootViewToScreen>(connection, decoder, replyEncoder, this, &WebPageProxy::rootViewToScreen);
    if (decoder.messageName() == Messages::WebPageProxy::AccessibilityScreenToRootView::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::AccessibilityScreenToRootView>(connection, decoder, replyEncoder, this, &WebPageProxy::accessibilityScreenToRootView);
    if (decoder.messageName() == Messages::WebPageProxy::RootViewToAccessibilityScreen::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::RootViewToAccessibilityScreen>(connection, decoder, replyEncoder, this, &WebPageProxy::rootViewToAccessibilityScreen);
    if (decoder.messageName() == Messages::WebPageProxy::RunBeforeUnloadConfirmPanel::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::RunBeforeUnloadConfirmPanel>(connection, decoder, replyEncoder, this, &WebPageProxy::runBeforeUnloadConfirmPanel);
    if (decoder.messageName() == Messages::WebPageProxy::PrintFrame::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::PrintFrame>(connection, decoder, replyEncoder, this, &WebPageProxy::printFrame);
    if (decoder.messageName() == Messages::WebPageProxy::DecidePolicyForNavigationActionSync::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::DecidePolicyForNavigationActionSync>(connection, decoder, replyEncoder, this, &WebPageProxy::decidePolicyForNavigationActionSync);
#if PLATFORM(IOS_FAMILY)
    if (decoder.messageName() == Messages::WebPageProxy::InterpretKeyEvent::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::InterpretKeyEvent>(connection, decoder, replyEncoder, this, &WebPageProxy::interpretKeyEvent);
#endif
    if (decoder.messageName() == Messages::WebPageProxy::BackForwardGoToItem::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::BackForwardGoToItem>(connection, decoder, replyEncoder, this, &WebPageProxy::backForwardGoToItem);
    if (decoder.messageName() == Messages::WebPageProxy::BackForwardItemAtIndex::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::BackForwardItemAtIndex>(connection, decoder, replyEncoder, this, &WebPageProxy::backForwardItemAtIndex);
    if (decoder.messageName() == Messages::WebPageProxy::BackForwardListCounts::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::BackForwardListCounts>(connection, decoder, replyEncoder, this, &WebPageProxy::backForwardListCounts);
    if (decoder.messageName() == Messages::WebPageProxy::CanUndoRedo::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::CanUndoRedo>(connection, decoder, replyEncoder, this, &WebPageProxy::canUndoRedo);
    if (decoder.messageName() == Messages::WebPageProxy::ExecuteUndoRedo::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::ExecuteUndoRedo>(connection, decoder, replyEncoder, this, &WebPageProxy::executeUndoRedo);
    if (decoder.messageName() == Messages::WebPageProxy::RequestDOMPasteAccess::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::RequestDOMPasteAccess>(connection, decoder, replyEncoder, this, &WebPageProxy::requestDOMPasteAccess);
    if (decoder.messageName() == Messages::WebPageProxy::ExceededDatabaseQuota::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::ExceededDatabaseQuota>(connection, decoder, replyEncoder, this, &WebPageProxy::exceededDatabaseQuota);
    if (decoder.messageName() == Messages::WebPageProxy::ReachedApplicationCacheOriginQuota::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::ReachedApplicationCacheOriginQuota>(connection, decoder, replyEncoder, this, &WebPageProxy::reachedApplicationCacheOriginQuota);
#if USE(UNIFIED_TEXT_CHECKING)
    if (decoder.messageName() == Messages::WebPageProxy::CheckTextOfParagraph::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::CheckTextOfParagraph>(connection, decoder, replyEncoder, this, &WebPageProxy::checkTextOfParagraph);
#endif
    if (decoder.messageName() == Messages::WebPageProxy::CheckSpellingOfString::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::CheckSpellingOfString>(connection, decoder, replyEncoder, this, &WebPageProxy::checkSpellingOfString);
    if (decoder.messageName() == Messages::WebPageProxy::CheckGrammarOfString::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::CheckGrammarOfString>(connection, decoder, replyEncoder, this, &WebPageProxy::checkGrammarOfString);
    if (decoder.messageName() == Messages::WebPageProxy::SpellingUIIsShowing::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::SpellingUIIsShowing>(connection, decoder, replyEncoder, this, &WebPageProxy::spellingUIIsShowing);
    if (decoder.messageName() == Messages::WebPageProxy::GetGuessesForWord::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::GetGuessesForWord>(connection, decoder, replyEncoder, this, &WebPageProxy::getGuessesForWord);
#if PLATFORM(COCOA)
    if (decoder.messageName() == Messages::WebPageProxy::ExecuteSavedCommandBySelector::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::ExecuteSavedCommandBySelector>(connection, decoder, replyEncoder, this, &WebPageProxy::executeSavedCommandBySelector);
#endif
#if PLATFORM(COCOA)
    if (decoder.messageName() == Messages::WebPageProxy::GetIsSpeaking::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::GetIsSpeaking>(connection, decoder, replyEncoder, this, &WebPageProxy::getIsSpeaking);
#endif
#if USE(APPKIT)
    if (decoder.messageName() == Messages::WebPageProxy::SubstitutionsPanelIsShowing::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::SubstitutionsPanelIsShowing>(connection, decoder, replyEncoder, this, &WebPageProxy::substitutionsPanelIsShowing);
#endif
#if PLATFORM(MAC)
    if (decoder.messageName() == Messages::WebPageProxy::DismissCorrectionPanelSoon::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::DismissCorrectionPanelSoon>(connection, decoder, replyEncoder, this, &WebPageProxy::dismissCorrectionPanelSoon);
#endif
#if USE(DICTATION_ALTERNATIVES)
    if (decoder.messageName() == Messages::WebPageProxy::DictationAlternatives::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::DictationAlternatives>(connection, decoder, replyEncoder, this, &WebPageProxy::dictationAlternatives);
#endif
    if (decoder.messageName() == Messages::WebPageProxy::LoadRecentSearches::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::LoadRecentSearches>(connection, decoder, replyEncoder, this, &WebPageProxy::loadRecentSearches);
#if ENABLE(PDFKIT_PLUGIN)
    if (decoder.messageName() == Messages::WebPageProxy::ShowPDFContextMenu::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::ShowPDFContextMenu>(connection, decoder, replyEncoder, this, &WebPageProxy::showPDFContextMenu);
#endif
#if ENABLE(WEB_CRYPTO)
    if (decoder.messageName() == Messages::WebPageProxy::WrapCryptoKey::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::WrapCryptoKey>(connection, decoder, replyEncoder, this, &WebPageProxy::wrapCryptoKey);
#endif
#if ENABLE(WEB_CRYPTO)
    if (decoder.messageName() == Messages::WebPageProxy::UnwrapCryptoKey::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::UnwrapCryptoKey>(connection, decoder, replyEncoder, this, &WebPageProxy::unwrapCryptoKey);
#endif
    if (decoder.messageName() == Messages::WebPageProxy::HandleSynchronousMessage::name())
        return IPC::handleMessageSynchronousWantsConnection<Messages::WebPageProxy::HandleSynchronousMessage>(connection, decoder, replyEncoder, this, &WebPageProxy::handleSynchronousMessage);
    if (decoder.messageName() == Messages::WebPageProxy::LoadSynchronousURLSchemeTask::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::LoadSynchronousURLSchemeTask>(connection, decoder, replyEncoder, this, &WebPageProxy::loadSynchronousURLSchemeTask);
#if ENABLE(ATTACHMENT_ELEMENT)
    if (decoder.messageName() == Messages::WebPageProxy::SerializedAttachmentDataForIdentifiers::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::SerializedAttachmentDataForIdentifiers>(connection, decoder, replyEncoder, this, &WebPageProxy::serializedAttachmentDataForIdentifiers);
#endif
    if (decoder.messageName() == Messages::WebPageProxy::SignedPublicKeyAndChallengeString::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::SignedPublicKeyAndChallengeString>(connection, decoder, replyEncoder, this, &WebPageProxy::signedPublicKeyAndChallengeString);
#if ENABLE(SPEECH_SYNTHESIS)
    if (decoder.messageName() == Messages::WebPageProxy::SpeechSynthesisVoiceList::name())
        return IPC::handleMessageSynchronous<Messages::WebPageProxy::SpeechSynthesisVoiceList>(connection, decoder, replyEncoder, this, &WebPageProxy::speechSynthesisVoiceList);
#endif
    UNUSED_PARAM(connection);
    UNUSED_PARAM(decoder);
    UNUSED_PARAM(replyEncoder);
#if ENABLE(IPC_TESTING_API)
    if (connection.ignoreInvalidMessageForTesting())
        return false;
#endif // ENABLE(IPC_TESTING_API)
    ASSERT_NOT_REACHED_WITH_MESSAGE("Unhandled synchronous message %s to %" PRIu64, description(decoder.messageName()), decoder.destinationID());
    return false;
}

} // namespace WebKit
