﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/elasticmapreduce/EMRRequest.h>
#include <aws/elasticmapreduce/EMR_EXPORTS.h>
#include <aws/elasticmapreduce/model/InstanceGroupModifyConfig.h>

#include <utility>

namespace Aws {
namespace EMR {
namespace Model {

/**
 * <p>Change the size of some instance groups.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/elasticmapreduce-2009-03-31/ModifyInstanceGroupsInput">AWS
 * API Reference</a></p>
 */
class ModifyInstanceGroupsRequest : public EMRRequest {
 public:
  AWS_EMR_API ModifyInstanceGroupsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ModifyInstanceGroups"; }

  AWS_EMR_API Aws::String SerializePayload() const override;

  AWS_EMR_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The ID of the cluster to which the instance group belongs.</p>
   */
  inline const Aws::String& GetClusterId() const { return m_clusterId; }
  inline bool ClusterIdHasBeenSet() const { return m_clusterIdHasBeenSet; }
  template <typename ClusterIdT = Aws::String>
  void SetClusterId(ClusterIdT&& value) {
    m_clusterIdHasBeenSet = true;
    m_clusterId = std::forward<ClusterIdT>(value);
  }
  template <typename ClusterIdT = Aws::String>
  ModifyInstanceGroupsRequest& WithClusterId(ClusterIdT&& value) {
    SetClusterId(std::forward<ClusterIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Instance groups to change.</p>
   */
  inline const Aws::Vector<InstanceGroupModifyConfig>& GetInstanceGroups() const { return m_instanceGroups; }
  inline bool InstanceGroupsHasBeenSet() const { return m_instanceGroupsHasBeenSet; }
  template <typename InstanceGroupsT = Aws::Vector<InstanceGroupModifyConfig>>
  void SetInstanceGroups(InstanceGroupsT&& value) {
    m_instanceGroupsHasBeenSet = true;
    m_instanceGroups = std::forward<InstanceGroupsT>(value);
  }
  template <typename InstanceGroupsT = Aws::Vector<InstanceGroupModifyConfig>>
  ModifyInstanceGroupsRequest& WithInstanceGroups(InstanceGroupsT&& value) {
    SetInstanceGroups(std::forward<InstanceGroupsT>(value));
    return *this;
  }
  template <typename InstanceGroupsT = InstanceGroupModifyConfig>
  ModifyInstanceGroupsRequest& AddInstanceGroups(InstanceGroupsT&& value) {
    m_instanceGroupsHasBeenSet = true;
    m_instanceGroups.emplace_back(std::forward<InstanceGroupsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_clusterId;

  Aws::Vector<InstanceGroupModifyConfig> m_instanceGroups;
  bool m_clusterIdHasBeenSet = false;
  bool m_instanceGroupsHasBeenSet = false;
};

}  // namespace Model
}  // namespace EMR
}  // namespace Aws
