﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/states/SFNRequest.h>
#include <aws/states/SFN_EXPORTS.h>
#include <aws/states/model/EncryptionConfiguration.h>
#include <aws/states/model/Tag.h>

#include <utility>

namespace Aws {
namespace SFN {
namespace Model {

/**
 */
class CreateActivityRequest : public SFNRequest {
 public:
  AWS_SFN_API CreateActivityRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateActivity"; }

  AWS_SFN_API Aws::String SerializePayload() const override;

  AWS_SFN_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the activity to create. This name must be unique for your Amazon
   * Web Services account and region for 90 days. For more information, see <a
   * href="https://docs.aws.amazon.com/step-functions/latest/dg/limits.html#service-limits-state-machine-executions">
   * Limits Related to State Machine Executions</a> in the <i>Step Functions
   * Developer Guide</i>.</p> <p>A name must <i>not</i> contain:</p> <ul> <li>
   * <p>white space</p> </li> <li> <p>brackets <code>&lt; &gt; { } [ ]</code> </p>
   * </li> <li> <p>wildcard characters <code>? *</code> </p> </li> <li> <p>special
   * characters <code>" # % \ ^ | ~ ` $ &amp; , ; : /</code> </p> </li> <li>
   * <p>control characters (<code>U+0000-001F</code>, <code>U+007F-009F</code>,
   * <code>U+FFFE-FFFF</code>)</p> </li> <li> <p>surrogates
   * (<code>U+D800-DFFF</code>)</p> </li> <li> <p>invalid characters (<code>
   * U+10FFFF</code>)</p> </li> </ul> <p>To enable logging with CloudWatch Logs, the
   * name should only contain 0-9, A-Z, a-z, - and _.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateActivityRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The list of tags to add to a resource.</p> <p>An array of key-value pairs.
   * For more information, see <a
   * href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/cost-alloc-tags.html">Using
   * Cost Allocation Tags</a> in the <i>Amazon Web Services Billing and Cost
   * Management User Guide</i>, and <a
   * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access_iam-tags.html">Controlling
   * Access Using IAM Tags</a>.</p> <p>Tags may only contain Unicode letters, digits,
   * white space, or these symbols: <code>_ . : / = + - @</code>.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateActivityRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateActivityRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Settings to configure server-side encryption.</p>
   */
  inline const EncryptionConfiguration& GetEncryptionConfiguration() const { return m_encryptionConfiguration; }
  inline bool EncryptionConfigurationHasBeenSet() const { return m_encryptionConfigurationHasBeenSet; }
  template <typename EncryptionConfigurationT = EncryptionConfiguration>
  void SetEncryptionConfiguration(EncryptionConfigurationT&& value) {
    m_encryptionConfigurationHasBeenSet = true;
    m_encryptionConfiguration = std::forward<EncryptionConfigurationT>(value);
  }
  template <typename EncryptionConfigurationT = EncryptionConfiguration>
  CreateActivityRequest& WithEncryptionConfiguration(EncryptionConfigurationT&& value) {
    SetEncryptionConfiguration(std::forward<EncryptionConfigurationT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;

  Aws::Vector<Tag> m_tags;

  EncryptionConfiguration m_encryptionConfiguration;
  bool m_nameHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
  bool m_encryptionConfigurationHasBeenSet = false;
};

}  // namespace Model
}  // namespace SFN
}  // namespace Aws
