﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/discovery/ApplicationDiscoveryServiceRequest.h>
#include <aws/discovery/ApplicationDiscoveryService_EXPORTS.h>
#include <aws/discovery/model/ExportFilter.h>

#include <utility>

namespace Aws {
namespace ApplicationDiscoveryService {
namespace Model {

/**
 */
class DescribeExportTasksRequest : public ApplicationDiscoveryServiceRequest {
 public:
  AWS_APPLICATIONDISCOVERYSERVICE_API DescribeExportTasksRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeExportTasks"; }

  AWS_APPLICATIONDISCOVERYSERVICE_API Aws::String SerializePayload() const override;

  AWS_APPLICATIONDISCOVERYSERVICE_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>One or more unique identifiers used to query the status of an export
   * request.</p>
   */
  inline const Aws::Vector<Aws::String>& GetExportIds() const { return m_exportIds; }
  inline bool ExportIdsHasBeenSet() const { return m_exportIdsHasBeenSet; }
  template <typename ExportIdsT = Aws::Vector<Aws::String>>
  void SetExportIds(ExportIdsT&& value) {
    m_exportIdsHasBeenSet = true;
    m_exportIds = std::forward<ExportIdsT>(value);
  }
  template <typename ExportIdsT = Aws::Vector<Aws::String>>
  DescribeExportTasksRequest& WithExportIds(ExportIdsT&& value) {
    SetExportIds(std::forward<ExportIdsT>(value));
    return *this;
  }
  template <typename ExportIdsT = Aws::String>
  DescribeExportTasksRequest& AddExportIds(ExportIdsT&& value) {
    m_exportIdsHasBeenSet = true;
    m_exportIds.emplace_back(std::forward<ExportIdsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>One or more filters.</p> <ul> <li> <p> <code>AgentId</code> - ID of the agent
   * whose collected data will be exported</p> </li> </ul>
   */
  inline const Aws::Vector<ExportFilter>& GetFilters() const { return m_filters; }
  inline bool FiltersHasBeenSet() const { return m_filtersHasBeenSet; }
  template <typename FiltersT = Aws::Vector<ExportFilter>>
  void SetFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters = std::forward<FiltersT>(value);
  }
  template <typename FiltersT = Aws::Vector<ExportFilter>>
  DescribeExportTasksRequest& WithFilters(FiltersT&& value) {
    SetFilters(std::forward<FiltersT>(value));
    return *this;
  }
  template <typename FiltersT = ExportFilter>
  DescribeExportTasksRequest& AddFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters.emplace_back(std::forward<FiltersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of volume results returned by
   * <code>DescribeExportTasks</code> in paginated output. When this parameter is
   * used, <code>DescribeExportTasks</code> only returns <code>maxResults</code>
   * results in a single page along with a <code>nextToken</code> response
   * element.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline DescribeExportTasksRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The <code>nextToken</code> value returned from a previous paginated
   * <code>DescribeExportTasks</code> request where <code>maxResults</code> was used
   * and the results exceeded the value of that parameter. Pagination continues from
   * the end of the previous results that returned the <code>nextToken</code> value.
   * This value is null when there are no more results to return.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  DescribeExportTasksRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::Vector<Aws::String> m_exportIds;

  Aws::Vector<ExportFilter> m_filters;

  int m_maxResults{0};

  Aws::String m_nextToken;
  bool m_exportIdsHasBeenSet = false;
  bool m_filtersHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace ApplicationDiscoveryService
}  // namespace Aws
