/*******************************************************************************
  TPI - flexible but useless plug-in framework.
  Copyright (C) 2002-2009 Silky

  This library is free software; you can redistribute it and/or modify it under
  the terms of the GNU Lesser General Public License as published by the Free
  Software Foundation; either version 2.1 of the License, or (at your option)
  any later version.

  This library is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
  for more details.

  You should have received a copy of the GNU Lesser General Public License along
  with this library; if not, write to the Free Software Foundation, Inc.,
  51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA

  $Id: dlg_make.cpp 364 2010-10-01 12:06:04Z sirakaba $
*******************************************************************************/

#include "lychee.h"

#include "dlg_make.h"
#include "frm_main.h"
#include "functions.h"

#include <wx/dirdlg.h>
#include <wx/arrimpl.cpp>

WX_DEFINE_OBJARRAY(ArrayTPI_FORMATINFO);

//******************************************************************************
// MakeDialog
//******************************************************************************

MakeDialog::MakeDialog(): wxDialog()
{
	this->uCommand = TPI_COMMAND_CREATE;
}

//******************************************************************************
// Event Table.
//******************************************************************************

BEGIN_EVENT_TABLE(MakeDialog, wxDialog)
	EVT_INIT_DIALOG(                 MakeDialog::OnInit)
	EVT_BUTTON(XRCID("btnDefault"),  MakeDialog::OnBtnDefault)
	EVT_BUTTON(XRCID("btnDesktop"),  MakeDialog::OnBtnDesktop)
	EVT_BUTTON(XRCID("btnCurrent"),  MakeDialog::OnBtnCurrent)
	EVT_BUTTON(XRCID("btnBrowse"),   MakeDialog::OnBtnBrowse)
	EVT_BUTTON(XRCID("btnBrowseKF"), MakeDialog::OnBtnBrowseKF)
	EVT_BUTTON(XRCID("btnOK"),       MakeDialog::OnBtnOK)
	EVT_BUTTON(XRCID("btnCancel"),   MakeDialog::OnBtnCancel)
	EVT_CHOICE(XRCID("chType"),      MakeDialog::OnChoice)
	EVT_CHECKBOX(XRCID("cbUnmask"),  MakeDialog::OnCbUnmask)
	EVT_CHECKBOX(XRCID("cbMakeSFX"), MakeDialog::OnCbMakeSFX)
	EVT_NOTEBOOK_PAGE_CHANGED(XRCID("nbTabs"), MakeDialog::OnTabChanged)
END_EVENT_TABLE()

//******************************************************************************
// Event handler.
//******************************************************************************

void MakeDialog::OnInit(wxInitDialogEvent&)
{
	// XRCと結びつけ。
	// "General"タブ
	this->cbDir             = XRCCTRL(* this, "cbDir",        wxComboBox);
	this->cbFileName        = XRCCTRL(* this, "cbFileName",   wxComboBox);
	this->cbOpenAfter       = XRCCTRL(* this, "cbOpenAfter",  wxCheckBox);
	this->cbIgnorePath      = XRCCTRL(* this, "cbIgnorePath", wxCheckBox);
	this->cbExitAfter       = XRCCTRL(* this, "cbExitAfter",  wxCheckBox);
	this->chType            = XRCCTRL(* this, "chType",       wxChoice);
	this->chDirMake         = XRCCTRL(* this, "chDirMake",    wxChoice);
	// "Config"タブ
	this->scLevel           = XRCCTRL(* this, "scLevel",      wxSpinCtrl);
	this->scRR              = XRCCTRL(* this, "scRR",         wxSpinCtrl);
	this->cbSolid           = XRCCTRL(* this, "cbSolid",      wxCheckBox);
	this->cbMMOptimize      = XRCCTRL(* this, "cbMMOptimize", wxCheckBox);
	this->cbCompressHeader  = XRCCTRL(* this, "cbCompressHeader", wxCheckBox);
	this->cbMakeSFX         = XRCCTRL(* this, "cbMakeSFX",    wxCheckBox);
	// "Comment"タブ
	this->tcComment         = XRCCTRL(* this, "tcComment",    wxTextCtrl);
	// "Encryption"タブ
	this->tcPassword        = XRCCTRL(* this, "tcPassword",   wxTextCtrl);
	this->tcKeyfile         = XRCCTRL(* this, "tcKeyfile",    wxTextCtrl);
	this->cbSplitSize       = XRCCTRL(* this, "cbSplitSize",  wxComboBox);
	this->cbUnmask          = XRCCTRL(* this, "cbUnmask",     wxCheckBox);
	this->cbEncryptHeader   = XRCCTRL(* this, "cbEncryptHeader", wxCheckBox);
	this->chEncryptMethod   = XRCCTRL(* this, "chEncryptMethod", wxChoice);
	// "Files"タブ
	this->lcFiles           = XRCCTRL(* this, "lcFiles",      myListCtrl2);

	// ListCtrlに列を追加。
	// wxGTKでは直接wxLC_VIRTUALを指定しないと反映されない。
	this->lcFiles->SetSingleStyle(wxLC_VIRTUAL);
	this->lcFiles->InsertColumn(0, _("Input"),  wxLIST_FORMAT_LEFT,  150);
	this->lcFiles->InsertColumn(1, _("Output"), wxLIST_FORMAT_LEFT,  290);
	this->lcFiles->asInput = & this->files;
	this->lcFiles->atDangerItem.SetTextColour(* wxRED);
	this->lcFiles->atExistItem.SetTextColour(* wxBLUE);

	::wxXmlResource::Get()->Unload(L_DIR_S_XRC wxT("dlg_make.xrc"));

	// 事前準備。
	MainFrame * frm_main = (MainFrame *) this->GetParent();
	wxString szArcPath = frm_main->fnArchive.GetPath();

	// パス履歴読み込み。
	for (size_t i = 0; i < frm_main->conf.GetHistoryCount(CONF_HISTORY_PATH); i++)
	{
		wxString sz = frm_main->conf.ReadHistory(CONF_HISTORY_PATH, i);
		if (sz.IsEmpty())
		{
			continue;
		}

		this->cbDir->Append(sz);
	}
	// 書庫名履歴読み込み。
	for (size_t i = 0; i < frm_main->conf.GetHistoryCount(CONF_HISTORY_NAME); i++)
	{
		wxString sz = frm_main->conf.ReadHistory(CONF_HISTORY_NAME, i);
		if (sz.IsEmpty())
		{
			continue;
		}

		this->cbFileName->Append(sz);
	}

	// パスを設定。
	this->cbDir->SetValue(szArcPath);

	// コマンド別の処理。
	switch (this->uCommand)
	{
	case TPI_COMMAND_EXTRACT:
		// 初期値を設定。
		this->SetTitle(_("Extract"));
		this->tcComment->SetValue(frm_main->aiArchive.szComment);
		this->cbDir->SetFocus();

		// コントロールを無効化。
		this->scLevel->Disable();
		this->scRR->Disable();
		this->cbSolid->Disable();
		this->cbMMOptimize->Disable();
		this->tcComment->SetEditable(false);
		this->cbEncryptHeader->Disable();
		this->chEncryptMethod->Disable();
	case TPI_COMMAND_ADD:
		// コントロールを無効化(展開時も)。
		this->cbFileName->Disable();
		this->chType->Disable();
		this->chDirMake->Enable();
		this->cbSplitSize->Disable();
		this->cbCompressHeader->Disable();
		this->cbMakeSFX->Disable();

		// 書庫名を設定。
		this->cbFileName->SetValue(frm_main->fnArchive.GetFullName());

		// 書庫形式欄を設定。
		this->afInfo.Add(frm_main->aiArchive.fiInfo);
		this->chType->Append(frm_main->aiArchive.fiInfo.szTypeName);
		this->chType->SetSelection(0);
		{
			wxCommandEvent e;
			e.SetInt(0);
			this->OnChoice(e);
		}

		if (this->uCommand != TPI_COMMAND_ADD)
		{
			break;
		}
		this->SetTitle(_("Add"));

		// コントロールを無効化(追加時のみ)。
		this->cbDir->Disable();
		this->chDirMake->Disable();
		XRCCTRL(* this, "btnDefault", wxButton)->Disable();
		XRCCTRL(* this, "btnDesktop", wxButton)->Disable();
		XRCCTRL(* this, "btnCurrent", wxButton)->Disable();
		XRCCTRL(* this, "btnBrowse",  wxButton)->Disable();
		break;
	case TPI_COMMAND_CREATE:
		this->SetTitle(_("Create"));

		// 書庫名を設定。初期化の都合上.を付加しておく。
		this->cbFileName->SetValue(frm_main->fnArchive.GetName() + wxT('.'));
		this->cbFileName->SetFocus();

		// ライブラリを検索。
		TPIHandle tpi;
		wxDir fs(L_DIR_B_LIB);
		wxString szTPIName;
		if (fs.GetFirst(& szTPIName,wxT("*" TPI_EXT)))
		{
			do
			{
				// ロード。
				wxString szLibName = L_DIR_B_LIB + szTPIName;
				if (tpi.InitLibrary(szLibName, wxEmptyString))
				{
					// 対応する形式名を取得。
					TPI_FORMATINFO fiInfo;
					if (tpi.GetFormatInformation(& fiInfo, true))
					{
						do
						{
							if (fiInfo.eSupportedCommand & TPI_COMMAND_CREATE && (this->lcFiles->asInput->GetCount() == 1 || fiInfo.fArchive))
							{
								fiInfo.szTPIName = szLibName;
								this->afInfo.Add(fiInfo);
								this->chType->Append(fiInfo.szTypeName);
							}
						}
						while (tpi.GetFormatInformation(& fiInfo));
					}
					tpi.FreeLibrary();
				}
			}
			while (fs.GetNext(& szTPIName));
		}

		if (this->chType->GetCount() == 0)
		{
			// 形式の候補が一つもない場合。
			XRCCTRL(* this, "btnOK", wxButton)->Disable();
			break;
		}

		// とりあえず最初の形式にしておく。
		this->chType->SetSelection(0);
		wxCommandEvent e;
		e.SetInt(0);
		this->OnChoice(e);
		break;
	}

	// 展開/格納先を予測。ただしDTVスキャンに時間がかかる場合はスキップ可能。
	size_t n = this->lcFiles->asInput->GetCount();
	this->lcFiles->asOutput.Alloc(n);
	this->lcFiles->apItem.Alloc(n);
	if (n < 3000 ||
		::AskDlg(
			this->uCommand == TPI_COMMAND_EXTRACT ?
				_("This archive contains so many files that it takes long to check Directory Traversal Vulnerability(DTV) problem. If you are sure this archive is safe, you can skip this scanning process. Do you want to scan for DTV problem?") :
				_("The files you want to store are too many, so it takes long to check Directory Traversal Vulnerability(DTV) problem. If you are sure the path of the files are no problem, you can skip this scanning process. Do you want to scan for DTV problem?"),
			this
		) == wxYES)
	{
		wxNotebookEvent e;
		e.SetSelection(4);
		this->OnTabChanged(e);
	}

	this->Raise();
}

void MakeDialog::OnBtnDefault(wxCommandEvent&)
{
	this->cbDir->SetValue(((MainFrame *) this->GetParent())->conf.ReadId(CONF_DEFAULT_PATH, (wxString) wxEmptyString));
}

void MakeDialog::OnBtnDesktop(wxCommandEvent&)
{
	wxFileName fn(wxFileName::GetHomeDir(), wxT("Desktop"));
	this->cbDir->SetValue(fn.GetFullPath());
}

void MakeDialog::OnBtnCurrent(wxCommandEvent&)
{
	this->cbDir->SetValue(((MainFrame *) this->GetParent())->fnArchive.GetPath());
}

void MakeDialog::OnBtnBrowse(wxCommandEvent&)
{
	wxDirDialog dd(this);
	dd.SetPath(this->cbDir->GetValue());
	if (dd.ShowModal() == wxID_OK)
	{
		this->cbDir->SetValue(dd.GetPath());
	}
}

void MakeDialog::OnBtnBrowseKF(wxCommandEvent&)
{
	wxFileDialog fd(this);
	fd.SetWindowStyleFlag(wxFD_OPEN | wxFD_FILE_MUST_EXIST);
	if (fd.ShowModal() == wxID_OK)
	{
		this->tcKeyfile->SetValue(fd.GetPath());
	}
}

void MakeDialog::OnBtnOK(wxCommandEvent&)
{
	// 履歴書き込み。
	if (this->uCommand != TPI_COMMAND_ADD)
	{
		MainFrame * frm_main = (MainFrame *) this->GetParent();
		wxFileName fnCurrent(this->cbDir->GetValue(), this->uCommand == TPI_COMMAND_CREATE ? this->cbFileName->GetValue() : (wxString) wxEmptyString);
		frm_main->conf.WriteHistory(CONF_HISTORY_FULL, fnCurrent.GetFullPath());
		frm_main->conf.WriteHistory(CONF_HISTORY_PATH, fnCurrent.GetPath());
		frm_main->conf.WriteHistory(CONF_HISTORY_NAME, fnCurrent.GetFullName());
	}
	this->EndModal(wxID_OK);
}

void MakeDialog::OnBtnCancel(wxCommandEvent&)
{
	this->EndModal(wxID_CANCEL);
}

void MakeDialog::OnChoice(wxCommandEvent& e)
{
	TPI_FORMATINFO * fiInfo = & this->afInfo[e.GetInt()];
	// 形式が各種設定に対応しているか。
	// 作成時/追加時/展開時設定。
	this->tcPassword->Enable(fiInfo->fEncryptPassword);
	this->tcKeyfile->Enable(fiInfo->fEncryptKeyFile);
	this->cbUnmask->Enable(fiInfo->fEncryptPassword);
	this->chEncryptMethod->Enable(fiInfo->fEncryptPassword || fiInfo->fEncryptKeyFile);
	XRCCTRL(* this, "btnBrowseKF", wxButton)->Enable(fiInfo->fEncryptKeyFile);
	if (this->uCommand == TPI_COMMAND_EXTRACT)
	{
		return;
	}

	// 作成時/追加時設定。
	this->scLevel->SetRange(fiInfo->nCompressLevelMin, fiInfo->nCompressLevelMax);
	this->scLevel->SetValue(fiInfo->nCompressLevelMax);
	this->scLevel->Enable(fiInfo->nCompressLevelMin != fiInfo->nCompressLevelMax);
	this->scRR->SetRange(fiInfo->nRecoveryRecordMin, fiInfo->nRecoveryRecordMax);
	this->scRR->SetValue(fiInfo->nRecoveryRecordMin);
	this->scRR->Enable(fiInfo->nRecoveryRecordMin != fiInfo->nRecoveryRecordMax);
	this->cbEncryptHeader->Enable(fiInfo->fEncryptHeader);
	this->cbCompressHeader->Enable(fiInfo->fCompressHeader);
	this->cbSolid->Enable(fiInfo->fSolid);
	this->cbMMOptimize->Enable(fiInfo->fMMOptimize);
	this->tcComment->Enable(fiInfo->fComment);
	if (this->uCommand == TPI_COMMAND_ADD)
	{
		return;
	}

	// 作成時設定。
	this->cbSplitSize->Enable(fiInfo->fMultiVolume);
	this->cbMakeSFX->Enable(fiInfo->fSFX);

	wxFileName fn(this->cbFileName->GetValue());
	fn.SetExt(this->cbMakeSFX->IsEnabled() && this->cbMakeSFX->IsChecked() ? EXE_EXT : fiInfo->szSuffix.BeforeFirst(wxT(';')));
	this->cbFileName->SetValue(fn.GetFullName());
}

void MakeDialog::OnCbUnmask(wxCommandEvent&)
{
	this->tcPassword->SetWindowStyle(this->tcPassword->GetWindowStyle() & (this->cbUnmask->IsChecked() ? ~ wxTE_PASSWORD : wxTE_PASSWORD));
	this->tcPassword->Refresh();
}

void MakeDialog::OnCbMakeSFX(wxCommandEvent&)
{
	wxFileName fn(this->cbFileName->GetValue());
	fn.SetExt(this->cbMakeSFX->IsChecked() ? EXE_EXT : this->afInfo[this->chType->GetSelection()].szSuffix.BeforeFirst(wxT(';')));
	this->cbFileName->SetValue(fn.GetFullName());
}

void MakeDialog::OnTabChanged(wxNotebookEvent& e)
{
	// "Files"タブのときは処理。
	if (e.GetSelection() != 4)
	{
		return;
	}
	this->lcFiles->DeleteAllItems();
	this->lcFiles->asOutput.Empty();
	this->lcFiles->apItem.Empty();

	bool fDTVWarning = false;
	switch (this->uCommand)
	{
	case TPI_COMMAND_EXTRACT:
	{
		// ファイルの出力先を推測。
		wxString szOutputRootDir = WillMakeDirByArcName((MainFrame *) this->GetParent(), this) ? MakeDirPath(wxFileName::DirName(this->cbDir->GetValue()), wxFileName(this->cbFileName->GetValue()).GetName(), false).GetPath() : this->cbDir->GetValue();

		// 各ファイルにパスを付加。
		for (size_t i = 0; i < this->lcFiles->asInput->GetCount(); i++)
		{
			wxString szOutputFile = szOutputRootDir + wxFileName::GetPathSeparator();
			wxFileName fnStored(this->lcFiles->asInput->Item(i));
			if (! this->cbIgnorePath->IsChecked())
			{
				szOutputFile += fnStored.GetPathWithSep();
			}
			wxFileName fnOutput(szOutputFile + fnStored.GetFullName());
			if (! fnOutput.Normalize(wxPATH_NORM_DOTS | wxPATH_NORM_ABSOLUTE | wxPATH_NORM_LONG) || ! fnOutput.GetFullPath().StartsWith(szOutputRootDir))
			{
				fDTVWarning = true;
				this->lcFiles->apItem.Add(& this->lcFiles->atDangerItem);
			}
			else
			{
				this->lcFiles->apItem.Add(fnOutput.FileExists() || ::wxDirExists(fnOutput.GetFullPath()) ? & this->lcFiles->atExistItem : NULL);
			}
			this->lcFiles->asOutput.Add(fnOutput.GetFullPath());
		}
		break;
	}
	case TPI_COMMAND_ADD:
	case TPI_COMMAND_CREATE:
		// 格納パスを推測。
		for (size_t i = 0; i < this->lcFiles->asInput->GetCount(); i++)
		{
			wxFileName fnStored(this->lcFiles->asInput->Item(i));
			this->lcFiles->asOutput.Add(this->cbIgnorePath->IsChecked() ? fnStored.GetFullName() : fnStored.GetFullPath());
			fnStored = wxFileName(this->lcFiles->asOutput.Item(i));
			if (fnStored.GetPathWithSep(wxPATH_UNIX).Find(wxT("../")) != wxNOT_FOUND)
			{
				fDTVWarning = true;
				this->lcFiles->apItem.Add(& this->lcFiles->atDangerItem);
			}
			else
			{
				this->lcFiles->apItem.Add(NULL);
			}
		}
		break;
	}

	// DTV検査で異常があれば警告。
	if (fDTVWarning)
	{
		wxString sz;
		sz.Printf(
			_("This archive may have Directory Traversal Vulnerability(DTV) problem, %s It is strongly recommended to ignore file path. Would you like to do so?"),
			(this->uCommand == TPI_COMMAND_EXTRACT ?
				_("and some danger files may be extracted to the unexpected system directory!") :
				_("and some danger files may be stored!")
			)
		);
		if (::AskDlg(sz, this) == wxYES)
		{
			this->cbIgnorePath->SetValue(true);
			this->OnTabChanged(e);
		}
	}

	// リストビューに表示。
	this->lcFiles->SetItemCount(this->lcFiles->asInput->GetCount());
}

//******************************************************************************
// myListCtrl2
//******************************************************************************

IMPLEMENT_DYNAMIC_CLASS(myListCtrl2, wxListView)

//******************************************************************************
// Event handler.
//******************************************************************************
wxString myListCtrl2::OnGetItemText(long i, long column) const
{
	// リストビューに項目を追加。
	switch (column)
	{
	case 0:
		return this->asInput->Item(i);
	case 1:
		return this->asOutput[i];
	default:
		return wxEmptyString;
	}
}

wxListItemAttr * myListCtrl2::OnGetItemAttr(long i) const
{
	return (wxListItemAttr *) this->apItem[i];
}
