// MyString.h
// A drop-in replacement for CString, the main difference being that the
// data is scrubbed by trashstring() in the destructor, thus leaving an attacker
// with a little less info to grovel for in the swap file / core dump / whatever
//
// Note that the CString member m_mystring is public - this is for cases where
// a function requires a CString (reference) as an argument.
// You could argue that implementing a cast-to-CString operator would be more elegant...
// 
// --- following comments are by {kjp} ---
// A CString is a unicode (16-bit character) string, however the basic character is
// a TCHAR, which on most versions of Windows is just an ordinary 8-bit char.  But
// WinCE/PocketPC is unicode at heart and a TCHAR is really 16-bits.  This gives us
// a bit of a problem cuz parts of PasswordSafe assume 8-bit chars.  E.g. there are
// parts that assume that operator LPCSTR(...) is the same as operator LPCTSTR.
// When UNICODE is defined this is not true - LPCSTR is char* and LPCTSTR is wchar_t*.
// And as for encrypting/decrypting directly to a CString's buffer - don't get me
// started!!
//
// Anyway enough of the rant - and onto the unicode support - I'll add an operator
// to convert a string of TCHAR into a string of CHAR, and to securely wipe this
// buffer.  However to my mind this opens up too much scope for programming error
// and I hope to replace it sometime with a proper implementation.
//-----------------------------------------------------------------------------

#ifndef _MYSTRING_H_
#define _MYSTRING_H_
#include <AFX.H>
//-----------------------------------------------------------------------------
class CMyString
{
public:
   CMyString();
   CMyString(LPCSTR lpsz);
   CMyString(LPCSTR lpsz, int nLength);
   CMyString(const CMyString& stringSrc);
   CMyString(const CString& stringSrc);
   ~CMyString();

   TCHAR operator[](int nIndex) const;
   void SetAt(int nIndex, TCHAR ch);
   operator CString() const;
   operator CString&();
   operator LPCTSTR() const;
#ifdef UNICODE
   operator LPCSTR() const;
#endif
   BOOL IsEmpty() const;
   BOOL LoadString(UINT nID);

   const CMyString& operator=(const CMyString& stringSrc);
   const CMyString& operator=(TCHAR ch);
   const CMyString& operator=(LPCSTR lpsz);
   const CMyString& operator=(LPCWSTR lpsz);
   const CMyString& operator=(const unsigned char* psz);

   const CMyString& operator+=(const CMyString& string);
   const CMyString& operator+=(TCHAR ch);
   const CMyString& operator+=(LPCTSTR lpsz);

   // CMytring operator+(LPCTSTR lpsz);

   friend CMyString AFXAPI operator+(const CMyString& string1,
                                     const CMyString& string2);
   friend CMyString AFXAPI operator+(const CMyString& string,
                                     TCHAR ch);
   friend CMyString AFXAPI operator+(TCHAR ch,
                                     const CMyString& string);
   friend CMyString AFXAPI operator+(const CMyString& string,
                                     LPCTSTR lpsz);
   friend CMyString AFXAPI operator+(LPCTSTR lpsz,
                                     const CMyString& string);

   CMyString Mid(int nFirst, int nCount) const;

   LPTSTR GetBuffer(int nMinBufLength);
   void ReleaseBuffer(int nNewLength = -1);
   int GetLength() const;

   int FindByte( char ch ) const;
   int Find(TCHAR ch) const;
   int Find(LPCTSTR lpszSub) const;
   CString Left(int nCount) const;
   CString Right(int nCount) const;
  void TrimRight() {m_mystring.TrimRight();}
  void TrimLeft() {m_mystring.TrimLeft();}
  void MakeLower() {m_mystring.MakeLower();}

  void Trash() {trashstring();}

private:
#ifdef UNICODE
  volatile PCHAR	char_buffer;
  volatile int		char_buffer_len;

  void trashbuffer();
#endif
  CString m_mystring;
  void trashstring();
  void init();
};
//-----------------------------------------------------------------------------

bool operator==(const CMyString& s1, const CMyString& s2);
bool operator==(const CMyString& s1, LPCTSTR s2);
bool operator==(LPCTSTR s1, const CMyString& s2);
bool operator!=(const CMyString& s1, const CMyString& s2);
bool operator!=(const CMyString& s1, LPCTSTR s2);
bool operator!=(LPCTSTR s1, const CMyString& s2);

#ifdef UNICODE
bool operator==(const CMyString& s1, LPCSTR s2);
bool operator!=(const CMyString& s1, LPCSTR s2);
#endif

//-----------------------------------------------------------------------------
#endif
//-----------------------------------------------------------------------------
// Local variables:
// mode: c++
// End:
