# -*- coding: utf-8 -*-
# @Time    : 2021/1/22 10:57 上午
# @Author  : xcy
# @File    : endpoint_client.py
# @Software: PyCharm
"""
This module provides a client class for ENDPOINT.
"""
import copy
import json
import logging
import uuid

from baidubce.bce_base_client import BceBaseClient
from baidubce.auth import bce_v1_signer
from baidubce.http import bce_http_client
from baidubce.http import handler
from baidubce.http import http_methods

from baidubce import compat

_logger = logging.getLogger(__name__)


class EndpointClient(BceBaseClient):
    """

       Endpoint base sdk client
    """

    prefix = b'/v1'
    path = b'/endpoint'

    def __init__(self, config=None):
        BceBaseClient.__init__(self, config)

    @staticmethod
    def _generate_default_client_token():
        """

        default client token by uuid1
        """
        return uuid.uuid1()

    def _merge_config(self, config):
        """

        :type config: baidubce.BceClientConfiguration
        :param config:
        :return:
        """
        if config is None:
            return self.config
        else:
            new_config = copy.copy(self.config)
            new_config.merge_non_none_values(config)
            return new_config

    def _send_request(self, http_method, path, body=None, headers=None,
                      params=None, config=None, body_parser=None):
        """

        :param http_method:
        :param path:
        :param body:
        :param headers:
        :param params:

        :type config: baidubce.BceClientConfiguration
        :param config:

        :param body_parser:

        :return: baidubce.BceResponse
        """
        config = self._merge_config(config)
        if body_parser is None:
            body_parser = handler.parse_json

        if headers is None:
            headers = {b'Accept': b'*/*',
                       b'Content-Type': b'application/json;charset=utf-8'}

        return bce_http_client.send_request(config, bce_v1_signer.sign,
                                            [handler.parse_error, body_parser],
                                            http_method, EndpointClient.prefix + path, body, headers,
                                            params)

    def list_services(self, config=None):
        """

        return all services
        """
        path = EndpointClient.path + b'/publicService'
        return self._send_request(http_methods.GET, path, config=config)

    def create_endpoint(self, vpc_id, subnet_id, name, service, billing, description=None, ip_address=None,
                        client_token=None, config=None):
        """

        The method of endpoint to be created.

        :param vpc_id:
            vpc id
        :type vpc_id: str

        :param subnet_id:
            subnet id
        :type subnet_id: str

        :param name:
            name
        :type name: str

        :param service:
            service
        :type service: str

        :param billing:
           order_configuration
        :type billing:Billing

        :param ip_address:
            ip_address ipv4 address
        :type ip_address: str

        :param description:
            description .
        :type description: string

        :param client_token:
            An ASCII string whose length is less than 64.
            The request will be idempotent if clientToken is provided.
            If the clientToken is not specified by the user, a random String generated by default algorithm will be used.
        :type client_token: string
        """
        if client_token is None:
            client_token = self._generate_default_client_token()
        params = {
            b'clientToken': client_token
        }

        body = {
            "vpcId": vpc_id,
            "subnetId": subnet_id,
            "name": name,
            "service": service,
            "billing": {
                "paymentTiming": billing.payment_timing
            }
        }
        if description is not None:
            body['description'] = description
        if ip_address is not None:
            body['ipAddress'] = ip_address
        return self._send_request(http_methods.POST, EndpointClient.path, body=json.dumps(body), params=params,
                                  config=config)

    def delete_endpoint(self, endpoint_id, client_token=None, config=None):
        """

        release endpoint

        :param endpoint_id:
            The id of endpoint.
        :type endpoint_id: string

        :param client_token:
            An ASCII string whose length is less than 64.
            The request will be idempotent if clientToken is provided.
            If the clientToken is not specified by the user, a random String generated by default algorithm will
            be used.
        :type client_token: string

        :param config:
        :type config: baidubce.BceClientConfiguration

        :return:
        :rtype baidubce.bce_response.BceResponse
        """
        path = EndpointClient.path + b'/' + compat.convert_to_bytes(endpoint_id)
        if client_token is None:
            client_token = self._generate_default_client_token()
        params = {
            b'clientToken': client_token
        }
        return self._send_request(http_methods.DELETE, path, params=params, config=config)

    def list_endpoints(self, vpc_id, name=None, ip_address=None, status=None, subnet_id=None, service=None, marker=None,
                       max_Keys=None, config=None):
        """

        return all endpoint about vpc

        :param vpc_id:
            vpc id
        :type vpc_id:string

        :param name:
            name
        :type name:string

        :param ip_address:
            ip address
        :type ip_address:string

        :param status:
            status
        :type status:string

        :param subnet_id:
            subnet id
        :type subnet_id:string

        :param service:
            service
        :type service:string

        :param marker:
            The optional parameter marker specified in the original request to specify
            where in the results to begin listing.
            Together with the marker, specifies the list result which listing should begin.
            If the marker is not specified, the list result will listing from the first one.
        :type marker: string

        :param max_Keys:
            The optional parameter to specifies the max number of list result to return.
            The default value is 1000.
        :type max_Keys: int

        :param config:
        :type config: baidubce.BceClientConfiguration

        :return:
        :rtype baidubce.bce_response.BceResponse
        """
        params = {b'vpcId': vpc_id}
        if name is not None:
            params[b'name'] = name
        if ip_address is not None:
            params[b'ipAddress'] = ip_address
        if status is not None:
            params[b'status'] = status
        if subnet_id is not None:
            params[b'subnetId'] = subnet_id
        if service is not None:
            params[b'service'] = service

        if marker is not None:
            params[b'marker'] = marker
        if max_Keys is not None:
            params[b'maxKeys'] = max_Keys

        return self._send_request(http_methods.GET, EndpointClient.path, params=params, config=config)

    def get_endpoint(self, endpoint_id, config=None):
        """

        Get the detail information of  endpoint.

        :param endpoint_id:
            The id of endpoint.
        :type endpoint_id: string

        :param config:
        :type config: baidubce.BceClientConfiguration

        :return:
        :rtype baidubce.bce_response.BceResponse
        """
        path = EndpointClient.path + b'/' + compat.convert_to_bytes(endpoint_id)

        return self._send_request(http_methods.GET, path, config=config)

    def update_endpoint(self, endpoint_id, name=None, description=None, client_token=None, config=None):
        """

        The method of endpoint to be update.

        :param endpoint_id: endpoint id
        :type endpoint_id: str

        :param name: endpoint name
        :type name: str

        :param description: the description of endpoint
        :type description: str

        :param client_token:
            An ASCII string whose length is less than 64.
            The request will be idempotent if clientToken is provided.
            If the clientToken is not specified by the user, a random String generated by default algorithm will be used.
        :type client_token: string

        :param config:
        :type config: baidubce.BceClientConfiguration

        :return:
        :rtype baidubce.bce_response.BceResponse
        """

        path = EndpointClient.path + b'/' + compat.convert_to_bytes(endpoint_id)

        if client_token is None:
            client_token = self._generate_default_client_token()
        params = {
            b'clientToken': client_token
        }

        body = {}
        if description is not None:
            body['description'] = description
        if name is not None:
            body['name'] = name

        return self._send_request(http_methods.PUT, path, body=json.dumps(body), params=params,
                                  config=config)
