package org.seasar.util;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.util.AbstractMap;
import java.util.AbstractSet;
import java.util.Iterator;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;

public final class SMap extends AbstractMap implements Map, Cloneable, Externalizable {

    private final static int INITIAL_CAPACITY = 17;
    private final static float LOAD_FACTOR = 0.75f;
    private final static long serialVersionUID = 1383533715771521971L;
    private transient int _threshold;
    private transient Entry[] _mapTable;
    private transient int _size = 0;
    private transient Set _entrySet = null;

    public SMap() {
        this(INITIAL_CAPACITY);
    }

    public SMap(int initialCapacity) {
        if (initialCapacity <= 0) {
            initialCapacity = INITIAL_CAPACITY;
        }
        _mapTable = new Entry[initialCapacity];
        _threshold = (int) (initialCapacity * LOAD_FACTOR);
    }
    
    public SMap(Map map) {
    	this((int) (map.size() / LOAD_FACTOR) + 1);
        putAll(map);
    }

    public final int size() {
        return _size;
    }

    public final boolean isEmpty() {
        return _size == 0;
    }

    public final boolean containsValue(final Object value) {
        Entry tab[] = _mapTable;
		if (value == null) {
		    for (int i = 0; i < tab.length; i++) {
	                for (Entry e = tab[i]; e != null ; e = e._next) {
			    if (e._value == null) {
				return true;
	                    }
	                }
	            }
		} else {
		    for (int i = 0; i < tab.length; i++) {
	                for (Entry e = tab[i]; e != null ; e = e._next) {
			    if (value.equals(e._value)) {
				return true;
	                    }
	                }
	            }
		}
		return false;
    }

    public final boolean containsKey(final Object key) {
        Entry[] tab = _mapTable;
        if (key != null) {
            int hashCode = key.hashCode();
            int index = (hashCode & 0x7FFFFFFF) % tab.length;
            for (Entry e = tab[index]; e != null; e = e._next) {
                if (e._hashCode == hashCode && key.equals(e._key)) {
                    return true;
                }
            }
        } else {
            for (Entry e = tab[0]; e != null; e = e._next) {
                if (e._key == null) {
                    return true;
                }
            }
        }
        return false;
    }

    public final Object get(final Object key) {
        Entry[] tab = _mapTable;
        if (key != null) {
            int hashCode = key.hashCode();
            int index = (hashCode & 0x7FFFFFFF) % tab.length;
            for (Entry e = tab[index]; e != null; e = e._next) {
                if (e._hashCode == hashCode && key.equals(e._key)) {
                    return e._value;
                }
            }
        } else {
            for (Entry e = tab[0]; e != null; e = e._next) {
                if (e._key == null) {
                    return e._value;
                }
            }
        }
        return null;
    }

    public final Object put(final Object key, final Object value) {
        Entry[] tab = _mapTable;
        int hashCode = 0;
        int index = 0;
        if (key != null) {
            hashCode = key.hashCode();
            index = (hashCode & 0x7FFFFFFF) % tab.length;
            for (Entry e = tab[index]; e != null; e = e._next) {
                if ((e._hashCode == hashCode) && key.equals(e._key)) {
                    return swapValue(e, value);
                }
            }
        } else {
            for (Entry e = tab[0]; e != null; e = e._next) {
                if (e._key == null) {
                    return swapValue(e, value);
                }
            }
        }
        ensureCapacity();
        tab = _mapTable;
        index = (hashCode & 0x7FFFFFFF) % tab.length;
        Entry e = new Entry(hashCode, key, value, tab[index]);
        tab[index] = e;
        _size++;
        return null;
    }

    public final Object remove(final Object key) {
        Entry e = removeMap(key);
        if (e != null) {
        	Object old = e._value;
        	e.clear();
            return old;
        } else {
            return null;
        }
    }

    public final void putAll(Map map) {
        for (Iterator i = map.entrySet().iterator(); i.hasNext(); ) {
            Map.Entry e = (Map.Entry) i.next();
            put(e.getKey(), e.getValue());
        }
    }

    public final void clear() {
        for (int i = 0; i < _mapTable.length; i++) {
            _mapTable[i] = null;
        }
        _size = 0;
    }

    public final boolean equals(Object o) {
		if (o == this) {
		    return true;
		}
		if (!(o instanceof Map) || o == null) {
		    return false;
	    }
		Map t = (Map) o;
		if (t.size() != size()) {
		    return false;
	    }
		Iterator i = entrySet().iterator();
		while (i.hasNext()) {
		    Entry e = (Entry) i.next();
		    Object key = e.getKey();
		    Object value = e.getValue();
		    if (value == null) {
			if (!(t.get(key)==null && t.containsKey(key)))
			    return false;
		    } else {
			if (!value.equals(t.get(key)))
			    return false;
		    }
		}
		return true;
    }


    public final Object clone() {
        try {
		    SMap copy = (SMap) super.clone();
		    copy._mapTable = new Entry[_mapTable.length];
		    for (int i = 0; i < _mapTable.length; i++) {
				copy._mapTable[i] = _mapTable[i] != null
				    ? (Entry) _mapTable[i].clone() : null;
		    }
		    copy._entrySet = null;
		    return copy;
		} catch (CloneNotSupportedException ex) {
		    throw new InternalError();
		}
    }

    public final Set entrySet() {
        if (_entrySet == null) {
            _entrySet =
                new AbstractSet() {
                    public Iterator iterator() {
                        return new SMapIterator();
                    }

                    public boolean contains(Object o) {
                        if (!(o instanceof Entry)) {
                            return false;
                        }
                        Entry entry = (Entry) o;
                        int index = (entry._hashCode & 0x7FFFFFFF) % _mapTable.length;
                        for (Entry e = _mapTable[index]; e != null; e = e._next) {
                            if (e.equals(entry)) {
                                return true;
                            }
                        }
                        return false;
                    }


                    public boolean remove(Object o) {
                        if (!(o instanceof Entry)) {
                            return false;
                        }
                        Entry entry = (Entry) o;
                        return SMap.this.remove(entry._key) != null;
                    }

                    public int size() {
                        return _size;
                    }

                    public void clear() {
                        SMap.this.clear();
                    }
                };
        }
        return _entrySet;
    }

    public final void writeExternal(final ObjectOutput out) throws IOException {
        Entry[] tab = _mapTable;
		out.writeInt(tab.length);
		out.writeInt(_size);
		for (int i = 0; i < tab.length; i++) {
		    Entry entry = tab[i];
		    while (entry != null) {
				out.writeObject(entry._key);
				out.writeObject(entry._value);
				entry = entry._next;
		    }
		}
    }

    public final void readExternal(final ObjectInput in)
            throws IOException, ClassNotFoundException {

		int num = in.readInt();
        _mapTable = new Entry[num];
        _threshold = (int) (num * LOAD_FACTOR);
        int size = in.readInt();
        for (int i = 0; i < size; i++) {
            Object key = in.readObject();
            Object value = in.readObject();
            put(key, value);
        }
    }

    private final Entry removeMap(Object key) {
        Entry[] tab = _mapTable;
        if (key != null) {
            int hashCode = key.hashCode();
            int index = (hashCode & 0x7FFFFFFF) % tab.length;
            for (Entry e = tab[index], prev = null; e != null; prev = e, e = e._next) {
                if ((e._hashCode == hashCode) && (key.equals(e._key))) {
                    if (prev != null) {
                        prev._next = e._next;
                    } else {
                        tab[index] = e._next;
                    }
                    _size--;
                    return e;
                }
            }
        } else {
            for (Entry e = tab[0], prev = null; e != null; prev = e, e = e._next) {
                if (e._key == null) {
                    if (prev != null) {
                        prev._next = e._next;
                    } else {
                        tab[0] = e._next;
                    }
                    _size--;
                    return e;
                }
            }
        }
        return null;
    }

    private final void ensureCapacity() {
        if (_size >= _threshold) {
            int oldCapacity = _mapTable.length;
            Entry[] oldTab = _mapTable;
            int newCapacity = oldCapacity * 2 + 1;
            Entry[] newTab = new Entry[newCapacity];
            _threshold = (int)(newCapacity * LOAD_FACTOR);
            _mapTable = newTab;
            for (int i = 0; i < oldCapacity; i++) {
                for (Entry old = oldTab[i] ; old != null; ) {
                    Entry e = old;
                    old = old._next;
                    int index = (e._hashCode & 0x7FFFFFFF) % newCapacity;
                    e._next = newTab[index];
                    newTab[index] = e;
                }
            }
        }
    }

    private final Object swapValue(final Entry entry, final Object value) {
        Object old = entry._value;
        entry._value = value;
        return old;
    }

    private final class SMapIterator implements Iterator {

		private int _index = SMap.this._mapTable.length;
		private Entry _entry = null;
		private Entry _lastReturned = null;

        public boolean hasNext() {
		    Entry e = _entry;
		    int i = _index;
		    Entry t[] = SMap.this._mapTable;
		    while (e == null && i > 0) {
				e = t[--i];
	        }
		    _entry = e;
		    _index = i;
		    return e != null;
		}

		public Object next() {
		    Entry et = _entry;
		    int i = _index;
		    Entry t[] = SMap.this._mapTable;
		    while (et == null && i > 0) {
				et = t[--i];
	        }
		    _entry = et;
		    _index = i;
		    if (et != null) {
				Entry e = _lastReturned = _entry;
				_entry = e._next;
				return e;
		    }
		    throw new NoSuchElementException();
		}

		public void remove() {
	    	if (_lastReturned == null) {
				throw new IllegalStateException();
            }
	    	SMap.this.remove(_lastReturned._key);
            _lastReturned = null;
		}
    }

    private static final class Entry implements Map.Entry, Externalizable {

        transient int _hashCode;
        transient Object _key;
        transient Object _value;
        transient Entry _next;

        public Entry(final int hashCode, final Object key, final Object value,
                final Entry next) {

            _hashCode = hashCode;
            _key = key;
            _value = value;
            _next = next;
        }

        public Object clone() {
            Object value = _value;
            if (value instanceof Cloneable) {
                value = Reflector.clone(value);
            }
            return new Entry(_hashCode, _key, value,
                    (_next == null ? null : (Entry) _next.clone()));
        }

        public Object getKey() {
            return _key;
        }

        public Object getValue() {
            return _value;
        }

        public Object setValue(final Object value) {
            Object oldValue = _value;
            _value = value;
            return oldValue;
        }
        
        public void clear() {
            _key = null;
            _value = null;
            _next = null;
        }

        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
            Entry e = (Entry) o;
            return (_key != null ? _key.equals(e._key) : e._key == null) &&
                    (_value != null ? _value.equals(e._value) : e._value == null);
        }

        public int hashCode() {
            return _hashCode;
        }

        public String toString() {
            return _key + "=" + _value;
        }

        public void writeExternal(final ObjectOutput s) throws IOException {
            s.writeInt(_hashCode);
            s.writeObject(_key);
            s.writeObject(_value);
            s.writeObject(_next);
        }

        public void readExternal(final ObjectInput s)
                 throws IOException, ClassNotFoundException {

            _hashCode = s.readInt();
            _key = s.readObject();
            _value = s.readObject();
            _next = (Entry) s.readObject();
        }
    }
}
