package org.seasar.system;

import java.io.EOFException;
import java.io.IOException;
import java.io.StreamCorruptedException;
import java.net.ServerSocket;
import java.net.Socket;

import org.seasar.log.Logger;
import org.seasar.util.NetUtil;
import org.seasar.util.SeasarException;
import org.seasar.util.ThreadUtil;

public abstract class AbstractSocketService implements Lifecycle, Runnable {

	private String _hostName = NetUtil.getHostAddress();
    private int _port = 0;
    private int _backlog = 50;
    private int _soTimeout = 0;
    private ServerSocket _serverSocket = null;
    private ThreadPool _threadPool = new ThreadPool();
    private boolean _started = false;


    public AbstractSocketService() {
    }
    
    public final String getHostName() {
    	return _hostName;
    }

    public final int getPort() {
        return _port;
    }

    public final void setPort(int port) {
        _port = port;
    }

    public final int getBacklog() {
        return _backlog;
    }

    public final void setBacklog(int backlog) {
        _backlog = backlog;
    }

    public final int getSoTimeout() {
        return _soTimeout;
    }

    public final void setSoTimeout(int soTimeout) {
        _soTimeout = soTimeout;
    }

    public final boolean isStarted() {
        return _started;
    }

    public void start() throws SeasarException {
        _threadPool.setMaxSize(_backlog);
        _threadPool.start();
        try {
            _serverSocket = new ServerSocket(_port, _backlog);
            if (_port == 0) {
                _port = _serverSocket.getLocalPort();
            }
            listen();
        } catch (IOException ex) {
            throw new SeasarException("ESSR0017", new Object[]{ex}, ex);
        }
        while (!_started) {
        	ThreadUtil.sleep(10);
        }
    }

    public void stop() throws SeasarException {
        try {
            ServerSocket ss = _serverSocket;
            _serverSocket = null;
            ss.close();
            _threadPool.stop();
            _started = false;
        } catch (Exception ex) {
        	Logger.getLogger(getClass()).log(ex);
        }
    }


    public final void run() {
        if (_serverSocket == null) {
            return;
        }
        Socket socket = accept();
        if (socket == null) {
            return;
        }
        listen();
        try {
            handleSocket(socket);
        } catch (EOFException ignore) {
        } catch (StreamCorruptedException ignore2) {
        } catch (Throwable t) {
       		Logger.getLogger(getClass()).log(t);
        } finally {
            try {
                socket.close();
            } catch (IOException ex) {
                Logger.getLogger(getClass()).log(ex);
            }
        }
    }

    protected Socket accept() {
        try {
            _started = true;
            Socket socket = _serverSocket.accept();
            socket.setSoTimeout(_soTimeout);
            return socket;
        } catch (IOException ex) {
            if (_serverSocket != null) {
                Logger.getLogger(getClass()).log(ex);
            }
            return null;
        }
    }

    protected abstract void handleSocket(final Socket socket) throws IOException;

    private void listen() {
        _threadPool.request(this);
    }
}
