package org.seasar.sql;

import java.sql.CallableStatement;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.SQLWarning;
import java.sql.Savepoint;
import java.sql.Statement;
import java.util.Map;

import org.seasar.util.SeasarRuntimeException;

public final class ConnectionImpl implements Connection {

    private XAConnectionImpl _xaConnection;
    private Connection _physicalConnection;

    public ConnectionImpl(final XAConnectionImpl xaConnection, final Connection physicalConnection) {
        if (xaConnection == null) {
            throw new SeasarRuntimeException("ESSR0007", new Object[]{"xaConnection"});
        }
        _xaConnection = xaConnection;
        if (physicalConnection == null) {
            throw new SeasarRuntimeException("ESSR0007", new Object[]{"physicalConnection"});
        }
        _physicalConnection = physicalConnection;
    }
    
    public Connection getPhysicalConnection() {
    	return _physicalConnection;
    }

    public Statement createStatement() throws SQLException {
    	try {
        	return _physicalConnection.createStatement();
    	} catch (SQLException ex) {
    		_xaConnection.notifyError(ex);
    		throw ex;
    	}
    }

    public PreparedStatement prepareStatement(final String sql) throws SQLException {
    	try {
        	return _physicalConnection.prepareStatement(sql);
        } catch (SQLException ex) {
    		_xaConnection.notifyError(ex);
    		throw ex;
    	}
    }

    public CallableStatement prepareCall(final String sql) throws SQLException {
    	try {
        	return _physicalConnection.prepareCall(sql);
    	} catch (SQLException ex) {
    		_xaConnection.notifyError(ex);
    		throw ex;
    	}
    }

    public String nativeSQL(final String sql) throws SQLException {
    	try {
        	return _physicalConnection.nativeSQL(sql);
        } catch (SQLException ex) {
    		_xaConnection.notifyError(ex);
    		throw ex;
    	}
    }

    public boolean isClosed() throws SQLException {
    	if (_physicalConnection == null) {
    		return true;
    	}
    	try {
        	return _physicalConnection.isClosed();
        } catch (SQLException ex) {
    		_xaConnection.notifyError(ex);
    		throw ex;
    	}
    }

    public DatabaseMetaData getMetaData() throws SQLException {
    	try {
        	return _physicalConnection.getMetaData();
        } catch (SQLException ex) {
    		_xaConnection.notifyError(ex);
    		throw ex;
    	}
    }

    public void setReadOnly(final boolean readOnly) throws SQLException {
    	try {
        	_physicalConnection.setReadOnly(readOnly);
        } catch (SQLException ex) {
    		_xaConnection.notifyError(ex);
    		throw ex;
    	}
    }

    public boolean isReadOnly() throws SQLException {
    	try {
        	return _physicalConnection.isReadOnly();
        } catch (SQLException ex) {
    		_xaConnection.notifyError(ex);
    		throw ex;
    	}
    }

    public void setCatalog(final String catalog) throws SQLException {
    	try {
        	_physicalConnection.setCatalog(catalog);
        } catch (SQLException ex) {
    		_xaConnection.notifyError(ex);
    		throw ex;
    	}
    }

    public String getCatalog() throws SQLException {
    	try {
        	return _physicalConnection.getCatalog();
        } catch (SQLException ex) {
    		_xaConnection.notifyError(ex);
    		throw ex;
    	}
    }

    public void close() throws SQLException {
    	if (_physicalConnection == null) {
    		return;
    	}
        _xaConnection.notifyClose();
    }

    public void setTransactionIsolation(final int level) throws SQLException {
    	try {
        	_physicalConnection.setTransactionIsolation(level);
        } catch (SQLException ex) {
    		_xaConnection.notifyError(ex);
    		throw ex;
    	}
    }

    public int getTransactionIsolation() throws SQLException {
    	try {
        	return _physicalConnection.getTransactionIsolation();
        } catch (SQLException ex) {
    		_xaConnection.notifyError(ex);
    		throw ex;
    	}
    }

    public SQLWarning getWarnings() throws SQLException {
    	try {
        	return _physicalConnection.getWarnings();
        } catch (SQLException ex) {
    		_xaConnection.notifyError(ex);
    		throw ex;
    	}
    }

    public void clearWarnings() throws SQLException {
    	try {
        	_physicalConnection.clearWarnings();
        } catch (SQLException ex) {
    		_xaConnection.notifyError(ex);
    		throw ex;
    	}
    }

    public void commit() throws SQLException {
        try {
        	_physicalConnection.commit();
        } catch (SQLException ex) {
    		_xaConnection.notifyError(ex);
    		throw ex;
    	}
    }

    public void rollback() throws SQLException {
        try {
        	_physicalConnection.rollback();
        } catch (SQLException ex) {
    		_xaConnection.notifyError(ex);
    		throw ex;
    	}
    }

    public void setAutoCommit(boolean autoCommit) throws SQLException {
    	try {
    		_physicalConnection.setAutoCommit(autoCommit);
    	} catch (SQLException ex) {
    		_xaConnection.notifyError(ex);
    		throw ex;
    	}
    }

    public boolean getAutoCommit() throws SQLException {
    	try {
        	return _physicalConnection.getAutoCommit();
        } catch (SQLException ex) {
    		_xaConnection.notifyError(ex);
    		throw ex;
    	}
    }

    public Statement createStatement(final int resultSetType, final int resultSetConcurrency)
             throws SQLException {

		try {
        	return _physicalConnection.createStatement(resultSetType, resultSetConcurrency);
        } catch (SQLException ex) {
    		_xaConnection.notifyError(ex);
    		throw ex;
    	}
    }

    public Map getTypeMap() throws SQLException {
    	try {
        	return _physicalConnection.getTypeMap();
        } catch (SQLException ex) {
    		_xaConnection.notifyError(ex);
    		throw ex;
    	}
    }

    public void setTypeMap(final Map map) throws SQLException {
    	try {
        	_physicalConnection.setTypeMap(map);
        } catch (SQLException ex) {
    		_xaConnection.notifyError(ex);
    		throw ex;
    	}
    }

    public PreparedStatement prepareStatement(final String sql, final int resultSetType,
            final int resultSetConcurrency) throws SQLException {

		try {
        	return _physicalConnection.prepareStatement(sql, resultSetType, resultSetConcurrency);
        } catch (SQLException ex) {
    		_xaConnection.notifyError(ex);
    		throw ex;
    	}
    }

    public CallableStatement prepareCall(final String sql, final int resultSetType,
            final int resultSetConcurrency) throws SQLException {

		try {
        	return _physicalConnection.prepareCall(sql, resultSetType, resultSetConcurrency);
        } catch (SQLException ex) {
    		_xaConnection.notifyError(ex);
    		throw ex;
    	}
    }

    public void setHoldability(int holdability) throws SQLException {
    	try {
        	_physicalConnection.setHoldability(holdability);
        } catch (SQLException ex) {
    		_xaConnection.notifyError(ex);
    		throw ex;
    	}
    }

    public int getHoldability() throws SQLException {
    	try {
        	return _physicalConnection.getHoldability();
        } catch (SQLException ex) {
    		_xaConnection.notifyError(ex);
    		throw ex;
    	}
    }

    public Savepoint setSavepoint() throws SQLException {
    	try {
        	return _physicalConnection.setSavepoint();
        } catch (SQLException ex) {
    		_xaConnection.notifyError(ex);
    		throw ex;
    	}
    }

    public Savepoint setSavepoint(String name) throws SQLException {
    	try {
        	return _physicalConnection.setSavepoint(name);
        } catch (SQLException ex) {
    		_xaConnection.notifyError(ex);
    		throw ex;
    	}
    }

    public void rollback(Savepoint savepoint) throws SQLException {
    	try {
        	_physicalConnection.rollback(savepoint);
        } catch (SQLException ex) {
    		_xaConnection.notifyError(ex);
    		throw ex;
    	}
    }

    public void releaseSavepoint(Savepoint savepoint) throws SQLException {
    	try {
        	_physicalConnection.releaseSavepoint(savepoint);
        } catch (SQLException ex) {
    		_xaConnection.notifyError(ex);
    		throw ex;
    	}
    }

    public Statement createStatement(int resultSetType,
    		int resultSetConcurrency, int resultSetHoldability)
            throws SQLException {
                            
        try {
        	return _physicalConnection.createStatement(resultSetType,
            	resultSetConcurrency, resultSetHoldability);
        } catch (SQLException ex) {
    		_xaConnection.notifyError(ex);
    		throw ex;
    	}
    }

    public PreparedStatement prepareStatement(String sql, int resultSetType,
    		int resultSetConcurrency, int resultSetHoldability) throws SQLException {
    			
    	try {
        	return _physicalConnection.prepareStatement(sql, resultSetType, resultSetConcurrency, resultSetHoldability);
        } catch (SQLException ex) {
    		_xaConnection.notifyError(ex);
    		throw ex;
    	}
    }

    public CallableStatement prepareCall(String sql, int resultSetType,
    		int resultSetConcurrency, int resultSetHoldability)
            throws SQLException {
            	
        try {
        	return _physicalConnection.prepareCall(sql, resultSetType, resultSetConcurrency, resultSetHoldability);
        } catch (SQLException ex) {
    		_xaConnection.notifyError(ex);
    		throw ex;
    	}
    }

    public PreparedStatement prepareStatement(String sql, int autoGeneratedKeys) throws SQLException {
    	try {
        	return _physicalConnection.prepareStatement(sql, autoGeneratedKeys);
        } catch (SQLException ex) {
    		_xaConnection.notifyError(ex);
    		throw ex;
    	}
    }

    public PreparedStatement prepareStatement(String sql, int[] columnIndexes) throws SQLException {
    	try {
        	return _physicalConnection.prepareStatement(sql, columnIndexes);
        } catch (SQLException ex) {
    		_xaConnection.notifyError(ex);
    		throw ex;
    	}
    }

    public PreparedStatement prepareStatement(String sql, String[] columnNames) throws SQLException {
    	try {
        	return _physicalConnection.prepareStatement(sql, columnNames);
        } catch (SQLException ex) {
    		_xaConnection.notifyError(ex);
    		throw ex;
    	}
    }
}