package org.seasar.mbean;

import java.util.Properties;

import org.seasar.log.Logger;
import org.seasar.naming.NamingContext;
import org.seasar.naming.NamingService;
import org.seasar.system.JMXService;
import org.seasar.system.JMXUtil;
import org.seasar.system.Lifecycle;
import org.seasar.system.MBeanProxy;
import org.seasar.util.Reflector;
import org.seasar.util.SeasarRuntimeException;
import org.seasar.util.StringUtil;

public final class MBeanFacade implements Lifecycle {

	private static final Logger _logger = Logger.getLogger(MBeanFacade.class);
	private String _className;
	private String _name;
	private Properties _properties = new Properties();
	private Object _mbean;
	private Object _mbeanProxy;
	
	public MBeanFacade(String className, String name) {
		if (className == null) {
			throw new SeasarRuntimeException("ESSR0007", new Object[]{"className"});
		}
		_className = className;
		if (name != null) {
			_name = name;
		} else {
        	_name = StringUtil.getLastToken(className, '.');
        }
	}
	
	public void setProperty(String name, String value) {
		_properties.setProperty(name, value);
	}
	
	public Object getMBean() {
		return _mbean;
	}
	
	public Object getMBeanProxy() {
		return _mbeanProxy;
	}
	
	public void start() {
		try {
			_mbean = Reflector.newInstance(_className);
			Reflector.setProperties(_mbean, _properties);
			if (_mbean instanceof Lifecycle) {
				((Lifecycle) _mbean).start();
			}
			JMXService.registerMBean(_mbean, JMXUtil.getMBeanName(_name));
			_mbeanProxy = MBeanProxy.create(
				JMXUtil.getMBeanInterface(_mbean.getClass()),
				JMXUtil.getMBeanName(_name));
			NamingContext ctx = NamingService.getNamingServer().getNamingContext();
			ctx.bind(JMXUtil.getJNDIName(_name), _mbeanProxy);
		} catch (Throwable t) {
			_logger.log(t);
		}
	}
	
	public void stop() {
		try {
			NamingContext ctx = NamingService.getNamingServer().getNamingContext();
			ctx.unbind(JMXUtil.getJNDIName(_name));
			JMXService.unregisterMBean(JMXUtil.getMBeanName(_name));
			if (_mbean instanceof Lifecycle) {
				((Lifecycle) _mbean).stop();
			}
		} catch (Throwable t) {
			_logger.log(t);
		} finally {
			_mbeanProxy = null;
			_mbean = null;
		}
	}
}
