/*
 * (C) Copyright 2003
 * Wolfgang Denk, DENX Software Engineering, wd@denx.de.
 *
 * SPDX-License-Identifier:	GPL-2.0+
 */

#include <common.h>
#include <asm/inca-ip.h>


/*******************************************************************************
*
* get_cpuclk - returns the frequency of the CPU.
*
* Gets the value directly from the INCA-IP hardware.
*
* RETURNS:
*          150.000.000 for 150 MHz
*          133.333.333 for 133 MHz (= 400MHz/3)
*          100.000.000 for 100 MHz (= 400MHz/4)
* NOTE:
*   This functions should be used by the hardware driver to get the correct
*   frequency of the CPU. Don't use the macros, which are set to init the CPU
*   frequency in the ROM code.
*/
uint incaip_get_cpuclk (void)
{
	/*-------------------------------------------------------------------------*/
	/* CPU Clock Input Multiplexer (MUX I)                                     */
	/* Multiplexer MUX I selects the maximum input clock to the CPU.           */
	/*-------------------------------------------------------------------------*/
	if (*((volatile ulong *) INCA_IP_CGU_CGU_MUXCR) &
	    INCA_IP_CGU_CGU_MUXCR_MUXI) {
		/* MUX I set to 150 MHz clock */
		return 150000000;
	} else {
		/* MUX I set to 100/133 MHz clock */
		if (*((volatile ulong *) INCA_IP_CGU_CGU_DIVCR) & 0x40) {
			/* Division value is 1/3, maximum CPU operating */
			/* frequency is 133.3 MHz                       */
			return 133333333;
		} else {
			/* Division value is 1/4, maximum CPU operating */
			/* frequency is 100 MHz                         */
			return 100000000;
		}
	}
}

/*******************************************************************************
*
* get_fpiclk - returns the frequency of the FPI bus.
*
* Gets the value directly from the INCA-IP hardware.
*
* RETURNS: Frquency in Hz
*
* NOTE:
*   This functions should be used by the hardware driver to get the correct
*   frequency of the CPU. Don't use the macros, which are set to init the CPU
*   frequency in the ROM code.
*   The calculation for the
*/
uint incaip_get_fpiclk (void)
{
	uint clkCPU;

	clkCPU = incaip_get_cpuclk ();

	switch (*((volatile ulong *) INCA_IP_CGU_CGU_DIVCR) & 0xC) {
	case 0x4:
		return clkCPU >> 1;	/* devided by 2 */
		break;
	case 0x8:
		return clkCPU >> 2;	/* devided by 4 */
		break;
	default:
		return clkCPU;
		break;
	}
}

int incaip_set_cpuclk (void)
{
	extern void ebu_init(long);
	extern void cgu_init(long);
	extern void sdram_init(long);
	char tmp[64];
	ulong cpuclk;

	if (getenv_f("cpuclk", tmp, sizeof (tmp)) > 0) {
		cpuclk = simple_strtoul (tmp, NULL, 10) * 1000000;
		cgu_init (cpuclk);
		ebu_init (cpuclk);
		sdram_init (cpuclk);
	}

	return 0;
}
