/*
 * Intracom TI6711/TI6412 DSP
 */

#include <common.h>
#include <post.h>

#include "mpc8xx.h"

struct ram_range {
	u32 start;
	u32 size;
};

#if defined(CONFIG_NETTA_6412)

static const struct ram_range int_ram[] = {
	{ 0x00000000U, 0x00040000U },
};

static const struct ram_range ext_ram[] = {
	{ 0x80000000U, 0x00100000U },
};

static const struct ram_range ranges[] = {
	{ 0x00000000U, 0x00040000U },
	{ 0x80000000U, 0x00100000U },
};

static inline u16 bit_invert(u16 d)
{
	register u8 i;
	register u16 r;
	register u16 bit;

	r = 0;
	for (i = 0; i < 16; i++) {
		bit = d & (1 << i);
		if (bit != 0)
			r |= 1 << (15 - i);
	}
	return r;
}

#else

static const struct ram_range int_ram[] = {
	{ 0x00000000U, 0x00010000U },
};

static const struct ram_range ext_ram[] = {
	{ 0x80000000U, 0x00100000U },
};

static const struct ram_range ranges[] = {
	{ 0x00000000U, 0x00010000U },
	{ 0x80000000U, 0x00100000U },
};

#endif

/*******************************************************************************************************/

static inline int addr_in_int_ram(u32 addr)
{
	int i;

	for (i = 0; i < sizeof(int_ram)/sizeof(int_ram[0]); i++)
		if (addr >= int_ram[i].start && addr < int_ram[i].start + int_ram[i].size)
			return 1;

	return 0;
}

static inline int addr_in_ext_ram(u32 addr)
{
	int i;

	for (i = 0; i < sizeof(ext_ram)/sizeof(ext_ram[0]); i++)
		if (addr >= ext_ram[i].start && addr < ext_ram[i].start + ext_ram[i].size)
			return 1;

	return 0;
}

/*******************************************************************************************************/

#define DSP_HPIC	0x0
#define DSP_HPIA	0x4
#define DSP_HPID1	0x8
#define DSP_HPID2	0xC

static u32 dummy_delay;
static volatile u32 *ti6711_delay = &dummy_delay;

static inline void dsp_go_slow(void)
{
	volatile memctl8xx_t *memctl = &((immap_t *)CFG_IMMR)->im_memctl;
#if defined(CONFIG_NETTA_6412)
	memctl->memc_or6 |= OR_SCY_15_CLK | OR_TRLX;
#else
	memctl->memc_or2 |= OR_SCY_15_CLK | OR_TRLX;
#endif
	memctl->memc_or5 |= OR_SCY_15_CLK | OR_TRLX;

	ti6711_delay = (u32 *)DUMMY_BASE;
}

static inline void dsp_go_fast(void)
{
	volatile memctl8xx_t *memctl = &((immap_t *)CFG_IMMR)->im_memctl;
#if defined(CONFIG_NETTA_6412)
	memctl->memc_or6 = (memctl->memc_or6 & ~(OR_SCY_15_CLK | OR_TRLX)) | OR_SCY_0_CLK;
#else
	memctl->memc_or2 = (memctl->memc_or2 & ~(OR_SCY_15_CLK | OR_TRLX)) | OR_SCY_3_CLK;
#endif
	memctl->memc_or5 = (memctl->memc_or5 & ~(OR_SCY_15_CLK | OR_TRLX)) | OR_SCY_0_CLK;

	ti6711_delay = &dummy_delay;
}

/*******************************************************************************************************/

static inline void dsp_delay(void)
{
	/* perform ti6711_delay chip select read to have a small delay */
	(void) *(volatile u32 *)ti6711_delay;
}

static inline u16 dsp_read_hpic(void)
{
#if defined(CONFIG_NETTA_6412)
	return bit_invert(*((volatile u16 *)DSP_BASE));
#else
	return *((volatile u16 *)DSP_BASE);
#endif
}

static inline void dsp_write_hpic(u16 val)
{
#if defined(CONFIG_NETTA_6412)
	*((volatile u16 *)DSP_BASE) = bit_invert(val);
#else
	*((volatile u16 *)DSP_BASE) = val;
#endif
}

static inline void dsp_reset(void)
{
#if defined(CONFIG_NETTA_6412)
	((volatile immap_t *)CFG_IMMR)->im_ioport.iop_pddat &= ~(1 << (15 - 15));
	udelay(500);
	((volatile immap_t *)CFG_IMMR)->im_ioport.iop_pddat |=  (1 << (15 - 15));
	udelay(500);
#else
	((volatile immap_t *)CFG_IMMR)->im_ioport.iop_pddat &= ~(1 << (15 - 7));
	udelay(250);
	((volatile immap_t *)CFG_IMMR)->im_ioport.iop_pddat |=  (1 << (15 - 7));
	udelay(250);
#endif
}

static inline u32 dsp_read_hpic_word(u32 addr)
{
	u32 val;
	volatile u16 *p;
#if defined(CONFIG_NETTA_6412)
	p = (volatile u16 *)((volatile u8 *)DSP_BASE + addr);

	val = ((u32) bit_invert(p[0]) << 16);
	/* dsp_delay(); */

	val |= bit_invert(p[1]);
	/* dsp_delay(); */
#else
	p = (volatile u16 *)((volatile u8 *)DSP_BASE + addr);

	val = ((u32) p[0] << 16);
	dsp_delay();

	val |= p[1];
	dsp_delay();
#endif
	return val;
}

static inline u16 dsp_read_hpic_hi_hword(u32 addr)
{
#if defined(CONFIG_NETTA_6412)
	return bit_invert(*(volatile u16 *)((volatile u8 *)DSP_BASE + addr));
#else
	return *(volatile u16 *)((volatile u8 *)DSP_BASE + addr);
#endif
}

static inline u16 dsp_read_hpic_lo_hword(u32 addr)
{
#if defined(CONFIG_NETTA_6412)
	return bit_invert(*(volatile u16 *)((volatile u8 *)DSP_BASE + addr + 2));
#else
	return *(volatile u16 *)((volatile u8 *)DSP_BASE + addr + 2);
#endif
}

static inline void dsp_wait_hrdy(void)
{
	int i;

	i = 0;
#if defined(CONFIG_NETTA_6412)
	while (i < 1000 && (dsp_read_hpic_word(DSP_HPIC) & 0x08) == 0) {
#else
	while (i < 1000 && (dsp_read_hpic() & 0x08) == 0) {
#endif
		dsp_delay();
		i++;
	}
}

static inline void dsp_write_hpic_word(u32 addr, u32 val)
{
	volatile u16 *p;
#if defined(CONFIG_NETTA_6412)
	p = (volatile u16 *)((volatile u8 *)DSP_BASE + addr);
	p[0] = bit_invert((u16)(val >> 16));
	/* dsp_delay(); */

	p[1] = bit_invert((u16)val);
	/* dsp_delay(); */
#else
	p = (volatile u16 *)((volatile u8 *)DSP_BASE + addr);
	p[0] = (u16)(val >> 16);
	dsp_delay();

	p[1] = (u16)val;
	dsp_delay();
#endif
}

static inline void dsp_write_hpic_hi_hword(u32 addr, u16 val_h)
{
#if defined(CONFIG_NETTA_6412)
	*(volatile u16 *)((volatile u8 *)DSP_BASE + addr) = bit_invert(val_h);
#else

	*(volatile u16 *)((volatile u8 *)DSP_BASE + addr) = val_h;
#endif
}

static inline void dsp_write_hpic_lo_hword(u32 addr, u16 val_l)
{
#if defined(CONFIG_NETTA_6412)
	*(volatile u16 *)((volatile u8 *)DSP_BASE + addr + 2) = bit_invert(val_l);
#else
	*(volatile u16 *)((volatile u8 *)DSP_BASE + addr + 2) = val_l;
#endif
}

/********************************************************************/

static inline void c62_write_word(u32 addr, u32 val)
{
	dsp_write_hpic_hi_hword(DSP_HPIA, (u16)(addr >> 16));
#if !defined(CONFIG_NETTA_6412)
	dsp_delay();
#endif
	dsp_write_hpic_lo_hword(DSP_HPIA, (u16)addr);
#if !defined(CONFIG_NETTA_6412)
	dsp_delay();
#endif

	dsp_wait_hrdy();
#if !defined(CONFIG_NETTA_6412)
	dsp_delay();
#endif
	dsp_write_hpic_hi_hword(DSP_HPID2, (u16)(val >> 16));
#if !defined(CONFIG_NETTA_6412)
	dsp_delay();

	/* dsp_wait_hrdy();
	dsp_delay(); */
#endif
	dsp_write_hpic_lo_hword(DSP_HPID2, (u16)val);
#if !defined(CONFIG_NETTA_6412)
	dsp_delay();
#endif
}

static u32 c62_read_word(u32 addr)
{
	u32 val;

	dsp_write_hpic_hi_hword(DSP_HPIA, (u16)(addr >> 16));
#if !defined(CONFIG_NETTA_6412)
	dsp_delay();
#endif
	dsp_write_hpic_lo_hword(DSP_HPIA, (u16)addr);
#if !defined(CONFIG_NETTA_6412)
	dsp_delay();
#endif

	/* FETCH */
#if defined(CONFIG_NETTA_6412)
	dsp_write_hpic_word(DSP_HPIC, 0x00100010);
#else
	dsp_write_hpic(0x10);
	dsp_delay();
#endif
	dsp_wait_hrdy();
#if !defined(CONFIG_NETTA_6412)
	dsp_delay();
#endif
	val = (u32)dsp_read_hpic_hi_hword(DSP_HPID2) << 16;
#if !defined(CONFIG_NETTA_6412)
	dsp_delay();

	/* dsp_wait_hrdy();
	dsp_delay(); */
#endif
	val |= dsp_read_hpic_lo_hword(DSP_HPID2);
#if !defined(CONFIG_NETTA_6412)
	dsp_delay();
#endif
	return val;
}

static inline void c62_read(u32 addr, u32 *buffer, int numdata)
{
	int i;

	if (numdata <= 0)
		return;

	for (i = 0; i < numdata; i++) {
		*buffer++ = c62_read_word(addr);
		addr += 4;
	}
}

static inline u32 c62_checksum(u32 addr, int numdata)
{
	int i;
	u32 chksum;

	chksum = 0;
	for (i = 0; i < numdata; i++) {
		chksum += c62_read_word(addr);
		addr += 4;
	}

	return chksum;
}

static inline void c62_write(u32 addr, const u32 *buffer, int numdata)
{
	int i;

	if (numdata <= 0)
		return;

	for (i = 0; i < numdata; i++) {
		c62_write_word(addr, *buffer++);
		addr += 4;
	}
}

static inline int c62_write_word_validated(u32 addr, u32 val)
{
	c62_write_word(addr, val);
	return c62_read_word(addr) == val ? 0 : -1;
}

static inline int c62_write_validated(u32 addr, const u32 *buffer, int numdata)
{
	int i, r;

	if (numdata <= 0)
		return 0;

	for (i = 0; i < numdata; i++) {
		r = c62_write_word_validated(addr, *buffer++);
		if (r < 0)
			return r;
		addr += 4;
	}
	return 0;
}

#if defined(CONFIG_NETTA_6412)

#define DRAM_REGS_BASE	0x1800000

#define GBLCTL	DRAM_REGS_BASE
#define CECTL1	(DRAM_REGS_BASE + 0x4)
#define CECTL0	(DRAM_REGS_BASE + 0x8)
#define CECTL2	(DRAM_REGS_BASE + 0x10)
#define CECTL3	(DRAM_REGS_BASE + 0x14)
#define SDCTL	(DRAM_REGS_BASE + 0x18)
#define SDTIM	(DRAM_REGS_BASE + 0x1C)
#define SDEXT	(DRAM_REGS_BASE + 0x20)
#define SESEC1	(DRAM_REGS_BASE + 0x44)
#define SESEC0	(DRAM_REGS_BASE + 0x48)
#define SESEC2	(DRAM_REGS_BASE + 0x50)
#define SESEC3	(DRAM_REGS_BASE + 0x54)

#define MAR128	0x1848200
#define MAR129	0x1848204

void dsp_dram_initialize(void)
{
	c62_write_word(GBLCTL, 0x120E4);
	c62_write_word(CECTL1, 0x18);
	c62_write_word(CECTL0, 0xD0);
	c62_write_word(CECTL2, 0x18);
	c62_write_word(CECTL3, 0x18);
	c62_write_word(SDCTL, 0x47115000);
	c62_write_word(SDTIM, 1536);
	c62_write_word(SDEXT, 0x534A9);
#if 0
	c62_write_word(SESEC1, 0);
	c62_write_word(SESEC0, 0);
	c62_write_word(SESEC2, 0);
	c62_write_word(SESEC3, 0);
#endif
	c62_write_word(MAR128, 1);
	c62_write_word(MAR129, 0);
}

#endif

static inline void dsp_init_hpic(void)
{
	int i;
	volatile u16 *p;
#if defined(CONFIG_NETTA_6412)
	dsp_go_fast();
#else
	dsp_go_slow();
#endif
	i = 0;
#if defined(CONFIG_NETTA_6412)
	while (i < 1000 && (dsp_read_hpic_word(DSP_HPIC) & 0x08) == 0) {
#else
	while (i < 1000 && (dsp_read_hpic() & 0x08) == 0) {
#endif
		dsp_delay();
		i++;
	}

	if (i == 1000)
		printf("HRDY stuck\n");

	dsp_delay();

	/* write control register */
	p = (volatile u16 *)DSP_BASE;
	p[0] = 0x0000;
	dsp_delay();
	p[1] = 0x0000;
	dsp_delay();

#if !defined(CONFIG_NETTA_6412)
	dsp_go_fast();
#endif
}

/***********************************************************************************************************/

#if !defined(CONFIG_NETTA_6412)

static const u8 bootstrap_rbin[5084] = {
	0x52, 0x42, 0x49, 0x4e, 0xc5, 0xa9, 0x9f, 0x1a, 0x00, 0x00, 0x00, 0x02,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x20, 0x00,
	0x00, 0x00, 0x11, 0xc0, 0x00, 0x17, 0x94, 0x2a, 0x00, 0x00, 0x00, 0x6a,
	0x00, 0x00, 0x03, 0x62, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00,
	0x00, 0x18, 0x00, 0xe2, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0xe2,
	0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0xe2, 0x00, 0x00, 0x60, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x18, 0x00, 0xe2, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0xe2,
	0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0xe2, 0x00, 0x00, 0x60, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x18, 0x00, 0xe2, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0xe2,
	0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0xe2, 0x00, 0x00, 0x60, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x18, 0x00, 0xe2, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0xe2,
	0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0xe2, 0x00, 0x00, 0x60, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x18, 0x00, 0xe2, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0xe2,
	0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0xe2, 0x00, 0x00, 0x60, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x17, 0x94, 0x2a, 0x00, 0x00, 0x00, 0x6a, 0x00, 0x00, 0x03, 0x62,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x18, 0x00, 0xe2,
	0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0xe2, 0x00, 0x00, 0x60, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x18, 0x00, 0xe2, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0xe2,
	0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0xe2, 0x00, 0x00, 0x60, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x18, 0x00, 0xe2, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0xe2,
	0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0xe2, 0x00, 0x00, 0x60, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x18, 0x00, 0xe2, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0xe2,
	0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0xe2, 0x00, 0x00, 0x60, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x18, 0x00, 0xe2, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0xe2,
	0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0xe2, 0x00, 0x00, 0x60, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x18, 0x00, 0xe2, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x90, 0x10, 0x5a,
	0x00, 0x19, 0x2e, 0x28, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x02, 0x64,
	0x02, 0x00, 0x00, 0xaa, 0x02, 0x10, 0xac, 0xe2, 0x00, 0x00, 0x20, 0x00,
	0x00, 0x00, 0x02, 0x64, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x9f, 0x7a,
	0x30, 0x00, 0x08, 0x10, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x04, 0x28,
	0x00, 0x00, 0xc6, 0x69, 0x02, 0x16, 0x4c, 0xa2, 0x02, 0x00, 0x90, 0x7a,
	0x00, 0x10, 0x02, 0xe4, 0x00, 0x00, 0x60, 0x00, 0x00, 0x02, 0xd6, 0xc8,
	0x00, 0x10, 0x02, 0xf4, 0x03, 0x10, 0x02, 0xe6, 0x00, 0x00, 0x60, 0x00,
	0x03, 0x1a, 0xf7, 0xca, 0x03, 0x10, 0x02, 0xf6, 0x00, 0x00, 0x04, 0x28,
	0x00, 0x00, 0xc6, 0x69, 0x02, 0x16, 0x4c, 0xa2, 0x02, 0x00, 0x90, 0x7a,
	0x02, 0x90, 0x02, 0xe6, 0x00, 0x00, 0x60, 0x00, 0x02, 0x97, 0xcf, 0x5a,
	0x02, 0x90, 0x02, 0xf6, 0x02, 0x90, 0x02, 0xe6, 0x00, 0x00, 0x60, 0x00,
	0x02, 0x96, 0x10, 0xca, 0x02, 0x90, 0x02, 0xf6, 0x00, 0x0c, 0x03, 0x62,
	0x00, 0x00, 0x80, 0x00, 0x02, 0x90, 0x10, 0x5a, 0x00, 0x00, 0x04, 0x28,
	0x00, 0x00, 0xc6, 0x69, 0x02, 0x16, 0x4c, 0xa2, 0x02, 0x00, 0x90, 0x7a,
	0x03, 0x10, 0x02, 0xe6, 0x00, 0x00, 0x60, 0x00, 0x03, 0x18, 0x2f, 0xda,
	0x03, 0x10, 0x02, 0xf6, 0x03, 0x10, 0x02, 0xe6, 0x00, 0x00, 0x60, 0x00,
	0x03, 0x1a, 0x10, 0x8a, 0x03, 0x10, 0x02, 0xf6, 0x00, 0x19, 0x2e, 0x28,
	0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x02, 0x64, 0x03, 0x00, 0x00, 0xaa,
	0x02, 0x98, 0xac, 0xe2, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x02, 0x64,
	0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0xbf, 0x7a, 0x22, 0x90, 0x02, 0xe6,
	0x00, 0x00, 0x60, 0x00, 0x22, 0x96, 0xd6, 0x8a, 0x22, 0x90, 0x02, 0xf6,
	0x22, 0x90, 0x02, 0xe6, 0x00, 0x00, 0x60, 0x00, 0x22, 0x96, 0xf7, 0x8a,
	0x22, 0x90, 0x02, 0xf6, 0x00, 0x0c, 0x03, 0x62, 0x00, 0x00, 0x80, 0x00,
	0x00, 0x0c, 0x03, 0x62, 0x00, 0x00, 0x80, 0x00, 0x00, 0x0c, 0x03, 0x62,
	0x00, 0x00, 0x80, 0x00, 0x00, 0x19, 0x2e, 0x28, 0x00, 0x00, 0x00, 0x68,
	0x00, 0x00, 0x02, 0x64, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x02, 0x64,
	0x00, 0x00, 0x60, 0x00, 0x00, 0x80, 0x4f, 0x58, 0x02, 0x00, 0x12, 0x2a,
	0x02, 0x00, 0xc8, 0x6a, 0x02, 0x90, 0x02, 0xe6, 0x00, 0x00, 0x60, 0x00,
	0x02, 0x95, 0x8c, 0xca, 0x02, 0x90, 0x02, 0xf6, 0x00, 0x00, 0x12, 0x28,
	0x00, 0x00, 0xc8, 0x68, 0x02, 0x00, 0x02, 0x66, 0x00, 0x00, 0x60, 0x00,
	0x02, 0x11, 0xad, 0xca, 0x02, 0x00, 0x02, 0x76, 0x92, 0x00, 0x12, 0x2a,
	0x92, 0x00, 0xc8, 0x6a, 0x92, 0x90, 0x02, 0xe6, 0x00, 0x00, 0x60, 0x00,
	0x92, 0x95, 0x6b, 0xca, 0x92, 0x90, 0x02, 0xf6, 0x80, 0x00, 0x12, 0x28,
	0x80, 0x00, 0xc8, 0x68, 0x82, 0x00, 0x02, 0x66, 0x80, 0x00, 0x12, 0x28,
	0x80, 0x00, 0xc8, 0x68, 0x00, 0x00, 0x20, 0x00, 0x82, 0x11, 0x6b, 0x8a,
	0x82, 0x00, 0x02, 0x76, 0x02, 0x00, 0x12, 0x2a, 0x02, 0x00, 0xc8, 0x6a,
	0x02, 0x90, 0x02, 0xe6, 0x00, 0x00, 0x60, 0x00, 0x02, 0x95, 0x4a, 0xca,
	0x02, 0x90, 0x02, 0xf6, 0x90, 0x00, 0x12, 0x28, 0x90, 0x00, 0xc8, 0x68,
	0x92, 0x00, 0x02, 0x66, 0x00, 0x00, 0x60, 0x00, 0x92, 0x11, 0x29, 0xca,
	0x92, 0x00, 0x02, 0x76, 0x82, 0x00, 0x12, 0x2a, 0x82, 0x00, 0xc8, 0x6a,
	0x82, 0x10, 0x02, 0xe6, 0x80, 0x00, 0x12, 0x28, 0x80, 0x00, 0xc8, 0x68,
	0x00, 0x00, 0x20, 0x00, 0x82, 0x11, 0x29, 0x8a, 0x82, 0x00, 0x02, 0x76,
	0x00, 0x00, 0x12, 0x28, 0x00, 0x00, 0xc8, 0x68, 0x02, 0x00, 0x02, 0x66,
	0x00, 0x00, 0x60, 0x00, 0x02, 0x11, 0x08, 0xca, 0x02, 0x00, 0x02, 0x76,
	0x02, 0x00, 0x12, 0x2a, 0x02, 0x00, 0xc8, 0x6a, 0x02, 0x90, 0x02, 0xe6,
	0x00, 0x00, 0x60, 0x00, 0x02, 0x97, 0x6f, 0x5a, 0x02, 0x90, 0x02, 0xf6,
	0x00, 0x00, 0x12, 0x28, 0x00, 0x00, 0xc8, 0x68, 0x01, 0x80, 0x02, 0x64,
	0x00, 0x00, 0x60, 0x00, 0x02, 0x0e, 0xff, 0x5a, 0x02, 0x00, 0x02, 0x76,
	0x02, 0x00, 0x12, 0x2a, 0x02, 0x00, 0xc8, 0x6a, 0x00, 0x10, 0x02, 0xe4,
	0x00, 0x00, 0x60, 0x00, 0x02, 0x83, 0xbf, 0x5a, 0x02, 0x90, 0x02, 0xf6,
	0x00, 0x00, 0x12, 0x28, 0x00, 0x00, 0xc8, 0x68, 0x01, 0x80, 0x02, 0x64,
	0x00, 0x00, 0x60, 0x00, 0x02, 0x0f, 0xdf, 0x5a, 0x02, 0x00, 0x02, 0x76,
	0x00, 0x00, 0x04, 0x28, 0x00, 0x00, 0xc8, 0x68, 0x02, 0x00, 0x02, 0x66,
	0x00, 0x00, 0x60, 0x00, 0x02, 0x12, 0xf7, 0xca, 0x02, 0x00, 0x02, 0x76,
	0x02, 0x00, 0x04, 0x2a, 0x02, 0x00, 0xc8, 0x6a, 0x02, 0x90, 0x02, 0xe6,
	0x00, 0x00, 0x60, 0x00, 0x02, 0x96, 0xd6, 0xca, 0x02, 0x90, 0x02, 0xf6,
	0x00, 0x00, 0x04, 0x28, 0x00, 0x00, 0xc8, 0x68, 0x02, 0x00, 0x02, 0x66,
	0x00, 0x00, 0x04, 0x28, 0x00, 0x00, 0xc8, 0x68, 0x00, 0x00, 0x20, 0x00,
	0x02, 0x10, 0x85, 0xca, 0x02, 0x00, 0x02, 0x76, 0x00, 0x00, 0x04, 0x28,
	0x00, 0x00, 0xc8, 0x68, 0x02, 0x80, 0x02, 0x66, 0x02, 0x00, 0x04, 0x2a,
	0x02, 0x00, 0xc8, 0x6a, 0x00, 0x00, 0x20, 0x00, 0x02, 0x96, 0x95, 0xca,
	0x02, 0x90, 0x02, 0xf6, 0x00, 0x00, 0x04, 0x28, 0x00, 0x00, 0xc8, 0x68,
	0x01, 0x80, 0x02, 0x64, 0x00, 0x00, 0x60, 0x00, 0x02, 0x0f, 0xdf, 0x5a,
	0x02, 0x00, 0x02, 0x76, 0x02, 0x00, 0x04, 0x2a, 0x02, 0x00, 0xc8, 0x6a,
	0x02, 0x90, 0x02, 0xe6, 0x00, 0x00, 0x60, 0x00, 0x02, 0x96, 0x10, 0xca,
	0x02, 0x90, 0x02, 0xf6, 0x00, 0x00, 0x04, 0x28, 0x00, 0x00, 0xc8, 0x68,
	0x02, 0x00, 0x02, 0x66, 0x00, 0x00, 0x60, 0x00, 0x02, 0x11, 0xef, 0xca,
	0x02, 0x00, 0x02, 0x76, 0x00, 0x00, 0x04, 0x28, 0x00, 0x00, 0xc8, 0x68,
	0x02, 0x80, 0x02, 0x66, 0x02, 0x00, 0x04, 0x2a, 0x02, 0x00, 0xc8, 0x6a,
	0x00, 0x00, 0x20, 0x00, 0x02, 0x95, 0xae, 0xca, 0x02, 0x90, 0x02, 0xf6,
	0x02, 0x00, 0x06, 0x2a, 0x02, 0x00, 0xc8, 0x6a, 0x02, 0x10, 0x02, 0xe6,
	0x00, 0x00, 0x06, 0x28, 0x00, 0x00, 0xc8, 0x68, 0x00, 0x00, 0x20, 0x00,
	0x02, 0x10, 0x21, 0x0a, 0x02, 0x00, 0x02, 0x76, 0x00, 0x00, 0x06, 0x28,
	0x00, 0x00, 0xc8, 0x68, 0x01, 0x80, 0x02, 0x64, 0x00, 0x00, 0x06, 0x28,
	0x00, 0x00, 0xc8, 0x68, 0x00, 0x00, 0x20, 0x00, 0x01, 0x8d, 0xae, 0xc8,
	0x01, 0x8d, 0x0c, 0x88, 0x01, 0x80, 0x02, 0x74, 0x00, 0x00, 0x06, 0x28,
	0x00, 0x00, 0xc8, 0x68, 0x02, 0x00, 0x02, 0x66, 0x00, 0x00, 0x06, 0x28,
	0x00, 0x00, 0xc8, 0x68, 0x00, 0x00, 0x20, 0x00, 0x02, 0x10, 0xa7, 0xca,
	0x02, 0x00, 0x02, 0x76, 0x00, 0x00, 0x06, 0x28, 0x00, 0x00, 0xc8, 0x68,
	0x00, 0x00, 0x02, 0x64, 0x02, 0x00, 0x06, 0x2a, 0x02, 0x00, 0xc8, 0x6a,
	0x00, 0x00, 0x20, 0x00, 0x00, 0x02, 0x31, 0xc8, 0x00, 0x02, 0x10, 0x88,
	0x00, 0x10, 0x02, 0xf4, 0x00, 0x00, 0x06, 0x28, 0x00, 0x00, 0xc8, 0x68,
	0x02, 0x80, 0x02, 0x66, 0x02, 0x00, 0x06, 0x2a, 0x02, 0x00, 0xc8, 0x6a,
	0x00, 0x00, 0x20, 0x00, 0x02, 0x96, 0x74, 0xca, 0x02, 0x90, 0x02, 0xf6,
	0x00, 0x00, 0x06, 0x28, 0x00, 0x00, 0xc8, 0x68, 0x02, 0x80, 0x02, 0x66,
	0x02, 0x00, 0x06, 0x2a, 0x02, 0x00, 0xc8, 0x6a, 0x00, 0x00, 0x20, 0x00,
	0x02, 0x96, 0x52, 0x8a, 0x02, 0x90, 0x02, 0xf6, 0x02, 0x00, 0x08, 0x2a,
	0x02, 0x00, 0xc8, 0x6a, 0x02, 0x10, 0x02, 0xe6, 0x00, 0x00, 0x08, 0x28,
	0x00, 0x00, 0xc8, 0x68, 0x00, 0x00, 0x20, 0x00, 0x02, 0x10, 0x21, 0x0a,
	0x02, 0x00, 0x02, 0x76, 0x00, 0x00, 0x08, 0x28, 0x00, 0x00, 0xc8, 0x68,
	0x00, 0x00, 0x02, 0x64, 0x02, 0x00, 0x08, 0x2a, 0x02, 0x00, 0xc8, 0x6a,
	0x00, 0x00, 0x20, 0x00, 0x00, 0x01, 0xae, 0xc8, 0x00, 0x01, 0x0c, 0x88,
	0x00, 0x10, 0x02, 0xf4, 0x02, 0x00, 0x08, 0x2a, 0x02, 0x00, 0xc8, 0x6a,
	0x02, 0x10, 0x02, 0xe6, 0x00, 0x00, 0x08, 0x28, 0x00, 0x00, 0xc8, 0x68,
	0x00, 0x00, 0x20, 0x00, 0x02, 0x10, 0xa7, 0xca, 0x02, 0x00, 0x02, 0x76,
	0x02, 0x00, 0x08, 0x2a, 0x02, 0x00, 0xc8, 0x6a, 0x02, 0x90, 0x02, 0xe6,
	0x02, 0x00, 0x08, 0x2a, 0x02, 0x00, 0xc8, 0x6a, 0x00, 0x00, 0x20, 0x00,
	0x02, 0x96, 0x31, 0xca, 0x02, 0x96, 0x10, 0x8a, 0x02, 0x90, 0x02, 0xf6,
	0x02, 0x00, 0x08, 0x2a, 0x02, 0x00, 0xc8, 0x6a, 0x02, 0x10, 0x02, 0xe6,
	0x00, 0x00, 0x08, 0x28, 0x00, 0x00, 0xc8, 0x68, 0x00, 0x00, 0x20, 0x00,
	0x02, 0x12, 0x74, 0xca, 0x02, 0x00, 0x02, 0x76, 0x02, 0x00, 0x08, 0x2a,
	0x02, 0x00, 0xc8, 0x6a, 0x02, 0x90, 0x02, 0xe6, 0x02, 0x00, 0x08, 0x2a,
	0x02, 0x00, 0xc8, 0x6a, 0x00, 0x00, 0x20, 0x00, 0x02, 0x96, 0x52, 0x8a,
	0x02, 0x90, 0x02, 0xf6, 0x90, 0x00, 0x1b, 0x10, 0x90, 0x19, 0x2e, 0x28,
	0x90, 0x00, 0x00, 0x68, 0x90, 0x00, 0x02, 0x64, 0x00, 0x00, 0x20, 0x00,
	0x00, 0x00, 0x0a, 0x28, 0x00, 0x00, 0xc8, 0x68, 0x01, 0x80, 0x02, 0x64,
	0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x0a, 0x28, 0x01, 0x8f, 0xbd, 0x88,
	0x00, 0x00, 0xc8, 0x68, 0x01, 0x80, 0x02, 0x74, 0x00, 0x00, 0x0a, 0x28,
	0x00, 0x00, 0xc8, 0x68, 0x00, 0x00, 0x02, 0x64, 0x02, 0x00, 0x0a, 0x2a,
	0x02, 0x00, 0xc8, 0x6a, 0x00, 0x00, 0x20, 0x00, 0x00, 0x03, 0x9c, 0x88,
	0x00, 0x10, 0x02, 0xf4, 0x02, 0x00, 0x0a, 0x2a, 0x02, 0x00, 0xc8, 0x6a,
	0x00, 0x10, 0x02, 0xe4, 0x02, 0x00, 0x0a, 0x2a, 0x02, 0x00, 0xc8, 0x6a,
	0x00, 0x00, 0x20, 0x00, 0x00, 0x03, 0x1b, 0xc8, 0x00, 0x02, 0x17, 0x88,
	0x00, 0x10, 0x02, 0xf4, 0x00, 0x19, 0x2e, 0x28, 0x00, 0x00, 0x00, 0x68,
	0x00, 0x00, 0x02, 0x64, 0x02, 0x00, 0x0a, 0x2a, 0x02, 0x00, 0xc8, 0x6a,
	0x02, 0x10, 0x02, 0xe6, 0x00, 0x00, 0x00, 0x00, 0x01, 0x82, 0x42, 0x64,
	0x00, 0x00, 0x0a, 0x28, 0x00, 0x00, 0xc8, 0x68, 0x00, 0x00, 0x00, 0x00,
	0x02, 0x10, 0x07, 0xca, 0x02, 0x0c, 0x9f, 0xfa, 0x02, 0x00, 0x02, 0x76,
	0x00, 0x19, 0x2e, 0x28, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x02, 0x64,
	0x00, 0x00, 0x60, 0x00, 0x03, 0x00, 0x80, 0x58, 0x03, 0x00, 0x10, 0x2a,
	0x02, 0x04, 0x03, 0xe2, 0x02, 0x18, 0x56, 0x15, 0x02, 0x93, 0xcf, 0x5a,
	0x00, 0x94, 0x03, 0xa2, 0x02, 0x18, 0x56, 0x14, 0x00, 0x00, 0x20, 0x01,
	0x00, 0x00, 0x00, 0x00, 0x02, 0x98, 0x56, 0x15, 0x00, 0x90, 0x4a, 0x58,
	0x81, 0x98, 0xa0, 0x14, 0x04, 0x04, 0x00, 0x59, 0x00, 0x00, 0x06, 0x12,
	0x00, 0x90, 0x4a, 0x59, 0x02, 0x98, 0x56, 0x15, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x1b, 0x40, 0x5b, 0x03, 0x80, 0x00, 0xf9, 0x02, 0x00, 0x00, 0xa9,
	0x81, 0x98, 0xa0, 0x14, 0x01, 0x20, 0x00, 0x59, 0x04, 0x04, 0x01, 0xa1,
	0x20, 0x00, 0x02, 0x13, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x10, 0x6c, 0xe1,
	0x00, 0x94, 0x4a, 0x59, 0x02, 0x98, 0x56, 0x14, 0xa3, 0x9c, 0x0f, 0xf9,
	0x20, 0x03, 0xe0, 0x5b, 0x81, 0x98, 0xa0, 0x14, 0x04, 0x04, 0x00, 0x59,
	0x01, 0x20, 0x01, 0xa0, 0x00, 0x94, 0x4a, 0x59, 0xa0, 0x10, 0x6c, 0xe0,
	0xa3, 0x9c, 0x0f, 0xf9, 0x81, 0x98, 0x60, 0x14, 0x04, 0x04, 0x00, 0x59,
	0x01, 0x20, 0x01, 0xa0, 0x00, 0x94, 0x4a, 0x59, 0xa0, 0x10, 0x6c, 0xe0,
	0xa3, 0x9c, 0x0f, 0xf9, 0x81, 0x98, 0x20, 0x14, 0x02, 0x84, 0x00, 0x59,
	0x01, 0x20, 0x01, 0xa0, 0xa0, 0x10, 0x6c, 0xe0, 0x01, 0x14, 0x01, 0xa1,
	0xa3, 0x9c, 0x0f, 0xf8, 0x00, 0x90, 0x03, 0xa2, 0xa0, 0x10, 0x6c, 0xe0,
	0xa3, 0x9c, 0x0f, 0xf8, 0x02, 0x00, 0x0c, 0x2b, 0x00, 0x00, 0x00, 0xa8,
	0x02, 0x00, 0xc8, 0x6b, 0x00, 0x00, 0x01, 0xe8, 0x00, 0x10, 0x02, 0xf4,
	0x02, 0x00, 0x0e, 0x2a, 0x02, 0x00, 0xc8, 0x6a, 0x03, 0x90, 0x02, 0xf4,
	0x00, 0x00, 0x10, 0x28, 0x00, 0x00, 0xc8, 0x68, 0x03, 0x80, 0x02, 0x74,
	0x00, 0x0c, 0x03, 0x62, 0x00, 0x00, 0x80, 0x00, 0x00, 0x19, 0x2e, 0x28,
	0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x02, 0x64, 0x00, 0x00, 0x60, 0x00,
	0x00, 0x00, 0x02, 0x64, 0x00, 0x00, 0x60, 0x00, 0x00, 0x80, 0x2f, 0x58,
	0x02, 0x00, 0x12, 0x2a, 0x02, 0x00, 0xc6, 0x6a, 0x02, 0x90, 0x02, 0xe6,
	0x00, 0x00, 0x60, 0x00, 0x02, 0x95, 0x8c, 0xca, 0x02, 0x90, 0x02, 0xf6,
	0x00, 0x00, 0x12, 0x28, 0x00, 0x00, 0xc6, 0x68, 0x02, 0x00, 0x02, 0x66,
	0x00, 0x00, 0x60, 0x00, 0x02, 0x11, 0xad, 0xca, 0x02, 0x00, 0x02, 0x76,
	0x92, 0x00, 0x12, 0x2a, 0x92, 0x00, 0xc6, 0x6a, 0x92, 0x90, 0x02, 0xe6,
	0x00, 0x00, 0x60, 0x00, 0x92, 0x95, 0x6b, 0xca, 0x92, 0x90, 0x02, 0xf6,
	0x80, 0x00, 0x12, 0x28, 0x80, 0x00, 0xc6, 0x68, 0x82, 0x00, 0x02, 0x66,
	0x80, 0x00, 0x12, 0x28, 0x80, 0x00, 0xc6, 0x68, 0x00, 0x00, 0x20, 0x00,
	0x82, 0x11, 0x6b, 0x8a, 0x82, 0x00, 0x02, 0x76, 0x02, 0x00, 0x12, 0x2a,
	0x02, 0x00, 0xc6, 0x6a, 0x02, 0x90, 0x02, 0xe6, 0x00, 0x00, 0x60, 0x00,
	0x02, 0x95, 0x4a, 0xca, 0x02, 0x90, 0x02, 0xf6, 0x90, 0x00, 0x12, 0x28,
	0x90, 0x00, 0xc6, 0x68, 0x92, 0x00, 0x02, 0x66, 0x00, 0x00, 0x60, 0x00,
	0x92, 0x11, 0x29, 0xca, 0x92, 0x00, 0x02, 0x76, 0x82, 0x00, 0x12, 0x2a,
	0x82, 0x00, 0xc6, 0x6a, 0x82, 0x10, 0x02, 0xe6, 0x80, 0x00, 0x12, 0x28,
	0x80, 0x00, 0xc6, 0x68, 0x00, 0x00, 0x20, 0x00, 0x82, 0x11, 0x29, 0x8a,
	0x82, 0x00, 0x02, 0x76, 0x00, 0x00, 0x12, 0x28, 0x00, 0x00, 0xc6, 0x68,
	0x02, 0x00, 0x02, 0x66, 0x00, 0x00, 0x60, 0x00, 0x02, 0x11, 0x08, 0xca,
	0x02, 0x00, 0x02, 0x76, 0x02, 0x00, 0x12, 0x2a, 0x02, 0x00, 0xc6, 0x6a,
	0x02, 0x90, 0x02, 0xe6, 0x00, 0x00, 0x60, 0x00, 0x02, 0x97, 0x6f, 0x5a,
	0x02, 0x90, 0x02, 0xf6, 0x00, 0x00, 0x12, 0x28, 0x00, 0x00, 0xc6, 0x68,
	0x01, 0x80, 0x02, 0x64, 0x00, 0x00, 0x60, 0x00, 0x02, 0x0e, 0xff, 0x5a,
	0x02, 0x00, 0x02, 0x76, 0x02, 0x00, 0x12, 0x2a, 0x02, 0x00, 0xc6, 0x6a,
	0x00, 0x10, 0x02, 0xe4, 0x00, 0x00, 0x60, 0x00, 0x02, 0x83, 0xbf, 0x5a,
	0x02, 0x90, 0x02, 0xf6, 0x00, 0x00, 0x12, 0x28, 0x00, 0x00, 0xc6, 0x68,
	0x01, 0x80, 0x02, 0x64, 0x00, 0x00, 0x60, 0x00, 0x02, 0x0f, 0xdf, 0x5a,
	0x02, 0x00, 0x02, 0x76, 0x00, 0x00, 0x04, 0x28, 0x00, 0x00, 0xc6, 0x68,
	0x02, 0x00, 0x02, 0x66, 0x00, 0x00, 0x60, 0x00, 0x02, 0x12, 0xf7, 0xca,
	0x02, 0x00, 0x02, 0x76, 0x02, 0x00, 0x04, 0x2a, 0x02, 0x00, 0xc6, 0x6a,
	0x02, 0x90, 0x02, 0xe6, 0x00, 0x00, 0x60, 0x00, 0x02, 0x96, 0xd6, 0xca,
	0x02, 0x90, 0x02, 0xf6, 0x00, 0x00, 0x04, 0x28, 0x00, 0x00, 0xc6, 0x68,
	0x02, 0x00, 0x02, 0x66, 0x00, 0x00, 0x04, 0x28, 0x00, 0x00, 0xc6, 0x68,
	0x00, 0x00, 0x20, 0x00, 0x02, 0x10, 0x85, 0xca, 0x02, 0x00, 0x02, 0x76,
	0x00, 0x00, 0x04, 0x28, 0x00, 0x00, 0xc6, 0x68, 0x02, 0x80, 0x02, 0x66,
	0x02, 0x00, 0x04, 0x2a, 0x02, 0x00, 0xc6, 0x6a, 0x00, 0x00, 0x20, 0x00,
	0x02, 0x96, 0x95, 0xca, 0x02, 0x90, 0x02, 0xf6, 0x00, 0x00, 0x04, 0x28,
	0x00, 0x00, 0xc6, 0x68, 0x01, 0x80, 0x02, 0x64, 0x00, 0x00, 0x60, 0x00,
	0x02, 0x0f, 0xdf, 0x5a, 0x02, 0x00, 0x02, 0x76, 0x02, 0x00, 0x04, 0x2a,
	0x02, 0x00, 0xc6, 0x6a, 0x02, 0x90, 0x02, 0xe6, 0x00, 0x00, 0x60, 0x00,
	0x02, 0x96, 0x10, 0xca, 0x02, 0x90, 0x02, 0xf6, 0x00, 0x00, 0x04, 0x28,
	0x00, 0x00, 0xc6, 0x68, 0x02, 0x00, 0x02, 0x66, 0x00, 0x00, 0x60, 0x00,
	0x02, 0x11, 0xef, 0xca, 0x02, 0x00, 0x02, 0x76, 0x00, 0x00, 0x04, 0x28,
	0x00, 0x00, 0xc6, 0x68, 0x02, 0x80, 0x02, 0x66, 0x02, 0x00, 0x04, 0x2a,
	0x02, 0x00, 0xc6, 0x6a, 0x00, 0x00, 0x20, 0x00, 0x02, 0x95, 0xae, 0xca,
	0x02, 0x90, 0x02, 0xf6, 0x02, 0x00, 0x06, 0x2a, 0x02, 0x00, 0xc6, 0x6a,
	0x02, 0x10, 0x02, 0xe6, 0x00, 0x00, 0x06, 0x28, 0x00, 0x00, 0xc6, 0x68,
	0x00, 0x00, 0x20, 0x00, 0x02, 0x10, 0x21, 0x0a, 0x02, 0x00, 0x02, 0x76,
	0x00, 0x00, 0x06, 0x28, 0x00, 0x00, 0xc6, 0x68, 0x01, 0x80, 0x02, 0x64,
	0x00, 0x00, 0x06, 0x28, 0x00, 0x00, 0xc6, 0x68, 0x00, 0x00, 0x20, 0x00,
	0x01, 0x8d, 0xae, 0xc8, 0x01, 0x8d, 0x0c, 0x88, 0x01, 0x80, 0x02, 0x74,
	0x00, 0x00, 0x06, 0x28, 0x00, 0x00, 0xc6, 0x68, 0x02, 0x00, 0x02, 0x66,
	0x00, 0x00, 0x06, 0x28, 0x00, 0x00, 0xc6, 0x68, 0x00, 0x00, 0x20, 0x00,
	0x02, 0x10, 0xa7, 0xca, 0x02, 0x00, 0x02, 0x76, 0x00, 0x00, 0x06, 0x28,
	0x00, 0x00, 0xc6, 0x68, 0x00, 0x00, 0x02, 0x64, 0x02, 0x00, 0x06, 0x2a,
	0x02, 0x00, 0xc6, 0x6a, 0x00, 0x00, 0x20, 0x00, 0x00, 0x02, 0x31, 0xc8,
	0x00, 0x02, 0x10, 0x88, 0x00, 0x10, 0x02, 0xf4, 0x00, 0x00, 0x06, 0x28,
	0x00, 0x00, 0xc6, 0x68, 0x02, 0x80, 0x02, 0x66, 0x02, 0x00, 0x06, 0x2a,
	0x02, 0x00, 0xc6, 0x6a, 0x00, 0x00, 0x20, 0x00, 0x02, 0x96, 0x74, 0xca,
	0x02, 0x90, 0x02, 0xf6, 0x00, 0x00, 0x06, 0x28, 0x00, 0x00, 0xc6, 0x68,
	0x02, 0x80, 0x02, 0x66, 0x02, 0x00, 0x06, 0x2a, 0x02, 0x00, 0xc6, 0x6a,
	0x00, 0x00, 0x20, 0x00, 0x02, 0x96, 0x52, 0x8a, 0x02, 0x90, 0x02, 0xf6,
	0x02, 0x00, 0x08, 0x2a, 0x02, 0x00, 0xc6, 0x6a, 0x02, 0x10, 0x02, 0xe6,
	0x00, 0x00, 0x08, 0x28, 0x00, 0x00, 0xc6, 0x68, 0x00, 0x00, 0x20, 0x00,
	0x02, 0x10, 0x21, 0x0a, 0x02, 0x00, 0x02, 0x76, 0x00, 0x00, 0x08, 0x28,
	0x00, 0x00, 0xc6, 0x68, 0x00, 0x00, 0x02, 0x64, 0x02, 0x00, 0x08, 0x2a,
	0x02, 0x00, 0xc6, 0x6a, 0x00, 0x00, 0x20, 0x00, 0x00, 0x01, 0xae, 0xc8,
	0x00, 0x01, 0x0c, 0x88, 0x00, 0x10, 0x02, 0xf4, 0x02, 0x00, 0x08, 0x2a,
	0x02, 0x00, 0xc6, 0x6a, 0x02, 0x10, 0x02, 0xe6, 0x00, 0x00, 0x08, 0x28,
	0x00, 0x00, 0xc6, 0x68, 0x00, 0x00, 0x20, 0x00, 0x02, 0x10, 0xa7, 0xca,
	0x02, 0x00, 0x02, 0x76, 0x02, 0x00, 0x08, 0x2a, 0x02, 0x00, 0xc6, 0x6a,
	0x02, 0x90, 0x02, 0xe6, 0x02, 0x00, 0x08, 0x2a, 0x02, 0x00, 0xc6, 0x6a,
	0x00, 0x00, 0x20, 0x00, 0x02, 0x96, 0x31, 0xca, 0x02, 0x96, 0x10, 0x8a,
	0x02, 0x90, 0x02, 0xf6, 0x02, 0x00, 0x08, 0x2a, 0x02, 0x00, 0xc6, 0x6a,
	0x02, 0x10, 0x02, 0xe6, 0x00, 0x00, 0x08, 0x28, 0x00, 0x00, 0xc6, 0x68,
	0x00, 0x00, 0x20, 0x00, 0x02, 0x12, 0x74, 0xca, 0x02, 0x00, 0x02, 0x76,
	0x02, 0x00, 0x08, 0x2a, 0x02, 0x00, 0xc6, 0x6a, 0x02, 0x90, 0x02, 0xe6,
	0x02, 0x00, 0x08, 0x2a, 0x02, 0x00, 0xc6, 0x6a, 0x00, 0x00, 0x20, 0x00,
	0x02, 0x96, 0x52, 0x8a, 0x02, 0x90, 0x02, 0xf6, 0x90, 0x00, 0x19, 0x90,
	0x90, 0x19, 0x2e, 0x28, 0x90, 0x00, 0x00, 0x68, 0x90, 0x00, 0x02, 0x64,
	0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x0a, 0x28, 0x00, 0x00, 0xc6, 0x68,
	0x01, 0x80, 0x02, 0x64, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x0a, 0x28,
	0x01, 0x8f, 0xbd, 0x88, 0x00, 0x00, 0xc6, 0x68, 0x01, 0x80, 0x02, 0x74,
	0x00, 0x00, 0x0a, 0x28, 0x00, 0x00, 0xc6, 0x68, 0x00, 0x00, 0x02, 0x64,
	0x02, 0x00, 0x0a, 0x2a, 0x02, 0x00, 0xc6, 0x6a, 0x00, 0x00, 0x20, 0x00,
	0x00, 0x03, 0x9c, 0x88, 0x00, 0x10, 0x02, 0xf4, 0x02, 0x00, 0x0a, 0x2a,
	0x02, 0x00, 0xc6, 0x6a, 0x00, 0x10, 0x02, 0xe4, 0x02, 0x00, 0x0a, 0x2a,
	0x02, 0x00, 0xc6, 0x6a, 0x00, 0x00, 0x20, 0x00, 0x00, 0x03, 0x1b, 0xc8,
	0x00, 0x02, 0x17, 0x88, 0x00, 0x10, 0x02, 0xf4, 0x00, 0x19, 0x2e, 0x28,
	0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x02, 0x64, 0x02, 0x00, 0x0a, 0x2a,
	0x02, 0x00, 0xc6, 0x6a, 0x02, 0x10, 0x02, 0xe6, 0x00, 0x00, 0x00, 0x00,
	0x01, 0x82, 0x22, 0x64, 0x00, 0x00, 0x0a, 0x28, 0x00, 0x00, 0xc6, 0x68,
	0x00, 0x00, 0x00, 0x00, 0x02, 0x10, 0x07, 0xca, 0x02, 0x0c, 0x9f, 0xfa,
	0x02, 0x00, 0x02, 0x76, 0x00, 0x19, 0x2e, 0x28, 0x00, 0x00, 0x00, 0x68,
	0x00, 0x00, 0x02, 0x64, 0x00, 0x00, 0x60, 0x00, 0x03, 0x00, 0x80, 0x58,
	0x03, 0x00, 0x10, 0x2a, 0x02, 0x04, 0x03, 0xe3, 0x00, 0x00, 0x00, 0x00,
	0x02, 0x18, 0x56, 0x15, 0x02, 0x93, 0xcf, 0x5a, 0x00, 0x94, 0x03, 0xa2,
	0x02, 0x18, 0x56, 0x14, 0x00, 0x00, 0x20, 0x00, 0x02, 0x98, 0x56, 0x15,
	0x00, 0x90, 0x2a, 0x58, 0x81, 0x98, 0xa0, 0x14, 0x04, 0x04, 0x00, 0x59,
	0x00, 0x00, 0x04, 0x12, 0x00, 0x90, 0x2a, 0x59, 0x02, 0x98, 0x56, 0x14,
	0x00, 0x1b, 0x40, 0x5b, 0x03, 0x80, 0x00, 0xf9, 0x02, 0x00, 0x00, 0xa9,
	0x81, 0x98, 0xa0, 0x14, 0x01, 0x20, 0x00, 0x59, 0x04, 0x04, 0x01, 0xa1,
	0x20, 0x00, 0x00, 0x12, 0xa0, 0x10, 0x6c, 0xe1, 0x00, 0x94, 0x2a, 0x59,
	0x02, 0x98, 0x56, 0x15, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0xa3, 0x9c, 0x0f, 0xf9, 0x20, 0x03, 0xe0, 0x5b, 0x81, 0x98, 0xa0, 0x14,
	0x04, 0x04, 0x00, 0x59, 0x01, 0x20, 0x01, 0xa0, 0x00, 0x94, 0x2a, 0x59,
	0xa0, 0x10, 0x6c, 0xe1, 0x00, 0x00, 0x00, 0x00, 0xa3, 0x9c, 0x0f, 0xf9,
	0x81, 0x98, 0x60, 0x14, 0x04, 0x04, 0x00, 0x59, 0x01, 0x20, 0x01, 0xa0,
	0x00, 0x94, 0x2a, 0x59, 0xa0, 0x10, 0x6c, 0xe0, 0xa3, 0x9c, 0x0f, 0xf9,
	0x81, 0x98, 0x20, 0x14, 0x02, 0x84, 0x00, 0x59, 0x01, 0x20, 0x01, 0xa0,
	0xa0, 0x10, 0x6c, 0xe0, 0x01, 0x14, 0x01, 0xa1, 0xa3, 0x9c, 0x0f, 0xf8,
	0x00, 0x90, 0x03, 0xa2, 0xa0, 0x10, 0x6c, 0xe0, 0xa3, 0x9c, 0x0f, 0xf8,
	0x02, 0x00, 0x0c, 0x2b, 0x00, 0x00, 0x00, 0xa8, 0x02, 0x00, 0xc6, 0x6b,
	0x00, 0x00, 0x01, 0xe8, 0x00, 0x10, 0x02, 0xf4, 0x02, 0x00, 0x0e, 0x2a,
	0x02, 0x00, 0xc6, 0x6a, 0x03, 0x90, 0x02, 0xf4, 0x00, 0x00, 0x10, 0x28,
	0x00, 0x00, 0xc6, 0x68, 0x03, 0x80, 0x02, 0x74, 0x00, 0x0c, 0x03, 0x62,
	0x00, 0x00, 0x80, 0x00, 0x00, 0x0c, 0x03, 0x62, 0x00, 0x00, 0x80, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x28, 0x00, 0x00, 0xc0, 0x69,
	0x02, 0x00, 0x10, 0x2a, 0x02, 0x00, 0x02, 0x76, 0x00, 0x0c, 0x03, 0x62,
	0x00, 0x00, 0x80, 0x00, 0x01, 0xbc, 0x54, 0xf6, 0x02, 0x04, 0x03, 0xe2,
	0x02, 0x13, 0xcf, 0x5a, 0x00, 0x90, 0x03, 0xa2, 0x02, 0x18, 0x50, 0x2a,
	0x02, 0x00, 0x00, 0x6a, 0x00, 0x10, 0x03, 0x62, 0x01, 0x97, 0x30, 0x2a,
	0x01, 0x80, 0x00, 0x6a, 0x00, 0x00, 0x40, 0x00, 0x00, 0x17, 0xb4, 0x28,
	0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x13, 0x62, 0x01, 0x97, 0x3c, 0x2a,
	0x01, 0x80, 0x00, 0x6a, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x04, 0x29,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x69, 0x02, 0x00, 0x10, 0x2a,
	0x02, 0x00, 0x02, 0x76, 0x00, 0x14, 0x4e, 0x28, 0x00, 0x00, 0x00, 0x68,
	0x00, 0x00, 0x13, 0x62, 0x01, 0x97, 0x52, 0x2a, 0x01, 0x80, 0x00, 0x6a,
	0x00, 0x00, 0x40, 0x00, 0x00, 0x11, 0x94, 0x28, 0x00, 0x00, 0x00, 0x68,
	0x00, 0x00, 0x13, 0x62, 0x01, 0x97, 0x5e, 0x2a, 0x01, 0x80, 0x00, 0x6a,
	0x00, 0x00, 0x40, 0x00, 0x02, 0x11, 0x4c, 0x2a, 0x02, 0x00, 0x00, 0x6a,
	0x00, 0x10, 0x03, 0x62, 0x01, 0x97, 0x6c, 0x2a, 0x01, 0x80, 0x00, 0x6a,
	0x02, 0x00, 0x00, 0xf8, 0x00, 0x00, 0x20, 0x00, 0x00, 0x11, 0x4c, 0x28,
	0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x13, 0x62, 0x01, 0x97, 0x7a, 0x2a,
	0x01, 0x80, 0x00, 0x6a, 0x02, 0x00, 0x00, 0xa8, 0x00, 0x00, 0x20, 0x00,
	0x02, 0x04, 0x03, 0xe2, 0x00, 0x12, 0x00, 0x28, 0x02, 0x00, 0x9f, 0xfa,
	0x00, 0x90, 0x03, 0xa2, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00,
	0x01, 0xbc, 0x52, 0xe6, 0x00, 0x00, 0x60, 0x00, 0x00, 0x0c, 0x03, 0x62,
	0x00, 0x00, 0x80, 0x00, 0x07, 0xae, 0xfe, 0x2a, 0x07, 0x80, 0x00, 0x6a,
	0x00, 0x10, 0x00, 0x28, 0x02, 0x80, 0x13, 0xa2, 0x0f, 0xff, 0xe3, 0x12,
	0x00, 0x00, 0x80, 0x00, 0x00, 0x0c, 0x03, 0x62, 0x00, 0x00, 0x80, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x0c, 0x03, 0x62, 0x00, 0x00, 0x80, 0x00, 0x00, 0x19, 0x30, 0x28,
	0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x02, 0x64, 0x00, 0x00, 0x60, 0x00,
	0x00, 0x80, 0x02, 0x64, 0x00, 0x00, 0x60, 0x00, 0x90, 0x00, 0x20, 0x90,
	0x01, 0x04, 0x2a, 0x58, 0x00, 0x00, 0x60, 0x00, 0xa0, 0x00, 0x0a, 0x10,
	0x00, 0x00, 0x80, 0x00, 0x02, 0x00, 0x00, 0xfa, 0x02, 0x00, 0xc0, 0x6a,
	0x02, 0x90, 0x02, 0xe6, 0x03, 0x00, 0x08, 0x2a, 0x00, 0x00, 0x40, 0x00,
	0x02, 0x94, 0xcd, 0xfa, 0x02, 0x90, 0x02, 0xf6, 0x00, 0x00, 0x00, 0xf8,
	0x00, 0x00, 0xc0, 0x68, 0x01, 0x80, 0x02, 0x64, 0x00, 0x00, 0x60, 0x00,
	0x01, 0x8d, 0x0d, 0xd8, 0x01, 0x80, 0x02, 0x74, 0x0f, 0xff, 0xfa, 0x90,
	0x00, 0x00, 0x80, 0x00, 0x02, 0x60, 0x80, 0x2a, 0x02, 0x00, 0xdb, 0xeb,
	0x01, 0x80, 0x00, 0xf8, 0x01, 0x90, 0x02, 0xf4, 0x02, 0x60, 0x80, 0x2a,
	0x02, 0x00, 0xdb, 0xeb, 0x02, 0x00, 0x04, 0x28, 0x02, 0x10, 0x02, 0xf4,
	0x02, 0x00, 0x22, 0x66, 0x02, 0x60, 0x88, 0x28, 0x02, 0x00, 0xdb, 0xe8,
	0x00, 0x00, 0x20, 0x00, 0x02, 0x10, 0x02, 0x76, 0x02, 0x80, 0x42, 0x66,
	0x02, 0x60, 0x8a, 0x2a, 0x02, 0x00, 0xdb, 0xea, 0x00, 0x00, 0x20, 0x00,
	0x02, 0x90, 0x02, 0xf6, 0x02, 0x00, 0xc2, 0x66, 0x02, 0x60, 0x92, 0x28,
	0x02, 0x00, 0xdb, 0xe8, 0x00, 0x00, 0x20, 0x00, 0x02, 0x10, 0x02, 0x76,
	0x02, 0x80, 0x62, 0x66, 0x02, 0x60, 0x8c, 0x2a, 0x02, 0x00, 0xdb, 0xea,
	0x00, 0x00, 0x20, 0x00, 0x02, 0x90, 0x02, 0xf6, 0x02, 0x00, 0x82, 0x66,
	0x02, 0x60, 0x8e, 0x28, 0x02, 0x00, 0xdb, 0xe8, 0x00, 0x00, 0x20, 0x00,
	0x02, 0x10, 0x02, 0x76, 0x00, 0x00, 0xa2, 0x64, 0x02, 0x60, 0x90, 0x2a,
	0x02, 0x00, 0xdb, 0xea, 0x00, 0x00, 0x20, 0x00, 0x00, 0x10, 0x02, 0xf4,
	0x02, 0x60, 0x80, 0x2a, 0x02, 0x00, 0xdb, 0xea, 0x01, 0x90, 0x02, 0xf4,
	0x02, 0x60, 0x80, 0x2a, 0x02, 0x00, 0xdb, 0xeb, 0x00, 0x00, 0x00, 0xa8,
	0x00, 0x10, 0x02, 0xf4, 0x00, 0x0c, 0x03, 0x62, 0x00, 0x00, 0x80, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x19, 0x2a, 0x2a,
	0x02, 0x84, 0x20, 0xfb, 0x02, 0x00, 0x00, 0x6a, 0x02, 0x90, 0x02, 0xf6,
	0x02, 0x98, 0xe0, 0x2a, 0x02, 0x19, 0x2c, 0x2a, 0x02, 0x00, 0x00, 0x6b,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x4a, 0x29, 0x02, 0x80, 0x00, 0x6a,
	0x03, 0x94, 0x10, 0x59, 0x00, 0x10, 0x02, 0xf4, 0x00, 0x00, 0x12, 0xaa,
	0x00, 0x80, 0x00, 0xa8, 0x04, 0x08, 0x00, 0x28, 0x04, 0x00, 0x00, 0x68,
	0x20, 0x03, 0xe0, 0x5b, 0x90, 0x14, 0x02, 0x64, 0x20, 0x00, 0x00, 0x12,
	0x93, 0x1c, 0x36, 0x74, 0x00, 0x00, 0x00, 0x00, 0x03, 0x20, 0x02, 0x65,
	0x02, 0x19, 0x2a, 0x29, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x79,
	0x01, 0xa0, 0x36, 0x65, 0x02, 0x00, 0x00, 0x68, 0x80, 0x87, 0xe0, 0x59,
	0x90, 0x14, 0x02, 0x75, 0x02, 0x90, 0x01, 0xa0, 0x00, 0x14, 0x02, 0x64,
	0x00, 0x00, 0x40, 0x00, 0x03, 0x1c, 0x36, 0x74, 0x00, 0x00, 0x60, 0x78,
	0x00, 0x14, 0x02, 0x74, 0x00, 0x19, 0x2a, 0x28, 0x00, 0x00, 0x00, 0x68,
	0x00, 0x18, 0xe0, 0x29, 0x00, 0x00, 0x02, 0x66, 0x00, 0x00, 0x00, 0x68,
	0x00, 0x00, 0x40, 0x00, 0x31, 0x80, 0x80, 0x59, 0x32, 0x19, 0x2e, 0x2a,
	0x32, 0x00, 0x00, 0x6a, 0x31, 0x90, 0x02, 0xf4, 0x30, 0x02, 0x9d, 0x41,
	0x32, 0x19, 0x30, 0x2a, 0x32, 0x00, 0x00, 0x6a, 0x30, 0x10, 0x02, 0xf4,
	0x30, 0x00, 0x09, 0x12, 0x00, 0x00, 0x80, 0x00, 0x02, 0x80, 0x00, 0xfa,
	0x02, 0x80, 0xc0, 0x6a, 0x03, 0x14, 0x02, 0xe6, 0x02, 0x00, 0x08, 0x2a,
	0x00, 0x00, 0x40, 0x00, 0x02, 0x18, 0x8d, 0xfa, 0x02, 0x14, 0x02, 0xf6,
	0x00, 0x00, 0x00, 0xf8, 0x00, 0x00, 0xc0, 0x68, 0x01, 0x80, 0x02, 0x64,
	0x00, 0x00, 0x60, 0x00, 0x01, 0x8d, 0x0d, 0xd8, 0x01, 0x80, 0x02, 0x74,
	0x0f, 0xff, 0xf9, 0x90, 0x00, 0x00, 0x80, 0x00, 0x00, 0x0c, 0x03, 0x62,
	0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

static int load_bootstrap(void)
{
	const u8 *s = bootstrap_rbin;
	u32 l = sizeof(bootstrap_rbin);
	const u8 *data, *hdr, *h;
	u32 chksum, chksum2;
	int i, j, rangenr;
	u32 start, length;

	if (l < 12) {
		printf("bootstrap image corrupted. (too short header)\n");
		return -1;
	}

	chksum  = ((u32)s[4] << 24) | ((u32)s[5] << 16) | ((u32)s[ 6] <<  8) |  (u32)s[ 7];
	rangenr = ((u32)s[8] << 24) | ((u32)s[9] << 16) | ((u32)s[10] <<  8) |  (u32)s[11];
	s += 12; l -= 12;

	hdr = s;
	s += 8 * rangenr; l -= 8 * rangenr;
	data = s;

	/* validate bootstrap image */
	h = hdr; s = data; chksum2 = 0;
	for (i = 0; i < rangenr; i++) {
		start  = ((u32)h[0] << 24) | ((u32)h[1] << 16) | ((u32)h[2] <<  8) |  (u32)h[3];
		length = ((u32)h[4] << 24) | ((u32)h[5] << 16) | ((u32)h[6] <<  8) |  (u32)h[7];
		h += 8;

		/* too short */
		if (l < length) {
			printf("bootstrap image corrupted. (too short data)\n");
			return -1;
		}
		l -= length;

		j = (int)length / 4;
		while (j-- > 0) {
			chksum2 += ((u32)s[0] << 24) | ((u32)s[1] << 16) | ((u32)s[2] <<  8) |  (u32)s[3];
			s += 4;
		}
	}

	/* checksum must match */
	if (chksum != chksum2) {
		printf("bootstrap image corrupted. (checksum error)\n");
		return -1;
	}

	/* nothing must be left */
	if (l != 0) {
		printf("bootstrap image corrupted. (garbage at the end)\n");
		return -1;
	}

	/* write the image */
	h = hdr;
	s = data;
	for (i = 0; i < rangenr; i++) {
		start  = ((u32)h[0] << 24) | ((u32)h[1] << 16) | ((u32)h[2] <<  8) |  (u32)h[3];
		length = ((u32)h[4] << 24) | ((u32)h[5] << 16) | ((u32)h[6] <<  8) |  (u32)h[7];
		h += 8;
		c62_write(start, (u32 *)s, length / 4);
		s += length;
	}

	/* and now validate checksum */
	h = hdr;
	s = data;
	chksum2 = 0;
	for (i = 0; i < rangenr; i++) {
		start  = ((u32)h[0] << 24) | ((u32)h[1] << 16) | ((u32)h[2] <<  8) |  (u32)h[3];
		length = ((u32)h[4] << 24) | ((u32)h[5] << 16) | ((u32)h[6] <<  8) |  (u32)h[7];
		h += 8;
		chksum2 += c62_checksum(start, length / 4);
		s += length;
	}

	/* checksum must match */
	if (chksum != chksum2) {
		printf("bootstrap in DSP memory is corrupted\n");
		return -1;
	}

	return 0;
}

struct host_init {
	u32 master_mode;
	struct {
		u8 port_id;
		u8 slot_id;
	} ch_serial_map[32];
	u32 clk_divider[2];
	/* pll */
	u32 initmode;
	u32 pllm;
	u32 div[4];
	u32 oscdiv1;
	u32 unused[10];
};

const struct host_init hi_default = {
	.master_mode	=
#if !defined(CONFIG_NETTA_ISDN)
		-1,
#else
		0,
#endif

	.ch_serial_map	= {
		[ 0]	= { .port_id = 2, .slot_id = 16 },
		[ 1]	= { .port_id = 2, .slot_id = 17 },
		[ 2]	= { .port_id = 2, .slot_id = 18 },
		[ 3]	= { .port_id = 2, .slot_id = 19 },
		[ 4]	= { .port_id = 2, .slot_id = 20 },
		[ 5]	= { .port_id = 2, .slot_id = 21 },
		[ 6]	= { .port_id = 2, .slot_id = 22 },
		[ 7]	= { .port_id = 2, .slot_id = 23 },
		[ 8]	= { .port_id = 2, .slot_id = 24 },
		[ 9]	= { .port_id = 2, .slot_id = 25 },
		[10]	= { .port_id = 2, .slot_id = 26 },
		[11]	= { .port_id = 2, .slot_id = 27 },
		[12]	= { .port_id = 2, .slot_id = 28 },
		[13]	= { .port_id = 2, .slot_id = 29 },
		[14]	= { .port_id = 2, .slot_id = 30 },
		[15]	= { .port_id = 2, .slot_id = 31 },
	},

	/*
	   dsp_clk(xin, pllm)         = xin * pllm
	   serial_clk(xin, pllm, div) = (dsp_clk(xin, pllm) / 2) / (div + 1)
	 */

	.clk_divider	= {
		[0]	= 47,		/* must be 2048Hz */
		[1] 	= 47,
	},

	.initmode	= 1,
	.pllm		=
#if !defined(CONFIG_NETTA_ISDN)
	8,		/* for =~ 25MHz 8 */
#else
	4,
#endif
	.div		= {
		[0]	= 0x8000,
		[1]	= 0x8000,	/* for =~ 25MHz 0x8000 */
		[2]	= 0x8001,	/* for =~ 25MHz 0x8001 */
		[3]	= 0x8001,	/* for =~ 25MHz 0x8001 */
	},

	.oscdiv1	= 0,
};

static void hi_write(const struct host_init *hi)
{
	u32 hi_buf[1 + sizeof(*hi) / sizeof(u32)];
	u32 *s;
	u32 chksum;
	int i;

	memset(hi_buf, 0, sizeof(hi_buf));

	s = hi_buf;
	s++;
	*s++ = hi->master_mode;
	for (i = 0; i < (sizeof(hi->ch_serial_map) / sizeof(hi->ch_serial_map[0])) / 2; i++)
		*s++ = ((u32)hi->ch_serial_map[i * 2 + 1].slot_id << 24) | ((u32)hi->ch_serial_map[i * 2 + 1].port_id << 16) |
				((u32)hi->ch_serial_map[i * 2 + 0].slot_id << 8) | (u32)hi->ch_serial_map[i * 2 + 0].port_id;

	for (i = 0; i < sizeof(hi->clk_divider)/sizeof(hi->clk_divider[0]); i++)
		*s++ = hi->clk_divider[i];

	*s++ = hi->initmode;
	*s++ = hi->pllm;
	for (i = 0; i < sizeof(hi->div)/sizeof(hi->div[0]); i++)
		*s++ = hi->div[i];
	*s++ = hi->oscdiv1;

	chksum = 0;
	for (i = 1; i < sizeof(hi_buf)/sizeof(hi_buf[0]); i++)
		chksum += hi_buf[i];
	hi_buf[0] = -chksum;

	c62_write(0x1000, hi_buf, sizeof(hi_buf) / sizeof(hi_buf[0]));
}

static void run_bootstrap(void)
{
	dsp_go_slow();

	hi_write(&hi_default);

	/* signal interrupt */
	dsp_write_hpic(0x0002);
	dsp_delay();

	dsp_go_fast();
}

#endif

/***********************************************************************************************************/

int board_post_dsp(int flags)
{
	u32 ramS, ramE;
	u32 data, data2;
	int i, j, k;
#if !defined(CONFIG_NETTA_6412)
	int r;
#endif
	dsp_reset();
	dsp_init_hpic();
#if !defined(CONFIG_NETTA_6412)
	dsp_go_slow();
#endif
	data = 0x11223344;
	dsp_write_hpic_word(DSP_HPIA, data);
	data2 = dsp_read_hpic_word(DSP_HPIA);
	if (data2 !=   0x11223344) {
		printf("HPIA: ** ERROR; wrote 0x%08X read 0x%08X **\n", data, data2);
		goto err;
	}

	data = 0xFFEEDDCC;
	dsp_write_hpic_word(DSP_HPIA, data);
	data2 = dsp_read_hpic_word(DSP_HPIA);
	if (data2 != 0xFFEEDDCC) {
		printf("HPIA: ** ERROR; wrote 0x%08X read 0x%08X **\n", data, data2);
		goto err;
	}
#if defined(CONFIG_NETTA_6412)
	dsp_dram_initialize();
#else
	r = load_bootstrap();
	if (r < 0) {
		printf("BOOTSTRAP: ** ERROR ** failed to load\n");
		goto err;
	}

	run_bootstrap();

	dsp_go_fast();
#endif
	printf("    ");

	/* test RAMs */
	for (k = 0; k < sizeof(ranges)/sizeof(ranges[0]); k++) {

		ramS = ranges[k].start;
		ramE = ranges[k].start + ranges[k].size;

		for (j = 0; j < 3; j++) {

			printf("\b\b\b\bR%d.%d", k, j);

			for (i = ramS; i < ramE; i += 4) {

				data = 0;
				switch (j) {
					case 0: data = 0xAA55AA55; break;
					case 1: data = 0x55AA55AA; break;
					case 2: data = (u32)i;     break;
				}

				c62_write_word(i, data);
				data2 = c62_read_word(i);
				if (data != data2) {
					printf(" ** ERROR at 0x%08X; wrote 0x%08X read 0x%08X **\n", i, data, data2);
					goto err;
				}
			}
		}
	}

	printf("\b\b\b\b    \b\b\b\bOK\n");
#if !defined(CONFIG_NETTA_6412)
	/* XXX assume that this works */
	load_bootstrap();
	run_bootstrap();
	dsp_go_fast();
#endif
	return 0;

err:
	return -1;
}

int board_dsp_reset(void)
{
#if !defined(CONFIG_NETTA_6412)
	int r;
#endif
	dsp_reset();
	dsp_init_hpic();
#if defined(CONFIG_NETTA_6412)
	dsp_dram_initialize();
#else
	dsp_go_slow();
	r = load_bootstrap();
	if (r < 0)
		return r;

	run_bootstrap();
	dsp_go_fast();
#endif
	return 0;
}
