﻿/**
FormulaDebugger.FormulaTextBox

Copyright (c) 2015 Shigeyuki Horimoto

This software is released under the MIT License.
http://opensource.org/licenses/mit-license.php
*/
using Formula;
using Formula.Node;
using FormulaDebugger.Utility;
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;

namespace FormulaDebugger
{
    /// <summary>
    /// 式組み立てようのテキストボックス。
    /// キー入力に従いパースをかけるのでちらつき防止用に
    /// 内部的にテキストボックスを作成してることに注意。
    /// 以下順に処理。
    /// １．テキストボックスの内容変更
    /// ２．評価用テキストボックスに式をコピー
    /// ３．評価用テキストボックスのTextChangedイベントないで式評価。書式設定
    /// ４．書式込のRTFをコピー。
    /// ５．キャレット位置を元に戻す
    /// </summary>
    public class FormulaTextBox : RichTextBox
    {
        /// <summary>
        /// デフォルトフォント
        /// </summary>
        private Font _DefaultFont { get; set; }
        /// <summary>
        /// デフォルト文字色
        /// </summary>
        private Color _DefaultForeColor { get; set; }
        /// <summary>
        /// デフォルト背景色
        /// </summary>
        private Color _DefaultBackColor { get; set; }
        /// <summary>
        /// 式評価用ビルダ
        /// </summary>
        public NodeTreeBuilder builder { get; set; }
        /// <summary>
        /// イベント：式の評価が終了した
        /// エラー発生の有無確認用を想定
        /// </summary>
        public event EventHandler OnEndEval;

        /// <summary>
        /// 式の解釈・書式付テキスト構築用ダミーテキストボックス
        /// </summary>
        private RichTextBox dummyTextBox {get;set;}

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public FormulaTextBox() : base()
        {
            this.dummyTextBox = new RichTextBox();
            this.dummyTextBox.TextChanged += dummyTextBox_TextChanged;
            this.TextChanged += _TextChanged;
        }
        

        /// <summary>
        /// クラスの初期化
        /// ・デフォルトのフォントを確保
        /// </summary>
        public void Initialize()
        {
            this._DefaultForeColor = this.ForeColor;
            this._DefaultBackColor = this.BackColor;
            this._DefaultFont = this.Font;
        }

        /// <summary>
        /// イベント：式が変更された
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void _TextChanged(object sender, EventArgs e)
        {
            this.TextChanged -= _TextChanged;
            int selectPos = this.SelectionStart;
            this.dummyTextBox.SelectionStart = 0;
            this.dummyTextBox.SelectionLength = this.dummyTextBox.Text.Length+1;
            this.dummyTextBox.ForeColor = this._DefaultForeColor;
            this.dummyTextBox.BackColor = this._DefaultBackColor;
            this.dummyTextBox.Font = this._DefaultFont;

            this.ForeColor = this._DefaultForeColor;
            this.BackColor = this._DefaultBackColor;
            this.Font = this._DefaultFont;
            this.dummyTextBox.Text = this.Text;
            this.Rtf = this.dummyTextBox.Rtf;
            this.SelectionStart = selectPos;
            this.TextChanged += _TextChanged;
        }

        /// <summary>
        /// 式のパース用ダミーテキストボックス
        /// イベント：式が変更された
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void dummyTextBox_TextChanged(object sender, EventArgs e)
        {
            if (this.builder == null) return;
            ColorUtility.getInstance().Reset();
            RichTextBox c = (RichTextBox)sender;
            string formula = c.Text;
            bool isOk = this.builder.analyzeFormula(formula , true);
            INode rootNode = this.builder.Tree;
            ColorSetVisitor.visit(c, rootNode);
            FormulaCheckVisitor.ErrorInformation errorInfo = this.builder.ErrorInfo;
            if (!isOk)
            {
                c.SelectionStart = errorInfo.ErrorNode.startPosition;
                c.SelectionLength = errorInfo.ErrorNode.endPosition
                    - errorInfo.ErrorNode.startPosition
                    + 1;
                c.ForeColor = Color.Red;
            }
            VariableManager man = new VariableManager(rootNode);
            man.loadVariable();
            List<string> varNameList = man.variableDictionary.Keys.ToList();
            HashSet<string> varNameSet = new HashSet<string>();
            foreach (string varName in varNameList)
                varNameSet.Add(varName);

            if (this.OnEndEval != null)
            {
                OnEndEval(this, new EndEvalEventArgs()
                {
                    IsError = !isOk
                    , ErrorMessage = errorInfo == null ? null : errorInfo.ErrorMessage
                    , ErrorNode = errorInfo == null ? null : errorInfo.ErrorNode
                    , Formula = formula
                    , VariableNameSet = varNameSet
                    , Node = rootNode
                });
            }
        }


        public class EndEvalEventArgs : EventArgs
        {
            /// <summary>
            /// エラーの有無
            /// </summary>
            public bool IsError { get; set; }
            /// <summary>
            /// エラーが発生してる場合、エラーが発生した箇所
            /// (エラーなしの場合、null)
            /// </summary>
            public INode ErrorNode { get; set; }
            /// <summary>
            /// エラーの内容
            /// (エラーなしの場合、null)
            /// </summary>
            public string ErrorMessage { get; set; }
            /// <summary>
            /// 評価された式
            /// </summary>
            public string Formula { get; set; }
            /// <summary>
            /// エラーが発生していない場合、
            /// パースされた変数のリスト
            /// </summary>
            public HashSet<string> VariableNameSet { get; set; }
            /// <summary>
            /// パースされた構文木
            /// （エラー発生時は最後まで構築されてないので注意）
            /// </summary>
            public INode Node { get; set; }
        }
    }
}
