﻿/**
Formula.PriorityChangeVisitor

Copyright (c) 2015 Shigeyuki Horimoto

This software is released under the MIT License.
http://opensource.org/licenses/mit-license.php
*/
using Formula.Node;
using Formula.Node.Impl;

namespace Formula
{
    /// <summary>
    /// 構文木を走査し、優先順に応じた組み換えを行います。
    /// 優先順はIOperatorNode.priorityの値を用います。
    /// IOperatorNode.priorityの値が大きいほど計算の優先順が高いと判断し、構文木の再構築を行います。
    /// </summary>
    public class PriorityChangeVisitor
    {
        /// <summary>
        /// 現在の処理対象となるノードの親
        /// </summary>
        public INode parentNode { get; set; }
        /// <summary>
        /// 処理対象となるノード
        /// </summary>
        public INode currentNode { get; set; }
        /// <summary>
        /// ノードの組み換え処理を行います。
        /// </summary>
        /// <returns>True:構文木を組み替えた False:組み換えの必要なし</returns>
        public bool visit()
        {
            //親ノード・子ノードが存在し、
            //両方が演算子、（ただし子ノードは括弧ではない。括弧は入れ替えない）
            //以上の条件を満たしたうえで、
            //子供の右ノードが存在しない場合（[1*-1]の場合、*の右は存在しない）
            //[-]
            //├[*]
            //│├1
            //│└  ここがnullの場合、置き換える必要がある。
            //└1
            //もしくは
            //子供の左ノードが存在かつ子供の重要度が親のそれを上回る([-1]や[+1]の場合、演算子の左は存在しない。)
            //[*]
            //├1
            //└[-]
            //　├  ここがnullの場合、-は続く1と同一視する必要があるので置き換えない
            //　└1 
            //場合に親と子供を置き換える。
            if (currentNode is IOperatorNode
                && parentNode != null
                && parentNode is IOperatorNode
                && !(currentNode is IBracketNode)
                //&& !(parentNode is IBracketNode)
                //&& !(currentNode is ISeparatorNode)
                && (
                        ((((IOperatorNode)currentNode).getRightNode() == null || ((IOperatorNode)currentNode).getRightNode() is BlankNode)
                            && !(currentNode is ISeparatorNode)
                            && ((IOperatorNode)parentNode).priority > ((IOperatorNode)currentNode).priority)
                        || ((((IOperatorNode)currentNode).getLeftNode() != null && !(((IOperatorNode)currentNode).getLeftNode() is BlankNode))
                            && ((IOperatorNode)parentNode).priority > ((IOperatorNode)currentNode).priority)
                    )
                )
            {
                IOperatorNode priorityParentNode = parentNode == null ? null : ((IOperatorNode)parentNode).getPriorityParentNode();
                INode rightNode = ((IOperatorNode)currentNode).getRightNode();
                if (priorityParentNode.ParentNode is IOperatorNode)
                    ((IOperatorNode)priorityParentNode.ParentNode).setLeftNode(currentNode);

                currentNode.ParentNode = priorityParentNode.ParentNode;

                ((IOperatorNode)this.parentNode).setLeftNode(rightNode);
                if (rightNode != null) rightNode.ParentNode = this.parentNode;

                ((IOperatorNode)currentNode).setRightNode(priorityParentNode);
                priorityParentNode.ParentNode = currentNode;

                return true;
            }
            else if (currentNode is IFunctionNode)
            {
                //カレントの処理をする必要がない場合、カレントの子供を順に処理する
                foreach (INode n in ((IFunctionNode)currentNode).Args)
                {
                    PriorityChangeVisitor visitor = new PriorityChangeVisitor();
                    visitor.parentNode = this.currentNode;
                    visitor.currentNode = n;
                    if (visitor.visit()) return true;
                }
            }
            return false;
        }
    }
}
