﻿/**
Formula.FormulaReader

Copyright (c) 2015 Shigeyuki Horimoto

This software is released under the MIT License.
http://opensource.org/licenses/mit-license.php
*/
using System.Collections.Generic;
using System.Text;

namespace Formula
{
    /// <summary>
    /// 式を予約語単位に分割します。
    /// </summary>
    public class FormulaReader : IEnumerator<string> , IEnumerable<string>
    {
        /// <summary>
        /// 予約語のリスト
        /// </summary>
        public List<string> reserved { get; set; }

        /// <summary>
        /// 分解対象の式
        /// </summary>
        public string formula { get; set; }

        /// <summary>
        /// エスケープシーケンス
        /// 通常は\(エンマーク)
        /// </summary>
        public string escapeSequence { get; set; }
        /// <summary>
        /// 文字列として認識させるための識別子。
        /// 通常は"(ダブルクォート)
        /// </summary>
        public string blockSignature { get; set; }

        /// <summary>
        /// 次に読みだす位置
        /// 読み出し対象はthis.formula
        /// </summary>
        private int position;
        public int Position{
            get { return this.position; }
        }
        /// <summary>
        /// 現在読み出し対象の文字列。
        /// （formulaを予約語で割った際の文字列）
        /// </summary>
        private string current;
        public string Current
        {
            get { return this.current; }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public FormulaReader()
        {
            this.reserved = new List<string>();
            this.escapeSequence = "\\";
            this.blockSignature = "\"";
            this.Reset();
        }

        /// <summary>
        /// 初期化
        /// </summary>
        public void initialize()
        {
            this.reserved.Add(this.escapeSequence);
            this.reserved.Add(this.blockSignature);
        }

        /// <summary>
        /// 次の要素を読みだす
        /// </summary>
        /// <returns>True:読み出し成功 False:読みだせない=末尾まで読み込み済み</returns>
        public bool MoveNext()
        {
            StringBuilder str = new StringBuilder();
            bool ret = false;
            for (; this.position < this.formula.Length;)
            {
                char c = this.formula[this.position++];
                if (reserved.Contains(c.ToString()))
                {
                    //先読みしてある場合は一つの演算子として扱う
                    StringBuilder msg = new StringBuilder();
                    msg.Append(c);
                    for (; this.position < this.formula.Length; )
                    {
                        msg = msg.Append( this.formula[this.position++]);
                        if(!reserved.Contains(msg.ToString()))
                        {
                            this.position--;
                            msg.Remove(msg.Length -1, 1);
                            break;
                        }
                    }

                    if (string.IsNullOrEmpty(str.ToString()))
                        this.current = msg.ToString();
                    else
                    {
                        this.current = str.ToString();
                        this.position -= msg.Length;
                    }
                    ret = true;
                    break;
                }
                str = str.Append(c);
                this.current = str.ToString();
                ret = true;

            }
            return ret;
        }

        /// <summary>
        /// 読み出し位置を初期化する
        /// </summary>
        public void Reset()
        {
            this.position = 0;
        }


        #region IEnumerable<string> メンバー

        public IEnumerator<string> GetEnumerator()
        {
            //throw new NotImplementedException();
            return this;
        }

        #endregion

        #region IEnumerable メンバー

        System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator()
        {
            //throw new NotImplementedException();
            return this;
        }

        #endregion

        #region IDisposable メンバー

        public void Dispose()
        {
            //throw new NotImplementedException();
        }

        #endregion

        #region IEnumerator メンバー

        object System.Collections.IEnumerator.Current
        {
            get { return this.current; }
        }

        #endregion
    }
}
