﻿/**
Formula.FormulaParser

Copyright (c) 2015 Shigeyuki Horimoto

This software is released under the MIT License.
http://opensource.org/licenses/mit-license.php
*/
using System.Collections.Generic;
using Formula.Node.Impl;
using Formula.Node;
using System.Text.RegularExpressions;
using System.Linq;
using System.Collections.Concurrent;

namespace Formula
{
    /// <summary>
    /// 要素のリストから構文木を作成します。
    /// 四則演算の優先順は考慮しません。
    /// </summary>
    public class FormulaParser
    {
        /// <summary>
        /// 読み出し対象の式を分割したリスト
        /// </summary>
        public IEnumerator<Word> Formula { get; set; }
        /// <summary>
        /// 演算子のリスト (string:シグネチャ
        /// </summary>
        public IDictionary<string, List<NodeInfo>> OperatorTable;
        /// <summary>
        /// メソッドのリスト (string:シグネチャ
        /// </summary>
        public IDictionary<string, NodeInfo> MethodTable;
        /// <summary>
        /// 変数タイプのリスト (string:識別するための正規表現
        /// </summary>
        public IDictionary<string, NodeInfo> ValueTable;
        /// <summary>
        /// 構文木を構成した際に「変数として」認識されたノードのリスト。
        /// </summary>
        public List<VariableNode> VariableList;

        /// <summary>
        /// 最後に処理をしたノード
        /// </summary>
        public INode LastNode { get; set; }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public FormulaParser()
        {
            this.OperatorTable = new ConcurrentDictionary<string, List<NodeInfo>>();
            this.MethodTable = new ConcurrentDictionary<string, NodeInfo>();
            this.VariableList = new List<VariableNode>();
            this.ValueTable = new ConcurrentDictionary<string, NodeInfo>();
        }

        /// <summary>
        /// 構文木を作成する
        /// </summary>
        /// <returns>作成した構文木のルートとなるノード</returns>
        public INode parse()
        {
            INode ret = null;
            //INode lastNode = null;
            while(this.Formula.MoveNext())
            {
                Word block = this.Formula.Current;
                if (block.word == null || (!block.isBlock && string.IsNullOrWhiteSpace(block.word.ToString())))
                    continue;
                string o = block.word.ToString().ToLower();
                if (!block.isBlock) o = o.Trim();
                INode currentNode = null;
                //===========================
                //算術演算子/論理演算子の処理
                //===========================
                if (!block.isBlock && this.OperatorTable.ContainsKey(o))
                {
                    //NodeInfo i = (NodeInfo)this.OperatorTable[o];
                    NodeInfo i = (from a in this.OperatorTable[o]
                                      //where ((IOperatorNode)a.sample).IsPermitLeftNodeBlank == (this.LastNode == null || this.LastNode is IFunctionNode)
                                  where ((IOperatorNode)a.sample).IsPermitLeftNodeBlank == (this.LastNode == null || (this.LastNode is IOperatorNode && !(this.LastNode is IBracketNode) && !(this.LastNode is ISeparatorNode)))
                                  select a).SingleOrDefault();
                    i = i ?? this.OperatorTable[o][0];
                    
                    currentNode = (INode)i.assembly.CreateInstance(i.type.FullName);
                    currentNode.startPosition = block.startPosition;
                    currentNode.endPosition = block.endPosition;
                    this.LastNode = currentNode;
                    if (currentNode is IBracketNode)
                    {
                        #region 括弧
                        //開き括弧の場合、再帰
                        if (((IBracketNode)currentNode).OpenBracket == o)
                        {
                            FormulaParser p = new FormulaParser();
                            p.MethodTable = this.MethodTable;
                            p.OperatorTable = this.OperatorTable;
                            p.ValueTable = this.ValueTable;
                            p.Formula = this.Formula;
                            INode cNode = p.parse();
                            if (p.LastNode != null) currentNode.endPosition = p.LastNode.endPosition;
                            ((IBracketNode)currentNode).Args.Add(cNode);
                            if(cNode != null) cNode.ParentNode = currentNode;
                            if (ret == null)
                                ret = currentNode;
                            else if (ret is IFunctionNode)
                            {
                                ((IFunctionNode)ret).Args.Add(currentNode);
                                currentNode.ParentNode = ret;
                            }
                            if (ret.ParentNode != null)
                                ret = ret.ParentNode;
                            continue;
                        }
                        //閉じ括弧の場合、呼び出し元に組み立てたnodeを返す
                        else if (((IBracketNode)currentNode).CloseBracket == o)
                        {
                            if (ret is IOperatorNode && ((IOperatorNode)ret).getRightNode() == null)
                                ((IOperatorNode)ret).setRightNode(new BlankNode());

                            return ret;
                        }
                        #endregion
                    }
                }
                if (currentNode != null)
                {
                    if(ret is IOperatorNode)
                    {
                        if(((IOperatorNode)ret).getRightNode() == null)
                            ((IOperatorNode)ret).setRightNode(new BlankNode());
                    }
                    ((AbstractOperatorNode)currentNode).Args.Add(ret);
                    if(ret != null) ret.ParentNode = currentNode;
                    ret = currentNode;
                    
                    continue;
                }

                //==========
                //関数の処理
                //==========
                if (!block.isBlock && this.MethodTable.ContainsKey(o))
                {
                    NodeInfo i = (NodeInfo)this.MethodTable[o];
                    currentNode = (INode)i.assembly.CreateInstance(i.type.FullName);
                    currentNode.startPosition = block.startPosition;
                    currentNode.endPosition = block.endPosition;
                }

                if (currentNode == null)
                {
                    //======================
                    //演算子・関数以外の処理
                    //======================
                    if (block.isBlock)
                    {
                        currentNode = new WordNode();
                        ((AbstractValueNode)currentNode).Value = block.word;
                    }
                    else
                    {
                        string w = block.word.ToString().Trim().ToUpper();
                        foreach (KeyValuePair<string , NodeInfo> v in this.ValueTable) {
                            if(Regex.IsMatch(w , v.Key))
                            {
                                currentNode = (INode)v.Value.assembly.CreateInstance(v.Value.type.FullName);
                                break;
                            }
                        }
                        if(currentNode == null)
                        {
                            currentNode = new VariableNode();
                            this.VariableList.Add((VariableNode)currentNode);
                        }

                        ((AbstractValueNode)currentNode).Value = block.word.ToString().Trim();
                    }
                }
                currentNode.startPosition = block.startPosition;
                currentNode.endPosition = block.endPosition;
                if (ret != null && ret is IFunctionNode)
                {
                    ((IFunctionNode)ret).Args.Add(currentNode);
                    currentNode.ParentNode = ret;
                }
                if (ret == null || currentNode is IFunctionNode)
                    ret = currentNode;
                this.LastNode = currentNode;
            }

            if (ret is IOperatorNode && ((IOperatorNode)ret).getRightNode() == null)
                ((IOperatorNode)ret).setRightNode(new BlankNode());

            return ret;
        }
    }
}

