!
! Copyright (C) 2001-2014 PWSCF group
! This file is distributed under the terms of the
! GNU General Public License. See the file `License'
! in the root directory of the present distribution,
! or http://www.gnu.org/copyleft/gpl.txt .
!
!-----------------------------------------------------------------------
!
! This program reads [prefix].dyn* and lambda*.dat
! then draw the dispersion plot
!
!-------------------------------------------------------------------------
module disp_width_vals
  !-----------------------------------------------------------------------
  !
  ! This module contains global variables for disp_width.x
  !
  integer,save :: &
  & natm, & !
  & nq,   & !
  & nm      !
  !
  real(8),save :: &
  & avec(3,3)
  !
  integer,allocatable,save :: &
  & qindx(:), &
  & mindx(:,:)
  !
  real(8),allocatable,save :: &
  & mass(:), &
  & proj(:,:,:), &
  & qx(:),     & 
  & qvec(:,:), &
  & lam(:,:),  &
  & omg(:,:)
  !
  complex(8),allocatable,save :: &
  & dyn(:,:,:)
  !
  character(100),save :: &
  & fildyn, &
  & prefix
  !
end module disp_width_vals
!
!-----------------------------------------------------------
module disp_width_routines
  !---------------------------------------------------------
  !
  ! This module contains subroutines for disp_width.x
  !
  implicit none
  !
contains
!
!------------------------------------------------------------
subroutine read_stdin
  !
  ! This routine Reads the number and the order of q points,
  ! and lattice vectors from standard inputs
  !
  use disp_width_vals, only : nq, fildyn, qindx, prefix
  !
  namelist /input/ nq, fildyn, prefix
  !
  read(*,input,err=100)
  write(*,*) "         # of q vectors : ", nq
  write(*,*) "  dyn. file : ", trim(fildyn)
  write(*,*) "     prefix : ", trim(prefix)
  !
  allocate(qindx(nq))
  !
  read(*,*) qindx(1:nq)
  !
  return
  !
100 write(*,*) "stop. reading namelist file"
  !
  stop
  !
end subroutine read_stdin
!
!------------------------------------------------------------------
subroutine read_dat()
  !----------------------------------------------------------------
  !
  ! This routine reads symmetry and bvec from data-file.xml
  !  ng(1:3) from standard input
  !
  use disp_width_vals, only : avec, nm, mass, prefix
  use iotk_module
  !
  integer :: ispc, iatm, fi = 10, natm, nspc
  character(iotk_namlenx) :: attr1, attr2, attr3
  real(8) :: alat
  real(8),allocatable :: mass0(:)
  !
  ! Open datafile.xml
  !
  write(*,'(a,a)') "   open data-file.xml"
  call iotk_open_read(fi, trim(prefix) // ".save/data-file.xml")
  !
  ! Read reciprocal lattice vecor
  ! 
  call iotk_scan_begin(fi,"CELL")
  !
  call iotk_scan_dat(fi,"LATTICE_PARAMETER",alat)
  write(*,*) "  lattice constant[a.u] : ", alat
  call iotk_scan_begin(fi,"DIRECT_LATTICE_VECTORS")
  call iotk_scan_dat(fi,"a1", avec(1:3,1))
  call iotk_scan_dat(fi,"a2", avec(1:3,2))
  call iotk_scan_dat(fi,"a3", avec(1:3,3))
  !
  avec(1:3,1:3) = avec(1:3,1:3) / alat
  write(*,*) "  Direct lattice vector[a] : "
  write(*,*) avec(1:3,1)
  write(*,*) avec(1:3,2)
  write(*,*) avec(1:3,3)
  write(*,*) ""
  call iotk_scan_end(fi,"DIRECT_LATTICE_VECTORS")
  !
  call iotk_scan_end(fi,"CELL")
  !
  ! Read Atomic masses
  !
  call iotk_scan_begin(fi,"IONS")
  call iotk_scan_dat(fi,"NUMBER_OF_ATOMS", natm)
  call iotk_scan_dat(fi,"NUMBER_OF_SPECIES", nspc)
  !
  nm = 3 * natm
  !
  write(*,*) "  # of atoms : ", natm
  write(*,*) "  # of species : ", nspc
  write(*,*) "  # of modes : ", nm
  !
  allocate(mass0(nspc), mass(natm))
  !
  do ispc = 1, nspc
     !
     write(attr1,*) ispc
     write(attr2,'(a,a)') "SPECIE.", trim(adjustl(attr1))
     !
     call iotk_scan_begin(fi,trim(attr2))
     call iotk_scan_dat(fi,"MASS", mass0(ispc))
     call iotk_scan_end(fi,trim(attr2))
     !
  end do
  !
  write(*,*) "Atomic masses [AMU]"
  !
  do iatm = 1, natm
     !
     write(attr1,*) iatm
     write(attr2,'(a,a)') "ATOM.", trim(adjustl(attr1))
     !
     attr3=""
     call iotk_scan_empty(fi,trim(attr2),attr3)
     call iotk_scan_attr(attr3,"INDEX",ispc)
     !
     mass(iatm) = mass0(ispc)
     !
     write(*,'(a,i3,a,e15.5)') "Atom.", iatm, " : ", mass(iatm)
     !
  end do
  !
  mass(1:natm) = mass(1:natm) * 911.444242132272d0
  !
  call iotk_scan_end(fi,"IONS")
  !
  call iotk_close_read(fi)
  !
  deallocate(mass0)
  !
end subroutine read_dat
!
!-------------------------------------------------------------------------
subroutine read_dyn()
  !-----------------------------------------------------------------------
  !
  ! This routine read the polarization vectors, frequencies, and lambda
  ! from [prefix].dyn* & lambda*.dat
  !
  use disp_width_vals, only : nq, nm, omg, dyn, fildyn, qvec, qindx, avec, lam, mass
  !
  integer :: iq, im, nline, fi = 10,  ii, jj, ntype
  real(8) :: qvec2(3), norm
  character :: tmp(3)
  character(100) :: fildyn2, ciq
  !
  allocate(omg(nm,nq), dyn(nm,nm,nq), lam(nm,nq), qvec(1:3,nq))
  !
  do iq = 1, nq
     !
     write(ciq,*) qindx(iq)
     write(fildyn2,'(a,a)') trim(fildyn), trim(adjustl(ciq))
     open(fi, file = trim(fildyn2))
     !
     do im = 1, 100000
        !
        read(fi,*) tmp(1)
        !
        if(tmp(1) == "*") then
           nline = im - 1
           exit
        end if
        !
     end do
     !
     do im = 1, nm
        !
!        read(fi,'(5x,"omega(",i2,") =",f15.6)') ii, omg(im,iq)
        read(fi,'(a22,f12.6)') ciq, omg(im,iq)
        !
        do ii = 1, nm / 3
           read(fi,'(a2,6f10.6)') ciq, dyn((ii - 1) * 3 + 1:(ii - 1) * 3 + 3, im, iq)
           dyn((ii - 1) * 3 + 1:(ii - 1) * 3 + 3, im, iq) = &
           &  dyn((ii - 1) * 3 + 1:(ii - 1) * 3 + 3, im, iq) * sqrt(mass(ii))
        end do
        !
        ! Normalize Dyn
        ! 
        norm = dble(dot_product(dyn(1:nm,im,iq), dyn(1:nm,im,iq)))
        dyn(1:nm,im,iq) = dyn(1:nm,im,iq) / sqrt(norm)
        !
     end do
     !
     rewind(fi)
     !
     do im = 1, nline - 1
        read(fi,*) tmp(1)
     end do
     !
     read(fi,*) tmp(1:3), qvec(1:3,iq)
     !
     ! q = GAMMA ?
     !
     qvec2(1:3) = matmul(qvec(1:3,iq), avec(1:3,1:3))
     qvec2(1:3) = abs(qvec2(1:3) - dble(nint(qvec2(1:3))))
     if(all(qvec2(1:3) < 1d-10)) omg(1:3,iq) = 0d0
     !
     close(fi)
     !
     ! Electron-phonon
     !
     write(ciq,*) qindx(iq)
     write(fildyn2,'(a,a,a)') "lambda", trim(adjustl(ciq)), ".dat"
     open(fi, file = trim(fildyn2), status = 'old', iostat = ii)
     !
     if(ii == 0) then
        !
        read(fi,*) ii
        read(fi,*) qvec2(1:3)
        !
        do im = 1, nm
           read(fi,*) qvec2(1), lam(im,iq)
        end do
        !
     else
        !
        write(*,*) trim(fildyn2), "is not exist."
        lam(1:nm,iq) = 0d0
        !
     end if
     !
  end do ! iq
  !
end subroutine read_dyn
!
!-------------------------------------------------------------
subroutine xaxis()
  !-----------------------------------------------------------
  !
  ! This routine set x axis for plotting the phonon dispersion.
  !
  use disp_width_vals, only : nq, qvec, qx
  !
  integer :: iq
  real(8) :: dq(3)
  !
  allocate(qx(nq))
  !
  qx(1) = 0d0
  !
  do iq = 2, nq
     !
     dq(1:3) = qvec(1:3,iq) - qvec(1:3, iq-1)
     qx(iq) = sqrt(dot_product(dq(1:3), dq(1:3)))
     qx(iq) = qx(iq - 1) + qx(iq)
     !
  end do
  !
end subroutine xaxis
!
!---------------------------------------------------------------------
subroutine mode_cross()
  !-------------------------------------------------------------------
  !
  ! This routine re-connect branches considering the crossing.
  !
  use disp_width_vals, only : nm, nq, dyn, omg, mindx
  !
  integer :: iq, im, jm, ovrpmax(2)
  real(8) :: ovrp(nm,nm), tmp(nm)
  complex(8) :: covrp
  !
  allocate(mindx(nm,nq))
  !
  do im = 1, nm
     mindx(im,1) = im
  end do
  !
  do iq = 2, nq
     !
     do im = 1, nm
        !
        do jm = 1, nm
           !
           covrp = dot_product(dyn(1:nm,im,iq), dyn(1:nm,jm,iq-1))
           ovrp(jm,im) = dble(conjg(covrp) * covrp)
           !
        end do ! jm
        !
     end do  ! im
     !
     do im = 1, nm
        !
        ovrpmax(1:2) = maxloc(ovrp(1:nm,1:nm))
        !
        mindx(ovrpmax(2),iq) = mindx(ovrpmax(1),iq - 1)
        !
        ovrp(ovrpmax(1),      1:nm) = - 1d8
        ovrp(      1:nm,ovrpmax(2)) = - 1d8
        !
     end do ! im
     !
  end do ! iq
  !
end subroutine mode_cross
!
!--------------------------------------------------------------
subroutine atom_proj()
  !--------------------------------------------------------------
  !  
  ! This routine compute contributions from each atoms.
  !
  use disp_width_vals, only : nq, nm, dyn, proj, natm
  !
  integer :: iq, im, iatm, first, last
  real(8) :: proj0(nm)
  !
  natm = nm / 3
  allocate(proj(natm,nm,nq))
  !
  do iq = 1, nq
     !
     do im = 1, nm
        !
        do iatm = 1, natm
           !
           first = 3 * (iatm - 1) + 1
           last  = 3 * (iatm - 1) + 3
           !
           proj(iatm,im,iq) = dble(dot_product(dyn(first:last,im,iq), &
           &                                   dyn(first:last,im,iq)  ))
           !
        end do ! iatm
        !
     end do ! im
     !
  end do ! iq
  !
end subroutine atom_proj
!
!----------------------------------------------------------------------------------
subroutine write_disp()
  !--------------------------------------------------------------------------------
  !
  ! This routine outputs the dispersion to a file (disp.dat).
  !
  use disp_width_vals, only : nq, qx, omg, lam, nm, mindx, proj, natm
  !
  integer :: fo = 20, im, iq, jm
  !
  open(fo, file = "disp.dat")
  !
  do im = 1, nm
     !
     do iq = 1, nq
        !
        do jm = 1, nm
           !
           if(mindx(jm,iq) /= im) cycle
           !
           write(fo,'(100e25.15)') qx(iq), omg(jm,iq), lam(jm,iq), proj(1:natm,jm,iq)
           !
        end do ! jm
        !
     end do ! iq
     !
     write(fo,*) ""
     write(fo,*) ""
     !
  end do ! im
  !
  close(fo)
  !
end subroutine write_disp
!
end module disp_width_routines
!
!-------------------------------------------------------------------------
program disp_width
  !-----------------------------------------------------------------------
  !
  ! This program reads [prefix].dyn* and lambda*.dat
  ! then draw the dispersion plot
  !
  use disp_width_routines, only : read_stdin, read_dyn, xaxis, &
  &                         mode_cross, write_disp, atom_proj, read_dat
  implicit none
  !
  call read_stdin()
  !
  call read_dat()
  !
  call read_dyn()
  !
  call xaxis()
  !
  call mode_cross()
  !
  call atom_proj()
  !
  call write_disp()
  !
end program disp_width
