/*
   This file is provided under the LGPL license ver 2.1.
   Written by K.Tanaka & Katsumi
   http://www.ze.em-net.ne.jp/~kenken/index.html
   http://hp.vector.co.jp/authors/VA016157/
*/

#include <xc.h>
#include "compiler.h"

/*
	Library functions follow
*/

void lib_out(int pos, int val){
	// pos must be between 0 and 15 or 16 and 18
	if (0<=pos && pos<=15) {
		// PORTB0-15
		// Set output vale
		if (val) {
			LATBSET=1<<pos;
		} else {
			LATBCLR=1<<pos;
		}
		// Enable output
		TRISBCLR=1<<pos;
		// Disable pulldown
		CNPUBCLR=1<<pos;
	} else if (16<=pos && pos<=18) {
		// PORTE5-7
		pos=pos-16+5;
		// Set output vale
		if (val) {
			LATESET=1<<pos;
		} else {
			LATECLR=1<<pos;
		}
		// Enable output
		TRISECLR=1<<pos;
		// Disable pulldown
		CNPUECLR=1<<pos;
	} else {
		return;
	}
}

void lib_out8h(int val){
	// Set output vale
	LATB=(LATB&0x00FF)|((val&0xff)<<8);
	// Enable output
	TRISBCLR=0xFF00;
	// Disable pulldown
	CNPUBCLR=0xFF00;
}

void lib_out8l(int val){
	// Set output vale
	LATB=(LATB&0xFF00)|(val&0xff);
	// Enable output
	TRISBCLR=0x00FF;
	// Disable pulldown
	CNPUBCLR=0x00FF;
}

int lib_out16(int val){
	// Set output vale
	LATB=(val&0xFFFF);
	// Enable output
	TRISBCLR=0xFFFF;
	// Disable pulldown
	CNPUBCLR=0xFFFF;
}

int lib_in(int pos){
	// pos must be between 0 and 15 or 16 and 18
	if (0<=pos && pos<=15) {
		// PORTB0-15
		// Enable pulldown
		CNPUBSET=1<<pos;
		// Enable input
		TRISBSET=1<<pos;
		ANSELBCLR=1<<pos;
		// Read value and return
		return (PORTB&(1<<pos)) ? 1:0;
	} else if (16<=pos && pos<=18) {
		// PORTE5-7
		pos=pos-16+5;
		// Enable pulldown
		CNPUESET=1<<pos;
		// Enable input
		TRISESET=1<<pos;
		ANSELECLR=1<<pos;
		// Read value and return
		return (PORTE&(1<<pos)) ? 1:0;
	} else {
		return 0;
	}
}

int lib_in8h(){
	// Enable pulldown
	CNPUBSET=0xFF00;
	// Enable input
	TRISBSET=0xFF00;
	ANSELBCLR=0xFF00;
	// Read value and return
	return (PORTB&0xFF00)>>8;
}

int lib_in8l(){
	// Enable pulldown
	CNPUBSET=0x00FF;
	// Enable input
	TRISBSET=0x00FF;
	ANSELBCLR=0x00FF;
	// Read value and return
	return PORTB&0x00FF;
}

int lib_in16(){
	// Enable pulldown
	CNPUBSET=0xFFFF;
	// Enable input
	TRISBSET=0xFFFF;
	ANSELBCLR=0xFFFF;
	// Read value and return
	return PORTB&0xFFFF;
}

int lib_analog(int pos){
	/*
		Analog to 12 bit digital converter function.
		AD1CON1=0x00E0;
		AD1CON1bits.ON=0;
		AD1CON1bits.SIDL=0;    // Continue module operation in Idle mode
		AD1CON1bits.FORM=0;    // Integer 16-bit
		AD1CON1bits.SSRC=8;    // Internal counter ends sampling and starts conversion (auto convert)
		AD1CON1bits.CLRASAM=0; // Normal operation, buffer contents will be overwritten by the next conversion sequence
		AD1CON1bits.ASAM=0;    // Sampling begins when SAMP bit is set.
		AD1CON1bits.SAMP=0;    // The ADC sample/hold amplifier is holding
		AD1CON1bits.DONE=0;    // Analog-to-digital conversion is not done or has not started
		AD1CON2=0;
		AD1CON2bits.VCFG=0;   // Voltage reference:AVdd-AVss
		AD1CON2bits.OFFCAL=0; // Disable Offset Calibration mode
		AD1CON2bits.CSCNA=0;  // Do not scan inputs
		AD1CON2bits.BUFS=0;   // Do not care, only valid when BUFM=1
		AD1CON2bits.SMPI=0;   // Do not care. Do not use interrupt.
		AD1CON2bits.BUFM=0;   // Buffer configured as one 16-word buffer ADC1BUFF-ADC1BUF0
		AD1CON2bits.ALTS=0;   // Always use Sample A input multiplexer settings
		AD1CON3=0;
		AD1CON3bits.ADRC=0; // Clock derived from Peripheral Bus Clock (PBCLK)
		AD1CON3bits.SAMC=0; // Auto-Sample not allowed
		AD1CON3bits.ADCS=0; // Do not care, only valid when ADRC=1.
		AD1CHS=0-27;
		AD1CHSbits.CN0NB=0;    // Do not care, only valid when using channel B
		AD1CHSbits.CH0SB=0;    // Do not care, only valid when using channel B
		AD1CHSbits.CH0NA=0;    // Channel 0 negative input is VREFL
		AD1CHSbits.CH0SA=0-27; // Set input channel here
		AD1CSSL; // Do not care, only valid when CSCNA=1;
	*/
	AD1CON1=0x00E0;
	AD1CON2=0x0000;
	AD1CON3=0x0000;
	// pos must be between 0 and 15 or 16 and 18
	if (0<=pos && pos<=15) {
		// RB0-RB15: AN0-AN15
		// Disable pulldown
		CNPUBCLR=1<<pos;
		// Enable input
		TRISBSET=1<<pos;
		// Enable analog
		ANSELBSET=1<<pos;
		// Select input pin
		AD1CHS=pos;
	} else if (16<=pos && pos<=18) {
		// RE5,6,7:AN22,23,27
		pos-pos-16+5;
		// Disable pulldown
		CNPUECLR=1<<pos;
		// Enable input
		TRISESET=1<<pos;
		// Enable analog
		ANSELESET=1<<pos;
		// Select input pin
		if (pos<=6) {
			pos=pos-5+22;
		} else {
			pos=pos-7+27;
		}
		AD1CHS=pos;
	} else {
		return 0;
	}
	// Enable ADC
	AD1CON1bits.ON=1;
	// Start 
	AD1CON1bits.SAMP=1;
	// Wait until done
	while(!AD1CON1bits.DONE){
		asm("nop");
	}
	// Disable ADC
	AD1CON1bits.ON=0;
	// Return value
	return ADC1BUF0;
}

/*
	Statements and functions implementations follow
*/

// Local prototyping
char* param2_statement(enum libs lib);

char* out_statement(){
	return param2_statement(LIB_SYSTEM | EXTRA_OUT);
}
char* out8h_statement(){
	char* err;
	err=get_value();
	if (err) return err;
	call_lib_code(LIB_SYSTEM | EXTRA_OUT8H);
	return 0;
}
char* out8l_statement(){
	char* err;
	err=get_value();
	if (err) return err;
	call_lib_code(LIB_SYSTEM | EXTRA_OUT8L);
	return 0;
}
char* out16_statement(){
	char* err;
	err=get_value();
	if (err) return err;
	call_lib_code(LIB_SYSTEM | EXTRA_OUT16);
	return 0;
}
char* in_function(){
	char* err;
	err=get_value();
	if (err) return err;
	call_lib_code(LIB_SYSTEM | EXTRA_IN);
	return 0;
}
char* in8h_function(){
	call_lib_code(LIB_SYSTEM | EXTRA_IN8H);
	return 0;
}
char* in8l_function(){
	call_lib_code(LIB_SYSTEM | EXTRA_IN8L);
	return 0;
}
char* in16_function(){
	call_lib_code(LIB_SYSTEM | EXTRA_IN16);
	return 0;
}
char* analog_function(){
	char* err;
	err=get_value();
	if (err) return err;
	call_lib_code(LIB_SYSTEM | EXTRA_ANALOG);
	return 0;
}
