/* $Id: game3.c,v 1.16 2007-04-29 18:14:12 localhost Exp $ */
/*
Copyright (c) 2016, k_mia
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice,
this list of conditions and the following disclaimer in the documentation and/or
other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
/*
 *
 * GAME III interpreter
 *
 * syntax/fuction expanded
 *  0. internal calculation by long
 *  1. long variable : a-z
 *  2. long indirect : A[1),a[1)
 *  3. ram expanded over 64K
 *  4. operator !=, ==, &, |, ^, >>, <<, %, ||, &&
 *  5. unary operator !, ~
 *  6. command ?"fmt" (is printf)
 *  7. debug - 10debug command, 10debug>3 command
 *  8. on goto, on gosub #=line1,line2,... !=...
 *  9. >=$FFE8 returns lap-time val (0.1s)
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#ifdef BIN_LINE_SEARCH
#include <stdlib.h>
#endif //BIN_LINE_SEARCH
#include <string.h>
#include <sys/time.h>
#include <time.h>


//#define NDEBUG
#include <assert.h>

#ifdef NDEBUG
  // no debugging mode
#  define INLINE static inline
#else //-NDEBUG
  // debugging mode
#  define PROTECTION_BREAK 1
#  define INLINE
#endif//-NDEBUG

// for faster line search (goto/gosub)
#define BIN_LINE_SEARCH 1

#define RAM_SIZE (256*1024)


static int verbose=0;
static int expand_mode=1;
static int binary=0;
static int debug=0;

#ifdef BIN_LINE_SEARCH
  static int* lineTop=NULL;
  static int lineTopMaxN=0;
  static int lineTopN=0;
#endif //BIN_LINE_SEARCH

/* A-Z */
short vars[26];
int program_start=0x100;
int program_end=0x100;
int ram_end=RAM_SIZE-1; //0xFFFF;
int lineNo=0;
int pc=0;
int program_exit=0;
long call_status=0;
/* memory */
unsigned char ram[256*1024];

/* a-z */
long varl[26];

//
int getch(int fd);
const char* getaline(int fd);

void usage(char* s) {
	int i;

	char* msgs[]={
		"usage: game3 [-v][-3][-s cmd][-g][-b][-d n] [file.gt]",
		" -v    : increase verbose",
		" -3    : not expand mode",
		" -s cmd: start command for run",
		" -g|-b : binary program",
		" -d n  : level n debug enable",
		"   file: GAME III ascii program file",
		"",
		" * GAME III interpreter  $Revision: 1.16 $",
		" *",
		" * syntax/fuction expanded",
		" *  0. internal calculation by long",
		" *  1. long variable : a-z",
		" *  2. long indirect : A[1),a[1)",
		" *  3. ram expanded over 64K",
		" *  4. operator !=, ==, &, |, ^, >>, <<, %, ||, &&",
		" *  5. unary operator !, ~",
		" *  6. command ?\"fmt\" (is printf)",
		" *  7. debug - 10debug command, 10debug>3 command",
		" *  8. on goto, on gosub #=line1,line2,... !=...",
		" *  9. >=$FFE8 returns lap-time val (0.1s)",
		NULL
	};

	if (s) { fprintf(stderr,"%s\n",s); }

	for (i=0; i<sizeof(msgs)/sizeof(msgs[0]) && msgs[i]; i++) {
		fprintf(stderr,"%s\n",msgs[i]);
	}
	exit(99);
}


//===============================================================================
// execution helpers
//===============================================================================
//-------------------------------------------------------------------------------
// stack system
#define STACK_SIZE 1024
struct stack {
	long val[STACK_SIZE];
	int pos;
};
typedef struct stack stack;
INLINE void clear_stack(stack* st) {
	st->pos=0;
}
INLINE void push_stack(stack* st, long v) {
	if (st->pos<=STACK_SIZE) {
		st->val[st->pos++]=v;
		return;
	}
	fprintf(stderr,"ERROR: stack overflow in %d\n",lineNo);
	exit(54);
}
INLINE long pop_stack(stack* st) {
	if (st->pos>=1) {
		return st->val[--st->pos];
	}
	fprintf(stderr,"ERROR: stack underflow in %d\n",lineNo);
	exit(54);
}
INLINE int get_stack_len(stack* st) { return st->pos; }

//-------------------------------------------------------------------------------
// subroutine stack
static stack sub_st;
static stack sub_lino_st;
void clear_sub_st()
{
	clear_stack(&sub_st);
	clear_stack(&sub_lino_st);
}
void push_sub_st(const char* p)
{
#if (DEBUG>9)
	printf("call from %d, stacksize=%d\n",lineNo,get_stack_len(&sub_st)+1);
#endif //DEBUG>9
	push_stack(&sub_st,(long)p);
	push_stack(&sub_lino_st,lineNo);
}
const char* pop_sub_st()
{
	lineNo=pop_stack(&sub_lino_st);
#if (DEBUG>9)
	printf("return to %d, stacksize=%d\n",lineNo,get_stack_len(&sub_st)-1);
#endif //DEBUG>9
	return (const char*)pop_stack(&sub_st);
}
//-------------------------------------------------------------------------------
// loop stack
static stack loop_st;
static stack loop_lino_st;
void clear_loop_st()
{
	clear_stack(&loop_st);
	clear_stack(&loop_lino_st);
}
void push_loop_st(const char* p)
{
	push_stack(&loop_st,(long)p);
	push_stack(&loop_lino_st,lineNo);
}
const char* pop_loop_st(const int forDrop)
{
	if (forDrop) {
		pop_stack(&loop_lino_st);
	}
	else {
		lineNo=pop_stack(&loop_lino_st);
	}
	return (const char*)pop_stack(&loop_st);
}
//-------------------------------------------------------------------------------
// for-next stack
static stack for_val_st;
static stack for_p_st;
static stack for_lino_st;
void clear_for_st()
{
	clear_stack(&for_val_st);
	clear_stack(&for_p_st);
	clear_stack(&for_lino_st);
}
void push_for_st(const char* p, const long val)
{
	push_stack(&for_val_st,val);
	push_stack(&for_p_st,(long)p);
	push_stack(&for_lino_st,lineNo);
}
long pop_for_st(const char* (*p))
{
	lineNo=pop_stack(&for_lino_st);
	*p=(const char*)pop_stack(&for_p_st);
	return pop_stack(&for_val_st);
}
//-------------------------------------------------------------------------------
int get_time(void) {
	static struct timeval last_tv;
	struct timeval tv;
	//struct timezone tz;

	if (gettimeofday(&tv, NULL)) {
		return -1;
	}
	if (timerisset(&last_tv)) {
		// 2nd or later
		unsigned long ss, us;
		if (last_tv.tv_usec>tv.tv_usec) {
			us=1000000+tv.tv_usec-last_tv.tv_usec;
			ss=tv.tv_sec-last_tv.tv_sec-1;
		}
		else {
			us=tv.tv_usec-last_tv.tv_usec;
			ss=tv.tv_sec-last_tv.tv_sec;
		}
		if (verbose>=1) {
			fprintf(stderr,"[%lus%lum%luu]",ss,us/1000,us%1000);
		}
		//return (ss*100+us/10000);
		return (ss*10+us/100000);
	}
	else {
		// 1st time
		last_tv=tv;
		return 0;
	}
}
//-------------------------------------------------------------------------------
// division
static long _division(const long mod, long v, const long d)
{
	// game3's modulo is INCORRECT value, but...
#ifdef CORRECT_MODULO
	static long last_v;
	static long last_d;

	if (mod) {
		return (last_v%last_d);
	}

	last_v=v;
	last_d=d;
	if (d==0) {
		// math error
		fprintf(stderr,"RUNTIME ERROR: divisor is zero in line %d\n",lineNo);
		exit(52);
	}
	return (v/d);
#else //-CORRECT_MODULO
	static long last_v;
	static long last_d;

	if (mod) {
		return (abs(last_v)%abs(last_d));
	}

	int quot_sign=0;
	if (v<0) { quot_sign++; }
	if (d<0) { quot_sign--; }
	last_v=v;
	last_d=d;
	if (d==0) {
		// math error
		fprintf(stderr,"RUNTIME ERROR: divisor is zero in line %d\n",lineNo);
		exit(52);
	}
	int quot=abs(v)/abs(d);
	if (quot_sign) { return -quot; }
	else { return quot; }
#endif //-CORRECT_MODULO
}
INLINE long division(const long v, const long d)
{
	return _division(0,v,d);
}
INLINE long modulo()
{
	return _division(1,0,0);
}
//-------------------------------------------------------------------------------
// random
//#define USE_LONG_LONG 1
#ifdef USE_LONG_LONG
#else //-USE_LONG_LONG
  INLINE int mod_64_32(unsigned long nnh, unsigned long nnl, unsigned long n)
  {
  	// ((nnh<<32)+nnl)%n
	unsigned long hh=nnh%n;
	unsigned long ll=nnl;
	unsigned long c=0;	// c|nnh|nnl
	int i;

	for (i=0; i<32; i++) {
		// (c|hh|ll)<<=1
		c=hh&0x80000000;
		hh<<=1;
		if (ll&0x80000000) {
			hh|=1;
		}
		ll<<=1;
		// (c|hh|ll) -= (n<<32) if can
		if (c) {
			//  (c|hh|ll) > (n<<32)
			hh-=n;
			//if (hh>n) { hh%=n; printf("xx\n");}
		}
		else if (hh>=n) {
			hh-=n;
			//if (hh>n) { hh%=n; printf("xx\n");}
		}
	}
	return hh;
  }
#endif //-USE_LONG_LONG
#if 0
static unsigned long r_next=1;
// GAMEIII original
INLINE long _myrand()
{
	r_next=r_next * 0x3D09 + 1;
	return (r_next&0x7FFFFFFF);
}
INLINE void mysrand(unsigned long n)
{
	r_next=n;
}
#endif //0
INLINE long _myrand() { return rand(); }
INLINE void mysrand(unsigned long n) { srand(n); }
INLINE long myrand(const long n)
{
	unsigned long h=n&0xFFFF0000;

	if (n==0) { return 0; }

	if (h==0 || h==0xFFFF0000) {
		// 16bit val
		return (_myrand()%n);
	}
    {
	// 32bit val
#ifdef USE_LONG_LONG
	unsigned long long nn=_myrand()&0xFFFF;
	nn=(nn<<16) + (_myrand()&0xFFFF);
	nn=(nn<<16) + (_myrand()&0xFFFF);
	return (unsigned long)(nn%n);
#else //-USE_LONG_LONG
	unsigned long nnl;
	unsigned long nnh=_myrand()&0xFFFF;
	nnl=_myrand()&0xFFFF;
	nnl=(nnl<<16)+(_myrand()&0xFFFF);
	// calc mod
	return mod_64_32(nnh,nnl,n);
#endif //-USE_LONG_LONG
    }
}
//-------------------------------------------------------------------------------
// variable access
INLINE void setVarVal(const int c, const long val)
{
	if ('A'<=c && c<='Z') {
		vars[c-'A']=val;
		return;
	}
	if (expand_mode) {
		if ('a'<=c && c<='z') {
			varl[c-'a']=val;
			return;
		}
	}
	if (c=='=') {
		program_start=val;
		return;
	}
	if (c=='&') {
		program_end=val;
		return;
	}
	if (c=='*') {
		ram_end=val;
		return;
	}
	fprintf(stderr,"ERROR: varnam %c = %ld\n",c,val);
}
INLINE long getVarVal(const int c)
{
	if ('A'<=c && c<='Z') {
		return vars[c-'A'];
	}
	if (expand_mode) {
		if ('a'<=c && c<='z') {
			return varl[c-'a'];
		}
	}
	if (c=='=') {
		return program_start;
	}
	if (c=='&') {
		return program_end;
	}
	if (c=='*') {
		return ram_end;
	}
	fprintf(stderr,"ERROR: varnam %c\n",c);
	return 0;
}
//-------------------------------------------------------------------------------
// indirect access
INLINE int inRam(int ofs)
{
	if (0<=ofs && ofs<RAM_SIZE) {
		return 1;
	}
	// out of ram
	return 0;
}
INLINE void setRam32Val(int c, long idx, long val)
{
	if ('A'<=c && c<='Z') {
		unsigned short v=vars[c-'A']+idx*4;
		ram[v]=val>>24;
		ram[(v+1)&0xFFFF]=val>>16;
		ram[(v+2)&0xFFFF]=val>>8;
		ram[(v+3)&0xFFFF]=val;
		return;
	}
	if (expand_mode) {
		if ('a'<=c && c<='z') {
			long v=vars[c-'a']+idx*4;
			if (inRam(v) && inRam(v+3)) {
				ram[v]=val>>24;
				ram[v+1]=val>>16;
				ram[v+2]=val>>8;
				ram[v+3]=val;
			}
			return;
		}
	}
	if (c=='=') {
		unsigned short v=program_start+idx*4;
		ram[v]=val>>24;
		ram[(v+1)&0xFFFF]=val>>16;
		ram[(v+2)&0xFFFF]=val>>8;
		ram[(v+3)&0xFFFF]=val;
		return;
	}
	if (c=='&') {
		unsigned short v=program_end+idx*4;
		ram[v]=val>>24;
		ram[(v+1)&0xFFFF]=val>>16;
		ram[(v+2)&0xFFFF]=val>>8;
		ram[(v+3)&0xFFFF]=val;
		return;
	}
	if (c=='*') {
		unsigned short v=ram_end+idx*4;
		ram[v]=val>>24;
		ram[(v+1)&0xFFFF]=val>>16;
		ram[(v+2)&0xFFFF]=val>>8;
		ram[(v+3)&0xFFFF]=val;
		return;
	}
	fprintf(stderr,"ERROR: varnam %c(%ld)=%ld\n",c,idx,val);
	return;
}
INLINE void setRam16Val(int c, long idx, long val)
{
	if ('A'<=c && c<='Z') {
		unsigned short v=vars[c-'A']+idx*2;
		ram[v]=(val&0xFFFF)>>8;
		ram[(v+1)&0xFFFF]=val;
		return;
	}
	if (expand_mode) {
		if ('a'<=c && c<='z') {
			long v=varl[c-'a']+idx*2;
			if (inRam(v) && inRam(v+1)) {
				ram[v]=(val&0xFFFF)>>8;
				ram[(v+1)&0xFFFF]=val;
			}
			return;
		}
	}
	if (c=='=') {
		unsigned short v=program_start+idx*2;
		ram[v]=(val&0xFFFF)>>8;
		ram[(v+1)&0xFFFF]=val;
		return;
	}
	if (c=='&') {
		unsigned short v=program_end+idx*2;
		ram[v]=(val&0xFFFF)>>8;
		ram[(v+1)&0xFFFF]=val;
		return;
	}
	if (c=='*') {
		unsigned short v=ram_end+idx*2;
		ram[v]=(val&0xFFFF)>>8;
		ram[(v+1)&0xFFFF]=val;
		return;
	}
	fprintf(stderr,"ERROR: varnam %c(%ld)=%ld\n",c,idx,val);
	return;
}
INLINE void setRam8Val(int c, long idx, long val)
{
	if ('A'<=c && c<='Z') {
		unsigned short v=vars[c-'A']+idx;
		ram[v]=val;
		return;
	}
	if (expand_mode) {
		if ('a'<=c && c<='z') {
			long v=varl[c-'a']+idx;
			if (inRam(v)) {
				ram[v]=val;
			}
			return;
		}
	}
	if (c=='=') {
		unsigned short v=program_start+idx*2;
		ram[v]=val;
		return;
	}
	if (c=='&') {
		unsigned short v=program_end+idx*2;
		ram[v]=val;
		return;
	}
	if (c=='*') {
		unsigned short v=ram_end+idx*2;
		ram[v]=val;
		return;
	}
	fprintf(stderr,"ERROR: varnam %c:%ld)=%ld\n",c,idx,val);
	return;
}
INLINE long getRam32Val(int c, long idx)
{
	if ('A'<=c && c<='Z') {
		unsigned short v=vars[c-'A']+idx*4;
		return ((ram[v]<<24)|(ram[(v+1)&0xFFFF]<<16)|(ram[(v+2)&0xFFFF]<<8)|ram[(v+3)&0xFFFF]);
	}
	if (expand_mode) {
		if ('a'<=c && c<='z') {
			long v=vars[c-'a']+idx*4;
			if (inRam(v) && inRam(v+3)) {
				return ((ram[v]<<24)|(ram[v+1]<<16)|(ram[v+2]<<8)|ram[v+3]);
			}
			return 0;
		}
	}
	if (c=='=') {
		unsigned short v=program_start+idx*4;
		return ((ram[v]<<24)|(ram[(v+1)&0xFFFF]<<16)|(ram[(v+2)&0xFFFF]<<8)|ram[(v+3)&0xFFFF]);
	}
	if (c=='&') {
		unsigned short v=program_end+idx*4;
		return ((ram[v]<<24)|(ram[(v+1)&0xFFFF]<<16)|(ram[(v+2)&0xFFFF]<<8)|ram[(v+3)&0xFFFF]);
	}
	if (c=='*') {
		unsigned short v=ram_end+idx*4;
		return ((ram[v]<<24)|(ram[(v+1)&0xFFFF]<<16)|(ram[(v+2)&0xFFFF]<<8)|ram[(v+3)&0xFFFF]);
	}
	fprintf(stderr,"ERROR: varnam %c(%ld)\n",c,idx);
	return 0;
}
INLINE long getRam16Val(int c, long idx)
{
	if ('A'<=c && c<='Z') {
		unsigned short v=vars[c-'A']+idx*2;
		return ((ram[v]<<8)|ram[(v+1)&0xFFFF]);
	}
	if (expand_mode) {
		if ('a'<=c && c<='z') {
			long v=vars[c-'a']+idx*2;
			if (inRam(v) && inRam(v+1)) {
				return ((ram[v]<<8)|ram[(v+1)&0xFFFF]);
			}
			return 0;
		}
	}
	if (c=='=') {
		unsigned short v=program_start+idx*2;
		return ((ram[v]<<8)|ram[(v+1)&0xFFFF]);
	}
	if (c=='&') {
		unsigned short v=program_end+idx*2;
		return ((ram[v]<<8)|ram[(v+1)&0xFFFF]);
	}
	if (c=='*') {
		unsigned short v=ram_end+idx*2;
		return ((ram[v]<<8)|ram[(v+1)&0xFFFF]);
	}
	fprintf(stderr,"ERROR: varnam %c(%ld)\n",c,idx);
	return 0;
}
INLINE long getRam8Val(int c, long idx)
{
	if ('A'<=c && c<='Z') {
		unsigned short v=vars[c-'A']+idx;
		return ram[v];
	}
	if (expand_mode) {
		if ('a'<=c && c<='z') {
			long v=vars[c-'a']+idx;
			if (inRam(v)) {
				return ram[v];
			}
			return 0;
		}
	}
	if (c=='=') {
		unsigned short v=program_start+idx;
		return ram[v];
	}
	if (c=='&') {
		unsigned short v=program_end+idx;
		return ram[v];
	}
	if (c=='*') {
		unsigned short v=ram_end+idx;
		return ram[v];
	}
	fprintf(stderr,"ERROR: varnam %c:%ld)\n",c,idx);
	return 0;
} 
//-------------------------------------------------------------------------------
INLINE unsigned char* pc_addr()
{
#ifdef PROTECTION_BREAK
	if (pc<program_start||program_end<pc) {
		fprintf(stderr,"ERROR: Illegal pc=$%x\n",pc);
		exit(10);
	}
#endif //PROTECTION_BREAK
	return &ram[pc];
}
INLINE int set_pc_addr(const char* pp)
{
	unsigned char* p=(unsigned char*)pp;
	if (ram<=p && p<ram+sizeof(ram)) {
		// ok
		pc=p-ram;
		return 0;
	}
	// not in ram
	return -1;
}
INLINE int pc_peek()
{
#ifdef PROTECTION_BREAK
	if (pc<program_start||program_end<pc) {
		fprintf(stderr,"ERROR: Illegal pc=$%x\n",pc);
		exit(10);
	}
#endif //PROTECTION_BREAK
	return ram[pc];
}
INLINE int pc_fetch()
{
#ifdef PROTECTION_BREAK
	if (pc<program_start||program_end<pc) {
		fprintf(stderr,"ERROR: Illegal pc=$%x\n",pc);
		exit(10);
	}
#endif //PROTECTION_BREAK
	return ram[pc++];
}
INLINE void pc_fetch_lineNo()
{
	lineNo=pc_fetch()<<8;
	lineNo+=pc_fetch();
#if (DEBUG>9)
	printf("lineNo=%d\n",lineNo);
#endif //DEBUG>9
}
//-------------------------------------------------------------------------------
INLINE int peek(const char* p)
{
	return *p;
}
INLINE int fetch(const char* p)
{
	int c=*p;
	if (verbose>=2) {
		if (isprint(c)) {
			fprintf(stderr,"%c",c);
		}
		else if (c==0) {
			fprintf(stderr,"\\0\n");
		}
		else {
			fprintf(stderr,"\\x%02x",c);
		}
	}
	set_pc_addr(p+1);
	return c;
}
//-------------------------------------------------------------------------------
long getNumber(const char* (*pp))
{
	long n=0;
	const char* p;

	assert(pp && *pp);

	p=*pp;
	while (isdigit(peek(p))) {
		int c=fetch(p++);
		n=n*10+c-'0';
	}
	*pp=p;
	return n;
}
//-------------------------------------------------------------------------------
long getHex(const char* (*pp))
{
	long n=0;
	const char* p;

	assert(pp && *pp);

	p=*pp;
	while (isxdigit(peek(p))) {
		int c=fetch(p++);
		if (isdigit(c)) {
			n=n*16+c-'0';
		}
		else if ('A'<=c && c<='F') {
			n=n*16+c-'A'+10;
		}
		else if ('a'<=c && c<='f') {
			n=n*16+c-'a'+10;
		}
	}
	*pp=p;
	return n;
}
//-------------------------------------------------------------------------------
// skip to next line - comment or if (false case)
void pc_skip_to_next_line(void)
{
	int c;
	do {
		// get from pc
		if (pc<program_start||program_end<pc) {
			fprintf(stderr,"ERROR: Illegal pc=$%x\n",pc);
			exit(10);
		}
		c=pc_fetch();
		//fprintf(stderr,"SKIP:%c\n",c);
	}while(c!=EOF && c!=0 && c!='\n');
}
//-------------------------------------------------------------------------------
// goto
// return: -1: end
//         >0: found no
int goto_line(int no)
{
	if (no<0 || no&0x8000) {
		// exit program
		program_exit=1;
		return -1;
	}
#if (DEBUG>9)
	printf("lineNo=%d,goto=%d\n",lineNo,no);
#endif //DEBUG>9
#ifdef BIN_LINE_SEARCH
	/* linear search
	int i;
	for (i=0; i<lineTopN; i++) {
		pos=lineTop[i];
		if (pos>=program_end) { break; }
		// get line number
		int n=(ram[pos]<<8)|ram[pos+1];
		if (n>=0x8000) {
			// exit program
			program_exit=1;
			pc=program_end;
			return -1;
		}
		if (n>=no) {
			// found
			pc=pos;
			return n;
		}
		// skip this line
	}
	*/
	// binary search
	int low=0;
	int high=lineTopN-1;
	int mid;
	if (((ram[lineTop[high]]<<8)|ram[lineTop[high]+1])<no) {
		// exit program
		program_exit=1;
		pc=program_end;
		return -1;
	}
	for (;;) {
		mid=(low+high)/2;
		// get [mid] line number
		int pos=lineTop[mid];
		int n=(ram[pos]<<8)|ram[pos+1];
#if (DEBUG>9)
		printf("%d-%d:%d-%d\n",low,mid,n,high);
#endif //DEBUG>9
		if (n<no) {
			low=mid+1;
			if (low>high) {
				pc=lineTop[low];
#ifndef NDEBUG //-NDEBUG
				if (low>0) {
				    int xx=(ram[pc]<<8)|ram[pc+1];
				    int yy=(ram[lineTop[low-1]]<<8)|ram[lineTop[low-1]+1];
				    assert(yy<no && no<=xx);
				}
#endif //-NDEBUG
				return (ram[pc]<<8)|ram[pc+1];
			}
			continue;
		}
		if (n>no) {
			high=mid-1;
			if (low>high) {
				pc=lineTop[low];
#ifndef NDEBUG //-NDEBUG
				if (low>0) {
				    int xx=(ram[pc]<<8)|ram[pc+1];
				    int yy=(ram[lineTop[low-1]]<<8)|ram[lineTop[low-1]+1];
				    assert(yy<no && no<=xx);
				}
#endif //-NDEBUG
				return (ram[pc]<<8)|ram[pc+1];
			}
			continue;
		}
		//n==no
		// found (match)
		pc=pos;
		return n;
	}
#else //-BIN_LINE_SEARCH
	unsigned short pos=program_start;

	if (lineNo>0 && lineNo<no) {
		pc_skip_to_next_line();
		pos=pc;
	}
	if (pos==program_end) {
		// end program
		program_exit=1;
		return -1;
	}
	for (; pos<program_end; ) {
		// get line number
		int n=(ram[pos]<<8)|ram[pos+1];

		if (n>=0x8000) {
			// exit program
			program_exit=1;
			pc=program_end;
			return -1;
		}
		if (n>=no) {
			// found
			pc=pos;
			return n;
		}
		// skip this line
		pos+=2;
		while (ram[pos++]!=0) {
			if (pos>=program_end) {
				break;
			}
		}
	}
#endif //-BIN_LINE_SEARCH
	// over???
	fprintf(stderr,"broken program while searching line %d\n",no);
	exit(51);
}

//===============================================================================
// arithmetic expression
//===============================================================================
INLINE int isVariable(const int c)
{
	if ('A'<=c && c<='Z') { return c; }
	if (c=='='||c=='&'||c=='*') { return c; }
	if (expand_mode) {
		if ('a'<=c && c<='z') { return c; }
	}
	return 0;
}
INLINE const char* eat_varnam(const char* (*pp))
{
	assert(pp && *pp);
	const char* p=*pp;
	for (;; fetch(p++)) {
		int c=peek(p);
		if (isalpha(c)) { continue; }
		if (c=='_') { continue; }
		// not var : *p
		*pp=p;
		return p;
	}
}
static long _artexp(const char* (*pp));
INLINE long artexp(const char* (*pp))
{
	long val=_artexp(pp);
	if (expand_mode) {
		return val;
	}
	return (val&0xFFFF);
}
static long _term(const char* (*pp));
INLINE long term(const char* (*pp))
{
	long val=_term(pp);
	if (expand_mode) {
		return val;
	}
	return (val&0xFFFF);
}
static long _term(const char* (*pp))
{
	assert(pp && *pp);
	long var;
	long val;
	long idx;
	int c=peek(*pp);

	if ((var=isVariable(c))!=0) {
		// VAR
		fetch((*pp)++);
		eat_varnam(pp);
		switch (peek(*pp)) {
		case '[': // A[1)=2
			if (expand_mode) {
				fetch((*pp)++);
				idx=artexp(pp);
				if (peek(*pp)!=')') {
					// syntax error
					fprintf(stderr,"ERROR: syntax error - %s in line %d\n",*pp,lineNo);
					exit(53);
				}
				fetch((*pp)++);
				return getRam32Val(var,idx);
			}
			// THRU
		default:  // A
			return getVarVal(var);
		case '(': // A(1)
			fetch((*pp)++);
			idx=artexp(pp);
			if (peek(*pp)!=')') {
				// syntax error
				fprintf(stderr,"ERROR: syntax error - %s in line %d\n",*pp,lineNo);
				exit(53);
			}
			fetch((*pp)++);
			return getRam16Val(var,idx);
		case ':': // A:1)=2
			fetch((*pp)++);
			idx=artexp(pp);
			if (peek(*pp)!=')') {
				// syntax error
				fprintf(stderr,"ERROR: syntax error - %s in line %d\n",*pp,lineNo);
				exit(53);
			}
			fetch((*pp)++);
			return getRam8Val(var,idx);
		}
	}
	if (isdigit(c)) {
	    return getNumber(pp);
	}
	switch (c) {
	case '"': // char constant
		fetch((*pp)++);
		val=fetch((*pp)++);
		if (val==0) {
			fprintf(stderr,"ERROR: syntax error - %s in line %d\n",*pp,lineNo);
			exit(53);
		}
		if (peek(*pp)!='"') {
			fprintf(stderr,"ERROR: syntax error - %s in line %d\n",*pp,lineNo);
			exit(53);
		}
		fetch((*pp)++);
		return val;
	case '\'': // random
		fetch((*pp)++);
		val=term(pp);
		if (expand_mode) {
		}
		else {
			// rand must 2..255
			val&=0xFF;
			if (val<=1) {
				// runtime error?
				return val;
			}
		}
		return (myrand(val)+1);
	case '$': // keyin, $hex
		fetch((*pp)++);
		c=peek(*pp);
		if (isxdigit(c)) {
			// $ABCD
			return getHex(pp);
		}
		return getch(fileno(stdin));
	case '(': // artexp
		fetch((*pp)++);
		val=artexp(pp);
		c=peek(*pp);
		if (c!=')') {
			// syntax error
			fprintf(stderr,"ERROR: syntax error - %s in line %d\n",*pp,lineNo);
			exit(53);
		}
		fetch((*pp)++);
		return val;
	case '#': // not
		fetch((*pp)++);
		val=term(pp);
		return (val?0:1);
	case '+': // abs
		fetch((*pp)++);
		val=term(pp);
		if (expand_mode) { }
		else { const short vs=val; return (vs<0?-vs:vs); }
		return (val<0?-val:val);
	case '-': // neg
		fetch((*pp)++);
		val=term(pp);
		if (expand_mode) { }
		else { const short vs=val; return (-vs); }
		return (-val);
	case '%': // mod
		fetch((*pp)++);
		val=term(pp);
		return (modulo());
	case '?': // input
		fetch((*pp)++);
		putc(':',stdout); fflush(stdout);
		const char* p=getaline(fileno(stdin));
		if (p==NULL) {
			// error
			return 0;
		}
		if (*p=='-' && p[1]=='$') {
			val= -(strtol(p+2,NULL,16));
		}
		else if (*p=='$') {
			val=strtol(p+1,NULL,16);
		}
		else {
			val=strtol(p,NULL,10);
		}
		if (expand_mode) { }
		else { const short vs=val; return (vs); }
		return val;
	case '>': // call status
		fetch((*pp)++);
		return call_status;
	case '~': // complement
		if (expand_mode) {
			fetch((*pp)++);
			val=term(pp);
			return (~val);
		}
		// THRU
	case '!': // not
		if (expand_mode) {
			fetch((*pp)++);
			val=term(pp);
			return (val?0:1);
		}
		// THRU
	default:
		// syntax error
		fprintf(stderr,"ERROR: syntax error - %s in line %d\n",*pp,lineNo);
		exit(53);
	}
}
long _artexp(const char* (*pp))
{
	assert(pp && *pp);
	long val;
	long val2;
	int c;

	val=term(pp);
	for (;;) {
		c=peek(*pp);
		if (c!=0 && c!='\n' && c!=' ' && c!='\t') {
			// binop
		}
		else {
			break;
		}
		//c=peek(*pp);
		switch (c) {
		case '+':
			fetch((*pp)++);
			val2=term(pp);
			val+=val2;
			continue;
		case '-':
			fetch((*pp)++);
			val2=term(pp);
			val-=val2;
			continue;
		case '*':
			fetch((*pp)++);
			val2=term(pp);
			val*=val2;
			continue;
		case '/':
			fetch((*pp)++);
			val2=term(pp);
			if (expand_mode) {
				val=division(val,val2);
			}
			else {
				// original div,mod (+/-,-/+,-/- => change +/+ and fix sign)
				short vs=val;
				short vs2=val2;
				int negflg=0;
				if (vs<0) { vs=-vs; negflg++; }
				if (vs2<0) { vs2=-vs2; negflg++; }
				val=division(vs,vs2);
				if (negflg&1) { val=-val; }
			}
			continue;
		case '<':
			fetch((*pp)++);
			if (peek(*pp)=='=') {
				// <=
				fetch((*pp)++);
				val2=term(pp);
				if (expand_mode) {
					val=val<=val2 ? 1:0;
				}
				else {
					const short vs=val;
					const short vs2=val2;
					val=vs<=vs2 ? 1:0;
				}
				continue;
			}
			else if (peek(*pp)=='>') {
				// !=
				fetch((*pp)++);
				val2=term(pp);
				val=val!=val2 ? 1:0;
				continue;
			}
			else {
				// <
				if (expand_mode) {
					if (peek(*pp)=='<') {
						// >>
						fetch((*pp)++);
						val2=term(pp);
						val<<=val2;
						continue;
					}
				}
				val2=term(pp);
				if (expand_mode) {
					val=val<val2 ? 1:0;
				}
				else {
					const short vs=val;
					const short vs2=val2;
					val=vs<vs2 ? 1:0;
				}
				continue;
			}
			continue;
		case '>':
			fetch((*pp)++);
			if (peek(*pp)=='=') {
				// >=
				fetch((*pp)++);
				val2=term(pp);
				if (expand_mode) {
					val=val>=val2 ? 1:0;
				}
				else {
					const short vs=val;
					const short vs2=val2;
					val=vs>=vs2 ? 1:0;
				}
				continue;
			}
			else {
				// >
				if (expand_mode) {
					if (peek(*pp)=='>') {
						// >>
						fetch((*pp)++);
						val2=term(pp);
						val>>=val2;
						continue;
					}
				}
				val2=term(pp);
				if (expand_mode) {
					val=val>val2 ? 1:0;
				}
				else {
					const short vs=val;
					const short vs2=val2;
					val=vs>vs2 ? 1:0;
				}
				continue;
			}
			continue;
		case '=':
			fetch((*pp)++);
			// =
			if (expand_mode) {
				if (peek(*pp)=='=') {
					// ==
					fetch((*pp)++);
				}
			}
			val2=term(pp);
			val=val==val2 ? 1:0;
			continue;
		case '%':
			if (expand_mode) {
				fetch((*pp)++);
				val2=term(pp);
				val%=val2;
				continue;
			}
			// THRU
		case '&':
			if (expand_mode) {
				fetch((*pp)++);
				if (peek(*pp)=='&') {
					fetch((*pp)++);
					val2=term(pp);
					val=(val&&val2)?1:0;
					continue;
				}
				val2=term(pp);
				val&=val2;
				continue;
			}
			// THRU
		case '|':
			if (expand_mode) {
				fetch((*pp)++);
				if (peek(*pp)=='|') {
					fetch((*pp)++);
					val2=term(pp);
					val=(val||val2)?1:0;
					continue;
				}
				val2=term(pp);
				val|=val2;
				continue;
			}
			// THRU
		case '^':
			if (expand_mode) {
				fetch((*pp)++);
				val2=term(pp);
				val^=val2;
				continue;
			}
			// THRU
		case '!':
			if (expand_mode) {
				fetch((*pp)++);
				if (*(*pp)=='=') {
					// !=
					fetch((*pp)++);
					val2=term(pp);
					val=val!=val2 ? 1:0;
					continue;
				}
			}
			// THRU
		default:
			// maybe ',' or SPC or EOL
			return val;
		}
	}
	return val;
}
//===============================================================================
// execution
//===============================================================================
INLINE void print_using(int w,long val)
{
	char nb[32];
	int len;
	if (expand_mode) {
		sprintf(nb,"%ld",val);
	}
	else {
		const short vs=val;
		sprintf(nb,"%d",vs);
	}
	len=strlen(nb);
	for (; len<w; len++) {
		putc(' ',stdout);
	}
	fputs(nb,stdout);
}
INLINE int comment(const char* (*pp)) {
	if (debug>0 && strncmp(*pp,"debug",5)==0) {
		// debug mode
		fetch((*pp)++); fetch((*pp)++); fetch((*pp)++);
		fetch((*pp)++); fetch((*pp)++);
		if (peek(*pp)=='>') {
			// debug>TERM
			fetch((*pp)++);
			int val=term(pp);
			if (debug>val) {
				// debug
			}
			else {
				// not level
				pc_skip_to_next_line();
				//continue;
				return 1;
			}
		}
		// debug (always)
		//break;
		return 0;
	}
	else {
		// comment line
		pc_skip_to_next_line();
		//continue;
		return 1;
	}
}
long execute(const char* cmdline)
{
	int c;
	const char* p=cmdline;

	assert(p);

	for (;;) {
		long var;
		long val;
		long idx;

		c=fetch(p++);
		if (c==0 || c=='\n') {
			// EOL
			if (set_pc_addr(p)) {
				// not in ram
				return 0;
			}
			for(;;) {
				if (pc_peek()>=0x80) {
					// program end
					return 0;
				}
				// linenumber
				pc_fetch_lineNo();
				p=(const char*)pc_addr();
				// is comment line?
				if (*p!=' ') {
					if (comment(&p)) {
						// EOL
						continue;
					}
					else {
						// statement
					}
				}
				if (verbose>=2) { fprintf(stderr,"%5d",lineNo); }
				break;
			}
			continue;
		}
		if (c==' ' || c=='\t') {
			// SPACE
			continue;
		}

		if ((var=isVariable(c))!=0) {
			// VAR
			eat_varnam(&p);
			switch (peek(p)) {
			case '=': // A=1
				fetch(p++);
				val=artexp(&p);
				setVarVal(var,val);
				if (peek(p)==',') {
					// A=1,100
					fetch(p++);
					val=artexp(&p);
					if (expand_mode) {
						push_for_st(p,val);
					}
					else {
						const short vs=val;
						push_for_st(p,vs);
					}
				}
				break;
			case '(': // A(1)=2
				fetch(p++);
				idx=artexp(&p);
				if (*p!=')' || *(p+1)!='=') {
					// syntax error
					fprintf(stderr,"ERROR: syntax error - %s in line %d\n",p,lineNo);
					exit(53);
				}
				fetch(p++); fetch(p++);
				val=artexp(&p);
				setRam16Val(var,idx,val);
				break;
			case ':': // A:1)=2
				fetch(p++);
				idx=artexp(&p);
				if (*p!=')' || *(p+1)!='=') {
					// syntax error
					fprintf(stderr,"ERROR: syntax error - %s in line %d\n",p,lineNo);
					exit(53);
				}
				fetch(p++); fetch(p++);
				val=artexp(&p);
				setRam8Val(var,idx,val);
				break;
			case '[': // A[1)=2
				if (expand_mode) {
					fetch(p++);
					idx=artexp(&p);
					if (*p!=')' || *(p+1)!='=') {
						// syntax error
						fprintf(stderr,"ERROR: syntax error - %s in line %d\n",p,lineNo);
						exit(53);
					}
					fetch(p++); fetch(p++);
					val=artexp(&p);
					setRam32Val(var,idx,val);
					break;
				}
				// THRU
			default:
				// syntax error
				fprintf(stderr,"ERROR: syntax error - %s in line %d\n",p,lineNo);
				exit(53);
			}
			//
			continue;
		}
		switch (c) {
		case ';': // if
			if (*p!='=') {
				// syntax error
				fprintf(stderr,"ERROR: syntax error - %s in line %d\n",p,lineNo);
				exit(53);
			}
			fetch(p++);
			val=artexp(&p);
			if (val) {
				// true -> do then
			}
			else {
				// false -> to next line
				if (set_pc_addr(p)) {
					// not in ram
					return 0;
				}
				pc_skip_to_next_line();
				for(;;) {
					if (pc_peek()>=0x80) {
						// program end
						return 0;
					}
					// linenumber
					pc_fetch_lineNo();
					p=(const char*)pc_addr();
					// is comment line?
					if (*p!=' ') {
						if (comment(&p)) {
							// EOL
							continue;
						}
						else {
							// statement
						}
					}
					if (verbose>=2) { fprintf(stderr,"\n%5d",lineNo); }
					break;
				}
				continue;
			}
			break;
		case '#': // goto
			if (*p!='=') {
				if (expand_mode) {
					// on goto line1, line2  #A=100,110,...
					int on_val=term(&p);
					if (peek(p)!='=') {
						// syntax error
						fprintf(stderr,"ERROR: syntax error - %s in line %d\n",p,lineNo);
						exit(53);
					}
					int nth=1;
					for (;;) {
						fetch(p++);
						val=term(&p);	// must be number
						if (on_val==nth) {
							// match : val is lineNo
							break;
						}
						else {
							// not match
							if (peek(p)!=',') {
								// nomore lineNos -> runtime error
								fprintf(stderr,"ERROR: on goto %dth not match in line %d\n\n",on_val,lineNo);
								exit(54);
							}
							nth++;
						}
					}
				}
				else {
					// syntax error
					fprintf(stderr,"ERROR: syntax error - %s in line %d\n",p,lineNo);
					exit(53);
				}
			}
			else {
				fetch(p++);
				val=artexp(&p);
			}
			if (val==0) {
				// #=0 -> nop
				continue;
			}
			// goto val line
			if (goto_line(val)<0) {
				// program end
				return 0;
			}
			for(;;) {
				if (pc_peek()>=0x80) {
					// program end
					return 0;
				}
				// linenumber
				pc_fetch_lineNo();
				p=(const char*)pc_addr();
				// is comment line?
				if (*p!=' ') {
					if (comment(&p)) {
						// EOL
						continue;
					}
					else {
						// statement
					}
				}
				if (verbose>=2) { fprintf(stderr,"\n%5d",lineNo); }
				break;
			}
			continue;
		case '!': // gosub
			if (*p!='=') {
				if (expand_mode) {
					// on gosub line1, line2  !A=100,110,...
					int on_val=term(&p);
					if (peek(p)!='=') {
						// syntax error
						fprintf(stderr,"ERROR: syntax error - %s in line %d\n",p,lineNo);
						exit(53);
					}
					int nth=1;
					for (;;) {
						fetch(p++);
						val=term(&p);	// must be number
						if (on_val==nth) {
							// match : val is lineNo -> skip to next statement
							for(; peek(p)==',';) {
								fetch(p++);
								term(&p);
							}
							break;
						}
						else {
							// not match
							if (peek(p)!=',') {
								// nomore lineNos -> runtime error
								fprintf(stderr,"ERROR: on gosub %dth not match in line %d\n\n",on_val,lineNo);
								exit(54);
							}
							nth++;
						}
					}
				}
				else {
					// syntax error
					fprintf(stderr,"ERROR: syntax error - %s in line %d\n",p,lineNo);
					exit(53);
				}
			}
			else {
				fetch(p++);
				val=artexp(&p);
			}
			if (val==0) {
				// !=0 -> nop
				continue;
			}
			// gosub val line
			push_sub_st(p);
			if (goto_line(val)<0) {
				// program end
				return 0;
			}
			for(;;) {
				if (pc_peek()>=0x80) {
					// program end
					return 0;
				}
				// linenumber
				pc_fetch_lineNo();
				p=(const char*)pc_addr();
				// is comment line?
				if (*p!=' ') {
					if (comment(&p)) {
						// EOL
						continue;
					}
					else {
						// statement
					}
				}
				if (verbose>=2) { fprintf(stderr,"\n%5d",lineNo); }
				break;
			}
			continue;
		case ']': // return
			p=pop_sub_st();
			continue;
		case '@': // do,until,next
			c=peek(p);
			if (c==0 || c==' '|| c=='\t') {
				// @
				push_loop_st(p);
				continue;
			}
			if (c!='=') {
				// syntax error
				fprintf(stderr,"ERROR: syntax error - %s in line %d\n",p,lineNo);
				exit(53);
			}
			fetch(p++);
			c=peek(p);
			if ((var=isVariable(c))!=0) {
				// @=A+1
				const char* pp;
				val=artexp(&p);
				setVarVal(var,val);
				int curLino=lineNo;
				idx=pop_for_st(&pp);
				if (expand_mode) {
					if (val<=idx) {
						// next
						p=pp;
						push_for_st(p,idx);
						continue;
					}
					// end for-next
					lineNo=curLino;
				}
				else {
					const short vs=val;
					if (vs<=idx) {
						// next
						p=pp;
						push_for_st(p,idx);
						continue;
					}
					// end for-next
					lineNo=curLino;
				}
			}
			else if (c=='(') {
				// @=(v)  loop if v<=0, next if v>0
				fetch(p++);
				val=artexp(&p);
				if (peek(p)!=')') {
					// syntax error
					fprintf(stderr,"ERROR: syntax error - %s in line %d\n",p,lineNo);
					exit(53);
				}
				fetch(p++);
				if (expand_mode) {
					if (val<=0) {
						// -> loop
						p=pop_loop_st(0);
						push_loop_st(p);
						continue;
					}
					// loop end -> throw stack
					pop_loop_st(1);
				}
				else {
					const short vs=val;
					if (vs<=0) {
						// -> loop
						p=pop_loop_st(0);
						push_loop_st(p);
						continue;
					}
					// loop end -> throw stack
					pop_loop_st(1);
				}
			}
			else {
				// syntax error
				fprintf(stderr,"ERROR: syntax error - %s in line %d\n",p,lineNo);
				exit(53);
			}
			continue;
		case '>': // call
			//if (expand_mode) {
				if (*p!='=') {
					// syntax error
					fprintf(stderr,"ERROR: syntax error - %s in line %d\n",p,lineNo);
					exit(53);
				}
				fetch(p++);
				val=artexp(&p);
				if (val==0xFFE8) {
					// 
					call_status=get_time();
					continue;
				}
			//}
			abort();
			break;
		case '\'':// set rand seed
			if (*p!='=') {
				// syntax error
				fprintf(stderr,"ERROR: syntax error - %s in line %d\n",p,lineNo);
				exit(53);
			}
			fetch(p++);
			val=artexp(&p);
			mysrand(val);
			break;
		case '$': // putchar
			if (*p!='=') {
				// syntax error
				fprintf(stderr,"ERROR: syntax error - %s in line %d\n",p,lineNo);
				exit(53);
			}
			fetch(p++);
			val=artexp(&p);
			putc(val,stdout);
			fflush(stdout);
			continue;
		case '?': // print dec,using,hex4,hex2
			c=peek(p);
			switch (c) {
			case '=': // ?=
				fetch(p++);
				val=artexp(&p);
				if (expand_mode) {
					printf("%ld",val);
					fflush(stdout);
				}
				else {
					const short vs=val;
					printf("%d",vs);
					fflush(stdout);
				}
				break;
			case '(': // ?(1)=
				fetch(p++);
				idx=artexp(&p);
				if (*p!=')' || *(p+1)!='=') {
					// syntax error
					fprintf(stderr,"ERROR: syntax error - %s in line %d\n",p,lineNo);
					exit(53);
				}
				fetch(p++); fetch(p++);
				val=artexp(&p);
				print_using(idx,val);
				break;
			case '?': // ??=
				fetch(p++);
				if (peek(p)!='=') {
					// syntax error
					fprintf(stderr,"ERROR: syntax error - %s in line %d\n",p,lineNo);
					exit(53);
				}
				fetch(p++);
				val=artexp(&p);
				printf("%04lX",val&0xFFFF);
				fflush(stdout);
				break;
			case '$': // ?$=
				fetch(p++);
				if (peek(p)!='=') {
					// syntax error
					fprintf(stderr,"ERROR: syntax error - %s in line %d\n",p,lineNo);
					exit(53);
				}
				fetch(p++);
				val=artexp(&p);
				printf("%02lX",val&0xFF);
				fflush(stdout);
				break;
			case '"': // ?"%d"= (is printf)
				fetch(p++);
				if (expand_mode) {
					char fmt[1024];
					int i;
					for (i=0; i<sizeof(fmt)-1; i++) {
						c=fetch(p++);
						if (c==0) {
							// EOF !?
							fprintf(stderr,"ERROR: syntax error - %s in line %d\n",p,lineNo);
							exit(53);
						}
						if (c=='"') {
							fmt[i]=0;
							break;
						}
						fmt[i]=c;
					}
					if (peek(p)!='=') {
						// syntax error
						fprintf(stderr,"ERROR: syntax error - %s in line %d\n",p,lineNo);
						exit(53);
					}
					fetch(p++);
					val=artexp(&p);
					if (expand_mode) {
						printf(fmt,val);
						fflush(stdout);
					}
					else {
						const short vs=val;
						printf(fmt,vs);
						fflush(stdout);
					}
					break;
				}
				// THRU
			default:
				// syntax error
				fprintf(stderr,"ERROR: syntax error - %s in line %d\n",p,lineNo);
				exit(53);
			}
			continue;
		case '.': // print spaces
			if (peek(p)!='=') {
				// syntax error
				fprintf(stderr,"ERROR: syntax error - %s in line %d\n",p,lineNo);
				exit(53);
			}
			fetch(p++);
			val=artexp(&p);
			for (; val>0; --val) {
				putc(' ',stdout);
			}
			fflush(stdout);
			continue;
		case '/': // print LF
			putc('\n',stdout);
			continue;
		case '"': // print string
			for (; peek(p)!='"';) {
				if (peek(p)==0) {
					// syntax error
					fprintf(stderr,"ERROR: syntax error - %s in line %d\n",p,lineNo);
					exit(53);
				}
				putc(fetch(p++),stdout);
			}
			fflush(stdout);
			fetch(p++);
			continue;
		default:
			// syntax error
			fprintf(stderr,"ERROR: syntax error - %c in line %d\n",c,lineNo);
			exit(53);
		}
	}
}

//===============================================================================
// load program
//===============================================================================
INLINE int getPRGx(FILE* fp)
{
	return getc(fp);
}
INLINE int getPRG(FILE* fp)
{
	int c=getPRGx(fp);
	if (verbose>=1) {
		if (binary && c==0) {
			// EOL
			putc('\n',stderr);
		}
		else {
			if (c!=EOF) {
				putc(c,stderr);
			}
		}
	}
	return c;
}
int load_program(FILE* fp)
{
	unsigned short pg=program_start;
	int c;
	int lines=1;
	int lino=0;

#ifdef BIN_LINE_SEARCH
	if (lineTop==NULL) {
		lineTopMaxN=64*1024/4;
		lineTop=(int*)malloc(lineTopMaxN*sizeof(lineTop[0]));
	}
#endif //BIN_LINE_SEARCH

	// readin file -> gen binary program on ram
	for (;;) {
		int no;
		if (binary) {
			if ((c=getPRGx(fp))==EOF) { break; }
			no=c<<8;
			if ((c=getPRGx(fp))==EOF) { break; }
			no+=c;
			if (verbose>=1) {
				fprintf(stderr,"%5d",no);
			}
			c=getPRG(fp);
			if (c==EOF || c=='\0' || c=='\n') {
				fprintf(stderr,"\nNo character after line number at line %d\n",lines);
				if (c==EOF) { break; }
				else { continue; }
				exit(3);
			}
		}
		else {
			if ((c=getPRG(fp))==EOF) { break; }
			// skip leading space
			if (c==' ' || c=='\t') {
				continue;
			}
			// line number
			if (! isdigit(c)) {
				fprintf(stderr,"\nNot LineNumber char %c at line %d\n",c,lines);
				exit(3);
			}
			no=0;
			for (; isdigit(c); c=getPRG(fp)) {
				no=no*10+c-'0';
			}
			if (c==EOF || c=='\n') {
				fprintf(stderr,"\nNo character after line number at line %d\n",lines);
				if (c==EOF) { break; }
				else { continue; }
				exit(3);
			}
		}
		if (expand_mode) {
			if (debug==0) {
				if (c!=' ') {
					// comment line -> remove it
					for (; c!='\n' && c!=0; c=getPRG(fp)) {
						if (c==EOF) {
							fprintf(stderr,"\nBroken line at line %d\n",lines);
							exit(3);
						}
					}
					continue;
				}
			}
		}
		if (pg>ram_end-3) {
			// too large program
			fprintf(stderr,"\nToo large program at line %d\n",lines);
			exit(3);
		}
		if (lino>=no) {
			// last line no >= cur no
			fprintf(stderr,"\nLineNumber reverse order %d>=%d\n",lino,no);
			exit(3);
		}
#ifdef BIN_LINE_SEARCH
		if (lines>=lineTopMaxN) { // must be 1 more for end of program
			lineTopMaxN*=2;
			lineTop=(int*)realloc(lineTop,lineTopMaxN*sizeof(lineTop[0]));
		}
		if (lineTop==NULL) {
			// cannot allocate
			fprintf(stderr,"\nno memory at line %d\n",lines);
			exit(3);
		}
		lineTop[lines-1]=pg;
#endif //BIN_LINE_SEARCH
		ram[pg++]=(no&0xFF00)>>8;
		ram[pg++]=no&0xFF;
		lino=no;
		// 
		for (; c!='\n' && c!=0; c=getPRG(fp)) {
			if (c==EOF) {
				fprintf(stderr,"\nBroken line at line %d\n",lines);
				exit(3);
			}
			ram[pg++]=c;
		}
		ram[pg++]=0;
		lines++;
	}
#ifdef BIN_LINE_SEARCH
	lineTopN=lines-1;
	lineTop[lines-1]=pg;	// allways ok
#endif //BIN_LINE_SEARCH
	program_end=pg;
	ram[program_end]=0xFF;

#ifdef BIN_LINE_SEARCH
# if (DEBUG>9)
	{
		int i;
		for (i=0; ; i++) {
			int n=ram[lineTop[i]];
			if (n>=0x80) { break; }
			n=(n<<8)+ram[lineTop[i]+1];
			printf("%5d : $%04x\n", n,lineTop[i]);
		}
	}
# endif //DEBUG>9
#endif //BIN_LINE_SEARCH

	return 0;
}


//===============================================================================
// list program
//===============================================================================
int list_program(char* start_cmd)
{
	char* p=start_cmd;
	int n;
	assert(p);
	assert(isdigit(start_cmd[0]));

	n=getNumber((const char**)&p);
	if (*p=='/' || *p=='-') {
		// list
		if (goto_line(n)<=0) {
			// program end
			return 0;
		}
		for (;;) {
			int c;

			n=pc_fetch()<<8;
			if (n>=0x8000) {
				// program end
				return 0;
			}
			n+=pc_fetch();
			printf("%5d",n);
			//
			while ((c=pc_fetch())!=0) {
				putc(c,stdout);
			}
			putc('\n',stdout);
		}
	}
	else {
		// syntax error
		fprintf(stderr,"ERROR: syntax error - %s\n",start_cmd);
		return 59;
	}

	return 0;
}

//===============================================================================
// main
//===============================================================================
int main(int ac, char** av)
{
	char* start_cmd="#=1";
	FILE* fp=stdin;
	int sts;

	while (ac>=2 && av[1][0]=='-') {
		if (ac>=3 && strcmp(av[1],"-s")==0) {
			// -s cmd : start command
			start_cmd=av[2];
			ac-=2;
			av+=2;
			continue;
		}
		if (strcmp(av[1],"-v")==0) {
			// -v
			verbose++;
			ac--;
			av++;
			continue;
		}
		if (strcmp(av[1],"-3")==0) {
			// -3 : restrict GAME III
			expand_mode=0;
			ac--;
			av++;
			continue;
		}
		if (strcmp(av[1],"-g")==0 || strcmp(av[1],"-b")==0) {
			// -g : game binary program source
			binary++;
			ac--;
			av++;
			continue;
		}
		if (ac>=3 && strcmp(av[1],"-d")==0) {
			// -d n : level n debug
			debug=atoi(av[2]);
			ac-=2;
			av+=2;
			continue;
		}
		usage("Illegal option");
	}

	if (ac<=1) {
		// from stdin
	}
	else if (ac==2) {
		// from file
		fp=fopen(av[1],"r");
		if (fp==NULL) {
			perror(av[1]);
			exit(2);
		}
	}
	else {
		usage(NULL);
	}

	sts=load_program(fp);
	if (fp!=stdin) {
		fclose(fp);
	}
	if (sts) {
		// error
		exit(3);
	}
	fp=NULL;

	if (expand_mode==0) {
		if (ram_end>0xFFFF) {
			ram_end=0xFFFF;
		}
	}
	pc=program_start;
	lineNo=0;

	if (isdigit(start_cmd[0])) {
		sts=list_program(start_cmd);
	}
	else {
		// start interpreter
		sts=execute(start_cmd);
	}
	//
	exit(sts);
}


//===============================================================================
// console io
//===============================================================================
#include <sys/types.h>
#include <termios.h>
#include <unistd.h>
int getch(int fd)
{
	struct termios cur_tio;
	struct termios tmp_tio;
	tcgetattr(fd,&cur_tio);
	tmp_tio=cur_tio;
	tmp_tio.c_lflag&= ~ECHO;
	tmp_tio.c_lflag&= ~ICANON;
	tcsetattr(fd,TCSANOW,&tmp_tio);
	unsigned char buf[2];
	int n=read(fd,buf,1);
	tcsetattr(fd,TCSANOW,&cur_tio);
	if (n!=1) {
		return -1;
	}
	return buf[0];
}

const char* getaline(int fd)
{
	static char buf[64];
	memset(buf,0,sizeof(buf));
	if (fgets(buf,sizeof(buf)-1,stdin)==NULL) {
		// error
		return NULL;
	}
	return buf;
}
//===============================================================================

