/*
 * $Id: convgame.c,v 1.6 2016/10/12 14:44:54 localhost Exp $
 */
/*
Copyright (c) 2016, k_mia
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice,
this list of conditions and the following disclaimer in the documentation and/or
other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>

void usage()
{
	fprintf(stderr,"usage: convgame [-g|-b|-t|-m] [file]\n");
	fprintf(stderr,"       -g: to game binary.\n");
	fprintf(stderr,"       -b: same as -g.\n");
	fprintf(stderr,"       -t: to text (default).\n");
	fprintf(stderr,"       -m: mzt to binary/text.\n");
}

int main(ac,av)
int ac;
char **av;
{
	int c;
	extern char *optarg;
	extern int optind;
	int errflg=0;
	int togame=0;
	FILE* fp=stdin;
	
	while ((c=getopt(ac,av,"gbtmh"))!= -1) {
		switch (c) {
		case 'g':
		case 'b':
			togame='g';
			break;
		case 't':
			togame=0;
			break;
		case 'm':
			togame='m';
			break;
		case '?':
			errflg++;
			break;
		case 'h':
			usage();
			exit(0);
		}
	}
	if (errflg) {
		usage();
		exit(2);
	}
	if (optind<ac) {
		FILE* ffp=fopen(av[optind],"r");
		if (ffp==NULL) {
			fprintf(stderr,"ERROR: %s cannot open.\n",av[optind]);
			exit (1);
		}
		fp=ffp;
	}

	if (togame=='g') {
		/* ascii -> game */
		int n;
		while ((c=getc(fp))!=EOF) {
			if (isspace(c)) continue;
			if (! isdigit(c)) {
				fprintf(stderr,"%c - Line number required\n",c);
				exit(2);
			}
			n=c-'0';
			while ((c=getc(fp))!=EOF) {
				if (! isdigit(c)) break;
				n=n*10+c-'0';
			}
			if (c==EOF||c=='\n') {
				fprintf(stderr,"%d - line contents required\n",n);
				exit(2);
			}
			printf("%c%c%c",(n>>8)&0xFF,n&0xFF,c);
			while ((c=getc(fp))!=EOF) {
				if (c=='\n') {
					printf("%c",0);
					break;
				}
				printf("%c",c);
			}
		}
		//printf("\377\000");
		putchar('\377');
		putchar('\000');
	}
	else if (togame==0) {
		/* game -> ascii */
		while ((c=getc(fp))!=EOF) {
			if (c>=0x80) break;
			c=(c<<8)+getc(fp);
			printf("%5d",c);
			while ((c=getc(fp))!=EOF) {
				if (c==0) {
					printf("\n");
					break;
				}
				printf("%c",c);
			}
		}
	}
	else if (togame=='m') {
		/* mzt -> game bin/text */
		int read_mzt(FILE* fp);
		int c=read_mzt(fp);
		if (c) {
			fprintf(stderr,"mzt format error (%d)\n",c);
			exit(3);
		}
	}
	return 0;
}


int read_mzt(FILE* fp)
{
	/*	00: attr
		01-11: filename
		12-13: filesize
		14-15: load addr
		16-17: start addr
		18-7F: reserved(blank)
		80-  : data
		xx-xx+1: csum
	*/
	// attr
	int attr=getc(fp);
	if (attr==EOF) { return -1; }
	// filename
	char fnam[32];
	int n=fread(fnam,1,17,fp);
	if (n<0) { return -2; }
	int i;
	for (i=0; i<17; i++) {
		if (fnam[i]<' '||0x7E<fnam[i]) { fnam[i]=0; break; }
	}
	fnam[i]=0;
	fprintf(stderr,"filename: %s\n",fnam);
	// filesize
	int siz=getc(fp);
	if (siz==EOF) { return -3; }
	int c=getc(fp);
	if (c==EOF) { return -4; }
	siz|=(c&0xFF)<<8;
	// load addr
	int loadfrom=getc(fp);
	if (loadfrom==EOF) { return -5; }
	c=getc(fp);
	if (c==EOF) { return -6; }
	loadfrom|=(c&0xFF)<<8;
	// start addr
	int startfrom=getc(fp);
	if (startfrom==EOF) { return -7; }
	c=getc(fp);
	if (c==EOF) { return -8; }
	startfrom|=(c&0xFF)<<8;
	fprintf(stderr,"filesize: 0x%04X (%d)\n",siz,siz);
	fprintf(stderr,"load adr: 0x%04X\n",loadfrom);
	fprintf(stderr,"startadr: 0x%04X\n",startfrom);
	// skip reserved
	for (i=0x18; i<0x80; i++) {
		c=getc(fp);
		if (c==EOF) { return -8; }
	}
	if (c==EOF) { return -9; }
	// data
	for (i=0; i<siz; i++) {
		c=getc(fp);
		if (c==EOF) { return -10; }
		printf("%c",c);
	}
	if (c==EOF) { return -11; }
	// csum (sometimes missing)
	int csum=getc(fp);
	if (csum==EOF) {
		fprintf(stderr,"no checksum\n");
		return 0;
	}
	c=getc(fp);
	if (c==EOF) { return -13; }
	csum|=(c&0xFF)<<8;
	fprintf(stderr,"checksum: 0x%04X (not checked)\n",csum);
	//
	return 0;
}
