package net.sf.amateras.nikocale.service;

import java.sql.SQLException;
import java.util.List;

import jp.sf.nikonikofw.persistence.jdbc.JdbcUtil;
import net.sf.amateras.nikocale.entity.GroupInfo;

public class GroupService {
	
	/**
	 * 指定したユーザが所属するグループのリストを取得します。
	 * 
	 * @param memberId ユーザID
	 * @return ユーザが所属するグループのリスト
	 * @throws SQLException SQLエラー
	 */
	public static List<GroupInfo> getMemberGroups(Long memberId) throws SQLException {
		return JdbcUtil.getResultList(GroupInfo.class,
				"SELECT G.ID, G.GROUP_NAME, G.DESCRIPTION " + 
				"FROM GROUP_INFO G, GROUP_MEMBER GM " +
				"WHERE G.ID = GM.GROUP_ID AND GM.MEMBER_ID = ? " +
				"ORDER BY G.ID ASC", memberId);
	}
	
	/**
	 * IDを指定してグループを取得します。
	 * グループが存在しない場合、nullを返します。
	 * 
	 * @param groupId グループID
	 * @return グループ
	 * @throws SQLException SQLエラー
	 */
	public static GroupInfo getGroup(Long groupId) throws SQLException {
		return JdbcUtil.getSingleResult(GroupInfo.class,
				"SELECT * FROM GROUP_INFO WHERE ID = ?", groupId);
	}
	
	/**
	 * すべてのグループを取得します。グループIDの昇順でソートされます。
	 * 
	 * @return すべてのグループ
	 * @throws SQLException SQLエラー
	 */
	public static List<GroupInfo> getGroups() throws SQLException {
		return JdbcUtil.getResultList(GroupInfo.class,
				"SELECT * FROM GROUP_INFO ORDER BY ID ASC", (Object[]) null);
	}
	
	/**
	 * グループを追加します。
	 * 
	 * @param group グループ
	 * @throws SQLException SQLエラー
	 */
	public static void addGroup(GroupInfo group) throws SQLException {
		JdbcUtil.insert(group);
	}
	
	/**
	 * グループを更新します。
	 * 
	 * @param group グループ
	 * @throws SQLException SQLエラー
	 */
	public static void updateGroup(GroupInfo group) throws SQLException {
		JdbcUtil.update(group);
	}
	
	/**
	 * グループを削除します。グループとメンバーの関連付けも削除されます。
	 * 
	 * @param group グループ
	 * @throws SQLException SQLエラー
	 */
	public static void deleteGroup(GroupInfo group) throws SQLException {
		JdbcUtil.execute("DELETE FROM GROUP_MEMBER WHERE GROUP_ID = ?", group.id);
		JdbcUtil.execute("DELETE FROM GROUP_INFO WHERE ID = ?", group.id);
	}
	
}
