package net.sf.amateras.nikocale.service;

import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jp.sf.nikonikofw.persistence.jdbc.JdbcUtil;
import jp.sf.nikonikofw.persistence.jdbc.annotation.Column;
import jp.sf.nikonikofw.util.IOUtils;
import net.sf.amateras.nikocale.entity.Entry;
import net.sf.amateras.nikocale.entity.Member;

public class EntryService {

	private static final String PREFIX = "net/sf/amateras/nikocale/service/EntryService_";
	
	/**
	 * 昨日更新されたニコカレエントリを取得します。
	 * 
	 * @return 昨日のニコカレ
	 */
	public static List<Entry> getYesterdayEntries(){
		Calendar cal = Calendar.getInstance();
		// TO: 今日まで
		cal.set(Calendar.HOUR_OF_DAY, 0);
		cal.set(Calendar.MINUTE, 0);
		cal.set(Calendar.SECOND, 0);
		cal.set(Calendar.MILLISECOND, 0);
		Date to = cal.getTime();
		
		// FROM: 昨日から
		cal.add(Calendar.DATE, -1);
		Date from = cal.getTime();
		
		return JdbcUtil.getResultList(Entry.class, 
				"SELECT M.NAME AS MEMBER_NAME, E.MEMBER_ID, E.MESSAGE, E.YEAR, E.MONTH, E.DAY, E.STATUS, E.UPDATE_DATE," +
				"(SELECT COUNT(ID) FROM COMMENT WHERE ENTRY_ID = E.ID) AS COMMENT_COUNT " +
				"FROM ENTRY E " +
				"INNER JOIN MEMBER M ON E.MEMBER_ID = M.ID " +
				"WHERE E.UPDATE_DATE >= ? AND E.UPDATE_DATE < ?", 
				new Timestamp(from.getTime()), 
				new Timestamp(to.getTime()));
	}
	
	/**
	 * 最新のニコカレエントリを取得します。
	 *
	 * @param limit 取得件数
	 * @return 最新のニコカレ
	 * @throws SQLException SQLエラー
	 */
	public static List<Entry> getRecentEntries(int limit) throws SQLException {
		return JdbcUtil.getResultList(Entry.class,
				IOUtils.read(PREFIX + "getRecentEntries.sql"),
				limit);
	}

	/**
	 * ユーザ毎の最新のニコカレエントリを取得します。
	 *
	 * @param memberId ユーザのID
	 * @param limit 取得件数
	 * @return 最新のニコカレ
	 * @throws SQLException SQLエラー
	 */
	public static List<Entry> getRecentEntriesByMember(Long memberId, int limit) throws SQLException {
		return JdbcUtil.getResultList(Entry.class,
				IOUtils.read(PREFIX + "getRecentEntriesByMember.sql"),
				memberId, limit);
	}
	
	/**
	 * グループ毎の最新のニコカレエントリを取得します。
	 *
	 * @param groupId グループのID
	 * @param limit 取得件数
	 * @return 最新のニコカレ
	 * @throws SQLException SQLエラー
	 */
	public static List<Entry> getRecentEntriesByGroup(Long groupId, int limit) throws SQLException {
		return JdbcUtil.getResultList(Entry.class,
				"SELECT E.ID, E.MEMBER_ID, E.YEAR, E.MONTH, E.DAY, E.STATUS, E.MESSAGE, " +
				"M.NAME AS MEMBER_NAME, E.UPDATE_DATE, " + 
				"(SELECT COUNT(ID) FROM COMMENT WHERE ENTRY_ID = E.ID) AS COMMENT_COUNT " +
				"FROM ENTRY E, MEMBER M, GROUP_MEMBER GM " +
				"WHERE E.MEMBER_ID = M.ID AND M.ID = GM.MEMBER_ID AND GM.GROUP_ID = ?" + 
				"ORDER BY UPDATE_DATE DESC LIMIT ?",
				groupId, limit);
	}
	
	public static Map<String, Entry> getMemberEntries(
			Long memberId, Integer year, Integer month) throws SQLException {
		List<Entry> result = JdbcUtil.getResultList(Entry.class,
				"SELECT ID, MEMBER_ID, YEAR, MONTH, DAY, STATUS, MESSAGE, " +
				"(SELECT COUNT(ID) FROM COMMENT WHERE ENTRY_ID = ENTRY.ID) AS COMMENT_COUNT " +
				"FROM ENTRY WHERE MEMBER_ID = ? AND YEAR = ? AND MONTH = ?",
				memberId, year, month);

		Map<String, Entry> entries = new HashMap<String, Entry>();
		for(Entry entry: result){
			entries.put(memberId + "-" + entry.day, entry);
		}

		return entries;
	}

	/**
	 * 引数で指定したIDのユーザが最初にニコカレをつけた年月日を取得します。
	 */
	public static String getFirstDate(Long memberId) throws SQLException {
		DateBean bean = JdbcUtil.getSingleResult(DateBean.class,
				"SELECT YEAR, MONTH, DAY " +
				"FROM ENTRY WHERE MEMBER_ID = ? " +
				"ORDER BY YEAR ASC, MONTH ASC, DAY ASC " +
				"LIMIT 1", memberId);
		
		if(bean == null){
			return null;
		}

		return String.format("%04d/%02d/%02d", bean.year, bean.month, bean.day);
	}

	/**
	 * 引数で指定したIDのユーザが最後にニコカレをつけた年月日を取得します。
	 */
	public static String getLatestDate(Long memberId) throws SQLException {
		DateBean bean = JdbcUtil.getSingleResult(DateBean.class,
				"SELECT YEAR, MONTH, DAY " +
				"FROM ENTRY WHERE MEMBER_ID = ? " +
				"ORDER BY YEAR DESC, MONTH DESC, DAY DESC " +
				"LIMIT 1", memberId);
		
		if(bean == null){
			return null;
		}

		return String.format("%04d/%02d/%02d", bean.year, bean.month, bean.day);
	}

	/**
	 * 引数で指定したIDのユーザのニコニコポイントを取得します。
	 */
	public static Integer getMemberPoint(Long memberId) throws SQLException {
		IntegerValueBean countBean = JdbcUtil.getSingleResult(IntegerValueBean.class,
				"SELECT SUM(CASE STATUS " +
				"  WHEN 0 THEN 10 " +
				"  WHEN 1 THEN 5 " +
				"  WHEN 2 THEN -5 " +
				"END) AS VALUE " +
				"FROM ENTRY WHERE MEMBER_ID = ? ", memberId);

		return countBean.value;
	}

	/**
	 * ニコニコ or ダメダメランキングを取得します。
	 *
	 * @param top ニコニコランキングの場合true、ダメダメランキングの場合false
	 * @return ニコニコポイントの上位 or 下位5名分のデータ
	 * @throws SQLException SQLエラー
	 */
	public static List<Member> getRanking(boolean top) throws SQLException {
		return JdbcUtil.getResultList(Member.class,
			String.format(
				"SELECT M.ID, M.NAME, SUM(CASE E.STATUS " +
				"  WHEN 0 THEN 10 " +
				"  WHEN 1 THEN 5 " +
				"  WHEN 2 THEN -5 " +
				"END) AS POINT " +
				"FROM ENTRY E, MEMBER M " +
				"WHERE E.MEMBER_ID = M.ID GROUP BY M.ID, M.NAME " +
				"ORDER BY POINT %s LIMIT 5", top ? "DESC" : "ASC"), (Object[]) null);
	}

	public static Map<String, Entry> getGroupEntries(Long groupId, Integer year, Integer month) throws SQLException {
		List<Entry> result = JdbcUtil.getResultList(Entry.class,
				"SELECT E.ID, E.MEMBER_ID, E.YEAR, E.MONTH, E.DAY, E.STATUS, E.MESSAGE, " +
				"(SELECT COUNT(ID) FROM COMMENT WHERE ENTRY_ID = E.ID) AS COMMENT_COUNT " +
				"FROM ENTRY E, MEMBER M, GROUP_MEMBER GM " +
				"WHERE E.MEMBER_ID = M.ID AND GM.MEMBER_ID = M.ID AND " +
				"GM.GROUP_ID = ? AND E.YEAR = ? AND E.MONTH = ?",
				groupId, year, month);

		Map<String, Entry> entries = new HashMap<String, Entry>();

		for(Entry entry: result){
			entries.put(entry.userId + "-" + entry.day, entry);
		}

		return entries;
	}

	public static Map<String, Entry> getEntries(Integer year, Integer month) throws SQLException {
		List<Entry> result = JdbcUtil.getResultList(Entry.class,
				"SELECT E.ID, E.MEMBER_ID, E.YEAR, E.MONTH, E.DAY, E.STATUS, E.MESSAGE, " +
				"(SELECT COUNT(ID) FROM COMMENT WHERE ENTRY_ID = E.ID) AS COMMENT_COUNT " +
				"FROM ENTRY E, MEMBER M " +
				"WHERE E.MEMBER_ID = M.ID AND E.YEAR = ? AND E.MONTH = ?",
				year, month);

		Map<String, Entry> entries = new HashMap<String, Entry>();

		for(Entry entry: result){
			entries.put(entry.userId + "-" + entry.day, entry);
		}

		return entries;
	}

	/**
	 * ニコカレのエントリーを作成または更新します。
	 *
	 * @param entry 保存するエントリ
	 * @throws SQLException SQLエラー
	 */
	public static void saveEntry(Entry entry) throws SQLException {
		if(entry.id == null){
			JdbcUtil.insert(entry);
		} else {
			JdbcUtil.update(entry);
		}
	}

	/**
	 * ニコカレのエントリを削除します。
	 *
	 * @param entry 削除するエントリ
	 * @throws SQLException SQLエラー
	 */
	public static void deleteEntry(Entry entry) throws SQLException {
		JdbcUtil.execute("DELETE FROM ENTRY WHERE ID = ?", entry.id);
	}

	public static Entry getEntry(Long entryId) throws SQLException {
		return JdbcUtil.getSingleResult(Entry.class, "SELECT * FROM ENTRY WHERE ID = ?", entryId);
	}
	
	public static Entry getEntry(Long memberId, Integer year, Integer month, Integer day) throws SQLException {
		return JdbcUtil.getSingleResult(Entry.class,
				"SELECT ID, MEMBER_ID, YEAR, MONTH, DAY, STATUS, MESSAGE, ADD_DATE, UPDATE_DATE, " +
				"(SELECT COUNT(ID) FROM COMMENT WHERE ENTRY_ID = E.ID) AS COMMENT_COUNT " +
				"FROM ENTRY E " +
				"WHERE MEMBER_ID = ? AND YEAR = ? AND MONTH = ? AND DAY = ?",
				memberId, year, month, day);
	}

	/**
	 * 指定したユーザのグラフデータを取得します。
	 *
	 * @param year 年
	 * @param members ユーザのリスト
	 * @return グラフデータ
	 * @throws SQLException SQLエラー
	 */
	public static List<Map<String, Object>> getMonthGroupGraph(
			Integer year, Integer month, Integer lastDay, List<Member> members) throws SQLException {

		List<Map<String, Object>> result = new ArrayList<Map<String,Object>>();

		for(Member member: members){
			Map<String, Object> map = new HashMap<String, Object>();
			map.put("label", member.name);

			List<Integer[]> data = new ArrayList<Integer[]>();
			map.put("data", data);

			// エントリの最大年月日を取得
			DateBean dateBean = JdbcUtil.getSingleResult(DateBean.class,
					"SELECT YEAR, MONTH, DAY FROM ENTRY " +
					"WHERE MEMBER_ID = ? " +
					"ORDER BY YEAR DESC, MONTH DESC, DAY DESC LIMIT 1", member.id);

			if(dateBean != null){
				String maxDate = String.format("%04d%02d%02d", dateBean.year, dateBean.month, dateBean.day);

				for(int i=1; i<=lastDay; i++){
					String date = String.format("%04d%02d%02d", year, month, i);

					if(Integer.parseInt(maxDate) >= Integer.parseInt(date)){
						IntegerValueBean countBean = JdbcUtil.getSingleResult(IntegerValueBean.class,
								"SELECT SUM( " +
								"    CASE STATUS " +
								"      WHEN 0 THEN 10 " +
								"      WHEN 1 THEN 5 " +
								"      WHEN 2 THEN -5 " +
								"    END) AS VALUE " +
								"FROM ENTRY WHERE MEMBER_ID = ? AND " +
								"(YEAR || CASEWHEN(LENGTH(MONTH) = 1, '0' || MONTH, CONVERT(MONTH, VARCHAR)) || CASEWHEN(LENGTH(DAY) = 1, '0' || DAY, CONVERT(DAY, VARCHAR))) <= ? ",
								member.id, date);

						if(countBean != null){
							Integer[] point = new Integer[]{i, countBean.value};
							data.add(point);
							continue;
						}
					}

					Integer[] point = new Integer[]{i, null};
					data.add(point);
				}
			} else {
				for(int i=1; i<=lastDay; i++){
					Integer[] point = new Integer[]{i, null};
					data.add(point);
				}
			}

			result.add(map);
		}

		return result;
	}

	/**
	 * 引数で指定したIDのユーザがニコカレをつけた回数を取得します。
	 */
	public static Integer getMemberEntryCount(Long memberId) throws SQLException {
		return JdbcUtil.getSingleResult(IntegerValueBean.class, 
				"SELECT COUNT(ID) AS VALUE FROM ENTRY WHERE MEMBER_ID=?",
				memberId).value;
	}
	
	public static class DateBean {
		@Column
		public Integer year;
		
		@Column
		public Integer month;
		
		@Column
		public Integer day;
	}
	
	public static class IntegerValueBean {
		@Column
		public Integer value;
	}

}
