package jp.sf.nikonikofw.util;

import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

/**
 * 入出力関係のユーティリティメソッドを提供します。
 * 
 * @author Naoki Takezoe
 */
public class IOUtils {
	
	/**
	 * 入力ストリームから出力ストリームにデータをコピーします。
	 * 
	 * @param in コピー元の入力ストリーム。処理の終了後はクローズされます。
	 * @param out コピー先の出力ストリーム。処理の終了後はクローズされません。
	 */
	public static void copy(InputStream in, OutputStream out){
		byte[] buf = new byte[1024 * 8];
		int length = 0;
		
		try {
			while((length = in.read(buf)) != -1){
				out.write(buf, 0, length);
			}
		} catch(Exception ex){
			throw new RuntimeException(ex);
		} finally {
			closeQuietly(in);
		}
	}
	
	/**
	 * クラスパス内のリソースからUTF-8で文字列として読み込みます。
	 * 
	 * @param path クラスパス内のパス
	 * @return 入力ストリームからUTF-8で読み込んだ文字列
	 */
	public static String read(String path){
		return read(Thread.currentThread().getContextClassLoader().getResourceAsStream(path));
	}
	
	/**
	 * クラスパス内のリソースから指定した文字コードで文字列として読み込みます。
	 * 
	 * @param path クラスパス内のパス
	 * @param charset 文字コード
	 * @return 入力ストリームから指定した文字コードで読み込んだ文字列
	 */
	public static String read(String path, String charset){
		return read(Thread.currentThread().getContextClassLoader().getResourceAsStream(path), charset);
	}
	
	/**
	 * 入力ストリームからUTF-8で文字列として読み込みます。
	 * 
	 * @param in 入力ストリーム
	 * @return 入力ストリームからUTF-8で読み込んだ文字列
	 */
	public static String read(InputStream in){
		return read(in, "UTF-8");
	}
	
	/**
	 * 入力ストリームから指定した文字コードで文字列として読み込みます。
	 * 
	 * @param in 入力ストリーム
	 * @param charset 文字コード
	 * @return 入力ストリームから指定した文字コードで読み込んだ文字列
	 */
	public static String read(InputStream in, String charset){
		try {
			byte[] bytes = readBytes(in);
			return new String(bytes, charset);
			
		} catch(IOException ex){
			throw new RuntimeException(ex);
		} finally {
			closeQuietly(in);
		}
	}
	
	/**
	 * ストリームをクローズします。
	 * クローズに失敗した場合でも例外をスローしません。
	 * 
	 * @param closeable クローズするストリーム
	 */
	public static void closeQuietly(Closeable closeable){
		if(closeable != null){
			try {
				closeable.close();
			} catch(IOException ex){
				;
			}
		}
	}
	
	/**
	 * クラスパス内のリソースをバイト配列として読み込みます。
	 * 
	 * @param path クラスパス内のパス
	 * @return 入力ストリームから読み込んだバイト配列
	 */
	public static byte[] readBytes(String path){
		return readBytes(Thread.currentThread().getContextClassLoader().getResourceAsStream(path));
	}
	
	/**
	 * 入力ストリームからバイト配列として読み込みます。
	 * 
	 * @param in 入力ストリーム
	 * @return 入力ストリームから読み込んだバイト配列
	 */
	public static byte[] readBytes(InputStream in){
		return readBytes(in, 0);
	}
	
	/**
	 * 入力ストリームからバイト配列として読み込みます。
	 * 
	 * @param in 入力ストリーム
	 * @param maxlength 最大バイト長（0もしくは負の値を指定した場合は無制限）
	 * @return 入力ストリームから読み込んだバイト配列
	 * @throws RuntimeException 最大長をこえていた場合
	 */
	public static byte[] readBytes(InputStream in, int maxlength){
		try {
			ByteArrayOutputStream out = new ByteArrayOutputStream();
			byte[] buf = new byte[1024 * 8];
			int length = 0;
			int totalLength = 0;
			while((length = in.read(buf)) != -1){
				totalLength = totalLength + length;
				if(maxlength > 0 && totalLength > maxlength){
					// TODO 例外はこれでいいのか？
					throw new RuntimeException("サイズ制限を超えています。");
				}
				out.write(buf, 0, length);
			}
			return out.toByteArray();
		} catch(IOException ex){
			throw new RuntimeException(ex);
		} finally {
			IOUtils.closeQuietly(in);
		}
	}
	
}
