/*
 * em_vango_v9260.h
 *
 *  Created on: 10/08/2021
 *      Author: alexrayne <alexraynepe196@gmail.com>
  ------------------------------------------------------------------------
    Copyright (c) alexrayne

   All rights reserved.
   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are met:
   - Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   - Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   - Neither the name of ARM nor the names of its contributors may be used
     to endorse or promote products derived from this software without
     specific prior written permission.
   *
   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL COPYRIGHT HOLDERS AND CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. *
  ------------------------------------------------------------------------
      VangoTech Energy-meter AFE V9260 chips UART API
 */

#ifndef BSP_DEV_METER_EM_VANGO_V9260_H_
#define BSP_DEV_METER_EM_VANGO_V9260_H_


#include "sys/pt.h"
#include "OsTime.h"
#include "halos_types.h"



/////////////////////////////////////////////////////////////////////////////////
///             App  provided API

/// App provided handles for IO
/// @return < 0 - error
///         >= 0 - writen/readen bytes
typedef int (*V9260IOWrite)(void* self, const void* src, unsigned len);
typedef int (*V9260IORead)(void* self, void* dst, unsigned len);
struct V9260Dev;

#ifndef V9260IO_TIMEOUT
#define V9260IO_TIMEOUT(self, sentlen)     (CLOCK_SECOND/5)
#endif

#ifndef V9260IO_RETRY
#define V9260IO_RETRY   3
#endif


/// @brief App should provide this routine for generate
///         RX reset signaling
void    v9260_rx_reset( bool onoff);

/// @brief singnaler on V9260 request finished
/// @nore weak, app should override it
void v9260_finished( struct V9260Dev* this);

/////////////////////////////////////////////////////////////////////////////////
#ifndef V9260IO_BUF_REGS
#define V9260IO_BUF_REGS    2
#endif

#if V9260IO_BUF_REGS < 1
#error "V9260IO_BUF_REGS  MUST >= 1"
#endif



/// @brief max v9260 message size, generated by driver
#define V9260_MSG_BYTES(regs)   ( 8 + 4*(regs-1))




typedef uint16_t    V9260Adr;
enum {

    V9260REG_Msk        = 0xfffu,

    // this flag in register adress, denotes, that writen data should be verifyed by read
    V9260REG_VERIFY     = 0x8000u,
};

union V9260DevStats{
    struct {
    unsigned        rd;
    unsigned        wr;
    unsigned        rd_retry;
    unsigned        wr_retry;
    unsigned        rd_fail;
    unsigned        wr_fail;
    unsigned        rdio_fail;
    unsigned        wrio_fail;
    };
    struct {
        unsigned        io[2];
        unsigned        retry[2];
        unsigned        err[4];
    };
};
typedef union V9260DevStats V9260DevStats;

struct V9260Dev{
    struct {
        V9260IOWrite    write;
        V9260IORead     read;
        unsigned        cs;     // A0A1 chip selector

        os_timeout_t    rd_to;

        unsigned        bufcnt;
        unsigned        bufneed;
        union {
            uint8_t         buf[8 + 4*(V9260IO_BUF_REGS)];
            uint32_t        w[2+V9260IO_BUF_REGS];
        };
    }           io;

    struct {
        unsigned        cnt;
        V9260Adr        adr;
        union {
            uint32_t*       dst;
            const uint32_t* src;
        };
        uint32_t        data;
        unsigned        retry;
        unsigned        retry_wr;
        int             state;

        void*           signal;
    }           task;

    V9260DevStats   stats;
};
typedef struct V9260Dev V9260Dev;

/// @brief буфер с данными последнего обмена - отправленое или принятое
///        он ссылается на структуру внутри отосланого/принятого сообщения
static inline
uint32_t*   v9260_data(V9260Dev* self) {
    return self->io.w+1;
}



void        v9260_init(V9260Dev* self);

static inline
void        v9260_connect(V9260Dev* self, V9260IOWrite write, V9260IORead read, unsigned cs){
    self->io.read   = read;
    self->io.write  = write;
    self->io.cs     = cs;
}

PTResult    v9260_wait(V9260Dev* self);

/// @arg dst - if NULL оставляет принятые данные в буферах
DevResult   v9260_ask(V9260Dev* self, V9260Adr adr, void* dst, unsigned cnt );

DevResult   v9260_post(V9260Dev* self, V9260Adr adr, const uint32_t* src, unsigned cnt );
DevResult   v9260_post1(V9260Dev* self, V9260Adr adr, uint32_t data);

DevResult   v9260_reset(V9260Dev* self);

/// @brief process receiving stream from device. not affects much, can invokes from isr
PTResult    v9260_receive(V9260Dev* self);

///@brief state-machine on device operation
PTResult    v9260_process(V9260Dev* self);

void        v9260_abort(V9260Dev* self);



/////////////////////////////////////////////////////////////////////////////////

enum V9260AddrID{
    V9260_ARRTU         = 0x104,        //< The raw total voltage RMS
    V9260_ARRTI         = 0x105,        //< The raw total current RMS

    V9260_ARTU          = 0x10D,        //< The instantaneous total voltage RMS
    V9260_ARTI          = 0x10E,        //< The instantaneous total current RMS

    V9260_AAP           = 0x119,        //< The average total active power
    V9260_AAQ           = 0x11A,        //< The average total reactive power
    V9260_AARTU         = 0x11B,        //< The instantaneous fundamental voltage RMS
    V9260_AARTI         = 0x11C,        //< The instantaneous fundamental current RMS

    V9260_SAFREQ        = 0x11D,        //< The line frequency per second
    V9260_AFREQ         = 0x11E,        //< The average line frequency

    V9260_BPFPARA       = 0x125,

    V9260_SYSCTRL       = 0x180,

    V9260_MTPARA0       = 0x183,
    V9260_MTPARA1       = 0x184,

    V9260_ANACTRL0      = 0x185,
    V9260_ANACTRL1      = 0x186,
    V9260_ANACTRL2      = 0x187,

    V9260_NONE          = 0x1000,
};

enum V9260_BPFPARAFields{
    /* This coefficient and the bits to be shifted in the  bandpass filter have effect on the filter response time,
     * the sensitivity to the frequency deviation, and the suppression on the harmonics and noise.*/

    /* Generally, it is set to “0x811D2BA7” for the  best performance. */
    V9260_BPFPARA_DEFAULT  = 0x811D2BA7,
};

enum V9260_SYSCTRLFields{
    /* The internal RAM will be self-checked immediately once a global reset event occurs.
     * The self-checking will be finished in 1.25ms. After the self-checking,
     * If this bit is read out as ‘1’, it indicates that the self-checking of the internal RAM fails.
     * If this bit is read out as ‘0’, it indicates that the internal RAM is ready to be accessed;
     *  but if this bit is read out as ‘1’ again after another reset event, it indicates that there is something
     *  wrongwith RAM.
     * */
    V9260_SYSCTRL_BISTERR   =   1u<<29,     //< The internal RAM self-check err

    /* When the input voltage on the pin “AVCC” is lower than 2.8 V, this bit will be set to ‘1’.
     * When the input is higher than 2.8 V, this bit will be cleared*/
    V9260_SYSCTRL_PDN       =   1u<<28,     //< Power-down interrupt flag bit

    V9260_SYSCTRL_HSEERR    =   1u<<27,     //< External crystal failure interrupt flag

    V9260_SYSCTRL_RSTSRC_Pos    =   24,
    V9260_SYSCTRL_RSTSRC_Msk    =   7u<<V9260_SYSCTRL_RSTSRC_Pos,

    V9260_SYSCTRL_RST_WDT       = 0,
    V9260_SYSCTRL_RST_POR       = 1u <<V9260_SYSCTRL_RSTSRC_Pos,
    V9260_SYSCTRL_RST_RX        = 2u <<V9260_SYSCTRL_RSTSRC_Pos,
    V9260_SYSCTRL_RST_SW        = 4u <<V9260_SYSCTRL_RSTSRC_Pos,

    /* Invert the value of “bit7” of “SysCtrl” and write it to “bit23”
        for the system control register self-checking */
    V9260_SYSCTRL_SELFCK        = 1u <<23,

    V9260_SYSCTRL_PDNF          = 1u <<22,
    V9260_SYSCTRL_REFERR        = 1u <<21,
    V9260_SYSCTRL_SYSERR        = 1u <<20,
    V9260_SYSCTRL_CHKERR        = 1u <<19,  //< Read this bit for the state of configuration verification

    V9260_SYSCTRL_DETCST        = 1u <<18,  //< Current detection interrupt flag
    V9260_SYSCTRL_USIGN         = 1u <<17,

    /* Invert the value of “bit0” (SLEEP) of “SysCtrl” and write it to “bit16” for
     * the system control register self-checking */
    V9260_SYSCTRL_SELFCK2       = 1u <<16,

    //<  baud rate for UART communication
    V9260_SYSCTRL_CKUDIV_Pos    = 2,
    V9260_SYSCTRL_CKUDIV_Msk    = 7u << V9260_SYSCTRL_CKUDIV_Pos,
    V9260_SYSCTRL_CKU_1200      = 0,
    V9260_SYSCTRL_CKU_2400      = 1<< V9260_SYSCTRL_CKUDIV_Pos,
    V9260_SYSCTRL_CKU_4800      = 2<< V9260_SYSCTRL_CKUDIV_Pos,
    V9260_SYSCTRL_CKU_9600      = 3<< V9260_SYSCTRL_CKUDIV_Pos,
    V9260_SYSCTRL_CKU_19200     = 4<< V9260_SYSCTRL_CKUDIV_Pos,
    V9260_SYSCTRL_CKU_38400     = 5<< V9260_SYSCTRL_CKUDIV_Pos,

    //  select the clock frequency for the Vango metering  architecture (MEACLK)
    V9260_SYSCTRL_CKMDIV        = 2,
    V9260_SYSCTRL_CKMDIV_819K   = 0,
    V9260_SYSCTRL_CKMDIV_3M2768 = V9260_SYSCTRL_CKMDIV,

    // Set this bit to ‘1’ to disable “CLK1” and “CLK2” and force the system to enter the Sleep Mode
    V9260_SYSCTRL_SLEEP         = 1,

};

enum V9260_MTPARA0Fields{
    /* T is calculated via the equation T = 2^WDT / f(CLK3)
     * where: WDT = 16+<WDT Field>
     * */
    V9260_MTPARA0_WDT_Pos   = 15,
    V9260_MTPARA0_WDT_Msk   = 7u <<V9260_MTPARA0_WDT_Pos,
    V9260_MTPARA0_WDT1      = 1u <<V9260_MTPARA0_WDT_Pos,
    V9260_MTPARA0_WDT_MAX   = V9260_MTPARA0_WDT_Msk,


    V9260_MTPARA0_IDTEN     = 1u << 12,     //< To enable current detection

    /*  width of filtering window for current detection
     * For instance, if “IDETLEN” is set to “0b0011”, the filtering window width is set to ‘4’,
     * which means only when four continuous current samples are higher than the threshold is it defined that
     * a current signal is caught.
     * */
    V9260_MTPARA0_IDTLEN_Pos    = 8,
    V9260_MTPARA0_IDTLEN_Msk    = 0xf <<V9260_MTPARA0_IDTLEN_Pos,
    V9260_MTPARA0_IDTLEN1       = 1u << V9260_MTPARA0_IDTLEN_Pos,

    V9260_MTPARA0_IEWDT         = 1u << 5,     //<  enable WDT overflow interrupt output
    V9260_MTPARA0_IEHSE         = 1u << 4,     //<  enable external crystal failure interrupt output
    V9260_MTPARA0_IEPDN         = 1u << 3,     //<  enable power-down interrupt output
    V9260_MTPARA0_IEIDET        = 1u << 2,     //<  enable current detection interrupt output
    V9260_MTPARA0_IESUL         = 1u << 1,     //<  enable voltage sign output
};

enum V9260_MTPARA1Fields{
    /* When this bit is set to ‘1’, only configuration verification will be deactivated. When this bit is cleared,
     * all functions of the Vango metering architecture will be activated.
     * */
    V9260_MTPARA1_CKSUM         = 1u<< 22,
    V9260_MTPARA1_IDET          = 1u<< 21,

    /* By default the high pass filter is enabled to remove the DC component from the signals,
     *  and only the AC component of the signals are sent for RMS and power calculation*/
    V9260_MTPARA1_BPHPF         = 1u<< 20,

    V9260_MTPARA1_ENHPC         = 1u<< 19,  //< enable phase compensation
    V9260_MTPARA1_PHCIU         = 1u<< 18,  //<  delay voltage for the phase compensation

    V9260_MTPARA1_ONI           = 1u<< 17,  //<  enable digital signal input of current channel for digital signal processing
    V9260_MTPARA1_ONU           = 1u<< 16,  //<  enable digital signal input of voltage channel for digital signal processing

    // absolute value for phase compensation. The resolution is 0.0055°/lsb, and a phase error up to 1.4° can be calibrated
    V9260_MTPARA1_PHC_Pos   = 8,
    V9260_MTPARA1_PHC_Msk   = 0xffu <<V9260_MTPARA1_PHC_Pos,
    V9260_MTPARA1_PHC1      = 1u <<V9260_MTPARA1_PHC_Pos,

    //  digital PGA gain of current input : 1/(2^N)
    V9260_MTPARA1_PGAI_Pos  = 4,
    V9260_MTPARA1_PGAI_Msk  = 0xfu <<V9260_MTPARA1_PGAI_Pos,
    V9260_MTPARA1_PGAI2     = 1u <<V9260_MTPARA1_PGAI_Pos,
    V9260_MTPARA1_PGAI4     = 2u <<V9260_MTPARA1_PGAI_Pos,
    V9260_MTPARA1_PGAI8     = 3u <<V9260_MTPARA1_PGAI_Pos,
    V9260_MTPARA1_PGAI16    = 4u <<V9260_MTPARA1_PGAI_Pos,
    V9260_MTPARA1_PGAI32    = 5u <<V9260_MTPARA1_PGAI_Pos,

    //  digital PGA gain of volt input : 1/(2^N)
    V9260_MTPARA1_PGAU_Pos  = 0,
    V9260_MTPARA1_PGAU_Msk  = 0xfu <<V9260_MTPARA1_PGAI_Pos,
    V9260_MTPARA1_PGAU2     = 1u <<V9260_MTPARA1_PGAU_Pos,
    V9260_MTPARA1_PGAU4     = 2u <<V9260_MTPARA1_PGAU_Pos,
    V9260_MTPARA1_PGAU8     = 3u <<V9260_MTPARA1_PGAU_Pos,
    V9260_MTPARA1_PGAU16    = 4u <<V9260_MTPARA1_PGAU_Pos,
    V9260_MTPARA1_PGAU32    = 5u <<V9260_MTPARA1_PGAU_Pos,

};


enum V9260_ANACTRL0Fields{

    V9260_ANACTRL0_DEFAULT      = 0x40,

    /* Clear this bit to enable the 3.2-MHz RC Clock.
     * It is mandatory to enable the Bandgap circuit and biasing circuit firstly.
     * The value of the bit is uncertain when the system is reset.
     * In the Sleep Mode, this bit is set to ‘1’ automatically.
     * In the Current Detection Mode, this bit is cleared automatically.
     * In the Metering Mode, when the chip operates with full functions,
     *  it is recommended to disable this circuit (Set the bit to ‘1’)
     * */
    V9260_ANACTRL0_PDRCLK       = 1u<<29,

    /* Set this bit to ‘1’ to enable the biasing circuit to provide global biasing current for ADCs and
     *  the 3.2-MHz RC oscillator. Therefore, in the Metering Mode, when the chip operates with  full functions,
     *      this bit must be set to ‘1’ before enabling ADCs and the 3.2-MHz RC oscillator.
     * By default the biasing circuit is disabled.
     * In the Sleep Mode, this bit is cleared automatically.
     * In the Current Detection Mode, this bit is set to ‘1’ automatically.
     * */
    V9260_ANACTRL0_BIASPDN      = 1u<<28,

    /* Set this bit to ‘1’ to enable the Bandgap circuit to provide ADCs and the 3.2-MHz RC oscillator with
     *      reference voltage and biasing voltage.
     * Therefore, in the Metering Mode, when the  chip operates with full functions, this bit
     *      must be set to ‘1’ before enabling ADCs and the 3.2-MHz RC oscillator.
     * By default Bandgap circuit is disabled.
     * In the Sleep Mode, this bit is cleared automatically.
     * In the Current Detection Mode, this bit is set to ‘1’ automatically.
     * */
    V9260_ANACTRL0_BGPDN        = 1u<<27,

    /* Set this bit to ‘1’ to enable Voltage Channel ADC. The Bandgap circuit must be enabled before this ADC.
     * Both in the Sleep Mode and in the Current Detection Mode, this bit is cleared automatically*/
    V9260_ANACTRL0_ADCUPDN      = 1u<<26,

    V9260_ANACTRL0_ADCIPDN      = 1u<<24,

    //< enable the function of stimulating the external crystal when it stops running
    V9260_ANACTRL0_XRSTEN       = 1u<<21,

    V9260_ANACTRL0_XTALPLD      = 1u<<20,   //< disable the crystal oscillation circuit.
    /* When a 3.2768-MHz external crystal is used, this bit must be set to ‘1’ to disable the 1/2 divider
     *  in the crystal oscillation circuit.
     * When a 6.5536-MHz crystal is used, this bit must be cleared to enable the 1/2 divider */
    V9260_ANACTRL0_XTAL3P2M     = 1u<<19,

    /* When a 3.2768-MHz crystal is used, it is mandatory to set this bit to ‘1’ to lower the power dissipation
     *  of the crystal oscillation circuit to a half. */
    V9260_ANACTRL0_XTALLP       = 1u<<18,

    /* To select the sampling frequency of the oversampling ADC (ADC clock, ADCCLK).
     * The sampling frequency of ADCs must be a quarter or one eighth of the metering clock (MEACLK) frequency
     *      when the chip operates with full functions in the Metering Mode
     * */
    V9260_ANACTRL0_ADCLKSEL_Pos = 16,
    V9260_ANACTRL0_ADCLKSEL_Msk = 3u << V9260_ANACTRL0_ADCLKSEL_Pos,
    V9260_ANACTRL0_ADCLKSEL_819K= 0,
    V9260_ANACTRL0_ADCLKSEL_409K= 1u << V9260_ANACTRL0_ADCLKSEL_Pos,
    V9260_ANACTRL0_ADCLKSEL_205K= 2u << V9260_ANACTRL0_ADCLKSEL_Pos,
    V9260_ANACTRL0_ADCLKSEL_102K= 3u << V9260_ANACTRL0_ADCLKSEL_Pos,

    /* To finely adjust the temperature coefficient of the Bandgap circuit.
     * In order to obtain the best metering performance and temperature performance during normal metering,
     *      it must be configured according to the calculated result.
     * */
    V9260_ANACTRL0_REST_Pos     = 12,
    V9260_ANACTRL0_REST_Msk     = 7u << V9260_ANACTRL0_REST_Pos,
    V9260_ANACTRL0_REST_10ppm   = 1,
    V9260_ANACTRL0_REST_20ppm   = 2,
    V9260_ANACTRL0_REST_30ppm   = 3,
    V9260_ANACTRL0_REST_m40ppm  = 4,
    V9260_ANACTRL0_REST_m30ppm  = 5,
    V9260_ANACTRL0_REST_m20ppm  = 6,
    V9260_ANACTRL0_REST_m10ppm  = 7,


    /* adjust the global bias current */
    V9260_ANACTRL0_IT_Pos       = 10,
    V9260_ANACTRL0_IT_Msk       = 3u << V9260_ANACTRL0_IT_Pos,
    V9260_ANACTRL0_IT_1D3       = 1u << V9260_ANACTRL0_IT_Pos,  //< -1/3
    V9260_ANACTRL0_IT_2D3       = 2u << V9260_ANACTRL0_IT_Pos,  //< -2/3
    V9260_ANACTRL0_IT_3D4       = 3u << V9260_ANACTRL0_IT_Pos,  //< -3/4

    /* To roughly adjust the temperature coefficient of the Bandgap circuit.
     *  In order to obtain the best metering performance and temperature performance during normal metering,
     *      it must be configured according to the calculated result.
     * */
    V9260_ANACTRL0_RESTL_Pos    = 8,
    V9260_ANACTRL0_RESTL_Msk    = 3u << V9260_ANACTRL0_RESTL_Pos,
    V9260_ANACTRL0_RESTL_70ppm  = 1u << V9260_ANACTRL0_RESTL_Pos,
    V9260_ANACTRL0_RESTL_m140ppm= 2u << V9260_ANACTRL0_RESTL_Pos,
    V9260_ANACTRL0_RESTL_m70ppm = 3u << V9260_ANACTRL0_RESTL_Pos,

    V9260_ANACTRL0_GU       = 1u<<7,    //  set analog PGA gain of analog input of Voltage Channel ADC
    V9260_ANACTRL0_GUx1     = V9260_ANACTRL0_GU,
    V9260_ANACTRL0_GUx4     = 0,

    /* To set analog PGA gain of analog input of Current Channel ADC.
     * The analog PGA gain is determined by the output signal of the sensor.
     * The product of the output signal and PGA gain (Both analog and digital) must be no more than the voltage reference.
     * */
    V9260_ANACTRL0_GI_Pos    = 0,
    V9260_ANACTRL0_GI_Msk    = 3u,
    V9260_ANACTRL0_GIx32     = 0,
    V9260_ANACTRL0_GIx16     = 1,
    V9260_ANACTRL0_GIx4      = 2,
    V9260_ANACTRL0_GIx1      = 3,
};

enum V9260_ANACTRL1Fields{
    V9260_ANACTRL1_DEFAULT  = 0x30000000u,
};

enum V9260_ANACTRL2Fields{

    /* To adjust the 3.2-MHz RC clock cycle. The resolution is 1% per LSB.
     * When these bits are in their default state, no adjustment is applied.
     * From “0b000000” to “0b100000”, the RC clock cycle is decreased by 1% per LSB;
     * from “0b100001” ~ “0b111111”, the RC clock cycle is increased by 1% per LSB.
     * When the chip operates with full functions in the Metering Mode, it is recommended to hold their
     *      default values for the best performance.
     * */
    V9260_ANACTRL2_RCTRIM_Pos = 24,
    V9260_ANACTRL2_RCTRIM_Msk = 0x3fu << V9260_ANACTRL2_RCTRIM_Pos,

    V9260_ANACTRL2_XRSEL1    = 1u<<19,   //< adjust the negative resistance of the crystal oscillator
    V9260_ANACTRL2_XRSEL0    = 1u<<18,   //< adjust the negative resistance of the crystal oscillator

    /* adjust the load capacitance of the crystal oscillator */
    V9260_ANACTRL2_XCSEL_Pos    = 16,
    V9260_ANACTRL2_XCSEL_Msk    = 3u << V9260_ANACTRL2_XCSEL_Pos,
    V9260_ANACTRL2_XCSEL_2PF    = 1u << V9260_ANACTRL2_XCSEL_Pos,
    V9260_ANACTRL2_XCSEL_4PF    = 2u << V9260_ANACTRL2_XCSEL_Pos,
    V9260_ANACTRL2_XCSEL_6PF    = 3u << V9260_ANACTRL2_XCSEL_Pos,

    /* adjust the DVCCLDO output voltage */
    V9260_ANACTRL2_LDOVSEL_Pos  = 12,
    V9260_ANACTRL2_LDOVSEL_Msk  = 7u << V9260_ANACTRL2_LDOVSEL_Pos,
    V9260_ANACTRL2_LDOVSEL_m100MV   = 1u << V9260_ANACTRL2_LDOVSEL_Pos,
    V9260_ANACTRL2_LDOVSEL_100MV    = 2u << V9260_ANACTRL2_LDOVSEL_Pos,
    V9260_ANACTRL2_LDOVSEL_200MV    = 3u << V9260_ANACTRL2_LDOVSEL_Pos,
    V9260_ANACTRL2_LDOVSEL_m400MV   = 4u << V9260_ANACTRL2_LDOVSEL_Pos,
    V9260_ANACTRL2_LDOVSEL_m500MV   = 5u << V9260_ANACTRL2_LDOVSEL_Pos,
    V9260_ANACTRL2_LDOVSEL_m200MV   = 6u << V9260_ANACTRL2_LDOVSEL_Pos,
    V9260_ANACTRL2_LDOVSEL_m300MV   = 7u << V9260_ANACTRL2_LDOVSEL_Pos,

    //< short the amplifier in the voltage channel to obtain the bias voltage of ADC itself
    V9260_ANACTRL2_SHORTU       = 1u<<10,

    //< short the amplifier in the voltage channel to obtain the bias voltage of ADC itself
    V9260_ANACTRL2_SHORTI       = 1u<<8,


};



#endif /* BSP_DEV_METER_EM_VANGO_V9260_H_ */
