/*
 
Copyright (C) 2008 NTT DATA INTELLILINK CORPORATION
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package jp.co.intellilink.hinemos.test.performance;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;

import jp.co.intellilink.hinemos.test.util.Config;
import jp.co.intellilink.hinemos.test.util.EjbConnectionManager;
import jp.co.intellilink.hinemos.test.util.Messages;
import jp.co.intellilink.hinemos.test.util.StringListComparator;
import jp.co.intellilink.hinemos.test.util.WriteCsvFile;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.performance.bean.CollectorProperty;
import com.clustercontrol.performance.ejb.bmp.RecordCollectorData;
import com.clustercontrol.performance.ejb.session.CollectorController;
import com.clustercontrol.repository.ejb.session.RepositoryController;

/**
 * 性能管理機能の実績収集結果をチェックするクラス<br>
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class CheckResultPerformance {
	private Date fromDate;
	private Date toDate;

	// ログ出力
	private static Log log = LogFactory.getLog(CheckResultPerformance.class);

	public CheckResultPerformance(Date fromDate, Date toDate){
		this.fromDate = fromDate;
		this.toDate = toDate;
	}
	
	/**
	 * 性能管理機能の実績収集結果をチェックしCSVフォーマット文字列を作成する。<br>
	 */
	public void checkResult() {
		log.info("[" + Messages.getMsg("CheckResult")+ "] " + Messages.getMsg("TestTool.Performance.Start"));

		//CSV出力配列を用意
		ArrayList<ArrayList<String>> csvOutput = new ArrayList<ArrayList<String>>();

		//収集項目リスト取得
		ArrayList idList = getPerformanceList();

		if(idList instanceof ArrayList){
			Iterator itr = idList.iterator();
			while(itr.hasNext()){
				ArrayList line = (ArrayList)itr.next();

				Object[] args = {(String)line.get(2)};
				log.info("[" + Messages.getMsg("CheckResult")+ "] " + Messages.getMsg("TestTool.CollectorID", args));

				CollectorProperty property = getPerformance((String)line.get(2));
				csvOutput.addAll(createCheckResult(property));
			}

			//収集項目でソート
			Collections.sort(csvOutput, new StringListComparator(2));
			//ファシリティIDでソート
			Collections.sort(csvOutput, new StringListComparator(1));
			//収集IDでソート
			Collections.sort(csvOutput, new StringListComparator(0));
		}

		SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd_HHmmss");

		WriteCsvFile.writeCsvFile(
				Config.getConfig("Check.Output.Dir") + "/PERFORMANCE_" + format.format(new Date()) + ".csv", 
				csvOutput);

		log.info("[" + Messages.getMsg("CheckResult")+ "] " + Messages.getMsg("TestTool.Performance.End"));
	}

	/**
	 * 性能管理機能の実績収集結果をチェックしCSVフォーマット文字列を作成する。<br>
	 * 
	 * @param property 性能管理機能の実績収集情報
	 * @return CSVフォーマット文字列
	 */
	public ArrayList<ArrayList<String>> createCheckResult(CollectorProperty property) {
		ArrayList<ArrayList<String>> csvFormat = new ArrayList<ArrayList<String>>();
		ArrayList<String> facilityIdList = new ArrayList<String>();

		//監視結果のファシリティIDリストを取得
		RecordCollectorData record = property.getCollectorData();
		facilityIdList.addAll(getNodeList(record.getFacilityID()));

		for(String facilityId : facilityIdList){
			CheckRecordData check = 
				new CheckRecordData(property, facilityId, fromDate, toDate);

			SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd_HHmmss");
			String fileName = Config.getConfig("Check.Output.Dir") + "/PERFORMANCE_(" + record.getCollectorID() + "_" + facilityId + ")_" + format.format(new Date()) + ".csv";

			check.write(fileName);

			csvFormat.addAll(check.check());
		}

		return csvFormat;
	}

	/**
	 * ノードのファシリティIDを取得する。<br>
	 */
	protected ArrayList<String> getNodeList(String facilityId) {

		RepositoryController repository = EjbConnectionManager.getConnectionManager().getRepositoryController();

		ArrayList<String> records = new ArrayList<String>();
		try {
			if(repository.isNode(facilityId)){
				records.add(facilityId);
			}
			else{
				records.addAll(repository.getExecTargetFacilityIdList(facilityId));
			}
		} catch (Exception e) {
			log.error("[" + Messages.getMsg("CheckResult")+ "] " + Messages.getMsg("TestTool.ConnectManagerFailed"), e);
			System.exit(14);
		}
		return records;
	}

	/**
	 * 性能管理機能の実績収集情報を取得する。<br>
	 * 
	 * @param collectorId 取得対象の収集ID
	 * @return 性能管理機能の実績収集情報
	 */
	protected CollectorProperty getPerformance(String collectorId) {

		CollectorController perf = EjbConnectionManager.getConnectionManager().getCollectorController();

		CollectorProperty property = null;
		try {
			property = perf.getCollectorProperty(collectorId);
		} catch (Exception e) {
			log.error("[" + Messages.getMsg("CheckResult")+ "] " + Messages.getMsg("TestTool.ConnectManagerFailed"), e);
			System.exit(14);
		}
		return property;
	}

	/**
	 * 収集項目リストを取得する。<br>
	 * 
	 * @return 収集項目一覧
	 */
	protected ArrayList getPerformanceList() {

		CollectorController perf = EjbConnectionManager.getConnectionManager().getCollectorController();

		ArrayList records = null;
		try {
			records = perf.getCollectionList();
		} catch (Exception e) {
			log.error("[" + Messages.getMsg("CheckResult")+ "] " + Messages.getMsg("TestTool.ConnectManagerFailed"), e);
			System.exit(14);
		}
		return records;
	}
}