/*
 
Copyright (C) 2008 NTT DATA INTELLILINK CORPORATION
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package jp.co.intellilink.hinemos.test.monitor;

import java.util.ArrayList;
import java.util.Collection;

import jp.co.intellilink.hinemos.test.util.EjbConnectionManager;
import jp.co.intellilink.hinemos.test.util.Messages;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.MonitorBlockConstant;
import com.clustercontrol.bean.NotifyTypeConstant;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.notify.bean.NotifyRelationInfo;
import com.clustercontrol.repository.ejb.session.RepositoryController;

/**
 * 監視情報を基に試験項目のCSVフォーマット文字列を作成するクラス<br>
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class ExportTestCaseMonitor {
	// ログ出力
	private static Log log = LogFactory.getLog(ExportTestCaseMonitor.class);
	
	/**
	 * 監視情報を基に試験項目のCSVフォーマット文字列を作成する。<br>
	 * 
	 * @param monitorInfo 監視情報
	 * @return CSVフォーマット文字列
	 */
	public ArrayList<ArrayList<String>> createTestCase(MonitorInfo monitorInfo) {
		ArrayList<ArrayList<String>> csvFormat = new ArrayList<ArrayList<String>>();
		ArrayList<String> facilityIdList = new ArrayList<String>();
		
        //監視結果のファシリティIDリストを取得
		facilityIdList.addAll(getFacilityIdList(monitorInfo));
		
		for(String facilityId : facilityIdList){
			ArrayList<String> csvLine = new ArrayList<String>();
			//監視項目ID
			csvLine.add(monitorInfo.getMonitorId());
			//ファシリティID
			csvLine.add(facilityId);
			
			//ステータスチェック
			csvLine.addAll(checkStatus(monitorInfo, facilityId));
			
			//イベントチェック（重要度別）
			csvLine.addAll(checkEvent(monitorInfo, facilityId));
			
			//抑制チェック
			csvLine.addAll(checkNotify(monitorInfo, facilityId));
			
			csvFormat.add(csvLine);
		}
		
		return csvFormat;
	}
	
	/**
	 * 監視情報を基にファシリティIDリストを取得する。<br>
	 * 
	 * @param monitorInfo 監視情報
	 * @return ファシリティIDリスト
	 */
	protected ArrayList<String> getFacilityIdList(MonitorInfo monitorInfo) {
		ArrayList<String> facilityIdList = new ArrayList<String>();
		
        //単位から監視結果のファシリティIDリストを取得
		if(monitorInfo.getMonitorBlock() == MonitorBlockConstant.TYPE_NODE)
			facilityIdList.addAll(getNodeList(monitorInfo.getFacilityId()));
		else
			facilityIdList.add(monitorInfo.getFacilityId());
		
		return facilityIdList;
	}
	
	/**
     * ステータス通知の試験項目のCSVフォーマット文字列を作成する。<br>
     * 
	 * @param monitorInfo 監視情報
	 * @param facilityId ファシリティID
	 * @return CSVフォーマット文字列
	 */
	protected ArrayList<String> checkStatus(MonitorInfo monitorInfo, String facilityId) {
		ArrayList<String> csvFormat = new ArrayList<String>();
		
		String result = "-";
		
		//監視が有効の場合
		if(monitorInfo.getValid() == YesNoConstant.TYPE_YES){
			String notifyId = null;
			if(monitorInfo.getNotifyId() != null){
				Collection<NotifyRelationInfo> notify = monitorInfo.getNotifyId();
				
				NotifyRelationInfo relationInfo;
				for (int i = 0; i < notify.size(); i++) {
					relationInfo = (NotifyRelationInfo)((ArrayList<NotifyRelationInfo>)notify).get(i);
					if(relationInfo.getNotifyType() == NotifyTypeConstant.TYPE_STATUS){
						notifyId = relationInfo.getNotifyId();
						break;
					}
				}
			}
			
			if(notifyId != null){
				result = "";
			}
		}
		
		csvFormat.add(result);
		
		return csvFormat;
	}
	
	/**
     * イベント通知の試験項目のCSVフォーマット文字列を作成する。<br>
     * 
	 * @param monitorInfo 監視情報
	 * @param facilityId ファシリティID
	 * @return CSVフォーマット文字列
	 */
	protected ArrayList<String> checkEvent(MonitorInfo monitorInfo, String facilityId) {
		ArrayList<String> csvFormat = new ArrayList<String>();

		String info = "-";
		String warning = "-";
		String critical = "-";
		String unknown = "-";
		
		//監視が有効の場合
		if(monitorInfo.getValid() == YesNoConstant.TYPE_YES){
			String notifyId = null;
			if(monitorInfo.getNotifyId() != null){
				Collection<NotifyRelationInfo> notify = monitorInfo.getNotifyId();
				
				NotifyRelationInfo relationInfo;
				for (int i = 0; i < notify.size(); i++) {
					relationInfo = (NotifyRelationInfo)((ArrayList<NotifyRelationInfo>)notify).get(i);
					if(relationInfo.getNotifyType() == NotifyTypeConstant.TYPE_EVENT){
						notifyId = relationInfo.getNotifyId();
						break;
					}
				}
			}
			
			if(notifyId != null){
				info = "";
				warning = "";
				critical = "";
				unknown = "";
			}
		}

		csvFormat.add(info);
		csvFormat.add(warning);
		csvFormat.add(critical);
		csvFormat.add(unknown);
		
		return csvFormat;
	}
    
	/**
     * 通知抑制の試験項目のCSVフォーマット文字列を作成する。<br>
     * 
	 * @param monitorInfo 監視情報
	 * @param facilityId ファシリティID
	 * @return CSVフォーマット文字列
	 */
	protected ArrayList<String> checkNotify(MonitorInfo monitorInfo, String facilityId) {
		ArrayList<String> csvFormat = new ArrayList<String>();
		
		String status = "-";
		String event = "-";
		String mail = "-";
		String job = "-";
		String syslog = "-";
		
		//監視が有効の場合
		if(monitorInfo.getValid() == YesNoConstant.TYPE_YES){
			if(monitorInfo.getNotifyId() != null){
				Collection<NotifyRelationInfo> notify = monitorInfo.getNotifyId();
				
				NotifyRelationInfo relationInfo;
				for (int i = 0; i < notify.size(); i++) {
					relationInfo = (NotifyRelationInfo)((ArrayList<NotifyRelationInfo>)notify).get(i);
					if(relationInfo.getNotifyType() == NotifyTypeConstant.TYPE_STATUS)
						status = "";
					else if(relationInfo.getNotifyType() == NotifyTypeConstant.TYPE_EVENT)
						event = "";
					else if(relationInfo.getNotifyType() == NotifyTypeConstant.TYPE_MAIL)
						mail = "";
					else if(relationInfo.getNotifyType() == NotifyTypeConstant.TYPE_JOB)
						job = "";
					else if(relationInfo.getNotifyType() == NotifyTypeConstant.TYPE_LOG_ESCALATE)
						syslog = "";
				}
			}
		}
		
		csvFormat.add(status);
		csvFormat.add(event);
		csvFormat.add(mail);
		csvFormat.add(job);
		csvFormat.add(syslog);
		
		return csvFormat;
	}

	/**
	 * ノードのファシリティIDを取得する。<br>
	 */
    protected ArrayList<String> getNodeList(String facilityId) {

		RepositoryController repository = EjbConnectionManager.getConnectionManager().getRepositoryController();

		ArrayList<String> records = new ArrayList<String>();
		try {
			if(repository.isNode(facilityId)){
				records.add(facilityId);
			}
			else{
				records.addAll(repository.getExecTargetFacilityIdList(facilityId));
			}
		} catch (Exception e) {
			log.error("[" + Messages.getMsg("ExportTestCase")+ "] " + Messages.getMsg("TestTool.ConnectManagerFailed"), e);
			System.exit(14);
		}
		return records;
	}
}