/*
 
Copyright (C) 2008 NTT DATA INTELLILINK CORPORATION
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package jp.co.intellilink.hinemos.test.monitor;

import java.rmi.RemoteException;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;

import jp.co.intellilink.hinemos.test.util.EjbConnectionManager;
import jp.co.intellilink.hinemos.test.util.Messages;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.ExclusionConstant;
import com.clustercontrol.bean.MonitorBlockConstant;
import com.clustercontrol.bean.NotifyTypeConstant;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.bean.StatusExpirationConstant;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.notify.bean.NotifyInfo;
import com.clustercontrol.notify.bean.NotifyRelationInfo;
import com.clustercontrol.notify.ejb.entity.NotifyActionInhibitionInfo;
import com.clustercontrol.notify.ejb.session.NotifyController;
import com.clustercontrol.notify.monitor.ejb.entity.EventLog;
import com.clustercontrol.notify.monitor.ejb.entity.StatusInfo;
import com.clustercontrol.repository.ejb.session.RepositoryController;

/**
 * 監視結果をチェックするクラス<br>
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class CheckResultMonitor {
	private Date fromDate;
	
	// ログ出力
	private static Log log = LogFactory.getLog(CheckResultMonitor.class);
	
	public CheckResultMonitor(Date fromDate){
		this.fromDate = fromDate;
	}
	
	/**
	 * 監視結果をチェックしCSVフォーマット文字列を作成する。<br>
	 * 
	 * @param monitorInfo 監視情報
	 * @return CSVフォーマット文字列
	 */
	public ArrayList<ArrayList<String>> createCheckResult(MonitorInfo monitorInfo) {
		ArrayList<ArrayList<String>> csvFormat = new ArrayList<ArrayList<String>>();
		ArrayList<String> facilityIdList = new ArrayList<String>();
		
        //監視結果のファシリティIDリストを取得
		facilityIdList.addAll(getFacilityIdList(monitorInfo));
		
		for(String facilityId : facilityIdList){
			ArrayList<String> csvLine = new ArrayList<String>();
			//監視項目ID
			csvLine.add(monitorInfo.getMonitorId());
			//ファシリティID
			csvLine.add(facilityId);
			
			//ステータスチェック
			csvLine.addAll(checkStatus(monitorInfo, facilityId));
			
			//イベントチェック（重要度別）
			csvLine.addAll(checkEvent(monitorInfo, facilityId));
			
			//抑制チェック
			csvLine.addAll(checkNotify(monitorInfo, facilityId));
			
			csvFormat.add(csvLine);
		}
		
		return csvFormat;
	}
	
	/**
	 * 監視情報を基にファシリティIDリストを取得する。<br>
	 * 
	 * @param monitorInfo 監視情報
	 * @return ファシリティIDリスト
	 */
	protected ArrayList<String> getFacilityIdList(MonitorInfo monitorInfo) {
		ArrayList<String> facilityIdList = new ArrayList<String>();
		
        //単位から監視結果のファシリティIDリストを取得
		if(monitorInfo.getMonitorBlock() == MonitorBlockConstant.TYPE_NODE)
			facilityIdList.addAll(getNodeList(monitorInfo.getFacilityId()));
		else
			facilityIdList.add(monitorInfo.getFacilityId());
		
		return facilityIdList;
	}
	
	/**
     * ステータス出力結果をチェックする。<br>
     * 
	 * @param monitorInfo 監視情報
	 * @param facilityId ファシリティID
	 * @return CSVフォーマット文字列
	 */
	protected ArrayList<String> checkStatus(MonitorInfo monitorInfo, String facilityId) {
		ArrayList<String> csvFormat = new ArrayList<String>();
		
		String result = "-";
		
		//監視が無効の場合
		if(monitorInfo.getValid() == YesNoConstant.TYPE_NO){
			csvFormat.add(result);
			return csvFormat;
		}
		
		String notifyId = null;
		if(monitorInfo.getNotifyId() != null){
			Collection<NotifyRelationInfo> notify = monitorInfo.getNotifyId();
			
			NotifyRelationInfo relationInfo;
			for (int i = 0; i < notify.size(); i++) {
				relationInfo = (NotifyRelationInfo)((ArrayList<NotifyRelationInfo>)notify).get(i);
				if(relationInfo.getNotifyType() == NotifyTypeConstant.TYPE_STATUS){
					notifyId = relationInfo.getNotifyId();
					break;
				}
			}
		}
		
		if(notifyId != null){
			result = "NG";
			
			Collection ct = 
				EjbConnectionManager.getConnectionManager().getStatus(facilityId);
			Iterator itr = ct.iterator();
			while (itr.hasNext()) {
				StatusInfo status = (StatusInfo) itr.next();
				try {
					if(status.getMonitorId().compareTo(monitorInfo.getMonitorId()) == 0){
						result = DateFormat.getDateTimeInstance().format(status.getGenerationDate());
						if(status.getExpirationFlg() != StatusExpirationConstant.TYPE_EXPIRATION)
							result = DateFormat.getDateTimeInstance().format(status.getGenerationDate());
						break;
					}
				} catch (RemoteException e) {
					log.error("[" + Messages.getMsg("CheckResult")+ "] " + Messages.getMsg("TestTool.ConnectManagerFailed"), e);
					System.exit(14);
				}
			}
		}
		else{
			result = "-";
		}

		csvFormat.add(result);
		
		return csvFormat;
	}
	
	/**
     * イベント出力結果をチェックする。<br>
     * 
	 * @param monitorInfo 監視情報
	 * @param facilityId ファシリティID
	 * @return CSVフォーマット文字列
	 */
	protected ArrayList<String> checkEvent(MonitorInfo monitorInfo, String facilityId) {
		ArrayList<String> csvFormat = new ArrayList<String>();

		String info = "-";
		String warning = "-";
		String critical = "-";
		String unknown = "-";
		
		//監視が無効の場合
		if(monitorInfo.getValid() == YesNoConstant.TYPE_NO){
			csvFormat.add(info);
			csvFormat.add(warning);
			csvFormat.add(critical);
			csvFormat.add(unknown);
			return csvFormat;
		}
		
		String notifyId = null;
		if(monitorInfo.getNotifyId() != null){
			Collection<NotifyRelationInfo> notify = monitorInfo.getNotifyId();
			
			NotifyRelationInfo relationInfo;
			for (int i = 0; i < notify.size(); i++) {
				relationInfo = (NotifyRelationInfo)((ArrayList<NotifyRelationInfo>)notify).get(i);
				if(relationInfo.getNotifyType() == NotifyTypeConstant.TYPE_EVENT){
					notifyId = relationInfo.getNotifyId();
					break;
				}
			}
		}
		
		if(notifyId != null){
			//通知
			info = checkEventByPriority(monitorInfo, facilityId, PriorityConstant.TYPE_INFO, fromDate);
			//警告
			warning = checkEventByPriority(monitorInfo, facilityId, PriorityConstant.TYPE_WARNING, fromDate);
			//危険
			critical = checkEventByPriority(monitorInfo, facilityId, PriorityConstant.TYPE_CRITICAL, fromDate);
			//不明
			unknown = checkEventByPriority(monitorInfo, facilityId, PriorityConstant.TYPE_UNKNOWN, fromDate);
		}

		csvFormat.add(info);
		csvFormat.add(warning);
		csvFormat.add(critical);
		csvFormat.add(unknown);
		
		return csvFormat;
	}
	
	/**
     * イベント出力結果をチェックする。<br>
     * 
	 * @param monitorInfo 監視情報
	 * @param facilityId ファシリティID
	 * @param priority 重要度
	 * @param fromDate 出力日時（開始）
	 * @return CSVフォーマット文字列
	 */
	protected String checkEventByPriority(MonitorInfo monitorInfo, String facilityId, int priority, Date fromDate) {
		String result = "";

		Collection ct = 
			EjbConnectionManager.getConnectionManager().getEventByPriority(
					facilityId, priority, fromDate);
		Iterator itr = ct.iterator();
		while (itr.hasNext()) {
			EventLog event = (EventLog) itr.next();
			try {
				String monitorId = event.getMonitorId();
				if(monitorId.compareTo(monitorInfo.getMonitorId()) == 0){
					result = DateFormat.getDateTimeInstance().format(event.getGenerationDate());
					break;
				}
			} catch (RemoteException e) {
				log.error("[" + Messages.getMsg("CheckResult")+ "] " + Messages.getMsg("TestTool.ConnectManagerFailed"), e);
				System.exit(14);
			}
		}
		
		return result;
	}
    
	/**
     * 通知抑制の結果をチェックする。<br>
     * 
	 * @param monitorInfo 監視情報
	 * @param facilityId ファシリティID
	 * @return CSVフォーマット文字列
	 */
	protected ArrayList<String> checkNotify(MonitorInfo monitorInfo, String facilityId) {
		ArrayList<String> csvFormat = new ArrayList<String>();
		
		String status = "-";
		String event = "-";
		String mail = "-";
		String job = "-";
		String syslog = "-";
		
		//監視が無効の場合
		if(monitorInfo.getValid() == YesNoConstant.TYPE_NO){
			csvFormat.add(status);
			csvFormat.add(event);
			csvFormat.add(mail);
			csvFormat.add(job);
			csvFormat.add(syslog);
			return csvFormat;
		}
		
		if(monitorInfo.getNotifyId() != null){
			Collection<NotifyRelationInfo> notify = monitorInfo.getNotifyId();
			
			NotifyRelationInfo relationInfo;
			for (int i = 0; i < notify.size(); i++) {
				relationInfo = (NotifyRelationInfo)((ArrayList<NotifyRelationInfo>)notify).get(i);
				if(relationInfo.getNotifyType() == NotifyTypeConstant.TYPE_STATUS)
					status = checkNotifyInhibit(monitorInfo, relationInfo.getNotifyId(), facilityId);
				else if(relationInfo.getNotifyType() == NotifyTypeConstant.TYPE_EVENT)
					event = checkNotifyInhibit(monitorInfo, relationInfo.getNotifyId(), facilityId);
				else if(relationInfo.getNotifyType() == NotifyTypeConstant.TYPE_MAIL)
					mail = checkNotifyInhibit(monitorInfo, relationInfo.getNotifyId(), facilityId);
				else if(relationInfo.getNotifyType() == NotifyTypeConstant.TYPE_JOB)
					job = checkNotifyInhibit(monitorInfo, relationInfo.getNotifyId(), facilityId);
				else if(relationInfo.getNotifyType() == NotifyTypeConstant.TYPE_LOG_ESCALATE)
					syslog = checkNotifyInhibit(monitorInfo, relationInfo.getNotifyId(), facilityId);
			}
		}
		
		csvFormat.add(status);
		csvFormat.add(event);
		csvFormat.add(mail);
		csvFormat.add(job);
		csvFormat.add(syslog);
		
		return csvFormat;
	}

	/**
     * 通知IDの抑制の結果をチェックする<br>
     * 
	 * @param monitorInfo 監視情報
	 * @param notifyId 通知ID
	 * @param facilityId ファシリティID
	 * @return 結果文字列
	 */
	protected String checkNotifyInhibit(MonitorInfo monitorInfo, String notifyId, String facilityId) {

		String result = "NG";
		
		NotifyInfo notify = getNotify(notifyId);
		if(notify.getInhibitionFlg().intValue() != ExclusionConstant.TYPE_NO){
		
			NotifyActionInhibitionInfo inhibitInfo = 
				EjbConnectionManager.getConnectionManager().getNotifyActionInhibition(
						monitorInfo.getNotifyGroupId(), notifyId, facilityId);
			try {
				if(inhibitInfo instanceof NotifyActionInhibitionInfo && inhibitInfo.getTotalCount() > 0){
					result = "OK";
				}
			} catch (RemoteException e) {
				log.error("[" + Messages.getMsg("CheckResult")+ "] " + Messages.getMsg("TestTool.ConnectManagerFailed"), e);
				System.exit(14);
			}
		}
		else
			result = "OK";
		
		return result;
	}
	
    /**
     * 通知情報を取得する。<br>
     * 
     * @param notifyId 取得対象の通知ID
     * @return 通知情報
     */
    protected NotifyInfo getNotify(String notifyId) {

        NotifyController notify = EjbConnectionManager.getConnectionManager()
                .getNotifyController();

        NotifyInfo notifyInfo = null;
        try {
        	notifyInfo = notify.getNotify(notifyId);
        } catch (Exception e) {
			log.error("[" + Messages.getMsg("CheckResult")+ "] " + Messages.getMsg("TestTool.ConnectManagerFailed"), e);
			System.exit(14);
        }
        return notifyInfo;
    }

	/**
	 * ノードのファシリティIDを取得する。<br>
	 */
    protected ArrayList<String> getNodeList(String facilityId) {

		RepositoryController repository = EjbConnectionManager.getConnectionManager().getRepositoryController();

		ArrayList<String> records = new ArrayList<String>();
		try {
			if(repository.isNode(facilityId)){
				records.add(facilityId);
			}
			else{
				records.addAll(repository.getExecTargetFacilityIdList(facilityId));
			}
		} catch (Exception e) {
			log.error("[" + Messages.getMsg("CheckResult")+ "] " + Messages.getMsg("TestTool.ConnectManagerFailed"), e);
			System.exit(14);
		}
		return records;
	}
}