/*
 
Copyright (C) 2008 NTT DATA INTELLILINK CORPORATION
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package jp.co.intellilink.hinemos.test.maintenance;

import java.rmi.AccessException;
import java.rmi.RemoteException;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.naming.NamingException;

import jp.co.intellilink.hinemos.test.util.Config;
import jp.co.intellilink.hinemos.test.util.EjbConnectionManager;
import jp.co.intellilink.hinemos.test.util.Messages;
import jp.co.intellilink.hinemos.test.util.StringListComparator;
import jp.co.intellilink.hinemos.test.util.WriteCsvFile;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.ExclusionConstant;
import com.clustercontrol.bean.NotifyTypeConstant;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.bean.StatusExpirationConstant;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.maintenance.bean.MaintenanceInfoDataImp;
import com.clustercontrol.maintenance.ejb.session.MaintenanceController;
import com.clustercontrol.notify.bean.NotifyInfo;
import com.clustercontrol.notify.bean.NotifyRelationInfo;
import com.clustercontrol.notify.ejb.entity.NotifyActionInhibitionInfo;
import com.clustercontrol.notify.ejb.session.NotifyController;
import com.clustercontrol.notify.monitor.ejb.entity.EventLog;
import com.clustercontrol.notify.monitor.ejb.entity.StatusInfo;

/**
 * メンテナンス結果をチェックするクラス<br>
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class CheckResultMaintenamce {
	private Date fromDate;
	
	// ログ出力
	private static Log log = LogFactory.getLog(CheckResultMaintenamce.class);
	
	public CheckResultMaintenamce(Date fromDate){
		this.fromDate = fromDate;
	}
	
    /**
     * メンテナンス結果をチェックしCSVフォーマット文字列を作成する。<br>
     */
    public void checkResult() {
    	log.info("[" + Messages.getMsg("CheckResult")+ "] " + Messages.getMsg("TestTool.Maintenance.Start"));
    	
    	//CSV出力配列を用意
    	ArrayList<ArrayList<String>> csvOutput = new ArrayList<ArrayList<String>>();
    	
    	//メンテナンスIDリスト取得
    	ArrayList idList = getMaintenanceIdList();

    	if(idList instanceof ArrayList){
			Iterator itr = idList.iterator();
			while(itr.hasNext()){
				ArrayList line = (ArrayList)itr.next();
				
				Object[] args = {(String)line.get(0)};
				log.info("[" + Messages.getMsg("CheckResult")+ "] " + Messages.getMsg("TestTool.MaintenanceID", args));
				
				MaintenanceInfoDataImp info = getMaintenance((String)line.get(0));
				csvOutput.addAll(createCheckResult(info));
			}
			
			//メンテナンスIDでソート
			Collections.sort(csvOutput, new StringListComparator(0));
    	}
    	
		SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd_HHmmss");
		
		WriteCsvFile.writeCsvFile(
				Config.getConfig("Check.Output.Dir") + "/MAINTENANCE_" + format.format(new Date()) + ".csv", 
				csvOutput);
		
		log.info("[" + Messages.getMsg("CheckResult")+ "] " + Messages.getMsg("TestTool.Maintenance.End"));
    }
    
	/**
	 * メンテナンス結果をチェックしCSVフォーマット文字列を作成する。<br>
	 * 
	 * @param maintenanceInfo メンテナンス情報
	 * @return CSVフォーマット文字列
	 */
	public ArrayList<ArrayList<String>> createCheckResult(MaintenanceInfoDataImp maintenanceInfo) {
		ArrayList<ArrayList<String>> csvFormat = new ArrayList<ArrayList<String>>();

		ArrayList<String> csvLine = new ArrayList<String>();
		//メンテナンスID
		csvLine.add(maintenanceInfo.getMaintenance_id());
		
		//ステータスチェック
		csvLine.addAll(checkStatus(maintenanceInfo, "INTERNAL"));
		
		//イベントチェック（重要度別）
		csvLine.addAll(checkEvent(maintenanceInfo, "INTERNAL"));
		
		//抑制チェック
		csvLine.addAll(checkNotify(maintenanceInfo, "INTERNAL"));
		
		csvFormat.add(csvLine);

		return csvFormat;
	}
	
	/**
     * ステータス出力結果をチェックする。<br>
     * 
	 * @param maintenanceInfo メンテナンス情報
	 * @param facilityId ファシリティID
	 * @return CSVフォーマット文字列
	 */
	protected ArrayList<String> checkStatus(MaintenanceInfoDataImp maintenanceInfo, String facilityId) {
		ArrayList<String> csvFormat = new ArrayList<String>();
		
		String result = "-";
		
		//監視が無効の場合
		if(maintenanceInfo.getValid_flg() == YesNoConstant.TYPE_NO){
			csvFormat.add(result);
			return csvFormat;
		}
		
		String notifyId = null;
		if(maintenanceInfo.getNotifyId() != null){
			Collection<NotifyRelationInfo> notify = maintenanceInfo.getNotifyId();
			
			NotifyRelationInfo relationInfo;
			for (int i = 0; i < notify.size(); i++) {
				relationInfo = (NotifyRelationInfo)((ArrayList<NotifyRelationInfo>)notify).get(i);
				if(relationInfo.getNotifyType() == NotifyTypeConstant.TYPE_STATUS){
					notifyId = relationInfo.getNotifyId();
					break;
				}
			}
		}
		
		if(notifyId != null){
			result = "NG";
			
			Collection ct = 
				EjbConnectionManager.getConnectionManager().getStatus(facilityId);
			Iterator itr = ct.iterator();
			while (itr.hasNext()) {
				StatusInfo status = (StatusInfo) itr.next();
				try {
					if(status.getMonitorId().compareTo(maintenanceInfo.getMaintenance_id()) == 0){
						result = DateFormat.getDateTimeInstance().format(status.getGenerationDate());
						if(status.getExpirationFlg() != StatusExpirationConstant.TYPE_EXPIRATION)
							result = DateFormat.getDateTimeInstance().format(status.getGenerationDate());
						break;
					}
				} catch (RemoteException e) {
					log.error("[" + Messages.getMsg("CheckResult")+ "] " + Messages.getMsg("TestTool.ConnectManagerFailed"), e);
					System.exit(14);
				}
			}
		}
		else{
			result = "-";
		}

		csvFormat.add(result);
		
		return csvFormat;
	}
	
	/**
     * イベント出力結果をチェックする。<br>
     * 
	 * @param maintenanceInfo メンテナンス情報
	 * @param facilityId ファシリティID
	 * @return CSVフォーマット文字列
	 */
	protected ArrayList<String> checkEvent(MaintenanceInfoDataImp maintenanceInfo, String facilityId) {
		ArrayList<String> csvFormat = new ArrayList<String>();

		String info = "-";
		String warning = "-";
		String critical = "-";
		String unknown = "-";
		
		//監視が無効の場合
		if(maintenanceInfo.getValid_flg() == YesNoConstant.TYPE_NO){
			csvFormat.add(info);
			csvFormat.add(warning);
			csvFormat.add(critical);
			csvFormat.add(unknown);
			return csvFormat;
		}
		
		String notifyId = null;
		if(maintenanceInfo.getNotifyId() != null){
			Collection<NotifyRelationInfo> notify = maintenanceInfo.getNotifyId();
			
			NotifyRelationInfo relationInfo;
			for (int i = 0; i < notify.size(); i++) {
				relationInfo = (NotifyRelationInfo)((ArrayList<NotifyRelationInfo>)notify).get(i);
				if(relationInfo.getNotifyType() == NotifyTypeConstant.TYPE_EVENT){
					notifyId = relationInfo.getNotifyId();
					break;
				}
			}
		}
		
		if(notifyId != null){
			//通知
			info = checkEventByPriority(maintenanceInfo, facilityId, PriorityConstant.TYPE_INFO, fromDate);
			//警告
			warning = checkEventByPriority(maintenanceInfo, facilityId, PriorityConstant.TYPE_WARNING, fromDate);
			//危険
			critical = checkEventByPriority(maintenanceInfo, facilityId, PriorityConstant.TYPE_CRITICAL, fromDate);
			//不明
			unknown = checkEventByPriority(maintenanceInfo, facilityId, PriorityConstant.TYPE_UNKNOWN, fromDate);
		}

		csvFormat.add(info);
		csvFormat.add(warning);
		csvFormat.add(critical);
		csvFormat.add(unknown);
		
		return csvFormat;
	}
	
	/**
     * イベント出力結果をチェックする。<br>
     * 
	 * @param maintenanceInfo メンテナンス情報
	 * @param facilityId ファシリティID
	 * @param priority 重要度
	 * @param fromDate 出力日時（開始）
	 * @return CSVフォーマット文字列
	 */
	protected String checkEventByPriority(MaintenanceInfoDataImp maintenanceInfo, String facilityId, int priority, Date fromDate) {
		String result = "";

		Collection ct = 
			EjbConnectionManager.getConnectionManager().getEventByPriority(
					facilityId, priority, fromDate);
		Iterator itr = ct.iterator();
		while (itr.hasNext()) {
			EventLog event = (EventLog) itr.next();
			try {
				String monitorId = event.getMonitorId();
				if(monitorId.compareTo(maintenanceInfo.getMaintenance_id()) == 0){
					result = DateFormat.getDateTimeInstance().format(event.getGenerationDate());
					break;
				}
			} catch (RemoteException e) {
				log.error("[" + Messages.getMsg("CheckResult")+ "] " + Messages.getMsg("TestTool.ConnectManagerFailed"), e);
				System.exit(14);
			}
		}
		
		return result;
	}
    
	/**
     * 通知抑制の結果をチェックする。<br>
     * 
	 * @param maintenanceInfo メンテナンス情報
	 * @param facilityId ファシリティID
	 * @return CSVフォーマット文字列
	 */
	protected ArrayList<String> checkNotify(MaintenanceInfoDataImp maintenanceInfo, String facilityId) {
		ArrayList<String> csvFormat = new ArrayList<String>();
		
		String status = "-";
		String event = "-";
		String mail = "-";
		String job = "-";
		String syslog = "-";
		
		//監視が無効の場合
		if(maintenanceInfo.getValid_flg() == YesNoConstant.TYPE_NO){
			csvFormat.add(status);
			csvFormat.add(event);
			csvFormat.add(mail);
			csvFormat.add(job);
			csvFormat.add(syslog);
			return csvFormat;
		}
		
		if(maintenanceInfo.getNotifyId() != null){
			Collection<NotifyRelationInfo> notify = maintenanceInfo.getNotifyId();
			
			NotifyRelationInfo relationInfo;
			for (int i = 0; i < notify.size(); i++) {
				relationInfo = (NotifyRelationInfo)((ArrayList<NotifyRelationInfo>)notify).get(i);
				if(relationInfo.getNotifyType() == NotifyTypeConstant.TYPE_STATUS)
					status = checkNotifyInhibit(maintenanceInfo, relationInfo.getNotifyId(), facilityId);
				else if(relationInfo.getNotifyType() == NotifyTypeConstant.TYPE_EVENT)
					event = checkNotifyInhibit(maintenanceInfo, relationInfo.getNotifyId(), facilityId);
				else if(relationInfo.getNotifyType() == NotifyTypeConstant.TYPE_MAIL)
					mail = checkNotifyInhibit(maintenanceInfo, relationInfo.getNotifyId(), facilityId);
				else if(relationInfo.getNotifyType() == NotifyTypeConstant.TYPE_JOB)
					job = checkNotifyInhibit(maintenanceInfo, relationInfo.getNotifyId(), facilityId);
				else if(relationInfo.getNotifyType() == NotifyTypeConstant.TYPE_LOG_ESCALATE)
					syslog = checkNotifyInhibit(maintenanceInfo, relationInfo.getNotifyId(), facilityId);
			}
		}
		
		csvFormat.add(status);
		csvFormat.add(event);
		csvFormat.add(mail);
		csvFormat.add(job);
		csvFormat.add(syslog);
		
		return csvFormat;
	}

	/**
     * 通知IDの抑制の結果をチェックする<br>
     * 
	 * @param maintenanceInfo メンテナンス情報
	 * @param notifyId 通知ID
	 * @param facilityId ファシリティID
	 * @return 結果文字列
	 */
	protected String checkNotifyInhibit(MaintenanceInfoDataImp maintenanceInfo, String notifyId, String facilityId) {

		String result = "NG";
		
		NotifyInfo notify = getNotify(notifyId);
		if(notify.getInhibitionFlg().intValue() != ExclusionConstant.TYPE_NO){
		
			NotifyActionInhibitionInfo inhibitInfo = 
				EjbConnectionManager.getConnectionManager().getNotifyActionInhibition(
						maintenanceInfo.getNotifyGroupId(), notifyId, facilityId);
			try {
				if(inhibitInfo instanceof NotifyActionInhibitionInfo && inhibitInfo.getTotalCount() > 0){
					result = "OK";
				}
			} catch (RemoteException e) {
				log.error("[" + Messages.getMsg("CheckResult")+ "] " + Messages.getMsg("TestTool.ConnectManagerFailed"), e);
				System.exit(14);
			}
		}
		else
			result = "OK";
		
		return result;
	}
	
    /**
     * 通知情報を取得する。<br>
     * 
     * @param notifyId 取得対象の通知ID
     * @return 通知情報
     */
    protected NotifyInfo getNotify(String notifyId) {

        NotifyController notify = EjbConnectionManager.getConnectionManager()
                .getNotifyController();

        NotifyInfo notifyInfo = null;
        try {
        	notifyInfo = notify.getNotify(notifyId);
        } catch (Exception e) {
			log.error("[" + Messages.getMsg("CheckResult")+ "] " + Messages.getMsg("TestTool.ConnectManagerFailed"), e);
			System.exit(14);
        }
        return notifyInfo;
    }
	
    /**
     * メンテナンス情報を取得する。<br>
     * 
     * @param maintenanceId 取得対象のメンテナンスID
     * @return メンテナンス情報
     */
    protected MaintenanceInfoDataImp getMaintenance(String maintenanceId) {

        MaintenanceController maintenance = EjbConnectionManager.getConnectionManager().getMaintenanceController();

        MaintenanceInfoDataImp info = null;
        try {
        	info = maintenance.getMaintenanceInfo(maintenanceId);
        } catch (RemoteException e) {
			if(e instanceof AccessException){
			}
        } catch (FinderException e) {
        } catch (NamingException e) {
        } catch (CreateException e) {
		}
        return info;
    }
    
	/**
	 * スケジュール情報を取得する。<br>
	 * 
	 * @param maintenanceId 取得対象のメンテナンスID
	 * @return スケジュール情報
	 */
    protected ArrayList getSchedule(String maintenanceId) {
		
	   	MaintenanceController maintenance = EjbConnectionManager.getConnectionManager().getMaintenanceController();
	   	
	   	ArrayList info = null;
		try {
			info = maintenance.getSchedule(maintenanceId);
        } catch (Exception e) {
			log.error("[" + Messages.getMsg("CheckResult")+ "] " + Messages.getMsg("TestTool.ConnectManagerFailed"), e);
			System.exit(14);
        }
		return info;
	}
    
    /**
     * メンテナンスIDリストを取得する。<br>
     * 
     * @return メンテナンスID一覧
     */
    protected ArrayList getMaintenanceIdList() {

        MaintenanceController maintenance = EjbConnectionManager.getConnectionManager().getMaintenanceController();

        ArrayList records = null;
        try {
            records = maintenance.getMaintenanceList();
        } catch (Exception e) {
			log.error("[" + Messages.getMsg("CheckResult")+ "] " + Messages.getMsg("TestTool.ConnectManagerFailed"), e);
			System.exit(14);
        }
        return records;
    }
}