/*

Copyright (C) 2009 NTT DATA INTELLILINK CORPORATION

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.

 */

package jp.co.intellilink.hinemos.importtool.conf.monitor;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.ejb.DuplicateKeyException;

import jp.co.intellilink.hinemos.importtool.util.CheckString;
import jp.co.intellilink.hinemos.importtool.util.Config;
import jp.co.intellilink.hinemos.importtool.util.EjbConnectionManager;
import jp.co.intellilink.hinemos.importtool.util.Messages;
import jp.co.intellilink.hinemos.importtool.util.ReadCsvFile;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.DataRangeConstant;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.monitor.run.bean.MonitorTabelDefine;
import com.clustercontrol.snmptrap.bean.MonitorSnmpTrapInfo;
import com.clustercontrol.snmptrap.bean.MonitorSnmpTrapOidInfo;
import com.clustercontrol.snmptrap.bean.SnmpTrapMasterInfo;
import com.clustercontrol.snmptrap.ejb.session.MonitorSnmpTrapController;

/**
 * SNMPTRAP監視情報をインポートするクラス<br>
 * 
 * @version 1.1.0
 * @since 1.1.0
 */
public class ImportMonitorSnmpTrap extends ImportMonitor {

	// ログ出力
	private static Log log = LogFactory.getLog(ImportMonitorSnmpTrap.class);

	@SuppressWarnings("unchecked")
	private ArrayList monitorList = null;

	private ArrayList<ArrayList<String>> monitorInfos = null;

	/**
	 * SNMPTRAP監視情報をインポートする。<br>
	 */
	public void importSnmpTrap() {
		log.info(Messages.getMsg("ImportTool.SnmpTrap.Start", new String[]{Messages.getMsg("ImportTool.Import")}));

		//SNMPTRAP監視情報のCSVファイルを読み込む
		monitorInfos = ReadCsvFile.readCsvFile(Config.getConfig("Input.Dir.Conf") + "/TRAP.csv");

		//監視項目リスト取得
		monitorList = getMonitorList();

		for(ArrayList<String> info : monitorInfos){
			MonitorInfo monitorInfo = createMonitorInfo(info);
			registerMonitorInfo(monitorInfo);
		}

		log.info(Messages.getMsg("ImportTool.SnmpTrap.End", new String[]{Messages.getMsg("ImportTool.Import")}));
	}

	/**
	 * 監視情報オブジェクトを作成する。<br>
	 * 
	 * @see jp.co.intellilink.hinemos.importtool.conf.monitor.ImportMonitor#createMonitorInfo(java.util.ArrayList)
	 */
	@Override
	protected MonitorInfo createMonitorInfo(ArrayList<String> info){
		MonitorSnmpTrapInfo monitorInfo = new MonitorSnmpTrapInfo();

		//監視基本情報
		createMonitorBase(info, monitorInfo);

		//監視情報
		createMonitorCheck(info.subList(3, 13), monitorInfo);

		//監視通知情報
		createMonitorNotify(info.subList(13, 18), monitorInfo);

		//アプリケーション
		createMonitorApplication(info.subList(18, 19), monitorInfo);

		//有効/無効
		createMonitorVaild(info.subList(19, 20), monitorInfo);

		return monitorInfo;
	}

	/**
	 * SNMPTRAP監視情報を設定する。<br>
	 * 
	 * @param info 監視情報配列
	 * @param monitorInfo 監視情報オブジェクト
	 * 
	 * @see jp.co.intellilink.hinemos.importtool.conf.monitor.ImportMonitor#createMonitorCheck(java.util.List, com.clustercontrol.monitor.run.bean.MonitorInfo)
	 */
	@Override
	protected void createMonitorCheck(List<String> info, MonitorInfo monitorInfo) {
		MonitorSnmpTrapInfo trapInfo = (MonitorSnmpTrapInfo)monitorInfo;

		//コミュニティ名
		String communityName = CheckString.checkLength(info.get(0), DataRangeConstant.VARCHAR_64);
		if(!communityName.equals(info.get(0))){
			String[] args = { com.clustercontrol.util.Messages.getString("community.name"), "64" };
			log.warn(Messages.getMsg("ImportTool.Common.2", args));
		}
		trapInfo.setCommunityName(communityName);
		//監視対象
		if(info.get(1).equals(com.clustercontrol.util.Messages.getString("OidListComposite.SpecificOid")))
			trapInfo.setCheckMode(MonitorSnmpTrapInfo.SPECIFIC_OID);
		else if(info.get(1).equals(com.clustercontrol.util.Messages.getString("OidListComposite.AllMaterOid")))
			trapInfo.setCheckMode(MonitorSnmpTrapInfo.ALL_OID);
		else if(info.get(1).equals(com.clustercontrol.util.Messages.getString("OidListComposite.NoMasterOid")))
			trapInfo.setCheckMode(MonitorSnmpTrapInfo.UNREGISTERED_OID);

		if(trapInfo.getCheckMode() == MonitorSnmpTrapInfo.SPECIFIC_OID){
			ArrayList<MonitorSnmpTrapOidInfo> oidInfos = new ArrayList<MonitorSnmpTrapOidInfo>();
			MonitorSnmpTrapOidInfo oidInfo = new MonitorSnmpTrapOidInfo();
			oidInfo.setMonitorId(trapInfo.getMonitorId());
			//トラップ名
			oidInfo.setTrapName(info.get(2));
			//OID
			oidInfo.setTrapOid(info.get(3));
			//generic_id
			oidInfo.setGenericId(Integer.parseInt(info.get(4)));
			//specific_id
			oidInfo.setSpecificId(Integer.parseInt(info.get(5)));
			//MIB
			SnmpTrapMasterInfo master = 
				EjbConnectionManager.getConnectionManager().getSnmpTrapMasterInfo(
						oidInfo.getTrapOid(), 
						oidInfo.getGenericId(), 
						oidInfo.getSpecificId());
			oidInfo.setMib(master.getMib());
			//有効・無効
			oidInfo.setValidFlg(ValidConstant.stringToType(info.get(6)));
			//重要度
			oidInfo.setPriority(PriorityConstant.stringToType(info.get(7)));
			//メッセージ
			oidInfo.setLogmsg(info.get(8));
			//詳細メッセージ
			oidInfo.setDescr(info.get(9));
			oidInfos.add(oidInfo);
			trapInfo.setOidInfos(oidInfos);
		}

		super.createMonitorCheck(info, monitorInfo);
	}

	/**
	 * 監視情報を登録する。<br>
	 * 
	 * @param info 監視情報オブジェクト
	 */
	protected void registerMonitorInfo(MonitorInfo info) {
		MonitorSnmpTrapController snmpTrap = EjbConnectionManager.getConnectionManager().getMonitorSnmpTrapController();

		try {
			@SuppressWarnings("unchecked")
			ArrayList list = checkMonitorList(info.getMonitorId(), monitorList);
			if(list instanceof ArrayList){
				Date createTime = (Date)list.get(MonitorTabelDefine.CREATE_TIME);
				String createUser = (String)list.get(MonitorTabelDefine.CREATE_USER);
				info.setRegDate(new Timestamp(createTime.getTime()));
				info.setRegUser(createUser);

				Object[] args = {info.getMonitorId()};
				log.info(Messages.getMsg("ImportTool.MonitorID", args) + 
						" (" + Messages.getMsg("modify") + ")");

				snmpTrap.modifyMonitor((MonitorSnmpTrapInfo)info);
			}
			else{
				Object[] args = {info.getMonitorId()};
				log.info(Messages.getMsg("ImportTool.MonitorID", args) + 
						" (" + Messages.getMsg("add") + ")");

				snmpTrap.addMonitor((MonitorSnmpTrapInfo)info);
			}
		} catch (DuplicateKeyException e) {
			String args[] = {info.getMonitorId()};
			log.error(Messages.getMsg("ImportTool.Common.1", args));
			System.exit(20);
		} catch (Exception e) {
			log.error(Messages.getMsg("ImportTool.ConnectManagerFailed"), e);
			System.exit(14);
		}
	}

	/**
	 * 監視項目リストを取得する。<br>
	 * 
	 * @return 監視項目一覧
	 */
	@SuppressWarnings("unchecked")
	protected ArrayList getMonitorList() {

		MonitorSnmpTrapController snmpTrap = EjbConnectionManager.getConnectionManager().getMonitorSnmpTrapController();

		ArrayList records = null;
		try {
			records = snmpTrap.getMonitorList();
		} catch (Exception e) {
			log.error(Messages.getMsg("ImportTool.ConnectManagerFailed"), e);
			System.exit(14);
		}
		return records;
	}
}