/*

Copyright (C) 2009 NTT DATA INTELLILINK CORPORATION

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.

 */

package jp.co.intellilink.hinemos.importtool.conf.job;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Locale;

import jp.co.intellilink.hinemos.importtool.util.Config;
import jp.co.intellilink.hinemos.importtool.util.EjbConnectionManager;
import jp.co.intellilink.hinemos.importtool.util.JobUtil;
import jp.co.intellilink.hinemos.importtool.util.Messages;
import jp.co.intellilink.hinemos.importtool.util.ReadCsvFile;
import jp.co.intellilink.hinemos.importtool.util.SelectArrayList;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.EndStatusConstant;
import com.clustercontrol.bean.JobConstant;
import com.clustercontrol.bean.JobParamTypeConstant;
import com.clustercontrol.bean.JudgmentObjectConstant;
import com.clustercontrol.bean.NotifyTypeConstant;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.bean.ProcessingMethodConstant;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.jobmanagement.bean.ConditionTypeConstant;
import com.clustercontrol.jobmanagement.bean.JobCommandInfo;
import com.clustercontrol.jobmanagement.bean.JobEndStatusInfo;
import com.clustercontrol.jobmanagement.bean.JobFileInfo;
import com.clustercontrol.jobmanagement.bean.JobInfo;
import com.clustercontrol.jobmanagement.bean.JobNotificationsInfo;
import com.clustercontrol.jobmanagement.bean.JobObjectInfo;
import com.clustercontrol.jobmanagement.bean.JobParameterInfo;
import com.clustercontrol.jobmanagement.bean.JobTreeItem;
import com.clustercontrol.jobmanagement.bean.JobWaitRuleInfo;
import com.clustercontrol.jobmanagement.bean.OperationConstant;
import com.clustercontrol.jobmanagement.bean.SystemParameterConstant;
import com.clustercontrol.jobmanagement.ejb.session.JobController;
import com.clustercontrol.notify.bean.NotifyRelationInfo;

/**
 * ジョブ管理情報をImportするクラス<br>
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class ImportJob {

	// ログ出力
	private static Log log = LogFactory.getLog(ImportJob.class);

	private ArrayList<ArrayList<String>> jobInfo = null;
	private ArrayList<ArrayList<String>> waitInfo = null;
	private ArrayList<ArrayList<String>> notifyInfo = null;
	private ArrayList<ArrayList<String>> delayInfo = null;
	private ArrayList<ArrayList<String>> paramInfo = null;

	private SimpleDateFormat formatTime1 = new SimpleDateFormat("HH:mm:ss");
	private SimpleDateFormat formatTime2 = new SimpleDateFormat("HH:mm");

	/**
	 * ジョブ管理情報をImportする。<br>
	 */
	public void importJob() {
		log.info(Messages.getMsg("ImportTool.Job.Start"));

		//ジョブ情報のCSVファイルを読み込む
		jobInfo = ReadCsvFile.readCsvFile(Config.getConfig("Input.Dir.Conf") + "/JOB.csv");

		//待ち条件情報のCSVファイルを読み込む
		waitInfo = ReadCsvFile.readCsvFile(Config.getConfig("Input.Dir.Conf") + "/JOB_WAIT.csv");

		//通知情報のCSVファイルを読み込む
		notifyInfo = ReadCsvFile.readCsvFile(Config.getConfig("Input.Dir.Conf") + "/JOB_NOTIFY.csv");

		//遅延情報のCSVファイルを読み込む
		delayInfo = ReadCsvFile.readCsvFile(Config.getConfig("Input.Dir.Conf") + "/JOB_DELAY.csv");

		//パラメータ情報のCSVファイルを読み込む
		paramInfo = ReadCsvFile.readCsvFile(Config.getConfig("Input.Dir.Conf") + "/JOB_PARAM.csv");

		//ジョブツリー作成
		JobTreeItem treeItem = createJobTree();
		
		//待ち条件のジョブIDのチェック
		log.info(Messages.getMsg("ImportTool.Job.Check.WaitRule"));
		if(JobUtil.checkWaitRule(treeItem)){
			//ファシリティIDのチェック
			log.info(Messages.getMsg("ImportTool.Job.Check.FacilityID"));
			if(JobUtil.checkFacilityId(treeItem)){
				//通知IDのチェック
				log.info(Messages.getMsg("ImportTool.Job.Check.NotifyID"));
				if(JobUtil.checkNotifyId(treeItem)){
					//ジョブ登録
					registerJob(treeItem);
				}
			}
		}

		log.info(Messages.getMsg("ImportTool.Job.End"));
	}

	/**
	 * ジョブツリー作成<br>
	 * 
	 * @return ジョブツリー
	 */
	public JobTreeItem createJobTree() {

		//JobTreeItemの最上位インスタンスを作成
		JobInfo info = new JobInfo("", JobConstant.STRING_COMPOSITE, JobConstant.TYPE_COMPOSITE);
		JobTreeItem tree = new JobTreeItem(null, info);

		//ジョブツリーのルートを生成
		info = new JobInfo("", com.clustercontrol.util.Messages.getString("job", Locale.getDefault()), JobConstant.TYPE_COMPOSITE);
		JobTreeItem item = new JobTreeItem(tree, info);

		//親ジョブIDが""のジョブ情報を取得
		ArrayList<ArrayList<String>> selectJobs = SelectArrayList.select(jobInfo, 3, com.clustercontrol.util.Messages.getString("jobunit", Locale.getDefault()));

		for(ArrayList<String> job : selectJobs){
			//ジョブツリーを作成する
			createJobTree(job, item);
		}

		return tree;
	}

	/**
	 * ジョブツリーを作成します。<br>
	 * 再帰呼び出しを行います。<br>
	 * 
	 * @param job ジョブ情報
	 * @param parent 親ジョブツリー
	 */
	protected void createJobTree(ArrayList<String> job, JobTreeItem parent) {
		
		JobTreeItem topItem = JobUtil.getTopJobTreeItem(parent);
		if(JobUtil.findJobId(job.get(0), topItem)){
			log.error(Messages.getMsg("ImportTool.Job.3", new String[]{job.get(0)}));
			System.exit(14);
		}
				
		//JobTreeItemに格納するジョブ情報(JobInfo)を作成
		JobInfo info = createJobData(job);

		//JobTreeItemを作成
		JobTreeItem item = new JobTreeItem(parent, info);

		//親ジョブIDでジョブ情報を取得
		ArrayList<ArrayList<String>> selectJobs = SelectArrayList.select(jobInfo, 4, job.get(0));

		for(ArrayList<String> childJob : selectJobs){
			//ジョブツリーを作成する
			createJobTree(childJob, item);
		}
	}

	/**
	 * ジョブ情報{@link com.clustercontrol.jobmanagement.bean.JobInfo}を作成します。<br>
	 * ジョブマスタを基に、ジョブ情報を作成します。<br>
	 * 
	 * @param job ジョブ情報
	 */
	protected JobInfo createJobData(ArrayList<String> job) {
		log.info(Messages.getMsg("ImportTool.JobID", new String[]{job.get(0)}));

		//JobInfoを作成
		JobInfo info = new JobInfo(job.get(0), job.get(1), JobConstant.stringToType(job.get(3)));
		info.setDescription(job.get(2));

		//待ち条件を取得
		JobWaitRuleInfo waitRule = null;

		//ジョブIDで遅延情報を取得
		ArrayList<ArrayList<String>> delayJobs = SelectArrayList.select(delayInfo, 0, job.get(0));
		if(delayJobs.size() > 0){
			//制御
			waitRule = new JobWaitRuleInfo();
			waitRule.setSuspend(YesNoConstant.stringToType(job.get(18)));
			waitRule.setCondition(ConditionTypeConstant.stringToType(job.get(12)));
			waitRule.setEndCondition(YesNoConstant.stringToType(job.get(13)));
//			waitRule.setEndStatus();
			waitRule.setEndValue(Integer.parseInt(job.get(14)));
			waitRule.setSkip(YesNoConstant.stringToType(job.get(19)));
//			waitRule.setSkipEndStatus();
			waitRule.setSkipEndValue(Integer.parseInt(job.get(20)));
			waitRule.setCalendar(YesNoConstant.stringToType(job.get(15)));
			if(job.get(16).length() > 0)
				waitRule.setCalendarId(job.get(16));
			waitRule.setCalendarEndValue(Integer.parseInt(job.get(17)));

			ArrayList<String> delayJob = delayJobs.get(0);

			//開始遅延
			waitRule.setStart_delay(YesNoConstant.stringToType(delayJob.get(1)));
			waitRule.setStart_delay_session(YesNoConstant.stringToType(delayJob.get(2)));
			waitRule.setStart_delay_session_value(Integer.parseInt(delayJob.get(3)));
			waitRule.setStart_delay_time(YesNoConstant.stringToType(delayJob.get(4)));
			try {
				waitRule.setStart_delay_time_value(formatTime1.parse(delayJob.get(5)));
			} catch (ParseException e) {
				try {
					waitRule.setStart_delay_time_value(formatTime2.parse(delayJob.get(5)));
				} catch (ParseException e1) {
				}
			}
			waitRule.setStart_delay_condition_type(ConditionTypeConstant.stringToType(delayJob.get(6)));
			waitRule.setStart_delay_notify(YesNoConstant.stringToType(delayJob.get(7)));
			waitRule.setStart_delay_notify_priority(PriorityConstant.stringToType(delayJob.get(8)));
			waitRule.setStart_delay_operation(YesNoConstant.stringToType(delayJob.get(9)));
			waitRule.setStart_delay_operation_type(OperationConstant.stringToType(delayJob.get(10)));
			waitRule.setStart_delay_operation_end_value(Integer.parseInt(delayJob.get(11)));

			//終了遅延
			waitRule.setEnd_delay(YesNoConstant.stringToType(delayJob.get(12)));
			waitRule.setEnd_delay_session(YesNoConstant.stringToType(delayJob.get(13)));
			waitRule.setEnd_delay_session_value(Integer.parseInt(delayJob.get(14)));
			waitRule.setEnd_delay_job(YesNoConstant.stringToType(delayJob.get(15)));
			waitRule.setEnd_delay_job_value(Integer.parseInt(delayJob.get(16)));
			waitRule.setEnd_delay_time(YesNoConstant.stringToType(delayJob.get(17)));
			try {
				waitRule.setEnd_delay_time_value(formatTime1.parse(delayJob.get(18)));
			} catch (ParseException e) {
				try {
					waitRule.setEnd_delay_time_value(formatTime2.parse(delayJob.get(18)));
				} catch (ParseException e1) {
				}
			}
			waitRule.setEnd_delay_condition_type(ConditionTypeConstant.stringToType(delayJob.get(19)));
			waitRule.setEnd_delay_notify(YesNoConstant.stringToType(delayJob.get(20)));
			waitRule.setEnd_delay_notify_priority(PriorityConstant.stringToType(delayJob.get(21)));
			waitRule.setEnd_delay_operation(YesNoConstant.stringToType(delayJob.get(22)));
			waitRule.setEnd_delay_operation_type(OperationConstant.stringToType(delayJob.get(23)));
			waitRule.setEnd_delay_operation_end_value(Integer.parseInt(delayJob.get(24)));

			//ジョブIDで待ち条件情報を取得
			ArrayList<ArrayList<String>> waitJobs = SelectArrayList.select(waitInfo, 0, job.get(0));

			//待ち条件
			ArrayList<JobObjectInfo> objectList = new ArrayList<JobObjectInfo>();
			if(waitJobs.size() > 0){
				for(ArrayList<String> waitJob : waitJobs){
					JobObjectInfo objectInfo = new JobObjectInfo();

					objectInfo.setType(JudgmentObjectConstant.stringToType(waitJob.get(1)));
					if (objectInfo.getType() == JudgmentObjectConstant.TYPE_JOB_END_STAUS) {
						objectInfo.setJobId(waitJob.get(2));
						objectInfo.setValue(EndStatusConstant.stringToType(waitJob.get(3)));
					}
					else if (objectInfo.getType() == JudgmentObjectConstant.TYPE_JOB_END_VALUE) {
						objectInfo.setJobId(waitJob.get(2));
						objectInfo.setValue(Integer.parseInt(waitJob.get(3)));
					}
					else if (objectInfo.getType() == JudgmentObjectConstant.TYPE_TIME) {
						try {
							objectInfo.setTime(formatTime1.parse(waitJob.get(3)));
						} catch (ParseException e) {
							try {
								objectInfo.setTime(formatTime2.parse(waitJob.get(3)));
							} catch (ParseException e1) {
							}
						}
					}
					objectList.add(objectInfo);
				}
			}

			waitRule.setObject(objectList);
			info.setWaitRule(waitRule);
		}

		//コマンド
		JobCommandInfo commandInfo = null;
		if(job.size() >= 22 && job.size() <= 29){
			commandInfo = new JobCommandInfo();
			if(com.clustercontrol.util.Messages.getString("fixed.value").compareTo(job.get(21)) == 0) {
				commandInfo.setFacilityID(job.get(22));
			} else {
				commandInfo.setFacilityID(SystemParameterConstant.HEADER + 
						SystemParameterConstant.FACILITY_ID + 
						SystemParameterConstant.FOOTER);
			}
			if(com.clustercontrol.util.Messages.getString("scope.process.all.nodes").compareTo(job.get(23)) == 0) {
				commandInfo.setProcessingMethod(ProcessingMethodConstant.TYPE_ALL_NODE);
			} else {
				commandInfo.setProcessingMethod(ProcessingMethodConstant.TYPE_RETRY);
			}
			commandInfo.setStartCommand(job.get(24));
			commandInfo.setStopCommand(job.get(25));
			commandInfo.setUser(job.get(26));
			commandInfo.setErrorEndFlg(YesNoConstant.stringToType(job.get(27)));
			commandInfo.setErrorEndValue(Integer.parseInt(job.get(28)));
			info.setCommand(commandInfo);
		}

		//ファイル転送
		JobFileInfo fileInfo = null;
		if(job.size() > 29){
			fileInfo = new JobFileInfo();
			if (com.clustercontrol.util.Messages.getString("scope.process.all.nodes").compareTo(job.get(32)) == 0) {
				fileInfo.setProcessingMethod(ProcessingMethodConstant.TYPE_ALL_NODE);
			} else {
				fileInfo.setProcessingMethod(ProcessingMethodConstant.TYPE_RETRY);
			}
			fileInfo.setSrcFacilityID(job.get(29));
			fileInfo.setDestFacilityID(job.get(31));
			fileInfo.setSrcFile(job.get(30));
			fileInfo.setSrcWorkDir("");
			fileInfo.setDestDirectory(job.get(33));
			fileInfo.setDestWorkDir("");
			fileInfo.setCompressionFlg(YesNoConstant.stringToType(job.get(34)));
			fileInfo.setCheckFlg(YesNoConstant.stringToType(job.get(35)));
			fileInfo.setUser(job.get(36));
			info.setFile(fileInfo);
		}

		//ジョブIDで通知情報を取得
		ArrayList<ArrayList<String>> notifyJobs = SelectArrayList.select(notifyInfo, 0, job.get(0));
		//通知
		if(notifyJobs.size() > 0){
			ArrayList<String> notifyJob = notifyJobs.get(0);

			String notifyGroupId = NotifyRelationInfo.createNotifyGroupIdJob(job.get(0),0);

			Collection<NotifyRelationInfo> notifyRelationList = new ArrayList<NotifyRelationInfo>();
			for(int i = 5; i < 10; i++){
				if(notifyJob.size() > i && notifyJob.get(i) instanceof String && notifyJob.get(i).length() > 0){
					NotifyRelationInfo relationInfo = new NotifyRelationInfo();
					relationInfo.setNotifyId(notifyJob.get(i));
					relationInfo.setNotifyGroupId(notifyGroupId);
					relationInfo.setNotifyFlg(0);

					if(i == 5)
						relationInfo.setNotifyType(NotifyTypeConstant.TYPE_STATUS);
					else if(i == 6)
						relationInfo.setNotifyType(NotifyTypeConstant.TYPE_EVENT);
					else if(i == 7)
						relationInfo.setNotifyType(NotifyTypeConstant.TYPE_MAIL);
					else if(i == 8)
						relationInfo.setNotifyType(NotifyTypeConstant.TYPE_JOB);
					else if(i == 9)
						relationInfo.setNotifyType(NotifyTypeConstant.TYPE_LOG_ESCALATE);

					notifyRelationList.add(relationInfo);
				}
			}
			
			ArrayList<JobNotificationsInfo> noticeList = new ArrayList<JobNotificationsInfo>();
			for(int i = 0; i < 4; i++){
				JobNotificationsInfo notificationsInfo = new JobNotificationsInfo();
				if(i == 0){
					notificationsInfo.setType(EndStatusConstant.TYPE_NORMAL);
					notificationsInfo.setPriority(PriorityConstant.stringToType(notifyJob.get(2)));
				}
				else if(i == 1){
					notificationsInfo.setType(EndStatusConstant.TYPE_WARNING);
					notificationsInfo.setPriority(PriorityConstant.stringToType(notifyJob.get(3)));
				}
				else if(i == 2){
					notificationsInfo.setType(EndStatusConstant.TYPE_ABNORMAL);
					notificationsInfo.setPriority(PriorityConstant.stringToType(notifyJob.get(4)));
				}
				else if(i == 3){
					notificationsInfo.setType(EndStatusConstant.TYPE_BEGINNING);
					notificationsInfo.setPriority(PriorityConstant.stringToType(notifyJob.get(1)));
				}
				notificationsInfo.setNotifyGroupId(notifyGroupId);

				//正常終了の場合のみ通知をセット
				if(notificationsInfo.getType() == EndStatusConstant.TYPE_NORMAL ){
					//通知情報の取得
					notificationsInfo.setNotifyId(notifyRelationList);
				}
				noticeList.add(notificationsInfo);
			}
			info.setNotifications(noticeList);
		}

		//終了状態
		ArrayList<JobEndStatusInfo> endList = new ArrayList<JobEndStatusInfo>();
		JobEndStatusInfo endInfo = new JobEndStatusInfo();
		endInfo.setType(EndStatusConstant.TYPE_NORMAL);
		endInfo.setValue(Integer.parseInt(job.get(5)));
		endInfo.setStartRangeValue(Integer.parseInt(job.get(6)));
		endInfo.setEndRangeValue(Integer.parseInt(job.get(7)));
		endList.add(endInfo);
		endInfo = new JobEndStatusInfo();
		endInfo.setType(EndStatusConstant.TYPE_WARNING);
		endInfo.setValue(Integer.parseInt(job.get(8)));
		endInfo.setStartRangeValue(Integer.parseInt(job.get(9)));
		endInfo.setEndRangeValue(Integer.parseInt(job.get(10)));
		endList.add(endInfo);
		endInfo = new JobEndStatusInfo();
		endInfo.setType(EndStatusConstant.TYPE_ABNORMAL);
		endInfo.setValue(Integer.parseInt(job.get(11)));
		endInfo.setStartRangeValue(0);
		endInfo.setEndRangeValue(0);
		endList.add(endInfo);
		info.setEndStatus(endList);

		//パラメータ
		ArrayList<ArrayList<String>> paramJobs = SelectArrayList.select(paramInfo, 0, job.get(0));
		if(paramJobs.size() > 0){
			ArrayList<JobParameterInfo> paramList = new ArrayList<JobParameterInfo>();
			for(ArrayList<String> paramJob : paramJobs){
				JobParameterInfo paramInfo = new JobParameterInfo();
				paramInfo.setParamId(paramJob.get(1));
				paramInfo.setType(JobParamTypeConstant.stringToType(paramJob.get(2)));
				paramInfo.setValue(paramJob.get(3));
				paramInfo.setDescription(paramJob.get(4));
				paramList.add(paramInfo);
			}
			info.setParam(paramList);
		}

		return info;
	}

	/**
	 * ジョブツリーを登録する。<br>
	 * 
	 * @param item ジョブツリー
	 */
	protected void registerJob(JobTreeItem item) {
		log.info(Messages.getMsg("ImportTool.Job.Register.Start"));
		
		JobController job = 
			EjbConnectionManager.getConnectionManager().getJobController();

		try {
			job.registerJob(item);
		} catch (Exception e) {
			log.error(Messages.getMsg("ExportTool.ConnectManagerFailed"), e);
			System.exit(14);
		}
		
		log.info(Messages.getMsg("ImportTool.Job.Register.End"));
	}
}