/*

Copyright (C) 2009 NTT DATA INTELLILINK CORPORATION

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.

 */

package jp.co.intellilink.hinemos.importtool.conf.repository;

import java.util.ArrayList;
import java.util.Collections;

import jp.co.intellilink.hinemos.util.CheckString;
import jp.co.intellilink.hinemos.util.Config;
import jp.co.intellilink.hinemos.util.EjbConnectionManager;
import jp.co.intellilink.hinemos.util.Messages;
import jp.co.intellilink.hinemos.util.ReadCsvFile;
import jp.co.intellilink.hinemos.util.StringListComparator;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.FacilityNotFoundException;
import com.clustercontrol.repository.bean.ScopeInfo;
import com.clustercontrol.repository.ejb.session.RepositoryController;

/**
 * スコープ情報をインポートするクラス<br>
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class ImportScope {

	// ログ出力
	private static Log log = LogFactory.getLog(ImportScope.class);

	private ArrayList<ArrayList<String>> scopeInfos = null;

	/**
	 * スコープ情報をインポートする。<br>
	 */
	@SuppressWarnings("unchecked")
	public void importScope() {
		log.info(Messages.getMsg("Scope.Start", new String[]{Messages.getMsg("ImportTool.Import")}));

		//スコープ情報のCSVファイルを読み込む
		scopeInfos = ReadCsvFile.readCsvFile(Config.getConfig("Import.Dir") + "/SCOPE.csv");

		//ファシリティIDでソート
		Collections.sort(scopeInfos, new StringListComparator(0));
		//親ファシリティIDでソート
		Collections.sort(scopeInfos, new StringListComparator(3));

		for(ArrayList<String> info : scopeInfos){
			String facilityId = info.get(0);
			if(!facilityId.matches("(INTERNAL|REGISTERED|UNREGISTERED)")){
				ScopeInfo property = createScopeInfo(info);

				String parentFacilityId = "";
				if(info.size() >= 4)
					parentFacilityId = info.get(3);

				registerScopeInfo(parentFacilityId, property);
			}
		}

		log.info(Messages.getMsg("Scope.End", new String[]{Messages.getMsg("ImportTool.Import")}));

	}

	/**
	 * スコープ情報を作成する。<br>
	 * 
	 * @param info スコープ情報配列
	 * @return スコープ情報オブジェクト
	 */
	public ScopeInfo createScopeInfo(ArrayList<String> info) {
		ScopeInfo scope = new ScopeInfo();

		//ファシリティID
		String id = CheckString.checkLength(info.get(0), 64);
		if(!id.equals(info.get(0))){
			String[] args = { com.clustercontrol.util.Messages.getString("facility.id"), "64" };
			log.error(Messages.getMsg("ImportTool.Common.2", args));
			System.exit(20);
		}
		scope.setFacilityId(id);
		//ファシリティ名
		String name = CheckString.checkLength(info.get(1), 256);
		if(!name.equals(info.get(1))){
			String[] args = { com.clustercontrol.util.Messages.getString("facility.name"), "256" };
			log.warn(Messages.getMsg("ImportTool.Common.2", args));
		}
		scope.setFacilityName(name);
		//説明
		if(info.size() >= 3){
			String description = CheckString.checkLength(info.get(2), 1024);
			if(!description.equals(info.get(2))){
				String[] args = { com.clustercontrol.util.Messages.getString("description"), "1024" };
				log.warn(Messages.getMsg("ImportTool.Common.2", args));
			}
			scope.setDescription(description);
		}

		return scope;
	}

	/**
	 * スコープ情報を登録する。<br>
	 * 
	 * @param scope スコープ情報
	 */
	protected void registerScopeInfo(String parentFacilityId, ScopeInfo scope) {

		RepositoryController repository = EjbConnectionManager.getConnectionManager().getRepositoryController();

		String facilityId = scope.getFacilityId();
		try {
			ScopeInfo line = getScope(facilityId);
			if(line instanceof ScopeInfo){
				Object[] args = {facilityId};
				log.info(Messages.getMsg("FacilityID", args) + 
						" (" + Messages.getMsg("modify") + ")");

				repository.modifyScope(scope);
			}
			else{
				Object[] args = {facilityId};
				log.info(Messages.getMsg("FacilityID", args) + 
						" (" + Messages.getMsg("add") + ")");

				repository.addScope(parentFacilityId, scope);
			}
		} catch (Exception e) {
			log.error(Messages.getMsg("ConnectManagerFailed"), e);
			System.exit(14);
		}
	}

	/**
	 * スコープを取得する。<br>
	 * 
	 * @param facilityId ファシリティID
	 * @return スコープ
	 */
	@SuppressWarnings("unchecked")
	protected ScopeInfo getScope(String facilityId) {

		RepositoryController repository = EjbConnectionManager.getConnectionManager().getRepositoryController();

		ScopeInfo scope = null;
		try {
			scope = repository.getScopeProperty(facilityId);
		} catch (FacilityNotFoundException e) {
		} catch (Exception e) {
			log.error(Messages.getMsg("ConnectManagerFailed"), e);
			System.exit(14);
		}
		return scope;
	}
}