/*

Copyright (C) 2009 NTT DATA INTELLILINK CORPORATION

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.

 */

package jp.co.intellilink.hinemos.importtool.conf.monitor;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import jp.co.intellilink.hinemos.util.CheckString;
import jp.co.intellilink.hinemos.util.Messages;
import jp.co.intellilink.hinemos.util.NotifyUtil;
import jp.co.intellilink.hinemos.util.RepositoryUtil;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.bean.MonitorBlockConstant;
import com.clustercontrol.bean.RunIntervalConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.hinemosagent.bean.AgentCheckInfo;
import com.clustercontrol.http.bean.HttpCheckInfo;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.notify.bean.NotifyRelationInfo;
import com.clustercontrol.notify.bean.NotifyTypeConstant;
import com.clustercontrol.performance.monitor.bean.PerfCheckInfo;
import com.clustercontrol.ping.bean.PingCheckInfo;
import com.clustercontrol.port.bean.PortCheckInfo;
import com.clustercontrol.process.bean.ProcessCheckInfo;
import com.clustercontrol.snmp.bean.SnmpCheckInfo;
import com.clustercontrol.snmptrap.bean.MonitorSnmpTrapInfo;
import com.clustercontrol.sql.bean.SqlCheckInfo;
import com.clustercontrol.syslogng.bean.LogFilterInfo;

/**
 * 監視情報を作成するクラス<br>
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class ImportMonitor {

	// ログ出力
	private static Log log = LogFactory.getLog(ImportMonitor.class);

	/**
	 * 監視情報オブジェクトを作成する。<br>
	 * 
	 * @param info 監視情報配列
	 * @return　監視情報オブジェクト
	 */
	protected MonitorInfo createMonitorInfo(ArrayList<String> info) {
		MonitorInfo monitorInfo = new MonitorInfo();

		return monitorInfo;
	}

	/**
	 * 監視基本情報を設定する。<br>
	 * 
	 * @param info 監視情報配列
	 * @param monitorInfo 監視情報オブジェクト
	 */
	protected void createMonitorBase(List<String> info, MonitorInfo monitorInfo) {
		//監視項目ID
		String id = CheckString.checkLength(info.get(0), 64);
		if(!id.equals(info.get(0))){
			String[] args = { com.clustercontrol.util.Messages.getString("monitor.id"), "64" };
			log.error(Messages.getMsg("ImportTool.Common.2", args));
			System.exit(20);
		}
		monitorInfo.setMonitorId(id);
		//説明
		String description = CheckString.checkLength(info.get(1), 256);
		if(!description.equals(info.get(1))){
			String[] args = { com.clustercontrol.util.Messages.getString("description"), "256" };
			log.warn(Messages.getMsg("ImportTool.Common.2", args));
		}
		monitorInfo.setDescription(description);
		//ファシリティID
		monitorInfo.setFacilityId(info.get(2));
		if(!RepositoryUtil.findFacilityId(info.get(2))){
			String args[] = {monitorInfo.getMonitorId(), info.get(2)};
			log.error(Messages.getMsg("ImportTool.Repository.1", args));
			System.exit(20);
		}
	}

	/**
	 * 監視単位情報を設定する。<br>
	 * 
	 * @param info 監視情報配列
	 * @param monitorInfo 監視情報オブジェクト
	 */
	protected void createMonitorBlock(List<String> info, MonitorInfo monitorInfo) {
		//単位
		if(info.get(0).compareTo(com.clustercontrol.util.Messages.getString("scope")) == 0)
			monitorInfo.setMonitorBlock(MonitorBlockConstant.TYPE_SCOPE);
		else
			monitorInfo.setMonitorBlock(MonitorBlockConstant.TYPE_NODE);
	}

	/**
	 * 監視条件情報を設定する。<br>
	 * 
	 * @param info 監視情報配列
	 * @param monitorInfo 監視情報オブジェクト
	 */
	protected void createMonitorRule(List<String> info, MonitorInfo monitorInfo) {
		//間隔
		monitorInfo.setRunInterval(RunIntervalConstant.stringToType(info.get(0)));
		//カレンダID
		if(info.get(1) != null && !"".equals(info.get(1)))
			monitorInfo.setCalendarId(info.get(1));
		else
			monitorInfo.setCalendarId(null);
	}

	/**
	 * 監視情報を設定する。<br>
	 * 
	 * @param info 監視情報配列
	 * @param monitorInfo 監視情報オブジェクト
	 */
	protected void createMonitorCheck(List<String> info, MonitorInfo monitorInfo) {
		monitorInfo.setMonitorTypeId(instanceToTypeId(monitorInfo));
	}

	/**
	 * 監視通知情報を設定する。<br>
	 * 
	 * @param info 監視情報配列
	 * @param monitorInfo 監視情報オブジェクト
	 */
	protected void createMonitorNotify(List<String> info, MonitorInfo monitorInfo) {
		monitorInfo.setNotifyGroupId(NotifyRelationInfo.createNotifyGroupId(instanceToTypeId(monitorInfo), monitorInfo.getMonitorId()));

		ArrayList<NotifyRelationInfo> notifyList = new ArrayList<NotifyRelationInfo>();

		for(int i = 0; i < 5; i++){
			if(info.get(i) instanceof String && info.get(i).length() > 0){
				if(!NotifyUtil.findNotifyId(info.get(i))){
					String args[] = {monitorInfo.getMonitorId(), info.get(i)};
					log.error(Messages.getMsg("ImportTool.Notify.1", args));
					System.exit(20);
				}

				NotifyRelationInfo relationInfo = new NotifyRelationInfo();
				relationInfo.setNotifyId(info.get(i));
				relationInfo.setNotifyGroupId(monitorInfo.getNotifyGroupId());
				relationInfo.setNotifyFlg(0);

				if(i == 0)
					relationInfo.setNotifyType(NotifyTypeConstant.TYPE_STATUS);
				else if(i == 1)
					relationInfo.setNotifyType(NotifyTypeConstant.TYPE_EVENT);
				else if(i == 2)
					relationInfo.setNotifyType(NotifyTypeConstant.TYPE_MAIL);
				else if(i == 3)
					relationInfo.setNotifyType(NotifyTypeConstant.TYPE_JOB);
				else if(i == 4)
					relationInfo.setNotifyType(NotifyTypeConstant.TYPE_LOG_ESCALATE);
				else if(i == 5)
					relationInfo.setNotifyType(NotifyTypeConstant.TYPE_COMMAND);

				notifyList.add(relationInfo);
			}
		}
		monitorInfo.setNotifyId(notifyList);
	}

	/**
	 * アプリケーションを設定する。<br>
	 * 
	 * @param info 監視情報配列
	 * @param monitorInfo 監視情報オブジェクト
	 */
	protected void createMonitorApplication(List<String> info, MonitorInfo monitorInfo) {
		//アプリケーション
		String application = CheckString.checkLength(info.get(0), 64);
		if(!application.equals(info.get(0))){
			String[] args = { com.clustercontrol.util.Messages.getString("application"), "64" };
			log.warn(Messages.getMsg("ImportTool.Common.2", args));
		}
		monitorInfo.setApplication(application);
	}

	/**
	 * 有効/無効を設定する。<br>
	 * 
	 * @param info 監視情報配列
	 * @param monitorInfo 監視情報オブジェクト
	 */
	protected void createMonitorVaild(List<String> info, MonitorInfo monitorInfo) {
		//有効/無効
		monitorInfo.setValid(ValidConstant.stringToType(info.get(0)));
	}

	/**
	 * 監視項目リストに指定した監視項目IDが存在するかチェックする。<br>
	 * 
	 * @param monitorId 監視項目ID
	 * @param monitorList 監視項目リスト
	 * @return 監視項目
	 */
	@SuppressWarnings("unchecked")
	protected MonitorInfo checkMonitorList(String monitorId, ArrayList monitorList) {

		if(monitorList instanceof ArrayList){
			Iterator itr = monitorList.iterator();
			while(itr.hasNext()){
				MonitorInfo info = (MonitorInfo)itr.next();
				if(info.getMonitorId().compareTo(monitorId) == 0)
					return info;
			}
		}

		return null;
	}

	/**
	 * 監視情報オブジェクトから監視種別を取得する
	 * 
	 * @param info 監視情報オブジェクト
	 * @return 監視種別
	 */
	protected String instanceToTypeId(MonitorInfo info){
		if (info.getAgentCheckInfo() instanceof AgentCheckInfo) {
			return HinemosModuleConstant.MONITOR_AGENT;
		} else if (info.getHttpCheckInfo() instanceof HttpCheckInfo) {
			return HinemosModuleConstant.MONITOR_HTTP;
		} else if (info.getPingCheckInfo() instanceof PingCheckInfo) {
			return HinemosModuleConstant.MONITOR_PING;
		} else if (info.getProcessCheckInfo() instanceof ProcessCheckInfo) {
			return HinemosModuleConstant.MONITOR_PROCESS;
		} else if (info.getSnmpCheckInfo() instanceof SnmpCheckInfo) {
			return HinemosModuleConstant.MONITOR_SNMP;
		} else if (info instanceof MonitorSnmpTrapInfo) {
			return HinemosModuleConstant.MONITOR_SNMPTRAP;
		} else if (info.getSqlCheckInfo() instanceof SqlCheckInfo) {
			return HinemosModuleConstant.MONITOR_SQL;
		} else if (info.getPerfCheckInfo() instanceof PerfCheckInfo) {
			return HinemosModuleConstant.MONITOR_PERFORMANCE;
		} else if (info.getPortCheckInfo() instanceof PortCheckInfo) {
			return HinemosModuleConstant.MONITOR_PORT;
		} else if (info instanceof LogFilterInfo) {
			return HinemosModuleConstant.MONITOR_SYSLOGNG;
		}
		return "";
	}
}