/*

Copyright (C) 2009 NTT DATA INTELLILINK CORPORATION

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.

 */

package jp.co.intellilink.hinemos.importtool.conf.performance;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import jp.co.intellilink.hinemos.util.Config;
import jp.co.intellilink.hinemos.util.EjbConnectionManager;
import jp.co.intellilink.hinemos.util.Messages;
import jp.co.intellilink.hinemos.util.ReadCsvFile;
import jp.co.intellilink.hinemos.util.RepositoryUtil;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.RunIntervalConstant;
import com.clustercontrol.performance.bean.CollectorItemInfo;
import com.clustercontrol.performance.bean.CollectorProperty;
import com.clustercontrol.performance.ejb.bmp.RecordCollectorData;
import com.clustercontrol.performance.ejb.session.CollectorController;

/**
 * 性能管理機能の実績収集情報をインポートするクラス<br>
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class ImportPerformance {

	// ログ出力
	private static Log log = LogFactory.getLog(ImportPerformance.class);

	@SuppressWarnings("unchecked")
	private ArrayList performanceList = null;

	private ArrayList<ArrayList<String>> monitorInfos = null;

	/**
	 * 性能管理機能の実績収集情報をインポートする。<br>
	 */
	public void importPerformance() {
		log.info(Messages.getMsg("Performance.Start", new String[]{Messages.getMsg("ImportTool.Import")}));

		//実績収集情報のCSVファイルを読み込む
		monitorInfos = ReadCsvFile.readCsvFile(Config.getConfig("Import.Dir") + "/PERFORMANCE.csv");

		//収集項目リスト取得
		performanceList = getPerformanceList();

		for(ArrayList<String> info : monitorInfos){
			CollectorProperty collectorProperty = createCollectorProperty(info);
			registerRecordColletor(collectorProperty);
		}

		log.info(Messages.getMsg("Performance.End", new String[]{Messages.getMsg("ImportTool.Import")}));
	}

	/**
	 * 収集設定を作成する。<br>
	 * 
	 * @param info 実績収集情報
	 * @return 収集設定
	 */
	protected CollectorProperty createCollectorProperty(ArrayList<String> info) {
		CollectorProperty collectorProperty = new CollectorProperty();   	
		RecordCollectorData collectorData = new RecordCollectorData();
		collectorProperty.setCollectorData(collectorData);

		//収集ID
		collectorData.setCollectorID(info.get(0));
		//説明
		collectorData.setLabel(info.get(1));
		//ファシリティID
		collectorData.setFacilityID(info.get(2));
		if(!RepositoryUtil.findFacilityId(info.get(2))){
			String args[] = {collectorData.getCollectorID(), info.get(2)};
			log.error(Messages.getMsg("ImportTool.Repository.1", args));
			System.exit(14);
		}
		//収集間隔
		collectorData.setIntervalSec(RunIntervalConstant.stringToType(info.get(3)));
		//収集期間
		if(info.get(4).equals(Messages.getMsg("UNLIMIT_COLLECT"))){
			collectorData.setPeriodMin(-1);
			collectorData.setPresavePeriod(Integer.parseInt(info.get(8)));
		}
		else{
			int	period = 
				Integer.parseInt(info.get(5)) * 24 * 60 + 
				Integer.parseInt(info.get(6)) * 60 + 
				Integer.parseInt(info.get(7));

			collectorData.setPeriodMin(period);
		}

		//収集項目
		List<CollectorItemInfo> itemInfoList = 
			CollectorItemCodeFactory.getAvailableCollectorItemList(collectorData.getFacilityID());

		List<CollectorItemInfo> list = new ArrayList<CollectorItemInfo>();
		List<String> names = info.subList(9, info.size());
		for(String name : names){
			for(CollectorItemInfo itemInfo : itemInfoList){
				String itemName = CollectorItemCodeFactory.getFullItemName(itemInfo);

				if(itemName.equals(name)){
					CollectorItemInfo addItemInfo = 
						new CollectorItemInfo(
								collectorData.getCollectorID(), 
								itemInfo.getCollectMethod(), 
								itemInfo.getItemCode(), 
								itemInfo.getDeviceName());
					list.add(addItemInfo);
					break;
				}
			}
		}
		collectorProperty.setItemList(list);

		return collectorProperty;
	}

	/**
	 * 収集項目を登録する。<br>
	 * 
	 * @param collectorProperty 収集設定
	 */
	@SuppressWarnings("unchecked")
	protected void registerRecordColletor(CollectorProperty collectorProperty) {
		CollectorController perf = EjbConnectionManager.getConnectionManager().getCollectorController();

		try {
			boolean check = false;
			if(performanceList instanceof ArrayList){
				Iterator itr = performanceList.iterator();
				while(itr.hasNext()){
					ArrayList line = (ArrayList)itr.next();

					String id = (String)line.get(2);
					if(id.compareTo(collectorProperty.getCollectorData().getCollectorID()) == 0){
						check = true;
						break;
					}
				}
			}

			if(!check){   
				Object[] args = {collectorProperty.getCollectorData().getCollectorID()};
				log.info(Messages.getMsg("CollectorID", args));

				perf.createRecordColletor(collectorProperty);
			}
		} catch (Exception e) {
			log.error(Messages.getMsg("ConnectManagerFailed"), e);
			System.exit(14);
		}
	}

	/**
	 * 収集項目リストを取得する。<br>
	 * 
	 * @return 収集項目一覧
	 */
	@SuppressWarnings("unchecked")
	protected ArrayList getPerformanceList() {
		CollectorController perf = EjbConnectionManager.getConnectionManager().getCollectorController();

		ArrayList records = null;
		try {
			records = perf.getCollectionList();
		} catch (Exception e) {
			log.error(Messages.getMsg("ConnectManagerFailed"), e);
			System.exit(14);
		}
		return records;
	}
}