/*
 
Copyright (C) 2009 NTT DATA INTELLILINK CORPORATION
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package jp.co.intellilink.hinemos.export.conf.maintenance;

import java.rmi.AccessException;
import java.rmi.RemoteException;
import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.naming.NamingException;

import jp.co.intellilink.hinemos.util.Config;
import jp.co.intellilink.hinemos.util.EjbConnectionManager;
import jp.co.intellilink.hinemos.util.Messages;
import jp.co.intellilink.hinemos.util.StringListComparator;
import jp.co.intellilink.hinemos.util.WriteCsvFile;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.DayOfWeekConstant;
import com.clustercontrol.bean.NotifyTypeConstant;
import com.clustercontrol.bean.Schedule;
import com.clustercontrol.bean.ScheduleConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.maintenance.bean.MaintenanceInfoDataImp;
import com.clustercontrol.maintenance.bean.MaintenanceKeepMstConstant;
import com.clustercontrol.maintenance.bean.MaintenanceTypeMstConstant;
import com.clustercontrol.maintenance.bean.ScheduleTableDefine;
import com.clustercontrol.maintenance.ejb.session.MaintenanceController;
import com.clustercontrol.notify.bean.NotifyRelationInfo;

/**
 * メンテナンス機能情報を取得するクラス<br>
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class ExportMaintenamce {

	// ログ出力
	private static Log log = LogFactory.getLog(ExportMaintenamce.class);
	
    /**
     * メンテナンス機能情報をExportする。<br>
     */
    public void exportMaintenance() {
    	log.info(Messages.getMsg("Maintenance.Start", new String[]{Messages.getMsg("ExportTool.Export")}));
    	
    	//CSV出力配列を用意
    	ArrayList<ArrayList<String>> csvOutput = new ArrayList<ArrayList<String>>();
    	
    	//メンテナンスIDリスト取得
    	ArrayList idList = getMaintenanceIdList();

    	if(idList instanceof ArrayList){
			Iterator itr = idList.iterator();
			while(itr.hasNext()){
				ArrayList line = (ArrayList)itr.next();
				
				Object[] args = {(String)line.get(0)};
				log.info(Messages.getMsg("MaintenanceID", args));
				
				MaintenanceInfoDataImp info = getMaintenance((String)line.get(0));
				csvOutput.add(createCsvFormat(info));
			}
			
			//メンテナンスIDでソート
			Collections.sort(csvOutput, new StringListComparator(0));
    	}
    	
		String timestamp = "";
		if(Config.getConfig("file.timestamp").equalsIgnoreCase("on")){
			SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd_HHmmss");
			timestamp = "_" + format.format(new Date());
		}
		WriteCsvFile.writeCsvFile(
				Config.getConfig("Export.Dir") + "/MAINTENANCE" + timestamp + ".csv", 
				csvOutput);
		
		log.info(Messages.getMsg("Maintenance.End", new String[]{Messages.getMsg("ExportTool.Export")}));
    }
    
    /**
	 * メンテナンス機能情報をExportする。<br>
	 * 
	 * @param info 通知情報
	 * @return CSVフォーマット文字列
	 */
	public ArrayList<String> createCsvFormat(MaintenanceInfoDataImp info) {
		ArrayList<String> csvFormat = new ArrayList<String>();
		
		//メンテナンスID
		csvFormat.add(info.getMaintenance_id());
		//説明
		csvFormat.add(info.getDescription());
		//メンテナンス種別
		csvFormat.add(com.clustercontrol.util.Messages.getString(
				MaintenanceTypeMstConstant.typeIdToNameId(info.getType_id())));
		//削除対象外の期間
		csvFormat.add(com.clustercontrol.util.Messages.getString(
				MaintenanceKeepMstConstant.keepIdToNameId(info.getKeep_id())));
		//カレンダID
		if (info.getCalendar_id() != null)
			csvFormat.add(info.getCalendar_id());
		else
			csvFormat.add("");
		
		//スケジュール
		csvFormat.addAll(createSchedule(info));
		
		//通知
		csvFormat.addAll(createNotify(info));
		
        //アプリケーション
        if (info.getApplication() != null)
        	csvFormat.add(info.getApplication());
        else
        	csvFormat.add("");
        
		//有効/無効
		csvFormat.add(ValidConstant.typeToString(info.getValid_flg()));

		return csvFormat;
	}
	
	/**
     * 通知情報をExportする。<br>
     * 
     * @param info メンテナンス情報
     * @return CSVフォーマット文字列
     */
	protected ArrayList<String> createNotify(MaintenanceInfoDataImp info) {
		ArrayList<String> csvFormat = new ArrayList<String>();
		
		String status = "";
		String event = "";
		String mail = "";
		String job = "";
		String syslog = "";
		
		if(info.getNotifyId() != null){
			Collection<NotifyRelationInfo> notify = info.getNotifyId();
			
			NotifyRelationInfo relationInfo;
			for (int i = 0; i < notify.size(); i++) {
				relationInfo = (NotifyRelationInfo)((ArrayList<NotifyRelationInfo>)notify).get(i);
				if(relationInfo.getNotifyType() == NotifyTypeConstant.TYPE_STATUS)
					status = relationInfo.getNotifyId();
				else if(relationInfo.getNotifyType() == NotifyTypeConstant.TYPE_EVENT)
					event = relationInfo.getNotifyId();
				else if(relationInfo.getNotifyType() == NotifyTypeConstant.TYPE_MAIL)
					mail = relationInfo.getNotifyId();
				else if(relationInfo.getNotifyType() == NotifyTypeConstant.TYPE_JOB)
					job = relationInfo.getNotifyId();
				else if(relationInfo.getNotifyType() == NotifyTypeConstant.TYPE_LOG_ESCALATE)
					syslog = relationInfo.getNotifyId();
			}
		}

		csvFormat.add(status);
		csvFormat.add(event);
		csvFormat.add(mail);
		csvFormat.add(job);
		csvFormat.add(syslog);
		
		return csvFormat;
	}
	
	/**
     * スケジュール情報をExportする。<br>
     * 
     * @param info メンテナンス情報
     * @return CSVフォーマット文字列
     */
	protected ArrayList<String> createSchedule(MaintenanceInfoDataImp info) {
		ArrayList<String> csvFormat = new ArrayList<String>();
		
		//スケジュール取得
        Schedule schedule = (Schedule)getSchedule(info.getMaintenance_id()).get(ScheduleTableDefine.SCHEDULE);

        //指定方式
        if (schedule.getType() == ScheduleConstant.TYPE_EVERY_WEEK)
        	csvFormat.add(com.clustercontrol.util.Messages.getString("weekday"));
        else
        	csvFormat.add(com.clustercontrol.util.Messages.getString("time"));
        
        //日時を設定
        if (schedule.getDate() instanceof Date) {
        	Calendar calendar = Calendar.getInstance();
        	calendar.setTime(schedule.getDate());

        	DecimalFormat format = new DecimalFormat("00");
        	
            if (schedule.getType() == ScheduleConstant.TYPE_EVERY_YEAR) {
		    	//月を設定
		    	csvFormat.add(format.format(calendar.get(Calendar.MONTH) + 1));
		    	//日を設定
		    	csvFormat.add(format.format(calendar.get(Calendar.DAY_OF_MONTH)));
		    	//時を設定
		    	csvFormat.add(format.format(calendar.get(Calendar.HOUR_OF_DAY)));
		    	//分を設定
		    	csvFormat.add(format.format(calendar.get(Calendar.MINUTE)));
            } else if (schedule.getType() == ScheduleConstant.TYPE_EVERY_MONTH) {
		    	//月を設定
            	csvFormat.add("");
		    	//日を設定
		    	csvFormat.add(format.format(calendar.get(Calendar.DAY_OF_MONTH)));
		    	//時を設定
		    	csvFormat.add(format.format(calendar.get(Calendar.HOUR_OF_DAY)));
		    	//分を設定
		    	csvFormat.add(format.format(calendar.get(Calendar.MINUTE)));
            } else if (schedule.getType() == ScheduleConstant.TYPE_EVERY_DAY) {
		    	//月を設定
            	csvFormat.add("");
		    	//日を設定
            	csvFormat.add("");
		    	//時を設定
		    	csvFormat.add(format.format(calendar.get(Calendar.HOUR_OF_DAY)));
		    	//分を設定
		    	csvFormat.add(format.format(calendar.get(Calendar.MINUTE)));
            } else if (schedule.getType() == ScheduleConstant.TYPE_EVERY_HOUR) {
		    	//月を設定
            	csvFormat.add("");
		    	//日を設定
            	csvFormat.add("");
		    	//時を設定
            	csvFormat.add("");
		    	//分を設定
		    	csvFormat.add(format.format(calendar.get(Calendar.MINUTE)));
            } else {
		    	//月を設定
            	csvFormat.add("");
		    	//日を設定
		    	csvFormat.add("");
		    	//時を設定
		    	csvFormat.add("");
		    	//分を設定
		    	csvFormat.add("");
        	}
        	
        	if (schedule.getType() == ScheduleConstant.TYPE_EVERY_WEEK){
	        	//曜日を設定
	        	csvFormat.add(DayOfWeekConstant.typeToString(schedule.getDayOfWeek()));
	        	//時を設定
	        	csvFormat.add(format.format(calendar.get(Calendar.HOUR_OF_DAY)));
	        	//分を設定
	        	csvFormat.add(format.format(calendar.get(Calendar.MINUTE)));
        	}
        	else{
        		//曜日を設定
        		csvFormat.add("");
		    	//時を設定
		    	csvFormat.add("");
		    	//分を設定
		    	csvFormat.add("");
        	}
        }
        else{
    		csvFormat.add("");
    		csvFormat.add("");
    		csvFormat.add("");
    		csvFormat.add("");
    		csvFormat.add("");
    		csvFormat.add("");
    		csvFormat.add("");
        }
		
		return csvFormat;
	}
	
    /**
     * メンテナンス情報を取得する。<br>
     * 
     * @param maintenanceId 取得対象のメンテナンスID
     * @return メンテナンス情報
     */
    protected MaintenanceInfoDataImp getMaintenance(String maintenanceId) {

        MaintenanceController maintenance = EjbConnectionManager.getConnectionManager().getMaintenanceController();

        MaintenanceInfoDataImp info = null;
        try {
        	info = maintenance.getMaintenanceInfo(maintenanceId);
        } catch (RemoteException e) {
			if(e instanceof AccessException){
			}
        } catch (FinderException e) {
        } catch (NamingException e) {
        } catch (CreateException e) {
		}
        return info;
    }
    
	/**
	 * スケジュール情報を取得する。<br>
	 * 
	 * @param maintenanceId 取得対象のメンテナンスID
	 * @return スケジュール情報
	 */
    protected ArrayList getSchedule(String maintenanceId) {
		
	   	MaintenanceController maintenance = EjbConnectionManager.getConnectionManager().getMaintenanceController();
	   	
	   	ArrayList info = null;
		try {
			info = maintenance.getSchedule(maintenanceId);
        } catch (Exception e) {
			log.error(Messages.getMsg("ConnectManagerFailed"), e);
			System.exit(14);
        }
		return info;
	}
    
    /**
     * メンテナンスIDリストを取得する。<br>
     * 
     * @return メンテナンスID一覧
     */
    protected ArrayList getMaintenanceIdList() {

        MaintenanceController maintenance = EjbConnectionManager.getConnectionManager().getMaintenanceController();

        ArrayList records = null;
        try {
            records = maintenance.getMaintenanceList();
        } catch (Exception e) {
			log.error(Messages.getMsg("ConnectManagerFailed"), e);
			System.exit(14);
        }
        return records;
    }
}