/*
 
Copyright (C) 2009 NTT DATA INTELLILINK CORPORATION
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package jp.co.intellilink.hinemos.export.conf.repository;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.Locale;

import jp.co.intellilink.hinemos.util.Config;
import jp.co.intellilink.hinemos.util.EjbConnectionManager;
import jp.co.intellilink.hinemos.util.Messages;
import jp.co.intellilink.hinemos.util.StringListComparator;
import jp.co.intellilink.hinemos.util.WriteCsvFile;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.FacilityInfo;
import com.clustercontrol.bean.FacilityTreeItem;
import com.clustercontrol.bean.Property;
import com.clustercontrol.bean.PropertyConstant;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.repository.ejb.session.RepositoryController;
import com.clustercontrol.repository.factory.ScopeProperty;
import com.clustercontrol.util.PropertyUtil;

/**
 * スコープ情報を取得するクラス<br>
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class ExportScope {
	
	// ログ出力
	private static Log log = LogFactory.getLog(ExportScope.class);

    /**
     * スコープ情報をExportする。<br>
     */
    public void exportScope() {
    	log.info(Messages.getMsg("Scope.Start", new String[]{Messages.getMsg("ExportTool.Export")}));
    	
    	//CSV出力配列を用意
    	ArrayList<ArrayList<String>> csvOutput = new ArrayList<ArrayList<String>>();
    	
    	//ファシリティツリー取得
    	FacilityTreeItem facilityTree = getScopeFacilityTree();
    	getFacilityTreeItem(facilityTree, csvOutput);
		
		//ファシリティIDでソート
		Collections.sort(csvOutput, new StringListComparator(0));
		
		SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd_HHmmss");
		WriteCsvFile.writeCsvFile(
				Config.getConfig("Export.Dir") + "/SCOPE_" + format.format(new Date()) + ".csv", 
				csvOutput);
		
		log.info(Messages.getMsg("Scope.End", new String[]{Messages.getMsg("ExportTool.Export")}));
    }
    
    /**
     * スコープ情報取得（再帰呼び出し）
     * 
     * @param treeItem
     * @param csvOutput CSV出力文字列配列
     */
    public void getFacilityTreeItem(FacilityTreeItem treeItem, ArrayList<ArrayList<String>> csvOutput) {

        //子ツリーを取得する
        if (treeItem.getData() instanceof FacilityInfo){
        	FacilityInfo info = treeItem.getData();
        	if(info.getFacilityId() != null && info.getFacilityId().compareTo("") != 0){
    			Object[] args = {info.getFacilityId()};
    			log.info(Messages.getMsg("FacilityID", args));
        		
    			Property property = getScope(info.getFacilityId());
    			ArrayList<String> csvFormat = createCsvFormat(property);
            	String parentFacilityId = "";
    			 if(treeItem.getParent() != null){
            		if (treeItem.getParent().getData() instanceof FacilityInfo){
            			FacilityInfo parentInfo = treeItem.getParent().getData();
            			if(parentInfo.getFacilityId() != null && parentInfo.getFacilityId().compareTo("") != 0){
            				parentFacilityId = parentInfo.getFacilityId();
            			}
            		}
            	}
    			csvFormat.add(parentFacilityId);
    			csvOutput.add(csvFormat);
        	}
        }
        
        
    	FacilityTreeItem[] childrenTreeItem = treeItem.getChildren();
        for (FacilityTreeItem childTreeItem : childrenTreeItem) {
        	getFacilityTreeItem(childTreeItem, csvOutput);
        }
    }
    
    /**
	 * スコープ情報をExportする。<br>
	 * 
	 * @param info スコープ情報
	 * @return CSVフォーマット文字列
	 */
	public ArrayList<String> createCsvFormat(Property info) {
		ArrayList<String> csvFormat = new ArrayList<String>();
		ArrayList values = null;
		
		//ファシリティID
		values = PropertyUtil.getPropertyValue(info, ScopeProperty.FACILITY_ID);
		csvFormat.add(getStringValue(values));
		//ファシリティ名
		values = PropertyUtil.getPropertyValue(info, ScopeProperty.FACILITY_NAME);
		csvFormat.add(getStringValue(values));
		//説明
		values = PropertyUtil.getPropertyValue(info, ScopeProperty.DESCRIPTION);
		csvFormat.add(getStringValue(values));
		
		return csvFormat;
	}
	
	/**
	 * 配列から文字列を取得する。<br>
	 * 
	 * @param values 配列
	 * @return 文字列
	 */
	public String getStringValue(ArrayList values) {
		if(values.size() > 0 && values.get(0) instanceof String)
			return (String)values.get(0);
		else
			return "";
	}
	
	/**
	 * 配列から数値を取得し文字列に変換する。<br>
	 * 
	 * @param values 配列
	 * @return 文字列
	 */
	public String getIntegerValue(ArrayList values) {
		if(values.size() > 0 && values.get(0) instanceof Integer)
			return ((Integer)values.get(0)).toString();
		else
			return "";
	}
	
	/**
	 * 配列からBooleanを取得し文字列に変換する。<br>
	 * 
	 * @param values 配列
	 * @return 文字列
	 */
	public String getBooleanValue(ArrayList values) {
		if(values.size() > 0 && values.get(0) instanceof Boolean)
			if(((Boolean)values.get(0)).booleanValue())
				return YesNoConstant.typeToString(YesNoConstant.TYPE_YES);
			else
				return YesNoConstant.typeToString(YesNoConstant.TYPE_NO);
		else
			return "";
	}
	
    /**
     * スコープ情報を取得する。<br>
     * 
     * @param facilityId 取得対象のファシリティID
     * @return スコープ情報
     */
    protected Property getScope(String facilityId) {

    	RepositoryController repository = EjbConnectionManager.getConnectionManager().getRepositoryController();

    	Property info = null;
        try {
        	info = repository.getScopeProperty(facilityId, PropertyConstant.MODE_SHOW, Locale.getDefault());
        } catch (Exception e) {
			log.error(Messages.getMsg("ConnectManagerFailed"), e);
			System.exit(14);
        }
        return info;
    }
    
    /**
     * スコープのみのファシリティツリーを取得する。<br>
     * 
     * @return スコープのみのファシリティツリー
     */
    protected FacilityTreeItem getScopeFacilityTree() {

    	RepositoryController repository = EjbConnectionManager.getConnectionManager().getRepositoryController();

    	FacilityTreeItem facilityTree = null;
        try {
        	facilityTree = repository.getScopeFacilityTree(Locale.getDefault());
        } catch (Exception e) {
			log.error(Messages.getMsg("ConnectManagerFailed"), e);
			System.exit(14);
		}
        return facilityTree;
    }
}