/*

Copyright (C) 2009 NTT DATA INTELLILINK CORPORATION

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.

 */

package jp.co.intellilink.hinemos.export.conf.job;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.Locale;

import jp.co.intellilink.hinemos.util.Config;
import jp.co.intellilink.hinemos.util.EjbConnectionManager;
import jp.co.intellilink.hinemos.util.Messages;
import jp.co.intellilink.hinemos.util.StringListComparator;
import jp.co.intellilink.hinemos.util.WriteCsvFile;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.EndStatusConstant;
import com.clustercontrol.bean.JobConstant;
import com.clustercontrol.bean.JobParamTypeConstant;
import com.clustercontrol.bean.JudgmentObjectConstant;
import com.clustercontrol.bean.NotifyTypeConstant;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.bean.ProcessingMethodConstant;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.jobmanagement.bean.ConditionTypeConstant;
import com.clustercontrol.jobmanagement.bean.JobCommandInfo;
import com.clustercontrol.jobmanagement.bean.JobEndStatusInfo;
import com.clustercontrol.jobmanagement.bean.JobFileInfo;
import com.clustercontrol.jobmanagement.bean.JobInfo;
import com.clustercontrol.jobmanagement.bean.JobNotificationsInfo;
import com.clustercontrol.jobmanagement.bean.JobObjectInfo;
import com.clustercontrol.jobmanagement.bean.JobParameterInfo;
import com.clustercontrol.jobmanagement.bean.JobTreeItem;
import com.clustercontrol.jobmanagement.bean.JobWaitRuleInfo;
import com.clustercontrol.jobmanagement.bean.OperationConstant;
import com.clustercontrol.jobmanagement.bean.SystemParameterConstant;
import com.clustercontrol.jobmanagement.ejb.session.JobController;
import com.clustercontrol.notify.bean.NotifyRelationInfo;

/**
 * ジョブ管理情報を取得するクラス<br>
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class ExportJob {

	// ログ出力
	private static Log log = LogFactory.getLog(ExportJob.class);

	/**
	 * ジョブ管理情報をExportする。<br>
	 */
	public void exportJob() {
		log.info(Messages.getMsg("Job.Start", new String[]{Messages.getMsg("ExportTool.Export")}));

		//ジョブツリー取得
		JobTreeItem jobTree = getJobTree();

		//ジョブ情報のCSV文字列を作成
		ArrayList<ArrayList<String>> csvOutput = new ArrayList<ArrayList<String>>();
		getJobData(jobTree, csvOutput);
		//ジョブIDでソート
		Collections.sort(csvOutput, new StringListComparator(0));

		SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd_HHmmss");
		WriteCsvFile.writeCsvFile(
				Config.getConfig("Export.Dir") + "/JOB_" + format.format(new Date()) + ".csv", 
				csvOutput);

		//待ち条件のCSV文字列を作成
		ArrayList<ArrayList<String>> csvOutputWait = new ArrayList<ArrayList<String>>();
		getJobWaitRule(jobTree, csvOutputWait);
		//ジョブIDでソート
		Collections.sort(csvOutputWait, new StringListComparator(0));

		WriteCsvFile.writeCsvFile(
				Config.getConfig("Export.Dir") + "/JOB_WAIT_" + format.format(new Date()) + ".csv", 
				csvOutputWait);

		//通知情報のCSV文字列を作成
		ArrayList<ArrayList<String>> csvOutputNotify = new ArrayList<ArrayList<String>>();
		getJobNotify(jobTree, csvOutputNotify);
		//ジョブIDでソート
		Collections.sort(csvOutputNotify, new StringListComparator(0));

		WriteCsvFile.writeCsvFile(
				Config.getConfig("Export.Dir") + "/JOB_NOTIFY_" + format.format(new Date()) + ".csv", 
				csvOutputNotify);

		//遅延情報のCSV文字列を作成
		ArrayList<ArrayList<String>> csvOutputDelay = new ArrayList<ArrayList<String>>();
		getJobDelay(jobTree, csvOutputDelay);
		//ジョブIDでソート
		Collections.sort(csvOutputDelay, new StringListComparator(0));

		WriteCsvFile.writeCsvFile(
				Config.getConfig("Export.Dir") + "/JOB_DELAY_" + format.format(new Date()) + ".csv", 
				csvOutputDelay);

		//パラメータ情報のCSV文字列を作成
		ArrayList<ArrayList<String>> csvOutputParam = new ArrayList<ArrayList<String>>();
		getJobParam(jobTree, csvOutputParam);
		//名前でソート
		Collections.sort(csvOutputParam, new StringListComparator(1));
		//ジョブIDでソート
		Collections.sort(csvOutputParam, new StringListComparator(0));

		WriteCsvFile.writeCsvFile(
				Config.getConfig("Export.Dir") + "/JOB_PARAM_" + format.format(new Date()) + ".csv", 
				csvOutputParam);

		log.info(Messages.getMsg("Job.End", new String[]{Messages.getMsg("ExportTool.Export")}));
	}

	/**
	 * ジョブ情報取得（再帰呼び出し）
	 * 
	 * @param treeItem ジョブツリー
	 * @param csvOutput CSV出力文字列配列
	 */
	public void getJobData(JobTreeItem treeItem, ArrayList<ArrayList<String>> csvOutput) {

		//子ツリーを取得する
		if (treeItem.getData() instanceof JobInfo){
			JobInfo info = treeItem.getData();
			if(info.getId() != null && info.getId().compareTo("") != 0){
				Object[] args = {info.getId()};
				log.info(Messages.getMsg("JobID", args));

				ArrayList<String> csvFormat = createCsvFormat(info);
				String parentJobId = "";
				if(treeItem.getParent() != null){
					if (treeItem.getParent().getData() instanceof JobInfo){
						JobInfo parentInfo = treeItem.getParent().getData();
						if(parentInfo.getId() != null && parentInfo.getId().compareTo("") != 0){
							parentJobId = parentInfo.getId();
						}
					}
				}
				csvFormat.add(4, parentJobId);
				csvOutput.add(csvFormat);
			}
		}


		JobTreeItem[] childrenTreeItem = treeItem.getChildren();
		for (JobTreeItem childTreeItem : childrenTreeItem) {
			getJobData(childTreeItem, csvOutput);
		}
	}

	/**
	 * ジョブ基本情報のCSV文字列を作成する。<br>
	 * 
	 * @param info ジョブ情報
	 * @return CSVフォーマット文字列
	 */
	public ArrayList<String> createBaseOutput(JobInfo info) {
		ArrayList<String> csvFormat = new ArrayList<String>();

		//ジョブID
		csvFormat.add(info.getId());
		//ジョブ名
		csvFormat.add(info.getName());
		//説明
		if (info.getDescription() != null)
			csvFormat.add(info.getDescription());
		else
			csvFormat.add("");
		//ジョブ種別
		csvFormat.add(JobConstant.typeToString(info.getType()));

		return csvFormat;
	}

	/**
	 * ジョブ情報からCSV文字列を作成する。<br>
	 * 
	 * @param info ジョブ情報
	 * @return CSVフォーマット文字列
	 */
	public ArrayList<String> createCsvFormat(JobInfo info) {
		ArrayList<String> csvFormat = new ArrayList<String>();

		//基本情報
		csvFormat.addAll(createBaseOutput(info));

		//終了状態
		csvFormat.addAll(createEndStatus(info));

		//待ち条件、制御
		csvFormat.addAll(createRule(info));

		//コマンド
		csvFormat.addAll(createCommand(info));

		//ファイル転送
		csvFormat.addAll(createFile(info));

		return csvFormat;
	}

	/**
	 * 終了状態のCSV文字列を作成する。<br>
	 * 
	 * @param info ジョブ情報
	 * @return CSVフォーマット文字列
	 */
	public ArrayList<String> createEndStatus(JobInfo info) {
		ArrayList<String> csvFormat = new ArrayList<String>();
		ArrayList<String> csvNormalOutput = new ArrayList<String>();
		ArrayList<String> csvWarningOutput = new ArrayList<String>();
		ArrayList<String> csvAbnormalOutput = new ArrayList<String>();

		for(int i=0; i < 3; i++){
			csvNormalOutput.add("");
			csvWarningOutput.add("");
		}
		csvAbnormalOutput.add("");

		for(JobEndStatusInfo endStatus : info.getEndStatus()){
			ArrayList<String> type = null;
			if(endStatus.getType() == EndStatusConstant.TYPE_NORMAL){
				//正常終了値
				csvNormalOutput.set(0, String.valueOf(endStatus.getValue()));
				//終了値の範囲(FROM)
				csvNormalOutput.set(1, String.valueOf(endStatus.getStartRangeValue()));
				//終了値の範囲(TO)
				csvNormalOutput.set(2, String.valueOf(endStatus.getEndRangeValue()));
			}
			else if(endStatus.getType() == EndStatusConstant.TYPE_WARNING){
				//警告終了値
				csvWarningOutput.set(0, String.valueOf(endStatus.getValue()));
				//終了値の範囲(FROM)
				csvWarningOutput.set(1, String.valueOf(endStatus.getStartRangeValue()));
				//終了値の範囲(TO)
				csvWarningOutput.set(2, String.valueOf(endStatus.getEndRangeValue()));
			}
			else if(endStatus.getType() == EndStatusConstant.TYPE_ABNORMAL){
				//異常終了値
				csvAbnormalOutput.set(0, String.valueOf(endStatus.getValue()));
			}
		}

		csvFormat.addAll(csvNormalOutput);
		csvFormat.addAll(csvWarningOutput);
		csvFormat.addAll(csvAbnormalOutput);

		return csvFormat;
	}

	/**
	 * 待ち条件及び制御のCSV文字列を作成する。<br>
	 * 
	 * @param info ジョブ情報
	 * @return CSVフォーマット文字列
	 */
	public ArrayList<String> createRule(JobInfo info) {
		ArrayList<String> csvFormat = new ArrayList<String>();

		JobWaitRuleInfo waitRule = info.getWaitRule();

		if (waitRule instanceof JobWaitRuleInfo) {
			//判定対象の条件関係
			csvFormat.add(ConditionTypeConstant.typeToString(waitRule.getCondition()));
			//条件を満たさなければ終了する
			csvFormat.add(YesNoConstant.typeToString(waitRule.getEndCondition()));
			//終了値（条件を満たさなければ終了する）
			csvFormat.add(String.valueOf(waitRule.getEndValue()));

			//カレンダ
			csvFormat.add(YesNoConstant.typeToString(waitRule.getCalendar()));
			//カレンダID
			if(waitRule.getCalendarId() != null)
				csvFormat.add(waitRule.getCalendarId());
			else
				csvFormat.add("");
			//終了値（カレンダ）
			csvFormat.add(String.valueOf(waitRule.getCalendarEndValue()));

			//保留
			csvFormat.add(YesNoConstant.typeToString(waitRule.getSuspend()));

			//スキップ
			csvFormat.add(YesNoConstant.typeToString(waitRule.getSkip()));
			//スキップ(終了値)
			csvFormat.add(String.valueOf(waitRule.getSkipEndValue()));
		} else {
			for(int i=0; i < 9; i++)
				csvFormat.add("");
		}

		return csvFormat;
	}

	/**
	 * コマンドのCSV文字列を作成する。<br>
	 * 
	 * @param info ジョブ情報
	 * @return CSVフォーマット文字列
	 */
	public ArrayList<String> createCommand(JobInfo info) {
		ArrayList<String> csvFormat = new ArrayList<String>();

		JobCommandInfo command = info.getCommand();

		if (command instanceof JobCommandInfo) {
			//スコープ指定
			if(command.getFacilityID().equals(
					SystemParameterConstant.HEADER + 
					SystemParameterConstant.FACILITY_ID + 
					SystemParameterConstant.FOOTER)){
				//スコープ指定方法
				csvFormat.add(com.clustercontrol.util.Messages.getString("job.parameter"));
				//ファシリティID
				csvFormat.add("");
			}
			else{
				//スコープ指定方法
				csvFormat.add(com.clustercontrol.util.Messages.getString("fixed.value"));
				//ファシリティID
				csvFormat.add(command.getFacilityID());
			}

			//処理方法設定
			if (command.getProcessingMethod() == ProcessingMethodConstant.TYPE_ALL_NODE) {
				csvFormat.add(com.clustercontrol.util.Messages.getString("scope.process.all.nodes"));
			} else {
				csvFormat.add(com.clustercontrol.util.Messages.getString("scope.process.retry.nodes"));
			}
			//起動コマンド設定
			if (command.getStartCommand() != null)
				csvFormat.add(command.getStartCommand());
			else
				csvFormat.add("");
			//停止コマンド設定
			if (command.getStopCommand() != null)
				csvFormat.add(command.getStopCommand());
			else
				csvFormat.add("");

			//実効ユーザ
			if (command.getUser() != null)
				csvFormat.add(command.getUser());
			else
				csvFormat.add("");

			//コマンド実効失敗時に終了する
			csvFormat.add(YesNoConstant.typeToString(command.getErrorEndFlg()));
			//終了値(コマンド実効失敗時に終了する)
			csvFormat.add(String.valueOf(command.getErrorEndValue()));

		} else {
			for(int i=0; i < 8; i++)
				csvFormat.add("");
		}

		return csvFormat;
	}

	/**
	 * ファイル転送のCSV文字列を作成する。<br>
	 * 
	 * @param info ジョブ情報
	 * @return CSVフォーマット文字列
	 */
	public ArrayList<String> createFile(JobInfo info) {
		ArrayList<String> csvFormat = new ArrayList<String>();

		JobFileInfo file = info.getFile();

		if (file instanceof JobFileInfo) {
			//転送ファシリティID
			if (file.getSrcFacilityID() != null)
				csvFormat.add(file.getSrcFacilityID());
			else
				csvFormat.add("");
			//転送ファイル
			if (file.getSrcFile() != null)
				csvFormat.add(file.getSrcFile());
			else
				csvFormat.add("");
			//受信ファシリティID
			if (file.getDestFacilityID() != null)
				csvFormat.add(file.getDestFacilityID());
			else
				csvFormat.add("");
			//処理方法設定
			if (file.getProcessingMethod() == ProcessingMethodConstant.TYPE_ALL_NODE) {
				csvFormat.add(com.clustercontrol.util.Messages.getString("scope.process.all.nodes"));
			} else {
				csvFormat.add(com.clustercontrol.util.Messages.getString("scope.process.retry.nodes"));
			}
			//受信ディレクトリ
			if (file.getDestDirectory() != null)
				csvFormat.add(file.getDestDirectory());
			else
				csvFormat.add("");
			//ファイル転送時に圧縮する
			csvFormat.add(YesNoConstant.typeToString(file.getCompressionFlg()));
			//転送ファイルのチェックを行う
			csvFormat.add(YesNoConstant.typeToString(file.getCheckFlg()));
			//実効ユーザ
			if (file.getUser() != null)
				csvFormat.add(file.getUser());
			else
				csvFormat.add("");
		} else {
			for(int i=0; i < 8; i++)
				csvFormat.add("");
		}

		return csvFormat;
	}

	/**
	 * 待ち条件情報取得（再帰呼び出し）
	 * 
	 * @param treeItem ジョブツリー
	 * @param csvOutput CSV出力文字列配列
	 */
	public void getJobWaitRule(JobTreeItem treeItem, ArrayList<ArrayList<String>> csvOutput) {

		//子ツリーを取得する
		if (treeItem.getData() instanceof JobInfo){
			JobInfo info = treeItem.getData();
			if(info.getId() != null && info.getId().compareTo("") != 0){
				Object[] args = {info.getId()};
				log.info(Messages.getMsg("JobID", args));

				if(info.getWaitRule() instanceof JobWaitRuleInfo){
					for(JobObjectInfo object : info.getWaitRule().getObject()){
						ArrayList<String> csvFormat = new ArrayList<String>();
						//ジョブID
						csvFormat.add(info.getId());
						csvFormat.addAll(createWairRuleCsvFormat(object));
						if(csvFormat.size() > 1)
							csvOutput.add(csvFormat);
					}
				}
			}
		}

		JobTreeItem[] childrenTreeItem = treeItem.getChildren();
		for (JobTreeItem childTreeItem : childrenTreeItem) {
			getJobWaitRule(childTreeItem, csvOutput);
		}
	}

	/**
	 * 待ち条件のCSV文字列を作成する。<br>
	 * 
	 * @param object 待ち条件情報
	 * @return CSVフォーマット文字列
	 */
	public ArrayList<String> createWairRuleCsvFormat(JobObjectInfo object) {
		ArrayList<String> csvFormat = new ArrayList<String>();
		SimpleDateFormat formatter = null;

		if(object instanceof JobObjectInfo){
			//待ち条件種別
			csvFormat.add(JudgmentObjectConstant.typeToString(object.getType()));
			if (object.getType() == JudgmentObjectConstant.TYPE_JOB_END_STAUS) {
				//ジョブID
				csvFormat.add(object.getJobId());
				//終了状態
				csvFormat.add(EndStatusConstant.typeToString(object.getValue()));
			}
			else if (object.getType() == JudgmentObjectConstant.TYPE_JOB_END_VALUE) {
				//ジョブID
				csvFormat.add(object.getJobId());
				//終了値
				csvFormat.add(String.valueOf(object.getValue()));
			}
			else if (object.getType() == JudgmentObjectConstant.TYPE_TIME) {
				//ジョブID
				csvFormat.add("");
				//時刻
				if(formatter == null)
					formatter = new SimpleDateFormat("HH:mm");
				csvFormat.add(formatter.format(object.getTime()));
			}
		}

		return csvFormat;
	}

	/**
	 * 通知情報取得（再帰呼び出し）
	 * 
	 * @param treeItem ジョブツリー
	 * @param csvOutput CSV出力文字列配列
	 */
	public void getJobNotify(JobTreeItem treeItem, ArrayList<ArrayList<String>> csvOutput) {

		//子ツリーを取得する
		if (treeItem.getData() instanceof JobInfo){
			JobInfo info = treeItem.getData();
			if(info.getId() != null && info.getId().compareTo("") != 0){
				Object[] args = {info.getId()};
				log.info(Messages.getMsg("JobID", args));

				csvOutput.add(createNotifyCsvFormat(info));
			}
		}

		JobTreeItem[] childrenTreeItem = treeItem.getChildren();
		for (JobTreeItem childTreeItem : childrenTreeItem) {
			getJobNotify(childTreeItem, csvOutput);
		}
	}

	/**
	 * 通知情報のCSV文字列を作成する。<br>
	 * 
	 * @param info ジョブ情報
	 * @return CSVフォーマット文字列
	 */
	public ArrayList<String> createNotifyCsvFormat(JobInfo info) {
		ArrayList<String> csvFormat = new ArrayList<String>();
		String start = "";
		String normal = "";
		String warning = "";
		String abnormal = "";

		JobNotificationsInfo normalNotify = null;

		//ジョブID
		csvFormat.add(info.getId());

		for(JobNotificationsInfo notify : info.getNotifications()){
			if (notify.getType() == EndStatusConstant.TYPE_BEGINNING) {
				start = PriorityConstant.typeToString(notify.getPriority());
			} else if (notify.getType() == EndStatusConstant.TYPE_NORMAL) {
				normalNotify = notify;
				normal = PriorityConstant.typeToString(notify.getPriority());
			} else if (notify.getType() == EndStatusConstant.TYPE_WARNING) {
				warning = PriorityConstant.typeToString(notify.getPriority());
			} else if (notify.getType() == EndStatusConstant.TYPE_ABNORMAL) {
				abnormal = PriorityConstant.typeToString(notify.getPriority());
			}
		}

		csvFormat.add(start);
		csvFormat.add(normal);
		csvFormat.add(warning);
		csvFormat.add(abnormal);
		csvFormat.addAll(createNotify(normalNotify));

		return csvFormat;
	}

	/**
	 * 通知IDのCSV文字列を作成する。<br>
	 * 
	 * @param nifityInfo ジョブ通知情報
	 * @return CSVフォーマット文字列
	 */
	protected ArrayList<String> createNotify(JobNotificationsInfo nifityInfo) {
		ArrayList<String> csvFormat = new ArrayList<String>();

		String status = "";
		String event = "";
		String mail = "";
		String job = "";
		String syslog = "";

		if(nifityInfo != null && nifityInfo.getNotifyId() != null){
			Collection<NotifyRelationInfo> notify = nifityInfo.getNotifyId();

			NotifyRelationInfo relationInfo;
			for (int i = 0; i < notify.size(); i++) {
				relationInfo = (NotifyRelationInfo)((ArrayList<NotifyRelationInfo>)notify).get(i);
				if(relationInfo.getNotifyType() == NotifyTypeConstant.TYPE_STATUS)
					status = relationInfo.getNotifyId();
				else if(relationInfo.getNotifyType() == NotifyTypeConstant.TYPE_EVENT)
					event = relationInfo.getNotifyId();
				else if(relationInfo.getNotifyType() == NotifyTypeConstant.TYPE_MAIL)
					mail = relationInfo.getNotifyId();
				else if(relationInfo.getNotifyType() == NotifyTypeConstant.TYPE_JOB)
					job = relationInfo.getNotifyId();
				else if(relationInfo.getNotifyType() == NotifyTypeConstant.TYPE_LOG_ESCALATE)
					syslog = relationInfo.getNotifyId();
			}
		}

		csvFormat.add(status);
		csvFormat.add(event);
		csvFormat.add(mail);
		csvFormat.add(job);
		csvFormat.add(syslog);

		return csvFormat;
	}

	/**
	 * 遅延情報取得（再帰呼び出し）
	 * 
	 * @param treeItem ジョブツリー
	 * @param csvOutput CSV出力文字列配列
	 */
	public void getJobDelay(JobTreeItem treeItem, ArrayList<ArrayList<String>> csvOutput) {

		//子ツリーを取得する
		if (treeItem.getData() instanceof JobInfo){
			JobInfo info = treeItem.getData();
			if(info.getId() != null && info.getId().compareTo("") != 0){
				Object[] args = {info.getId()};
				log.info(Messages.getMsg("JobID", args));

				if (info.getType() != JobConstant.TYPE_JOBUNIT && 
						info.getWaitRule() instanceof JobWaitRuleInfo) {
					csvOutput.add(createDelayCsvFormat(info));
				}
			}
		}

		JobTreeItem[] childrenTreeItem = treeItem.getChildren();
		for (JobTreeItem childTreeItem : childrenTreeItem) {
			getJobDelay(childTreeItem, csvOutput);
		}
	}

	/**
	 * 遅延情報のCSV文字列を作成する。<br>
	 * 
	 * @param info ジョブ情報
	 * @return CSVフォーマット文字列
	 */
	public ArrayList<String> createDelayCsvFormat(JobInfo info) {
		ArrayList<String> csvFormat = new ArrayList<String>();

		JobWaitRuleInfo waitRule = info.getWaitRule();

		//ジョブID
		csvFormat.add(info.getId());

		if (waitRule instanceof JobWaitRuleInfo) {
			//開始遅延
			csvFormat.add(YesNoConstant.typeToString(waitRule.getStart_delay()));
			//セッション開始後の時間
			csvFormat.add(YesNoConstant.typeToString(waitRule.getStart_delay_session()));
			//セッション開始後の時間の値
			csvFormat.add(String.valueOf(waitRule.getStart_delay_session_value()));
			//時刻
			csvFormat.add(YesNoConstant.typeToString(waitRule.getStart_delay_time()));
			//時刻の値
			if (waitRule.getStart_delay_time_value() instanceof Date) {
				SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
				csvFormat.add(formatter.format(waitRule.getStart_delay_time_value()));
			}
			else
				csvFormat.add("");
			//条件関係設定
			csvFormat.add(ConditionTypeConstant.typeToString(waitRule.getStart_delay_condition_type()));
			//通知
			csvFormat.add(YesNoConstant.typeToString(waitRule.getStart_delay_notify()));
			//通知の重要度
			csvFormat.add(PriorityConstant.typeToString(waitRule.getStart_delay_notify_priority()));
			//操作
			csvFormat.add(YesNoConstant.typeToString(waitRule.getStart_delay_operation()));
			//操作の名前
			if (waitRule.getStart_delay_operation_type() == OperationConstant.TYPE_STOP_SKIP)
				csvFormat.add(OperationConstant.STRING_STOP_SKIP);
			else if (waitRule.getStart_delay_operation_type() == OperationConstant.TYPE_STOP_WAIT)
				csvFormat.add(OperationConstant.STRING_STOP_WAIT);
			else
				csvFormat.add(OperationConstant.STRING_STOP_SKIP);
			//操作の終了値
			csvFormat.add(String.valueOf(waitRule.getStart_delay_operation_end_value()));

			//終了遅延
			csvFormat.add(YesNoConstant.typeToString(waitRule.getEnd_delay()));
			//セッション開始後の時間
			csvFormat.add(YesNoConstant.typeToString(waitRule.getEnd_delay_session()));
			//セッション開始後の時間の値
			csvFormat.add(String.valueOf(waitRule.getEnd_delay_session_value()));
			//ジョブ開始後の時間
			csvFormat.add(YesNoConstant.typeToString(waitRule.getEnd_delay_job()));
			//ジョブ開始後の時間の値
			csvFormat.add(String.valueOf(waitRule.getEnd_delay_job_value()));
			//時刻
			csvFormat.add(YesNoConstant.typeToString(waitRule.getEnd_delay_time()));
			//時刻の値
			if (waitRule.getEnd_delay_time_value() instanceof Date) {
				SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
				csvFormat.add(formatter.format(waitRule.getEnd_delay_time_value()));
			}
			else
				csvFormat.add("");
			//条件関係設定
			csvFormat.add(ConditionTypeConstant.typeToString(waitRule.getEnd_delay_condition_type()));
			//通知
			csvFormat.add(YesNoConstant.typeToString(waitRule.getEnd_delay_notify()));
			//通知の重要度
			csvFormat.add(PriorityConstant.typeToString(waitRule.getEnd_delay_notify_priority()));
			//操作
			csvFormat.add(YesNoConstant.typeToString(waitRule.getEnd_delay_operation()));
			//操作の名前
			if (waitRule.getEnd_delay_operation_type() == OperationConstant.TYPE_STOP_AT_ONCE)
				csvFormat.add(OperationConstant.STRING_STOP_AT_ONCE);
			else if (waitRule.getEnd_delay_operation_type() == OperationConstant.TYPE_STOP_SUSPEND)
				csvFormat.add(OperationConstant.STRING_STOP_SUSPEND);
			else if (waitRule.getEnd_delay_operation_type() == OperationConstant.TYPE_STOP_SET_END_VALUE)
				csvFormat.add(OperationConstant.STRING_STOP_SET_END_VALUE);
			else
				csvFormat.add(OperationConstant.STRING_STOP_AT_ONCE);
			//操作の終了値
			csvFormat.add(String.valueOf(waitRule.getEnd_delay_operation_end_value()));
		} else {
			for(int i=0; i < 24; i++)
				csvFormat.add("");
		}

		return csvFormat;
	}

	/**
	 * パラメータ情報取得（再帰呼び出し）
	 * 
	 * @param treeItem ジョブツリー
	 * @param csvOutput CSV出力文字列配列
	 */
	public void getJobParam(JobTreeItem treeItem, ArrayList<ArrayList<String>> csvOutput) {

		//子ツリーを取得する
		if (treeItem.getData() instanceof JobInfo){
			JobInfo info = treeItem.getData();
			if(info.getId() != null && info.getId().compareTo("") != 0){
				Object[] args = {info.getId()};
				log.info(Messages.getMsg("JobID", args));

				if(info.getParam() != null){
					ArrayList<JobParameterInfo> paramList = info.getParam();
					for(JobParameterInfo param : paramList){
						ArrayList<String> csvFormat = new ArrayList<String>();
						//ジョブID
						csvFormat.add(info.getId());
						csvFormat.addAll(createParamCsvFormat(param));
						if(csvFormat.size() > 1)
							csvOutput.add(csvFormat);
					}
				}
			}
		}

		JobTreeItem[] childrenTreeItem = treeItem.getChildren();
		for (JobTreeItem childTreeItem : childrenTreeItem) {
			getJobParam(childTreeItem, csvOutput);
		}
	}

	/**
	 * パラメータ情報のCSV文字列を作成する。<br>
	 * 
	 * @param info ジョブ情報
	 * @return CSVフォーマット文字列
	 */
	public ArrayList<String> createParamCsvFormat(JobParameterInfo info) {
		ArrayList<String> csvFormat = new ArrayList<String>();

		if(info instanceof JobParameterInfo){
			//名前
			csvFormat.add(info.getParamId());
			//種別
			csvFormat.add(JobParamTypeConstant.typeToString(info.getType()));
			//値
			if (info.getType() == JobParamTypeConstant.TYPE_USER)
				csvFormat.add(String.valueOf(info.getValue()));
			else
				csvFormat.add("");
			//説明
			csvFormat.add(info.getDescription());
		}

		return csvFormat;
	}

	/**
	 * ジョブツリーを取得する。<br>
	 * 
	 * @return ジョブツリー
	 */
	protected JobTreeItem getJobTree() {

		JobController job = EjbConnectionManager.getConnectionManager().getJobController();

		JobTreeItem jobTree = null;
		try {
			jobTree = job.getJobTree(false, Locale.getDefault());
		} catch (Exception e) {
			log.error(Messages.getMsg("ConnectManagerFailed"), e);
			System.exit(14);
		}
		return jobTree;
	}
}