/*

Copyright (C) 2009 NTT DATA INTELLILINK CORPORATION

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.

 */

package jp.co.intellilink.hinemos.export.history;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.Iterator;

import jp.co.intellilink.hinemos.export.util.Config;
import jp.co.intellilink.hinemos.export.util.EjbConnectionManager;
import jp.co.intellilink.hinemos.export.util.Messages;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.performance.bean.CollectorProperty;
import com.clustercontrol.performance.ejb.bmp.RecordCollectorData;
import com.clustercontrol.performance.ejb.session.CollectorController;
import com.clustercontrol.repository.ejb.session.RepositoryController;

/**
 * 性能管理機能の実績収集結果をCSV出力するクラス<br>
 * 
 * @version 1.1.0
 * @since 1.1.0
 */
public class ExportPerformance {

	// ログ出力
	private static Log log = LogFactory.getLog(ExportPerformance.class);

	/**
	 * 性能管理機能の実績収集結果をCSV出力する。<br>
	 * 
	 * @param startDate 開始日時
	 */
	public void export(Date startDate) {
		log.info(Messages.getMsg("ExportTool.Performance.Start"));

		//CSV出力配列を用意
		ArrayList<ArrayList<String>> csvOutput = new ArrayList<ArrayList<String>>();

		//収集項目リスト取得
		ArrayList idList = getPerformanceList();

		Calendar cal = Calendar.getInstance();
		cal.setTime(startDate);
		cal.set(Calendar.HOUR_OF_DAY, 0);
		cal.set(Calendar.MINUTE, 0);
		cal.set(Calendar.SECOND, 0);
		cal.set(Calendar.MILLISECOND, 0);
		Date startTime = cal.getTime();

		cal.add(Calendar.DATE, 1);
		cal.add(Calendar.MILLISECOND, -1);
		Date endTime = cal.getTime();

		if(idList instanceof ArrayList){
			Iterator itr = idList.iterator();
			while(itr.hasNext()){
				ArrayList line = (ArrayList)itr.next();

				Object[] args = {(String)line.get(2)};
				log.info(Messages.getMsg("ExportTool.CollectorID", args));

				CollectorProperty property = getPerformance((String)line.get(2));
				createPerformance(property, startTime, endTime);
			}
		}

		log.info(Messages.getMsg("ExportTool.Performance.End"));
	}

	/**
	 * 性能管理機能の実績収集結果のCSVファイルを出力する。<br>
	 * 
	 * @param property 性能管理機能の実績収集情報
	 * @param startTime 開始日時
	 * @param endTime 終了日時
	 */
	public void createPerformance(CollectorProperty property, Date startTime, Date endTime) {
		ArrayList<String> facilityIdList = new ArrayList<String>();

		//監視結果のファシリティIDリストを取得
		RecordCollectorData record = property.getCollectorData();
		facilityIdList.addAll(getNodeList(record.getFacilityID()));

		for(String facilityId : facilityIdList){
			Object[] args = {facilityId};
			log.info(Messages.getMsg("ExportTool.FacilityID", args));

			ExportRecordData export = 
				new ExportRecordData(property, facilityId, startTime, endTime);

			SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd");
			String fileName = Config.getConfig("Output.Dir.History") + "/PERF_" + record.getCollectorID() + "_" + facilityId + "_" + format.format(startTime) + ".csv";

			export.write(fileName);
		}
	}

	/**
	 * ノードのファシリティIDを取得する。<br>
	 * 
	 * @param facilityId ファシリティID
	 * @return ノード一覧
	 */
	protected ArrayList<String> getNodeList(String facilityId) {

		RepositoryController repository = EjbConnectionManager.getConnectionManager().getRepositoryController();

		ArrayList<String> records = new ArrayList<String>();
		try {
			if(repository.isNode(facilityId)){
				records.add(facilityId);
			}
			else{
				records.addAll(repository.getExecTargetFacilityIdList(facilityId));
			}
		} catch (Exception e) {
			log.error(Messages.getMsg("ExportTool.ConnectManagerFailed"), e);
			System.exit(14);
		}
		return records;
	}

	/**
	 * 性能管理機能の実績収集情報を取得する。<br>
	 * 
	 * @param collectorId 取得対象の収集ID
	 * @return 性能管理機能の実績収集情報
	 */
	protected CollectorProperty getPerformance(String collectorId) {

		CollectorController perf = EjbConnectionManager.getConnectionManager().getCollectorController();

		CollectorProperty property = null;
		try {
			property = perf.getCollectorProperty(collectorId);
		} catch (Exception e) {
			log.error(Messages.getMsg("ExportTool.ConnectManagerFailed"), e);
			System.exit(14);
		}
		return property;
	}

	/**
	 * 収集項目リストを取得する。<br>
	 * 
	 * @return 収集項目一覧
	 */
	protected ArrayList getPerformanceList() {

		CollectorController perf = EjbConnectionManager.getConnectionManager().getCollectorController();

		ArrayList records = null;
		try {
			records = perf.getCollectionList();
		} catch (Exception e) {
			log.error(Messages.getMsg("ExportTool.ConnectManagerFailed"), e);
			System.exit(14);
		}
		return records;
	}
}