/*
 
Copyright (C) 2008 NTT DATA INTELLILINK CORPORATION
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package jp.co.intellilink.hinemos.export.job;

import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;

import jp.co.intellilink.hinemos.export.util.Config;
import jp.co.intellilink.hinemos.export.util.EjbConnectionManager;
import jp.co.intellilink.hinemos.export.util.Messages;
import jp.co.intellilink.hinemos.export.util.StringListComparator;
import jp.co.intellilink.hinemos.export.util.WriteCsvFile;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.DayOfWeekConstant;
import com.clustercontrol.bean.Schedule;
import com.clustercontrol.bean.ScheduleConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.jobmanagement.bean.ScheduleTableDefine;
import com.clustercontrol.jobmanagement.ejb.session.JobController;

/**
 * スケジュール情報を取得するクラス<br>
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class ExportSchedule {
	
	// ログ出力
	private static Log log = LogFactory.getLog(ExportSchedule.class);

    /**
     * スケジュール情報をExportする。<br>
     */
    public void exportSchedule() {
    	log.info(Messages.getMsg("ExportTool.Job.Schedule.Start"));
    	
    	//CSV出力配列を用意
    	ArrayList<ArrayList<String>> csvOutput = new ArrayList<ArrayList<String>>();
    	
    	//スケジュール一覧取得
    	ArrayList list = getScheduleList();
    	if(list instanceof ArrayList){
			Iterator itr = list.iterator();
			while(itr.hasNext()){
				ArrayList line = (ArrayList)itr.next();
				
				Object[] args = {(String)line.get(ScheduleTableDefine.SCHE_ID)};
				log.info(Messages.getMsg("ExportTool.ScheduleID", args));
				csvOutput.add(createCsvFormat(line));
			}
			
			//スケジュールIDでソート
			Collections.sort(csvOutput, new StringListComparator(0));
    	}
		
		SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd_HHmmss");
		WriteCsvFile.writeCsvFile(
				Config.getConfig("Output.Dir") + "/JOB_SCHEDULE_" + format.format(new Date()) + ".csv", 
				csvOutput);
		
		log.info(Messages.getMsg("ExportTool.Job.Schedule.End"));
    }
    
    /**
	 * スケジュール情報をExportする。<br>
	 * 
	 * @param info スコープ情報
	 * @return CSVフォーマット文字列
	 */
	public ArrayList<String> createCsvFormat(ArrayList info) {
		ArrayList<String> csvFormat = new ArrayList<String>();
		
        //スケジュールID
        csvFormat.add(getStringValue(info, ScheduleTableDefine.SCHE_ID));

        //スケジュール名設定
        csvFormat.add(getStringValue(info, ScheduleTableDefine.SCHE_NAME));
        
        //ジョブID設定
        csvFormat.add(getStringValue(info, ScheduleTableDefine.JOB_ID));

        //ジョブ名設定
        csvFormat.add(getStringValue(info, ScheduleTableDefine.JOB_NAME));

        //カレンダID
        csvFormat.add(getStringValue(info, ScheduleTableDefine.CALENDAR_ID));
        
        //スケジュール設定
        csvFormat.addAll(createSchedule((Schedule)info.get(ScheduleTableDefine.SCHEDULE)));
        
		//有効/無効
		csvFormat.add(ValidConstant.typeToString((Integer)info.get(ScheduleTableDefine.VALID)));
		
		return csvFormat;
	}
	
	/**
     * スケジュール情報をExportする。<br>
     * 
     * @param info メンテナンス情報
     * @return CSVフォーマット文字列
     */
	protected ArrayList<String> createSchedule(Schedule schedule) {
		ArrayList<String> csvFormat = new ArrayList<String>();

        //指定方式
        if (schedule.getType() == ScheduleConstant.TYPE_EVERY_WEEK)
        	csvFormat.add(com.clustercontrol.util.Messages.getString("weekday"));
        else
        	csvFormat.add(com.clustercontrol.util.Messages.getString("time"));
        
        //日時を設定
        if (schedule.getDate() instanceof Date) {
        	Calendar calendar = Calendar.getInstance();
        	calendar.setTime(schedule.getDate());

        	DecimalFormat format = new DecimalFormat("00");
        	
            if (schedule.getType() == ScheduleConstant.TYPE_EVERY_YEAR) {
		    	//月を設定
		    	csvFormat.add(format.format(calendar.get(Calendar.MONTH) + 1));
		    	//日を設定
		    	csvFormat.add(format.format(calendar.get(Calendar.DAY_OF_MONTH)));
		    	//時を設定
		    	csvFormat.add(format.format(calendar.get(Calendar.HOUR_OF_DAY)));
		    	//分を設定
		    	csvFormat.add(format.format(calendar.get(Calendar.MINUTE)));
            } else if (schedule.getType() == ScheduleConstant.TYPE_EVERY_MONTH) {
		    	//月を設定
            	csvFormat.add("");
		    	//日を設定
		    	csvFormat.add(format.format(calendar.get(Calendar.DAY_OF_MONTH)));
		    	//時を設定
		    	csvFormat.add(format.format(calendar.get(Calendar.HOUR_OF_DAY)));
		    	//分を設定
		    	csvFormat.add(format.format(calendar.get(Calendar.MINUTE)));
            } else if (schedule.getType() == ScheduleConstant.TYPE_EVERY_DAY) {
		    	//月を設定
            	csvFormat.add("");
		    	//日を設定
            	csvFormat.add("");
		    	//時を設定
		    	csvFormat.add(format.format(calendar.get(Calendar.HOUR_OF_DAY)));
		    	//分を設定
		    	csvFormat.add(format.format(calendar.get(Calendar.MINUTE)));
            } else if (schedule.getType() == ScheduleConstant.TYPE_EVERY_HOUR) {
		    	//月を設定
            	csvFormat.add("");
		    	//日を設定
            	csvFormat.add("");
		    	//時を設定
            	csvFormat.add("");
		    	//分を設定
		    	csvFormat.add(format.format(calendar.get(Calendar.MINUTE)));
            } else {
		    	//月を設定
            	csvFormat.add("");
		    	//日を設定
		    	csvFormat.add("");
		    	//時を設定
		    	csvFormat.add("");
		    	//分を設定
		    	csvFormat.add("");
        	}
        	
        	if (schedule.getType() == ScheduleConstant.TYPE_EVERY_WEEK){
	        	//曜日を設定
	        	csvFormat.add(DayOfWeekConstant.typeToString(schedule.getDayOfWeek()));
	        	//時を設定
	        	csvFormat.add(format.format(calendar.get(Calendar.HOUR_OF_DAY)));
	        	//分を設定
	        	csvFormat.add(format.format(calendar.get(Calendar.MINUTE)));
        	}
        	else{
        		//曜日を設定
        		csvFormat.add("");
		    	//時を設定
		    	csvFormat.add("");
		    	//分を設定
		    	csvFormat.add("");
        	}
        }
        else{
    		csvFormat.add("");
    		csvFormat.add("");
    		csvFormat.add("");
    		csvFormat.add("");
    		csvFormat.add("");
    		csvFormat.add("");
    		csvFormat.add("");
        }
		
		return csvFormat;
	}
	
	/**
	 * 配列から文字列を取得する。<br>
	 * 
	 * @param values 配列
	 * @param index インデックス
	 * @return 文字列
	 */
	public String getStringValue(ArrayList values, int index) {
		if(values.size() > index - 1 && values.get(index) instanceof String)
			return (String)values.get(index);
		else
			return "";
	}
    
    /**
     * スケジュール一覧を取得する。<br>
     * 
     * @return スケジュール一覧
     */
    protected ArrayList getScheduleList() {

    	JobController job = EjbConnectionManager.getConnectionManager().getJobController();

    	ArrayList list = null;
        try {
        	list = job.getScheduleList();
        } catch (Exception e) {
			log.error(Messages.getMsg("ExportTool.ConnectManagerFailed"), e);
			System.exit(14);
		}
        return list;
    }
}