/**
 *  Impulser2
 *
 *  Copyright (C) 2006-2014 Teru Kamogashira
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef __Freeverb_H
#define __Freeverb_H

#define EFFECT_NAME "Freeverb3_Impulser2"

#include <new>
#include <vector>
#include <string>
#include <cstdio>
#include <cstdlib>
#include <locale.h>
#include <wchar.h>
#include <cstring>

#include <audioeffectx.h>

#include <libsamplerate2/samplerate2.h>
#include <sndfile.h>
#include <sndfile.hh>

#include <freeverb/irmodel3.hpp>
#include <freeverb/limitmodel.hpp>
#include <freeverb/slot.hpp>
#include <freeverb/utils.hpp>
#include <freeverb/fv3_ch_tool.hpp>
#include <freeverb/efilter.hpp>
#include "CFileLoader.hpp"
#include "CChunk.hpp"
#include "IRPrograms.hpp"
#include "ProcessBlock.hpp"
#include "FileLog.hpp"
#include "Locker.hpp"

#if WIN32
#include <freeverb/irmodel3w.hpp>
#else
#include <freeverb/irmodel3p.hpp>
#endif

#include "fv3_config.h"

#ifdef PLUGDOUBLE
typedef fv3::irmodel3_ IR2;
#if WIN32
typedef fv3::irmodel3w_ IR3W;
#else
typedef fv3::irmodel3p_ IR3W;
#endif
typedef fv3::limitmodel_ LIMITM;
typedef fv3::utils_ UTILS;
typedef fv3::CFileLoader_ CFILELOADER;
typedef double pfloat_t;
#else
typedef fv3::irmodel3_f IR2;
#if WIN32
typedef fv3::irmodel3w_f IR3W;
#else
typedef fv3::irmodel3p_f IR3W;
#endif
typedef fv3::limitmodel_f LIMITM;
typedef fv3::utils_f UTILS;
typedef fv3::CFileLoader_f CFILELOADER;
typedef float pfloat_t;
#endif

class MidiChannelHandle
{
public:
  MidiChannelHandle()
  {
    Status = Note = Velocity = MSB = LSB = 0;
  }
  ~MidiChannelHandle(){}
  VstInt32 Status, Note, Velocity, MSB, LSB, Program;
};

// midi Event Executer

enum {
  KMEventNull = 0,
  KMEventLoadFG = 1,
  KMEventLoadBG = 2,
  KMEventUnloadFG = 3,
  KMEventUnloadBG = 4,
};

typedef struct {
  int kmevent, channel;
  void *fv, *fve;
  std::string targetFilename;
  pfloat_t ta, th, td, ts, tr;
  bool errorMessage;
} lfEventExecuterInfo;

typedef struct {
  void *fv, **fve;
  std::vector<lfEventExecuterInfo> * ecue;
  volatile int * flags;
  Locker * cueLocker;
#ifndef WIN32
  PthreadEvent * event;
#endif
} lfEventClassInfo;

#define FV_EVENT_PREFIX L"FV_EVENT"
#define MODEL_SLOT_SIZE 10

class Freeverb :
  public AudioEffectX,
  public ProcessBlock,
  public FileLog
{
public:
  Freeverb(audioMasterCallback audioMaster) ALIGN_ARG_POINTER;
  virtual ~Freeverb();
  virtual VstInt32 processEvents(VstEvents* ev);
  virtual VstPlugCategory getPlugCategory();
  virtual void getProgramName(char *name);
  virtual void setProgramName(char *name);
  virtual bool getProgramNameIndexed(VstInt32 category, VstInt32 index, char* text);
  virtual void getParameterLabel(VstInt32 index, char *label);
  virtual void getParameterDisplay(VstInt32 index, char *text);
  virtual void getParameterName(VstInt32 index, char *text);
  virtual bool getEffectName(char* name);
  virtual bool getVendorString(char* text);
  virtual bool getProductString(char* text);
  virtual VstInt32 canDo(char* text);
  virtual VstInt32 getChunk(void ** data, bool isPreset);
  virtual VstInt32 setChunk(void * data, VstInt32 byteSize, bool isPreset);
  virtual bool setBypass(bool onOff);
  virtual bool getInputProperties(VstInt32 index, VstPinProperties* properties);
  virtual bool getOutputProperties(VstInt32 index, VstPinProperties* properties);

  virtual void  suspend() ALIGN_ARG_POINTER;
  virtual void  resume() ALIGN_ARG_POINTER;
  virtual void  setSampleRate (float sampleRate) ALIGN_ARG_POINTER;
  virtual float getSampleRate() ALIGN_ARG_POINTER;
  virtual void  setParameter(VstInt32 index, float value) ALIGN_ARG_POINTER;
  virtual float getParameter(VstInt32 index) ALIGN_ARG_POINTER;
  virtual void process               (float **inputs, float **outputs, VstInt32 sampleFrames) ALIGN_ARG_POINTER;
  virtual void processReplacing      (float **inputs, float **outputs, VstInt32 sampleFrames) ALIGN_ARG_POINTER;
#ifdef PLUGDOUBLE
  virtual void processDoubleReplacing(double **inputs, double **outputs, VstInt32 sampleFrames) ALIGN_ARG_POINTER;
#endif
  virtual void processLRModel(pfloat_t *inL, pfloat_t *inR, pfloat_t *ouL, pfloat_t *outR, VstInt32 sampleFrames) ALIGN_ARG_POINTER;
  
  void setThreadPriority(int priority);
  void setConverterType(int type);  
  int  getConverterType();
  void setLFragmentSize(int size);
  void setFragmentSize(int size);
  void setFactor(int size);
  void setGUIZoomFactor(float factor);
  void setZeroLatency(int on);
  void setMT(int on);
  
  void setIRProgsFileName(char * filename);
  int  registerIRPrograms();
  
  void setSlotFileNameA(int slot, const char * path);
  const char * getSlotFileNameA(int slot);
  
  void pushLoadCueA(int slot, const char * filename);
  void pushReloadCue(int slot);
  void loadToSlot(int slot, int rear, long size, const pfloat_t * L, const pfloat_t * R);
  void unloadSlot(int slot, int rear);
  int  getCurrentSlot();
  void setCurrentSlot(int slot);
  float getNRTParameter(VstInt32 index);
  void  setNRTParameter(VstInt32 index, float value);
  
private:
  Freeverb(const Freeverb& x);
  Freeverb& operator=(const Freeverb& x);
  void initModels();
  // Global
  int callProcess, callDouble, callReplace, callEvents;
  char programName[32];
  double currentFs;
  VstInt32 currentBlockSize;
  CChunk chunk;
  IRPrograms irPrograms;
  int converter_type, conf_fragmentSize, conf_factor;
  int currentSlot;
  bool zl, conf_mt, byPass, skipLimiter;
  float dryValue, dryValueDB;
  pfloat_t dryValueR;

  // Slot {
  IR2 * model[MODEL_SLOT_SIZE][2];
  bool muteWet[MODEL_SLOT_SIZE][2], m2s[MODEL_SLOT_SIZE], swap[MODEL_SLOT_SIZE];
  float conf_idelay[MODEL_SLOT_SIZE][2], conf_stretch[MODEL_SLOT_SIZE][2],
    conf_attack[MODEL_SLOT_SIZE][2], conf_hold[MODEL_SLOT_SIZE][2],
    conf_decay[MODEL_SLOT_SIZE][2], conf_sustain[MODEL_SLOT_SIZE][2],
    conf_release[MODEL_SLOT_SIZE][2];
  std::string slotFileNameA[MODEL_SLOT_SIZE];
  // }

  // Limiter
  LIMITM limiter;
  
  // lockModel
  bool validModel;
  Locker eventCueLocker, irSlotLocker[MODEL_SLOT_SIZE][2], globalLocker;

  volatile int threadFlags;
  lfEventClassInfo hostThreadData;
#ifdef WIN32
  HANDLE lEventThreadHandle, event;
  unsigned int threadId;
  wchar_t eventName[_MAX_PATH];
#else
  pthread_t lEventThreadHandle;
  PthreadEvent event;
#endif

  std::vector<lfEventExecuterInfo> eventCue;

  // IR programs
  std::string irProgramsFilename;

  // Midi Channel Handler (0~15)
  MidiChannelHandle MCHandle[0x10];
};

// VST Automation Parameters

// SlotParameters{KNumParams}[MODEL_SLOT_SIZE]
// Global Parameters

enum {
  // <<Slot Parameters>>
  // Realtime parameters
  KWidth = 0, KWet = 1, KMuteWet = 2, KLPF = 3, KHPF = 4, KM2S = 5, KSwap = 6, KDelay = 7, KLRBalance = 8,
  KWet2 = 9, KMuteWet2 = 10, KDelay2 = 11, KLRBalance2 = 12,
  KMaxParams, KNumParams = 16, // <- must not be changed
  // valueChangedTag
  kKickTag = 100, kUnloadTag = 101, kAbout = 102, kResetTag = 103,
  // Internal (controllable from GUI only)
  KInternal = 300, KIRStretch = 300, KIRAttack = 301, KIRHold = 302, KIRDecay = 303, KIRSustain = 304, KIRRelease = 305,
  KIMaxParams, KINumParamMax = 316,
  // SlotSelector
  KSlotHead = 400,
  // overallTag
  KOATag = 1000,
  // realNumberTag
  KRealTag = 10000,
};


enum {
  KChunkFileName = 0, KChunkStretch = 1, KChunkOldNum = 2,
  KChunkAttack = 2, KChunkHold = 3, KChunkDecay = 4, KChunkSustain = 5, KChunkRelease = 6,
  KChunkNums, KChunkNumMax = 32, KChunkIParamsH = 1000,
};

// Global parameters
enum {
  KOARelease = 0, KOACeiling = 1, KOAThreshold = 2, KOADry = 3, KOASkipLimiter = 4, KOAMaxParams,
  KOANumParams = 16,
};


#define KNumNRTParams (KINumParamMax-KInternal)
#define KNRTParam(slot, param) (slot*KNumNRTParams+param)
#define KRTParam(slot, param) (slot*KNumParams+param)
#define KATParam(param) (MODEL_SLOT_SIZE*KNumParams+param)

// Wet Dry 0.0~1.0 -> -100~20[dB], 0.0->-inf
// setwet/dry(MIN+value*(MAX-MIN));
#define DB_RANGE_MIN (-100.0f)
#define DB_RANGE_MAX (20.0f)
#define PARAM2DB(val) (DB_RANGE_MIN+val*(DB_RANGE_MAX-DB_RANGE_MIN))
#define DB2PARAM(val) ((val-DB_RANGE_MIN)/(DB_RANGE_MAX-DB_RANGE_MIN))

#define DBA_RANGE_MIN (-20.0f)
#define DBA_RANGE_MAX (20.0f)
#define PARAM2DBA(val) (DBA_RANGE_MIN+val*(DBA_RANGE_MAX-DBA_RANGE_MIN))
#define DBA2PARAM(val) ((val-DBA_RANGE_MIN)/(DBA_RANGE_MAX-DBA_RANGE_MIN))

// DelayTime 0.0~1.0 -> -1000~1000
#define DELAY_MIN (-1000.0f)
#define DELAY_MAX (1000.0f)
#define PARAM2DELAYTIME(val) (DELAY_MIN+val*(DELAY_MAX-DELAY_MIN))
#define DELAYTIME2PARAM(val) ((val-DELAY_MIN)/(DELAY_MAX-DELAY_MIN))

#define RELEASE_MIN (0.0f)
#define RELEASE_MAX (500.0f)
#define PARAM2RELEASE(val) (RELEASE_MIN+val*(RELEASE_MAX-RELEASE_MIN))
#define RELEASE2PARAM(val) ((val-RELEASE_MIN)/(RELEASE_MAX-RELEASE_MIN))

// LRBalance 0.0~1.0 -> -1~1
#define LRB_MIN (-1.f)
#define LRB_MAX (1.f)
#define PARAM2LRB(val) (LRB_MIN+val*(LRB_MAX-LRB_MIN))
#define LRB2PARAM(val) ((val-LRB_MIN)/(LRB_MAX-LRB_MIN))

// Stretch ~v2.5.2  0.0 ~ 50  ~ 100  ->  x0.5 ~ x1.0 ~ x2
//         v2.5.3~  0.0 ~ 0.5 ~ 1.0  ->  x0.1 ~ x1.0 ~ x10.0
#define PARAM2STRETCH(val) (std::pow(100.0, (val-0.5)))
#define STRETCH2PARAM(val) (std::log(val)/std::log(100.0)+0.5)
#define PARAM2STRETCH_OLD_1(val) std::pow(std::sqrt(2.0),((val/100.0)+0.5))
#define STRETCH2PARAM_OLD_1(val) ((((std::log(val)*2)/std::log(2.0))-0.5)*100.0)

#define IRATTACK2PARAM(val) val
#define PARAM2IRATTACK(val) val

#define IRHOLD2PARAM(val) val
#define PARAM2IRHOLD(val) val

#define IRDECAY2PARAM(val) val
#define PARAM2IRDECAY(val) val

#define IRSUSTAIN2PARAM(val) val
#define PARAM2IRSUSTAIN(val) val

#define IRRELEASE2PARAM(val) val
#define PARAM2IRRELEASE(val) val

#define FV3_IR2_CHUNK_VERSION_OLD_1   1 // ~2.5.6
#define FV3_IR2_CHUNK_VERSION_CURRENT 2 // 2.5.6~

#define DEFAULT_FRAGMENT_SIZE 1024
#define DEFAULT_FACTOR 16

#define PARAMLENGTH 10

#endif
