-- Classical setup for 1D Euler equations.
-- This intial condition features a discontinuity in the middle of the domain,
-- resulting in a Riemann problem.
-- The test setup makes use of a seeder generated mesh, though this is not
-- necessary, it illustrates the use of such a mesh.
require('seeder')
-- seeder.lua sets a different timing file name, set to the usual one.
timing_file = 'timing.res'

-- global simulation options
simulation_name = 'toro1_x_euler_modg_1d'
sim_control = {
  time_control = {
    min = 0,
    max = 0.025
  }
}

-- Mesh definitions --
-- Using a mesh generated by Seeder.
-- In 1D this usually is not needed, except you want to make use of local
-- refinements.
mesh = 'mesh/'

-- Equation definitions --
equation = {
  name      = 'euler_1d',
  isen_coef = 1.4,
  r         = 296.0,
  material = {
    characteristic = 0.0,
    relax_velocity = 0.0,
    relax_temperature = 0.0
  }
}
-- (cv) heat capacity and (r) ideal gas constant
equation["cv"] = equation["r"] / (equation["isen_coef"] - 1.0)

-- Tracking
tracking = {
  label = 'probe_density_Q4_toro_x',
  folder = './',
  variable = {'density'},
  shape = {
    kind = 'canoND',
    object= {
      origin = {
        (channel_length/2.0) + epsx,
        epsx,
        epsx
      }
    }
  },
  time_control = {
    min = 0,
    max = sim_control.time_control.max,
    interval = sim_control.time_control.max/20.0
  },
  output = { format = 'ascii', ndofs = 1 }
}

-- The initial conditions for the Riemann problem
-- ... left state
rho_l = 1.0
u_l = 0.0
p_l = 1.0
-- ... right state
rho_r = 0.125
u_r = 0.0
p_r = 0.1

function rho(x,y,z)
  if ( x < channel_length/2.0 ) then
    return rho_l
  else
    return rho_r
  end
end

function p(x,y,z)
  if ( x < channel_length/2.0 ) then
    return p_l
  else
    return p_r
  end
end

function u(x,y,z)
  if( x < channel_length/2.0 ) then
    return u_l
  else
    return u_r
  end
end

projection = {
   kind = 'fpt',
   factor = 1.0,
   blocksize = 32
}

initial_condition = {
  density = rho,
  pressure = p,
  velocity = u,
}

-- Scheme definitions --
scheme = {
  spatial = {
    name = 'modg_1d',
    m = 3
  },
  temporal = {
    name = 'explicitRungeKutta',
    steps = 4,
    control = {
      name = 'cfl',
      cfl  = 0.6
    }
  }
}

-- Boundary conditions
boundary_condition = {
  {
    label = 'inlet',
    kind = 'inflow_normal',
    density = rho_l,
    v_norm = u_l,
  },
  {
    label = 'outlet',
    kind = 'outflow',
    pressure = p_r,
  },
  {
    label = 'bottom',
    kind = 'slipwall',
  },
  {
    label = 'top',
    kind = 'slipwall',
  },
  {
    label = 'south',
    kind = 'slipwall',
  },
  {
    label = 'north',
    kind = 'slipwall',
  }
}
