package org.opengion.hayabusa.taglib;

import java.util.Set;											// 6.4.3.4 (2016/03/11)
import java.util.regex.Matcher;									// 7.0.1.1 (2018/10/22)
import java.util.regex.Pattern;									// 7.0.1.1 (2018/10/22)
// import java.util.StringJoiner;									// 7.0.1.2 (2018/11/04)

import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.io.JsChartData;
import org.opengion.fukurou.util.ArraySet;						// 6.4.3.4 (2016/03/11)
import org.opengion.fukurou.util.ToString;
import org.opengion.fukurou.util.ColorMap;						// 6.7.7.0 (2017/03/31)
import org.opengion.fukurou.util.StringUtil ;

import static org.opengion.fukurou.util.StringUtil.nval ;

/**
 * 設定された値をJsChartDataに設定し、
 * JsChartTagのJsChartDataリストに追加するタグです。
 * 
 * @og.formSample
 * ●形式：&lt;og:jsChartData chartColumn="…" … /&gt;
 * ●body:なし
 * 
 * ●Tag定義：
 * &lt;og:jsChartData
 *  ===================    data:datasets: の 要素の属性です。
 *      chartColumn     ○【TAG】ﾁｬｰﾄのｶﾗﾑ名を指定します(必須)。
 *      label             【TAG】凡例の値を指定します。
 *      type              【TAG】複合ﾁｬｰﾄの種類を指定します[line/bar]													// 7.0.1.1 (2018/10/22)
 *      fill              【TAG】線下を塗りつぶすかどうか[true/false]を指定します(初期値:false)。
 *      tension           【TAG】線の伸張を指定します。0で直線になります(初期値:0.4)。
 *      backgroundColor   【TAG】ﾃﾞｰﾀの背景色を指定します(色,色番号,VIVID,PASTEL,V0～,P0～)。
 *      borderColor       【TAG】線の色を指定します(色,色番号,VIVID,PASTEL,V0～,P0～)。
 *  X   colorNo           【廃止】線の色(borderColor)をColorMapの色番号で指定します。
 *      borderWidth       【TAG】線の幅を指定します。
 *      borderDash        【TAG】点線のスタイルを配列で指定します。
 *      pointStyle        【TAG】点のスタイル(circle,triangle,rect,rectRot,cross,crossRot,star,line,dash)を指定します。 // 6.8.5.0 (2018/01/09)
 *      pointRadius       【TAG】点の大きさを指定します。																// 6.8.5.0 (2018/01/09)
 *      showLine          【TAG】ラインを表示するかどうか[true/false]を指定します(初期値:null)。						// 6.8.5.0 (2018/01/09)
 *      spanGaps          【TAG】spanGaps属性を行うかどうか[true/false]を指定します(初期値:null)。						// 7.0.1.2 (2018/11/04)
 *      pointBGColor      【TAG】pointBackgroundColor属性を指定します(色,色番号,VIVID,PASTEL,V0～,P0～)。				// 7.0.1.2 (2018/11/04)
 *  ===================    options:scales:yAxes の 要素の属性です。
 *      useAxis           【TAG】y軸表示を行うかどうか[true/false]を指定します(初期値:null)。							// 7.0.1.1 (2018/10/22)
 *      id                【TAG】y軸のid(自動採番 'y'+連番)																// 7.0.1.1 (2018/10/22)
 *      position          【TAG】y軸の表示位置[left,right]を指定します(初期値:null)。									// 7.0.1.1 (2018/10/22)
 *      scaleType         【TAG】y軸のｽｹｰﾙﾀｲﾌﾟ[linear/category/realtime]を指定します(初期値:linear)						// 7.0.1.1 (2018/10/22)
 *      categoryList      【TAG】y軸のﾒﾓﾘﾘｽﾄをカンマ区切りで指定します(scaleTypeがcategoryの場合に有効)					// 7.0.1.1 (2018/10/22)
 *      ylabel            【TAG】scaleLabel:y軸に表示するﾗﾍﾞﾙ文字														// 7.0.1.1 (2018/10/22)
 *      beginAtZero       【TAG】ticks:y軸を０から書き始まるかどうか[true/false]を指定(初期値:true)(円形の場合もこの値)	// 7.0.1.1 (2018/10/22)
 *      fontColor         【TAG】ticks:y軸のフォントの色(色,色番号,VIVID,PASTEL,V0～,P0～)								// 7.0.1.1 (2018/10/22)
 *      scaleCallback     【TAG】ticks:y軸ｺｰﾙﾊﾞｯｸを指定します。															// 7.0.1.1 (2018/10/22)
 *      max               【TAG】ticks:y軸の最大値を指定します(scaleTypeがlinearの場合に有効)							// 7.0.1.1 (2018/10/22)
 *      min               【TAG】ticks:y軸の最小値を指定します(scaleTypeがlinearの場合に有効)							// 7.0.1.1 (2018/10/22)
 *      stepSize          【TAG】ticks:y軸のメモリ幅を指定します(scaleTypeがlinearの場合に有効)							// 7.0.1.1 (2018/10/22)
 *      ticks             【TAG】ticks属性(他のticks属性とは、同時に使用できません)										// 7.0.1.1 (2018/10/22)
 *      gridColor         【TAG】gridLines:color属性( gridLines:{ color:'red', } を生成)								// 7.0.1.1 (2018/10/22)
 *      gridLines         【TAG】gridLines属性(gridColorは、同時に使用できません)										// 7.0.1.1 (2018/10/22)
 *  ===================
 *      optDataset        【TAG】その他data:datasetのｵﾌﾟｼｮﾝを指定します。												// 7.0.1.2 (2018/11/04)
 *      optAxis           【TAG】その他options:scales:yAxesのｵﾌﾟｼｮﾝを指定します。										// 7.0.1.2 (2018/11/04)
 *      optTicks          【TAG】その他options:scales:yAxes:ticksのｵﾌﾟｼｮﾝを指定します。									// 7.0.1.2 (2018/11/04)
 *      optScaleLabel     【TAG】その他options:scales:yAxes:scaleLabelのｵﾌﾟｼｮﾝを指定します。							// 7.0.1.2 (2018/11/04)
 *      optGridLines      【TAG】その他options:scales:yAxes:gridLinesのｵﾌﾟｼｮﾝを指定します。								// 7.0.1.2 (2018/11/04)
 *  ===================
 *      caseKey           【TAG】このタグ自体を利用するかどうかの条件キーを指定します(初期値:null)
 *      caseVal           【TAG】このタグ自体を利用するかどうかの条件値を指定します(初期値:null)
 *      caseNN            【TAG】指定の値が、null/ゼロ文字列 でない場合(Not Null=NN)は、このタグは使用されます(初期値:判定しない)
 *      caseNull          【TAG】指定の値が、null/ゼロ文字列 の場合は、このタグは使用されます(初期値:判定しない)
 *      caseIf            【TAG】指定の値が、true/TRUE文字列の場合は、このタグは使用されます(初期値:判定しない)
 *      debug             【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)					// 7.0.1.1 (2018/10/22)
 *  /&gt;
 * 
 * ●使用例
 * &lt;og:jsChart...&gt;
 *     &lt;og:jsChartData
 *         chartColumn ="CLM1"
 *         label       ="ラベル"
 *         fill        ="true"
 *         tension     ="0"
 *         borderColor ="rbga(150,150,150,0.7)"
 *         borderWidth ="2"
 *     /&gt;
 * &lt;/og:jsChart&gt;
 * 
 * @og.rev 5.9.17.2 (2017/02/08) 新規作成
 * @og.rev 7.0.1.1 (2018/10/22) 大幅見直し
 * @og.group 画面表示
 * 
 * @version	5.9.17.2		2017/02/08
 * @author	T.OTA
 * @since	JDK7.0
 * 
 */
public class JsChartDataTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。{@VALUE} */
	private static final String VERSION = "7.0.1.3 (2018/11/12)" ;
	private static final long serialVersionUID = 701320181112L ;

	private static final boolean	USE_QUOTE		= false;
	private static final boolean	NO_QUOTE		= true;		// IS_NUMBER か、!USE_QUOTE か、

	private static final Set<String> SET_TYPE		= new ArraySet<>( "line", "bar" );
	private static final Set<String> SET_PSTYLE		= new ArraySet<>( "circle","triangle","rect","rectRot","cross","crossRot","star","line","dash" );
	private static final Set<String> SET_POSITION	= new ArraySet<>( "left", "right" );
	private static final Set<String> SET_SCALE		= new ArraySet<>( "linear", "category", "realtime" );
	private static final Set<String> SET_BOOLEAN	= new ArraySet<>( "true", "false" );

	private JsChartData jsData  = new JsChartData();

	private String	yAxisID				;	// 7.0.1.1 (2018/10/22) y軸のid(自動採番 'y'+連番)

	private boolean	fill				;	// 7.0.1.1 (2018/10/22) lineﾁｬｰﾄの下部塗りつぶし(初期値:falseが、chartJS の初期値と異なるので、後付する)
	private String	borderColor			;	// borderColor は、colorNo と競合するので、最後に判定します。
	private String	backgroundColor		;	// backgroundColor が未設定の場合は、borderColor を使用します。

	private static final String D_TENSION	= "0.4";	// 7.0.1.1 (2018/10/22) 初期値

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.9.7.0 (2018/05/14) PMD Each class should declare at least one constructor
	 */
	public JsChartDataTag() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * Taglibの終了タグが見つかった時に処理する doEndTag() を オーバーライドします。
	 * 
	 * @og.rev 6.7.6.0 (2017/03/17) タグの使用を決める共通属性の追加
	 * @og.rev 6.7.7.0 (2017/03/31) backgroundColor が未設定の場合は、borderColor を使用します。
	 * @og.rev 6.8.5.0 (2018/01/09) pointStyle , pointRadius , showLine 属性の追加。
	 * @og.rev 7.0.1.1 (2018/10/22) rightAxis 属性の追加。
	 * 
	 * @return 後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();
		if( !useTag() ) { return EVAL_PAGE ; }			// 6.7.6.0 (2017/03/17)

		final JsChartTag jsChartTag = (JsChartTag) findAncestorWithClass( this, JsChartTag.class );

		if( jsChartTag == null ) {
			final String errMsg = "jsChart タグが見つかりませんでした。";
			throw new HybsSystemException( errMsg );
		}

		final int size = jsChartTag.getJsChartDataSize();		// 登録順で、現時点で持っている個数
		if( yAxisID == null ) { yAxisID = "y" + size ; }		// 指定しない場合は、y軸のid(自動採番 'y'+連番)

		if( size == 0 ) { jsData.setUseAxis( true ); }			// 要方法検討。false でもY軸が表示されるが不完全
		jsData.setId( yAxisID );

		// borderColor と、backgroundColor の設定
//		setBorderOrBackColor( size , jsChartTag.isOneColor() );
		setBorderOrBackColor( jsChartTag.isOneColor() ? size : -1 );		// 7.0.1.3 (2018/11/12) 変数の集約

		// fill は、未設定時に、false をあえて設定する必要がある。
		jsData.addDataset( "fill" , String.valueOf( fill ) , NO_QUOTE );	// 数値(boolean)

		jsChartTag.addJsChartData( jsData );

		return EVAL_PAGE;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 * 
	 * @og.rev 6.7.7.0 (2017/03/31) jsDataのローカル変数化。
	 * @og.rev 6.8.5.0 (2018/01/09) pointStyle , pointRadius , showLine 属性の追加。
	 * @og.rev 7.0.1.1 (2018/10/22) rightAxis 属性の追加。
	 * @og.rev 7.0.1.1 (2018/10/22) 初期値は、ﾃﾞﾌｫﾙﾄ(出力しない)に変更。
	 */
	@Override
	protected void release2() {
		super.release2();
		jsData				= new JsChartData();

		yAxisID				= null;		// 7.0.1.1 (2018/10/22) y軸のid(自動採番 'y'+連番)

		fill				= false;	// 7.0.1.1 (2018/10/22) lineﾁｬｰﾄの下部塗りつぶし(初期値:falseが、chartJS の初期値と異なるので、後付する)
		borderColor			= null;		// borderColor は、colorNo と競合するので、最後に判定します。
		backgroundColor		= null;		// backgroundColor が未設定の場合は、borderColor を使用します。
	}

	/**
	 * borderColorとbackgroundColor の設定
	 *
	 * borderColorとbackgroundColor は、どちらか一方が設定されている場合は、
	 * もう片方も、そちらにあわせます。
	 * どちらも設定されていない場合は、ﾁｬｰﾄの番号から、色コードを自動で割り当てます。
	 * また、ｷｰﾜｰﾄﾞ PASTELとVIVID が指定された場合は、グラフごとに、色を変える配列を設定します。
	 * 
	 * @og.rev 7.0.1.3 (2018/11/12) 色情報を返すメソッドを集約します。
	 * 
	 * @param cnt 現在のﾁｬｰﾄの番号(マイナスの場合は、JavaScript配列形式で返します。)
//	 * @param isOneColor １色しか使用できないかどうか [true:１色のみ/false:色配列可]
	 */
//	private void setBorderOrBackColor( final int cnt , final boolean isOneColor ) {
	private void setBorderOrBackColor( final int cnt ) {
		// 7.0.1.3 (2018/11/12) 色情報を返すメソッドを集約します。
		if( borderColor == null ) {					// borderColorが未設定
			backgroundColor = makeColor( backgroundColor , cnt );
			borderColor = backgroundColor;
		}
		else if( backgroundColor == null ) {		// backgroundColorが未設定
			borderColor		= makeColor( borderColor	 , cnt );
			backgroundColor = borderColor;
		}
		else {
			backgroundColor = makeColor( backgroundColor , cnt );
			borderColor		= makeColor( borderColor	 , cnt );
		}

//		if( borderColor == null && backgroundColor != null ) {		// borderColorが未設定
//			borderColor = backgroundColor;
//		}
//		else if( backgroundColor == null && borderColor != null ) {	// backgroundColorが未設定
//			backgroundColor = borderColor;
//		}
//		else if( borderColor == null && backgroundColor == null ) {	// 両方未設定
//			borderColor = isOneColor ? ColorMap.getColorKey( cnt )		// cnt 番号の色番号を1色指定します。
//									 : "['" + String.join( "','", ColorMap.getColorKeys() ) + "']";
//			backgroundColor = borderColor;
//		}
//
//		if( "PASTEL".equalsIgnoreCase( borderColor ) ) {
//			borderColor = isOneColor ? ColorMap.getPastelKey( cnt )		// cnt 番号の色番号を1色指定します。
//									 : "['" + String.join( "','", ColorMap.getPastelKeys() ) + "']";
//		}
//		else if( "VIVID".equalsIgnoreCase( borderColor ) ) {
//			borderColor = isOneColor ? ColorMap.getVividKey( cnt )			// cnt 番号の色番号を1色指定します。
//									 : "['" + String.join( "','", ColorMap.getVividKeys() ) + "']";
//		}
//
//		if( "PASTEL".equalsIgnoreCase( backgroundColor ) ) {
//			backgroundColor = isOneColor ? ColorMap.getPastelKey( cnt )		// cnt 番号の色番号を1色指定します。
//										 : "['" + String.join( "','", ColorMap.getPastelKeys() ) + "']";
//		}
//		else if( "VIVID".equalsIgnoreCase( backgroundColor ) ) {
//			backgroundColor = isOneColor ? ColorMap.getVividKey( cnt )			// cnt 番号の色番号を1色指定します。
//										 : "['" + String.join( "','", ColorMap.getVividKeys() ) + "']";
//		}

		jsData.addDataset( "borderColor"		, borderColor		, NO_QUOTE );	// 文字はすでにクオート付き、配列の場合はクオート不用
		jsData.addDataset( "backgroundColor"	, backgroundColor	, NO_QUOTE );	// 文字はすでにクオート付き、配列の場合はクオート不用
	}

	/**
	 * ﾊﾟﾗﾒｰﾀﾁｪｯｸ用ﾒｿｯﾄﾞ。
	 * 
	 * @param trg		ターゲット
	 * @param set		使用可能なキーワードのSet
	 * @param trgStr	ターゲットの名称
	 */
	private void checkPara( final String trg, final Set<String> set, final String trgStr ) {
		if( StringUtil.isNotNull( trg ) && !check( trg, set ) ) {						// 6.8.5.0 (2018/01/09)
			final StringBuilder errMsg = new StringBuilder( BUFFER_MIDDLE )
				.append( "指定の" ).append( trgStr ).append( "は指定できません。" ).append( CR )
				.append( trgStr ).append( "=[" ).append( trg ).append( ']' ).append( CR )
				.append( set );		// org.opengion.fukurou.util.ArraySet の toStringメソッド

			throw new HybsSystemException( errMsg.toString() );
		}
	}

	/**
	 * 色情報を返します。
	 * 
	 * 通常の、#XXXXXX形式の16bitRGB表記や、rgb(r,g,b)や、rgba(r,g,b,a) などが設定可能です。
	 * 色の代わりに、ColorMapの色番号や色記号を指定できます。
	 * 
	 * 特殊ｷｰﾜｰﾄﾞとして、VIVIDとPASTEL やビビッド、0～11 (V0～V11) , パステル、12～23 (P0～P11) 
	 * を指定できます。
	 * CSV形式の場合、cnt で指定された番号の色を使用します。-1 の場合は、JavaScriptの配列文字列で返します。
	 * 
	 * キーがnull の場合は、色番号から初期設定の値を返します。
	 * 
	 * @og.rev 7.0.1.3 (2018/11/12) 色情報を返すメソッドを集約します。
	 * 
	 * @param colKey	色を表すｷｰﾜｰﾄﾞ(色,色番号,VIVID,PASTEL,V0～,P0～)
	 * @param cnt		CSV形式か、VIVID,PASTEL の場合、指定の番号の色を使用します。
	 * @return 色文字列
	 */
	private String makeColor( final String colKey, final int cnt ) {
		// cnt < 0 の場合、CSV形式なら、JavaScript配列で色を返します。
		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );
		if( cnt < 0 ) {
			final String[] cols = ColorMap.getColorKeys( colKey );		// nullの場合は、ビビッドとパステルの全24色
			if( cols.length == 1 ) {		// 配列が1の場合、配列にせず、値をそのまま設定します。
				buf.append( '\'' ).append( cols[0] ).append( '\'' );
			}
			else {
				buf.append( "['" ).append( String.join( "','", cols ) ).append( "']" );
			}
		}
		else {
			// 色順指定されているので、1色だけ返します。
			final String[] cols = StringUtil.csv2Array( colKey );
			if( cols.length == 0 ) {		// 元のcolKeyがnullかゼロ文字列
				buf.append( '\'' ).append( ColorMap.getColorKey( cnt ) ).append( '\'' );		// cnt に応じた自動設定
			}
//			else if( cols.length == 1 ) {	// 一つしかない。
//				buf.append( '\'' ).append( ColorMap.getColorKey( cols[0],cols[0] ) ).append( '\'' );
//			}
			else {
				final String col = cols[cnt % cols.length];					// オーバーする場合は、繰返しになります。
				buf.append( '\'' ).append( ColorMap.getColorKey( col , col ) ).append( '\'' );
			}
		}

		return buf.toString();
	}

	/**
	 * 【TAG】ﾁｬｰﾄのｶﾗﾑ名を指定します(必須)。
	 *
	 * @og.tag
	 * 
	 * @param clm ﾁｬｰﾄのｶﾗﾑ名
	 */
	public void setChartColumn( final String clm ) {
		jsData.setChartColumn( nval( getRequestParameter( clm ),null ) );
	}

	/**
	 * 【TAG】凡例の値を指定します。
	 *
	 * @og.tag
	 * 
	 * @param lbl 凡例
	 */
	public void setLabel( final String lbl ) {
		jsData.addDataset( "label" , nval( getRequestParameter( lbl ),null ) , USE_QUOTE );		// 文字
	}

	/**
	 * 【TAG】複合ﾁｬｰﾄの種類を指定します[line/bar]。
	 *
	 * 通常は、JsChartTagタグのchartTypeで指定しますが、複合グラフの場合は、個々のJsChartDataTag でタイプを指定します。
	 * なお、複合グラフ時には、JsChartTagタグのchartTypeを、"bar" にしておかないと、きちんと表示しないようです。
	 *
	 * @og.tag
	 * 
	 * @param type 種類 [line/bar]
	 */
	public void setType( final String type ) {
		final String ctype = nval( getRequestParameter( type ),null );

		checkPara( ctype, SET_TYPE, "type" );
		jsData.addDataset( "type" , ctype , USE_QUOTE );		// 文字
	}

	/**
	 * 【TAG】線下を塗りつぶすかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * フィル(線より下の塗りつぶし) を設定します。
	 * 
	 * @param fill 塗りつぶすかどうか [true/false]
	 */
	public void setFill( final String fill ) {
		// 7.0.1.1 (2018/10/22) lineﾁｬｰﾄの下部塗りつぶし(初期値:falseが、chartJS の初期値と異なるので、後付する)
		this.fill = nval( getRequestParameter( fill ),this.fill );
	}

	/**
	 * 【TAG】線の伸張を指定します。0で直線になります(初期値:0.4)。
	 *
	 * @og.tag
	 * 伸張 を設定します。
	 *
	 * @og.rev 7.0.1.1 (2018/10/22) 初期値は、ﾃﾞﾌｫﾙﾄ(出力しない)に変更。
	 *
	 * @param tension 線の伸張
	 */
	public void setTension( final String tension ) {
		jsData.addDataset( "tension" , nval( getRequestParameter( tension ),D_TENSION ) , NO_QUOTE );		// 数値
	}

	/**
	 * 【TAG】ﾃﾞｰﾀの背景色を指定します。
	 *
	 * @og.tag
	 * backgroundColor = "BLUE" とすると、すべての背景色を指定できます。
	 * 配列で指定すると、データの順番に適用されます。
	 * 例：backgroundColor = "['#ffaaaa','#ffffaa','#aaffaa','#aaaaff','#aaaaff']"
	 * 
	 * 特殊ｷｰﾜｰﾄﾞとして、PASTELとVIVID を指定することで、パステルカラーやビビッドカラーの色ｺｰﾄﾞを指定できます。
	 * 
	 * 背景色を指定しない場合、線の色(borderColor)を使用します。
	 *
	 * @og.rev 6.9.9.2 (2018/09/18) パステルカラーの色文字列のｶﾝﾏ区切り文字列
	 * 
	 * @param bgColor 背景色
	 * @see		#setBorderColor(String)
	 */
	public void setBackgroundColor( final String bgColor ) {
		backgroundColor = nval( getRequestParameter( bgColor ),null );
	}

	/**
	 * 【TAG】線の色を指定します。
	 *
	 * @og.tag
	 * borderColor = "BLUE" とすると、すべての線の色を指定できます。
	 * 配列で指定すると、データの順番に適用されます。
	 * 例：borderColor = "['#ffaaaa','#ffffaa','#aaffaa','#aaaaff','#aaaaff']"
	 * 
	 * 色の代わりに、ColorMapの色番号を指定したい場合は、colorNo を指定します。
	 * 両方指定した場合は、borderColor が優先されます。
	 * どちらも指定しない場合は、JsChartTagに登録した順番に色コードで指定されます。
	 * 
	 * 特殊ｷｰﾜｰﾄﾞとして、PASTELとVIVID を指定することで、パステルカラーやビビッドカラーの
	 * 色ｺｰﾄﾞ配列を指定できます。
	 * 
	 * @param color 線の色
//	 * @see		#setColorNo(String)
	 */
	public void setBorderColor( final String color ) {
		// colorNo で、初期値設定されている可能性があるので、nval の初期値は、borderColor にしておく。
		borderColor = nval( getRequestParameter( color ),borderColor );
	}

//	/**
//	 * 【TAG】線の色(borderColor)をColorMapの色番号で指定します(CSV指定可能)。
//	 *
//	 * @og.tag
//	 * 色の代わりに、ColorMapの色番号を指定します。
//	 * borderColorも指定した場合は、borderColor が優先されます。
//	 * この引数は、色に変換後、borderColor に設定されます。
//	 * backgroundColor を指定しない場合は、このborderColorが使用されます。
//	 *
//	 * 特殊な色番号として、ビビッドを、0～11 (V0～V11) , パステルを、11～23 (P0～P11) とします。
//	 * また、ここで指定する値は、json として直接設定されないため、複数指定したい場合は、
//	 * 単純なカンマ区切りで指定します。
//	 *
//	 * @og.rev 6.7.7.0 (2017/03/31) ColorMapの色番号で指定
//	 * @og.rev 7.0.1.2 (2018/11/04) 色番号をCSVで指定できるようにします。
//	 * 
//	 * @param colorNo 線の色の番号
//	 * @see		ColorMap#getColorKeys()
//	 */
//	public void setColorNo( final String colorNo ) {
//		if( borderColor == null ) {		// borderColor が、未設定の場合のみ、色番号を利用する。
//			final String[] cols = getCSVParameter( colorNo );
//			if( cols.length == 1 ) {								// 1件の場合は、配列にしない。
//				borderColor = ColorMap.getColorKey( cols[0] );		// ColorMap に無ければ、null が返される。
//			}
//			else {
//				final StringJoiner sj = new StringJoiner("','", "['", "']");
//				for( final String col : cols ) {
//					sj.add( ColorMap.getColorKey( col,"rgba( 255,255,255,0 )" ) );		// キーが無ければ、透明
//				}
//				borderColor = sj.toString();
//			}
//		}
//	}

	/**
	 * 【TAG】線の幅を指定します。
	 *
	 * @og.tag
	 * 
	 * @param width 線の幅
	 */
	public void setBorderWidth( final String width ) {
		jsData.addDataset( "borderWidth" , nval( getRequestParameter( width ),null ) , NO_QUOTE );		// 数値
	}

	/**
	 * 【TAG】点線のスタイルを配列で指定します。
	 *
	 * ダッシュ線のスタイルは、配列で指定します。
	 * borderDash="[5,2]" とすれば、線の長さが5px , 線と線の間が2px になります。
	 *
	 * @og.tag
	 *
	 * @og.rev 7.0.1.3 (2018/11/12) 点線のスタイル追加
	 * 
	 * @param dash 点線のスタイル
	 */
	public void setBorderDash( final String dash ) {
		jsData.addDataset( "borderDash" , nval( getRequestParameter( dash ),null ) , NO_QUOTE );		// 配列
	}

	/**
	 * 【TAG】点のスタイル[circle,triangle,rect,rectRot,cross,crossRot,star,line,dash]を指定します。
	 *
	 * @og.tag
	 * jsChartTag で、usePointStyle = "true" をセットした場合に有効になります。
	 * 点のスタイルは、circle,triangle,rect,rectRot,cross,crossRot,star,line,dash が、
	 * 
	 * @og.rev 6.8.5.0 (2018/01/09) 新規登録
	 * 
	 * @param ptStyle 点のスタイル [circle,triangle,rect,rectRot,cross,crossRot,star,line,dash]
	 */
	public void setPointStyle( final String ptStyle ) {
		final String pointStyle = nval( getRequestParameter( ptStyle ),null );

		checkPara( pointStyle, SET_PSTYLE, "pointStyle" );
		jsData.addDataset( "pointStyle" , pointStyle , USE_QUOTE );		// 文字
	}

	/**
	 * 【TAG】点の大きさを指定します。
	 *
	 * @og.tag
	 * jsChartTag で、usePointStyle = "true" をセットした場合に有効になります。
	 * 
	 * @og.rev 6.8.5.0 (2018/01/09) 新規登録
	 * 
	 * @param ptRadius 点の大きさを指定します。
	 */
	public void setPointRadius( final String ptRadius ) {
		jsData.addDataset( "pointRadius" , nval( getRequestParameter( ptRadius ),null ) , NO_QUOTE );		// 数値
	}

	/**
	 * 【TAG】ラインを表示するかどうか[true/false]を指定します(初期値:null)。
	 *
	 * @og.tag
	 * jsChartTag で、usePointStyle = "true" をセットした場合に有効になります。
	 * 初期値(null)は、showLine 属性を設定しませんが、chartJS 自体の初期値が true
	 * なので、表示されます。
	 * 
	 * @og.rev 6.8.5.0 (2018/01/09) 新規登録
	 * 
	 * @param show ラインを表示するかどうか [true:表示する/false:表示しない]
	 */
	public void setShowLine( final String show ) {
		jsData.addDataset( "showLine" , nval( getRequestParameter( show ),null ) , NO_QUOTE );		// Boolean
	}

	/**
	 * 【TAG】spanGaps属性を行うかどうか[true/false]を指定します(初期値:null)。
	 *
	 * @og.tag
	 * trueの場合、データがない点またはヌルの点との間に線が描画されます。
	 * falseの場合、 NaN データ点では線が途切れます。
	 * 
	 * @og.rev 7.0.1.2 (2018/11/04) 新規登録
	 * 
	 * @param flag spanGaps属性を行うかどうか [true/false]
	 */
	public void setSpanGaps( final String flag ) {
		jsData.addDataset( "spanGaps" , nval( getRequestParameter( flag ),null ) , NO_QUOTE );		// Boolean
	}

	/**
	 * 【TAG】pointBackgroundColor属性を指定します(初期値:null)。
	 *
	 * @og.tag
	 * 点の塗りつぶしの色を指定します。属性名が長いので、短縮しています。
	 * 単独文字列の場合は、すべての点を同じ色で塗ります。配列([]で囲う)の場合は、
	 * 点の並び順に応じて、色付けを行います。
	 * 
	 * 配列([]で囲う)か、var定義変数を想定していますので、前後にクオートを付けません。
	 * 単独文字列を指定する場合は、"'red'" のように、クオートを付けてください。
	 * 通常は、backgroundColorが使用されますので、単独文字で色指定は行わないと思います。
	 *
	 * ポイントの色指定に、ColorMapの色コードは使えません。
	 *
	 * @og.rev 6.8.5.0 (2018/01/09) 新規登録
	 * 
	 * @param cols 点の塗りつぶしの色(単独、配列)
	 */
	public void setPointBGColor( final String cols ) {
		jsData.addDataset( "pointBackgroundColor" , nval( getRequestParameter( cols ),null ) , NO_QUOTE );		// 配列[]か、変数なので、クオート無しにします。
	}

	//========================================================================================

	/**
	 * 【TAG】このﾃﾞｰﾀのy軸を表示するかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * true にセットした場合、jsChartTag で、yAxis に対して、一連の設定を行います。
	 * 初期値(false)ですが、1つのデータセットは必ず表示されるようです。
	 * 
	 * @og.rev 7.0.1.1 (2018/10/22) useAxis 属性の追加。
	 * 
	 * @param use 右側のy軸表示するかどうか [true:表示する/false:表示しない]
	 */
	public void setUseAxis( final String use ) {
		jsData.setUseAxis( nval( getRequestParameter( use ), false ) );
	}

	/**
	 * 【TAG】ﾃﾞｰﾀﾁｬｰﾄのIDを指定します。
	 *
	 * @og.tag
	 * 指定しない場合は、y軸のid(自動採番 'y'+連番) になります。
	 * options:scales:yAxes の 要素の属性です。
	 *
	 * @og.rev 7.0.1.1 (2018/10/22) 属性の追加。
	 *
	 * @param   id 固有の名前
	 */
	@Override
	public void setId( final String id ) {
		yAxisID  = nval( getRequestParameter( id ),null );
	}

	/**
	 * 【TAG】y軸の表示位置[left,right]を指定します(初期値:null)。
	 *
	 * @og.tag
	 * 複合グラフ表示で、指定のデータのy軸を、右に表示したい場合は、right を指定します。
	 * 初期値(null)は、左に表示されます。
	 * options:scales:yAxes の 要素の属性です。
	 * 
	 * @og.rev 7.0.1.1 (2018/10/22) 属性の追加。
	 * 
	 * @param pos y軸の表示位置 [left,right]
	 */
	public void setPosition( final String pos ) {
		final String position = nval( getRequestParameter( pos ),null );

		checkPara( position, SET_POSITION, "position" );

		jsData.addAxis( "position" , position , USE_QUOTE );		// 文字
	}

	/**
	 * 【TAG】y軸のｽｹｰﾙﾀｲﾌﾟ[linear/category]を指定します(初期値:null)。
	 *
	 * @og.tag
	 * 未指定(null)の場合は、linear になります。
	 * options:scales:yAxes の 要素の属性です。
	 * 
	 * @og.rev 7.0.1.1 (2018/10/22) 属性の追加。
	 * 
	 * @param type y軸のｽｹｰﾙﾀｲﾌﾟ [linear/category]
	 */
	public void setScaleType( final String type ) {
		final String sType = nval( getRequestParameter( type ),null );

		// プラグインなどで独自の type を指定することがあるため、警告だけにします。
		try {
			checkPara( sType, SET_SCALE, "type" );
		}
		catch( final HybsSystemException ex ) {
			System.err.println( ex.getMessage() );
		}

		jsData.addAxis( "type" , sType , USE_QUOTE );		// 文字
	}

	/**
	 * 【TAG】y軸のﾒﾓﾘﾘｽﾄをカンマ区切りで指定します(scaleTypeがcategoryの場合に有効)。
	 *
	 * @og.tag
	 * ※ 通常のCSVで指定します。
	 * 
	 * @og.rev 7.0.1.1 (2018/10/22) 属性の追加。
	 * 
	 * @param categoryList y軸のﾒﾓﾘﾘｽﾄ
	 */
	public void setCategoryList( final String categoryList ) {
		final String lbls = nval( getRequestParameter( categoryList ),null );

		if( lbls != null ) {
			// 「,」を「','」に変換して設定。(,前後の半角ｽﾍﾟｰｽは除去する)
			final String regex = " *, *";
			final Pattern pttn = Pattern.compile( regex );
			final Matcher mtch = pttn.matcher( lbls );

			// y軸ｶﾃｺﾞﾘｰﾘｽﾄの設定
			final String labels = "['" + mtch.replaceAll( "','" ) + "']" ;

			jsData.addAxis( "labels" , labels , NO_QUOTE );		// 配列なので、クオート不用
		}
	}

	/**
	 * 【TAG】scaleLabel:y軸に表示するﾗﾍﾞﾙ文字を指定します(初期値:null)。
	 *
	 * @og.tag
	 * 横軸に表示する文字を指定します。
	 * options:scales:yAxes:scaleLabel の 要素の属性です。
	 * scaleLabel: { display: true, labelString: 'ﾗﾍﾞﾙ文字', } がセットされます。
	 * 
	 * @og.rev 7.0.1.1 (2018/10/22) 属性の追加。
	 * 
	 * @param label y軸に表示するﾗﾍﾞﾙ文字
	 */
	public void setYlabel( final String label ) {
		final String lbl = nval( getRequestParameter( label ),null );
		if( lbl != null ) {
			final String scLbl = "{display: true,labelString:'" + lbl + "'}" ;
			jsData.addAxis( "scaleLabel" , scLbl , NO_QUOTE );		// カンマが不要なのは判っている
		}
	}

	/**
	 * 【TAG】y軸を０から書き始まるかどうか[true/false]を指定します(初期値:null)。
	 *
	 * @og.tag
	 * ticks と同時には使用できません。
	 * 初期値(null)は、０から書き始めます。
	 * options:scales:yAxes:ticks の 要素の属性です。
	 * 
	 * @og.rev 7.0.1.1 (2018/10/22) 属性の追加。
	 * 
	 * @param atZero y軸を０から書き始まるかどうか [true/false]
	 */
	public void setBeginAtZero( final String atZero ) {
		final String beginAtZero =  nval( getRequestParameter( atZero ),null );

		checkPara( beginAtZero, SET_BOOLEAN, "beginAtZero" );
		jsData.addTicks( "beginAtZero" , beginAtZero , NO_QUOTE );		// 数値(boolean)
	}

	/**
	 * 【TAG】y軸のフォントの色を指定(初期値:null)。
	 *
	 * @og.tag
	 * ticks と同時には使用できません。
	 * options:scales:yAxes:ticks の 要素の属性です。
	 * 
	 * ColorMapの色コード(色,色番号,VIVID,PASTEL,V0～,P0～)が使えます。
	 * 
	 * @og.rev 7.0.1.1 (2018/10/22) 属性の追加。
	 * @og.rev 7.0.1.3 (2018/11/12) 色情報を返すメソッドを集約します。
	 * 
	 * @param fontColor y軸のフォントの色
	 */
	public void setFontColor( final String fontColor ) {
		final String col = nval( getRequestParameter( fontColor ),null );
		if( col != null ) {
			jsData.addTicks( "fontColor" , ColorMap.getColorKey( col , col ) , USE_QUOTE );		// 文字
		}

//		jsData.addTicks( "fontColor" , nval( getRequestParameter( fontColor ),null ) , USE_QUOTE );		// 文字
	}

	/**
	 * 【TAG】y軸ｺｰﾙﾊﾞｯｸを指定します。
	 *
	 * @og.tag
	 * y軸のメモリ編集用スケールバックを設定します。
	 * options:scales:yAxes:ticks の 要素の属性です。
	 * 
	 * @og.rev 7.0.1.1 (2018/10/22) 属性の追加。
	 * 
	 * @param callback y軸ｺｰﾙﾊﾞｯｸ
	 */
	public void setScaleCallback( final String callback ) {
		jsData.addTicks( "callback" , nval( getRequestParameter( callback ),null ) , NO_QUOTE );		// ファンクションは、クオートしない
	}

	/**
	 * 【TAG】y軸の最大値を指定します(scaleTypeがlinearの場合に有効)。
	 *
	 * @og.tag
	 * options:scales:yAxes:ticks の 要素の属性です。
	 * 
	 * @og.rev 7.0.1.1 (2018/10/22) 属性の追加。
	 * 
	 * @param max メモリの最大値
	 */
	public void setMax( final String max ) {
		jsData.addTicks( "max" , nval( getRequestParameter( max ),null ) , NO_QUOTE );		// 数値
	}

	/**
	 * 【TAG】y軸の最小値を指定します(scaleTypeがlinearの場合に有効)。
	 *
	 * @og.tag
	 * options:scales:yAxes:ticks の 要素の属性です。
	 * 
	 * @og.rev 7.0.1.1 (2018/10/22) 属性の追加。
	 * 
	 * @param min メモリの最小値
	 */
	public void setMin( final String min ) {
		jsData.addTicks( "min" , nval( getRequestParameter( min ),null ) , NO_QUOTE );		// 数値
	}

	/**
	 * 【TAG】y軸のメモリ幅を指定します(scaleTypeがlinearの場合に有効)。
	 *
	 * @og.tag
	 * options:scales:yAxes:ticks の 要素の属性です。
	 * 
	 * @og.rev 7.0.1.1 (2018/10/22) 属性の追加。
	 * 
	 * @param stepSize y軸のメモリ幅
	 */
	public void setStepSize( final String stepSize ) {
		jsData.addTicks( "stepSize" , nval( getRequestParameter( stepSize ),null ) , NO_QUOTE );		// 数値
	}

	/**
	 * 【TAG】y軸のticks属性を指定(初期値:null)。
	 *
	 * @og.tag
	 * ticks に登録する内容をそのまま書き込みます。
	 * tics = "{ beginAtZero:true,fontColor:'blue'  }" という感じに、{} なども含めて書きます。
	 * この設定と、beginAtZero、fontColor を同時に設定した場合の動作は、不定です。
	 * options:scales:yAxes:ticks の 要素の属性です。
	 * 
	 * @og.rev 7.0.1.1 (2018/10/22) 属性の追加。
	 * 
	 * @param ticks y軸のticks属性
	 */
	public void setTicks( final String ticks ) {
		jsData.addAxis( "ticks" , nval( getRequestParameter( ticks ),null ) , USE_QUOTE );		// 文字
	}

	/**
	 * 【TAG】gridLinesのcolor属性( gridLines:{ color:'red', } を生成)(初期値:null)。
	 *
	 * @og.tag
	 * gridLines と同時には使用できません。
	 * options:scales:yAxes:gridLines の 要素の属性です。
	 * 
	 * ColorMapの色コード(色,色番号,VIVID,PASTEL,V0～,P0～)が使えます。
	 * 
	 * @og.rev 7.0.1.1 (2018/10/22) 属性の追加。
	 * @og.rev 7.0.1.3 (2018/11/12) 色情報を返すメソッドを集約します。
	 * 
	 * @param gridColor y軸のフォントの色
	 */
	public void setGridColor( final String gridColor ) {
		final String col = nval( getRequestParameter( gridColor ),null );
		if( col != null ) {
//			final String grid = "{color:'" + col + "'}" ;
			final String grid = "{color:'" + ColorMap.getColorKey( col , col ) + "'}" ;
			jsData.addAxis( "gridLines" , grid , NO_QUOTE );		// カンマが不要なのは判っている
		}
	}

	/**
	 * 【TAG】gridLines属性(gridColorを同時に設定した場合は、不定です)(初期値:null)。
	 *
	 * @og.tag
	 * gridLines に登録する内容をそのまま書き込みます。
	 * gridLines = "{ color:'rgba(256,0,0,0.2)' }" という感じに、{} なども含めて書きます。
	 * この設定と、gridColor を同時に設定した場合の動作は、不定です。
	 * options:scales:yAxes:gridLines の 要素の属性です。
	 * 
	 * @og.rev 7.0.1.1 (2018/10/22) 属性の追加。
	 * 
	 * @param gridLines y軸のgridLines属性
	 */
	public void setGridLines( final String gridLines ) {
		jsData.addAxis( "gridLines" , nval( getRequestParameter( gridLines ),null ) , USE_QUOTE );		// 文字
	}

	//========================================================================================

	/**
	 * 【TAG】その他data:datasetのｵﾌﾟｼｮﾝを指定します。
	 *
	 * @og.tag
	 *
	 * @og.rev 7.0.1.2 (2018/11/04) 属性の追加。
	 * 
	 * @param attri その他data:datasetのｵﾌﾟｼｮﾝ
	 */
	public void setOptDataset( final String attri ) {
		jsData.addOptions( JsChartData.DATASET , nval( getRequestParameter( attri ),null ) );
	}

	/**
	 * 【TAG】その他options:scales:yAxesのｵﾌﾟｼｮﾝを指定します。
	 * 
	 * @og.tag
	 * options:scales:yAxes の 要素の属性です。
	 *  ※ chartJS上は、Axes(axisの複数形)と、Axis を使い分けていますが、属性は、axis で統一します。
	 *
	 * @og.rev 7.0.1.2 (2018/11/04) 属性の追加。
	 * 
	 * @param attri その他options:scales:yAxesのｵﾌﾟｼｮﾝ
	 */
	public void setOptAxis( final String attri ) {
		jsData.addOptions( JsChartData.AXIS , nval( getRequestParameter( attri ),null ) );
	}

	/**
	 * 【TAG】その他options:scales:yAxes:ticksのｵﾌﾟｼｮﾝを指定します。
	 * 
	 * @og.tag
	 * options:scales:yAxes:ticks の 要素の属性です。
	 *
	 * @og.rev 7.0.1.2 (2018/11/04) 属性の追加。
	 * 
	 * @param attri その他options:scales:yAxes:ticksのｵﾌﾟｼｮﾝ
	 */
	public void setOptTicks( final String attri ) {
		jsData.addOptions( JsChartData.TICKS , nval( getRequestParameter( attri ),null ) );
	}

	/**
	 * 【TAG】その他options:scales:yAxes:scaleLabelのｵﾌﾟｼｮﾝを指定します。
	 * 
	 * @og.tag
	 * options:scales:yAxes:scaleLabel の 要素の属性です。
	 *
	 * @og.rev 7.0.1.2 (2018/11/04) 属性の追加。
	 * 
	 * @param attri その他options:scales:yAxes:scaleLabelのｵﾌﾟｼｮﾝ
	 */
	public void setOptScaleLabel( final String attri ) {
		jsData.addOptions( JsChartData.SCALE_LABEL , nval( getRequestParameter( attri ),null ) );
	}

	/**
	 * 【TAG】その他options:scales:yAxes:gridLinesのｵﾌﾟｼｮﾝを指定します。
	 * 
	 * @og.tag
	 * options:scales:yAxes:gridLines の 要素の属性です。
	 *
	 * @og.rev 7.0.1.2 (2018/11/04) 属性の追加。
	 * 
	 * @param attri その他options:scales:yAxes:gridLinesのｵﾌﾟｼｮﾝ
	 */
	public void setOptGridLines( final String attri ) {
		jsData.addOptions( JsChartData.GRID_LINES , nval( getRequestParameter( attri ),null ) );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 * 
	 * @return このクラスの文字列表現
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
			.println( "VERSIION"			, VERSION	)
			.println( "JsChartData"			, jsData	)
			.fixForm().toString();
	}
}
