/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.table;

import java.io.File;
import java.io.PrintWriter;
import java.util.Map;

import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.FileUtil;
import org.opengion.fukurou.util.FixLengthData;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.AbstractTableFilter;
import org.opengion.hayabusa.db.DBTableModel;

/**
 * TableFilter_SEQUENCE は、TableUpda インターフェースを継承した、DBTableModel 処理用の
 * 実装クラスです。
 *
 * ここでは、シーケンス一覧の検索結果より、GF09 のシーケンス定義テーブルから
 * 必要な情報を取得し、シーケンス作成スクリプトを作成します。
 *
 * この処理を実行するには、DBTableModelのカラムとして、
 *  SYSTEM_ID,TBLSYU,SEQNAME,INCREBY,STARTVAL,MINVAL,MAXVAL,FGCYCLE,SUCACHE
 * が必要です。
 *
 * パラメータは、tableFilterタグの keys, vals にそれぞれ記述するか、BODY 部にCSS形式で記述します。
 * 【パラメータ】
 *  {
 *       DIR : {&#064;BASE_DIR}/sql/03_SEQ ;    出力ファイルの基準フォルダ(必須)
 *       XML : false ;                          XML出力を行うかどうか[true/false]を指定します(初期値:false)。
 *  }
 *
 * @og.formSample
 * ●形式：
 *      select SYSTEM_ID,TBLSYU,SEQNAME,INCREBY,STARTVAL,MINVAL,MAXVAL,FGCYCLE,SUCACHE from GF09
 * 
 *      ① &lt;og:tableFilter classId="SEQUENCE" keys="DIR,XML" vals='"{&#064;BASE_DIR}/sql/03_SEQ,"' /&gt;
 *
 *      ② &lt;og:tableFilter classId="SEQUENCE" &gt;
 *               {
 *                   DIR : {&#064;BASE_DIR}/sql/03_SEQ ;
 *                   XML : false ;
 *               }
 *         &lt;/og:tableFilter&gt;
 *
 * @og.rev 5.1.9.0 (2010/08/01) DB定義DB・シーケンス定義追加
 * @og.rev 5.6.6.0 (2013/07/05) keys の整合性チェックを追加
 *
 * @version  0.9.0  2010/08/01
 * @author   Hiroki Nakamura
 * @since    JDK1.1,
 */
public class TableFilter_SEQUENCE extends AbstractTableFilter {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.6.6.1 (2013/07/12)" ;

	/**
	 * keys の整合性チェックを行うための初期設定を行います。
	 *
	 * @og.rev 5.6.6.1 (2013/07/12) keys の整合性チェック対応
	 *
	 * @param	keysMap keys の整合性チェックを行うための Map
	 */
	@Override
	protected void init( final Map<String,String> keysMap ) {
		keysMap.put( "DIR"	, "出力ファイルの基準フォルダ(必須)"						);
		keysMap.put( "XML"	, "XML出力を行うかどうか[true/false]を指定(初期値:false)"	);
	}

	private static final String[] DBKEY = {"SYSTEM_ID","TBLSYU","SEQNAME","INCREBY","STARTVAL","MINVAL","MAXVAL","FGCYCLE","SUCACHE" };

	/** データのアクセス用の配列番号 {@value} */
	protected static final int SYSTEM_ID		= 0;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int TBLSYU			= 1;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int SEQNAME			= 2;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int INCREBY			= 3;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int STARTVAL			= 4;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int MINVAL			= 5;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int MAXVAL			= 6;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int FGCYCLE			= 7;
	/** データのアクセス用の配列番号 {@value} */
	protected static final int SUCACHE			= 8;

	private static final String ENCODE = "UTF-8" ; // 4.3.6.6 (2009/05/15)

	private static final String CMNT  = "************************************************************************" ;

	private static final int X = FixLengthData.X ;
	private static final int K = FixLengthData.K ;

	/** 各種定数 */
	protected static final String XML_START_TAG	= "<?xml version='1.0' encoding='UTF-8'?>" + CR + "<ROWSET tableName='xxx'>";
	protected static final String XML_END_TAG	= "</ROWSET>";
	protected static final String EXEC_START_TAG= "<EXEC_SQL>";
	protected static final String EXEC_END_TAG	= "</EXEC_SQL>";

	/** XML形式かどうか  */
	protected boolean		isXml				= false; // 4.3.7.0 (2009/06/01)

	/**
	 * DBTableModel処理を実行します。
	 *
	 * @og.rev 5.5.2.6 (2012/05/25) protected変数を、private化したため、getterメソッドで取得するように変更
	 *
	 * @return	実行結果のテーブルモデル
	 */
	public DBTableModel execute() {
		DBTableModel table = getDBTableModel();		// 5.5.2.6 (2012/05/25) インターフェースにgetterメソッド追加

		isXml = StringUtil.nval( getValue( "XML" ), false );

		int[] clmNo = getTableColumnNo( DBKEY );
		int rowCnt = table.getRowCount();

		File dir = new File( getValue( "DIR" ) );
		if( ! dir.exists() && ! dir.mkdirs() ) {
			String errMsg = "所定のフォルダが作成できませんでした。[" + dir + "]" ;
			// 4.3.4.4 (2009/01/01)
			throw new HybsSystemException( errMsg );
		}

		String[] data  = null;
		PrintWriter writer = null;
		for( int row=0; row<rowCnt; row++ ) {
			data  = table.getValues( row );
			String seqName = data[clmNo[SEQNAME]];
			try {
				writer = FileUtil.getPrintWriter( new File( dir,seqName + ( isXml ? ".xml" : ".sql" ) ),ENCODE );
				if( isXml ) { writer.println( XML_START_TAG ); }
				writer.print( makeHeadLine( clmNo,data ) );
				writer.println( makeLineList( clmNo,data ) );
			}
			catch( RuntimeException ex ) {
				ErrorMessage errMessage = makeErrorMessage( "TableFilter_SEQNENCE Error",ErrorMessage.NG );
				errMessage.addMessage( row+1,ErrorMessage.NG,"SEQUENCE",ex.getMessage() );
				errMessage.addMessage( row+1,ErrorMessage.NG,"SEQUENCE",StringUtil.array2csv( data ) );
				errMessage.addMessage( row+1,ErrorMessage.NG,"SEQUENCE","SEQ=[" + seqName + "]" );
				System.out.println( errMessage );
			}
			if( isXml ) { writer.println( XML_END_TAG ); }
			if( writer != null ) { writer.close(); }
		}

		return table;
	}

	/**
	 * ヘッダー部分の処理を実行します。
	 *
	 * @og.rev 5.6.6.0 (2013/07/05) FixLengthData の簡易コンストラクタを使用
	 *
	 * @param	clmNo	カラム番号配列
	 * @param	data	１行分のデータ配列
	 *
	 * @return	ヘッダー部分の文字列
	 */
	protected String makeHeadLine( final int[] clmNo,final String[] data ) {
		String seqName = data[clmNo[SEQNAME]];
		String LINE1 = seqName;
		String LINE2 = "Created : " + HybsSystem.getDate() ;

		// 5.6.6.0 (2013/07/05) FixLengthData の簡易コンストラクタを使用
		int[] addLen = new int[] { 0,0,0 };	// 各データ間のスペース
		int[] type   = new int[] { X,K,X };	// 各データの種別 X:半角 S:空白前埋め K:全角混在
		FixLengthData fixData = new FixLengthData( addLen,type );

		String[][] outData = new String[][] {
			{ "/**",	CMNT ,	"**/" },
			{ "/* ",	LINE1,	" */" },
			{ "/* ",	LINE2,	" */" },
			{ "/**",	CMNT ,	"**/" },
		};

		fixData.addAllListData( outData );

		return fixData.getAllFixData();
	}

	/**
	 * シーケンス作成の処理を実行します。
	 *
	 * @param	clmNo	カラム番号配列
	 * @param	data	１行分のデータ配列
	 *
	 * @return	シーケンス作成
	 */
	protected String makeLineList( final int[] clmNo,final String[] data ) {
		StringBuilder buf = new StringBuilder();

		if( isXml ) { buf.append( EXEC_START_TAG ).append( CR ); }

		buf.append( "CREATE SEQUENCE " ).append( data[clmNo[SEQNAME]] ).append( CR );
		buf.append( "  START WITH " ).append( data[clmNo[STARTVAL]] );
		buf.append( " INCREMENT BY " ).append( data[clmNo[INCREBY]] );
		buf.append( " MINVALUE " ).append( data[clmNo[MINVAL]] );
		buf.append( " MAXVALUE " ).append( data[clmNo[MAXVAL]] );

		if( "1".equals( data[clmNo[FGCYCLE]] ) ) {
			buf.append( " CYCLE" );
		}
		else {
			buf.append( " NOCYCLE" );
		}

		if( Integer.valueOf( data[clmNo[SUCACHE]] ) > 0 ) {
			buf.append( " CACHE " ).append( data[clmNo[SUCACHE]] );
		}
		else {
			buf.append( " NOCACHE" );
		}

		if( isXml )	{ buf.append( CR ).append( EXEC_END_TAG ); }
		else		{ buf.append( " ;" ); }

		return buf.toString();
	}
}
