/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.fukurou.util.HybsEntry;

import org.opengion.fukurou.util.StringUtil ;
import static org.opengion.fukurou.util.StringUtil.nval ;

/**
 * WriteTableTag にパラメーターを渡す為のタグクラスです。
 *
 * writeTable タグに対して、EXEC_SQL 情報や、tableName情報を付加することができます。
 * WriteTable の、writerClass によって、使用できるキーが異なります。
 * writerClass="XML"
 * 　　TableName ：XML 形式の ROWSET の table 属性になります。
 * 　　First     ：最初に記載して、初期処理(データクリア等)を実行させる、EXEC_SQL 情報になります。
 * 　　Last      ：最後に記載して、項目の設定(整合性登録)を行う、EXEC_SQL 情報になります。
 * 　　MergeSql  ：このSQL文で UPDATEして、結果が０件ならINSERTを行う、MERGE_SQL 情報になります。
 * writerClass="JSON"
 * 　　JsonName  ：JSON形式で、配列をオブジェクトとしてまとめる場合に使います。
 * 　　LowerCase ：カラム名(=パラメータ名)を小文字にする場合、true をセットします(初期値:false)。
 * writerClass="CalcDef"
 * 　　Size ：レコードのデータ件数(初期値:25)
 *
 * 値は、value 属性で指定するか、なければ BODY 部に記述します。
 *
 * tableName情報は、XMLファイルのROWSET属性にセットすることで、XMLファイルの登録テーブル名を
 * 指定することができます。
 * EXEC_SQL 情報とは、タブ区切りファイルやXML形式ファイルの先頭(key="First")
 * または、最後(key="Last")に、SQL文を記述することで、ファイル取り込み時の
 * 前処理、後処理を処理する為の情報です。
 * key="MergeSql" で、MERGE_SQL 情報をセットできます。MERGE_SQL を登録すると、
 * そのSQL文で、UPDATEして、結果が０件ならINSERTを行います。
 *
 * この情報は、複数件登録できるため、通常の writeTable タグに属性を追加すると、
 * 複雑になるため、複数登録できる用に、内部にタグを持てる構造にします。
 *
 * @og.formSample
 * ●形式：
 *     &lt;og:writeTableParam key="[First|Last|TableName]" &gt;
 *          delete from GE12 where SYSTEM_ID='**' and KBSAKU='0'
 *     &lt;/og:writeTableParam
 *
 * ●body：あり(EVAL_BODY_BUFFERED:BODYを評価し、{&#064;XXXX} を解析します)
 *
 * ●Tag定義：
 *   &lt;og:writeTableParam
 *       key              ○【TAG】パラメータとして渡すキー情報([First|Last|MergeSql|TableName|Size|JsonName|LowerCase])を指定しま(必須)。
 *       value              【TAG】パラメータとして渡す設定値を指定します(初期値:null)
 *       caseKey            【TAG】このタグ自体を利用するかどうかの条件キーを指定します(初期値:null)
 *       caseVal            【TAG】このタグ自体を利用するかどうかの条件値を指定します(初期値:null)
 *       caseNN             【TAG】指定の値が、null/ゼロ文字列 でない場合(Not Null=NN)は、このタグは使用されます(初期値:true)
 *       caseNull           【TAG】指定の値が、null/ゼロ文字列 の場合は、このタグは使用されます(初期値:true)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:writeTableParam&gt;
 *
 * ●使用例
 *     &lt;og:writeTable ･････ &gt;
 *         &lt;og:writeTableParam
 *             key  = "Tablename"  value="GE12"
 *         /&gt;
 *         &lt;og:writeTableParam
 *             key  = "First"             First:最初に登録
 *         &gt;
 *                      insert into GE12bk
 *                         select * from GE12
 *                         where SYSTEM_ID='**'
 *         &lt;/og:writeTableParam
 *         &lt;og:writeTableParam
 *             key  = "First"             First:の２番目に登録
 *         &gt;
 *              delete from GE12 where SYSTEM_ID='**' and KBSAKU='0'
 *         &lt;/og:writeTableParam
 *         &lt;og:writeTableParam
 *             key  = "Last"              Last:最後に登録
 *         &gt;
 *              update GE12 set XXXX='YYYY' where SYSTEM_ID='**' and KBSAKU='0'
 *         &lt;/og:writeTableParam
 *     &lt;/og:writeTableParam
 *
 * @og.rev 4.0.0.0 (2005/01/31) 新規作成
 * @og.rev 5.6.6.1 (2013/07/12) MERGE_SQL 対応
 * @og.group ファイル出力
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class WriteTableParamTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.6.6.1 (2013/07/12)" ;

	private static final long serialVersionUID = 566120130712L ;

	private static final String[] KEY_LIST = new String[] { "First","Last","MergeSql|","TableName","Size","JsonName","LowerCase" }; // 5.6.6.1 (2013/07/12)

	private String key		= null;		// "First","Last","MergeSql|","TableName","Size","JsonName","LowerCase" のどれか
	private String value	= null ;	// 実行するSQL文字列

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @og.rev 5.6.6.1 (2013/07/12) caseKey 、caseVal 属性対応
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doStartTag() {
		// 5.6.6.1 (2013/07/12) caseKey 、caseVal 属性対応
		// 6.0.0.1 (2014/04/25) These nested if statements could be combined
		if( useTag() && value == null ) { return EVAL_BODY_BUFFERED ; }	// Body を評価する。( extends BodyTagSupport 時)

		return SKIP_BODY ;					// Body を評価しない
	}

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {
		value = getBodyString().trim();
		return SKIP_BODY ;
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 5.6.6.1 (2013/07/12) caseKey 、caseVal 属性対応
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)
		// 5.6.6.1 (2013/07/12) caseKey 、caseVal 属性対応
		if( useTag() ) {
			WriteTableTag writeTable = (WriteTableTag)findAncestorWithClass( this,WriteTableTag.class );
			if( writeTable == null ) {
				String errMsg = "<b>" + getTagName() + "タグは、WriteTableTagの内側(要素)に記述してください。</b>";
				throw new HybsSystemException( errMsg );
			}

			writeTable.addParam( new HybsEntry( key,value ) );
		}
		return EVAL_PAGE ;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();		// 3.5.6.0 (2004/06/18) 追加(抜けていた)
		key		= null;		// KEY_LIST のどれか
		value	= null ;	// 実行するSQL文字列
	}

	/**
	 * 【TAG】パラメータとして渡すキー情報([First|Last|MergeSql||TableName|Size|JsonName|LowerCase])を指定しま。
	 *
	 * @og.tag
	 * WriteTable に渡すパラメータのキー情報です。
	 * writerClass によって、使用できるキーが異なります。
	 * writerClass="XML"
	 * 　　TableName ：XML 形式の ROWSET の table 属性になります。
	 * 　　First     ：最初に記載して、初期処理(データクリア等)を実行させる、EXEC_SQL 情報になります。
	 * 　　Last      ：最後に記載して、項目の設定(整合性登録)を行う、EXEC_SQL 情報になります。
	 * 　　MergeSql  ：このSQL文で UPDATEして、結果が０件ならINSERTを行う、MERGE_SQL 情報になります。
	 * writerClass="JSON"
	 * 　　JsonName  ：JSON形式で、配列をオブジェクトとしてまとめる場合に使います。
	 * 　　LowerCase ：カラム名(=パラメータ名)を小文字にする場合、true をセットします(初期値:false)。
	 * writerClass="CalcDef"
	 * 　　Size ：レコードのデータ件数(初期値:25)
	 *
	 * 値は、value 属性で指定するか、なければ BODY 部に記述します。
	 *
	 * @param   prmKey パラメータとして渡すキー情報([First|Last|MergeSql||TableName|Size|JsonName|LowerCase])
	 */
	public void setKey( final String prmKey ) {
		key = nval( getRequestParameter( prmKey ),key );
		if( ! check( key, KEY_LIST ) ) {
			String errMsg = "指定の値は、セットできません。key=[" + key + "]" + HybsSystem.CR
					+ "次の中から指定してください。(" + StringUtil.array2csv( KEY_LIST ) + ")" ;
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 【TAG】パラメータとして渡す設定値を指定します(初期値:null)。
	 *
	 * @og.tag
	 * WriteTable に渡すパラメータの設定値です。
	 * First と Last を指定した場合は、XML 形式で出力する EXEC_SQL 情報になります。
	 * TableName の場合は、XML 形式の ROWSET の table 属性になります。
	 * 値は、value 属性で指定するか、なければ BODY 部に記述します。
	 * BODY 部に記述された場合は、文字列を trim() します。
	 * 設定値は、value 属性が優先です。ここの値が、null の場合は、
	 * BODY 要素を値として使用します。
	 *
	 * @param   val パラメータとして渡す設定値
	 */
	public void setValue( final String val ) {
		value = nval( getRequestParameter( val ),value );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	@Override
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "key"			,key		)
				.println( "value"		,value		)
				.println( "KEY_LIST"	,KEY_LIST	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
