/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.io;

import java.util.List;
import java.awt.GradientPaint;
import java.awt.Color;
import java.awt.BasicStroke;

import org.jfree.chart.plot.Plot;
import org.jfree.chart.plot.ValueMarker;
import org.jfree.chart.axis.CategoryAxis;
import org.jfree.chart.axis.DateAxis;
import org.jfree.chart.renderer.category.CategoryItemRenderer;
import org.jfree.chart.labels.StandardCategoryToolTipGenerator;
import org.jfree.data.category.CategoryDataset;
import org.jfree.ui.Layer;

/**
 * ChartPlot_Time は、ChartPlot インターフェースを継承した実体クラスです。
 * JFreeChart では、CategoryPlot 関係の プロットを構築して、レンデラーや、データセットを
 * 設定していきます。
 * ここでは、複数のデータセットをサポートしており、ChartCreate の getDatasetList で
 * 得られる List オブジェクトを順に設定していきます。
 *
 * @version  0.9.0	2007/06/21
 * @author	 Kazuhiko Hasegawa
 * @since	 JDK1.1,
 */
public class ChartPlot_Time implements ChartPlot {

	private static final BasicStroke DOT_LINE =
										new BasicStroke(
											2.0f,							// 太さ
											BasicStroke.CAP_ROUND,
											BasicStroke.JOIN_ROUND,
											1.0f,
											new float[] { 10.0f, 6.0f },	// 破線の形状
											0.0f);

	/**
	 * Plot オブジェクトを取得します。
	 *
	 * Plot オブジェクト には、その種類の応じた、データセットやレンデラーを
	 * 設定する必要があります。
	 * また、複数のデータセットや、それに関係する属性情報も、設定する必要が
	 * あります。
	 * Plot は、JFreeChart オブジェクトにつき、一つ用意しなければなりません。
	 * チャート合成時でも、Plot は一つです。
	 *
	 * @og.rev 3.8.9.2 (2007/07/28) シリーズ指定時の色、シェープ対応
	 * @og.rev 4.1.1.0 (2008/02/04) クリッカブル・マップの設定を ChartDataset に移動
	 * @og.rev 4.1.1.0 (2008/02/04) CategoryPlotの設定を ChartCreate に移動
	 * @og.rev 4.1.1.0 (2008/02/19) Shapes の設定をChartDatasetに移動。
	 * @og.rev 5.1.4.0 (2010/03/01) jfreechart-1.0.13 対応。CategoryDataset は、setRangeAxis より先に行う必要がある。
	 *
	 * @param	create	ChartCreateオブジェクト
	 *
	 * @return	Plotオブジェクト
	 */
	public Plot getPlot( final ChartCreate create ) {
		int serNo = create.getSeriesPikup();
		HybsCategoryPlot cPlot = create.makeCategoryPlot();

		CategoryItemRenderer rend ;

		// 日付タイプの横軸を作成
		DateAxis daxis = create.getDateAxis();
		cPlot.setRangeAxis( daxis );

		// クリッカブル・マップ
		HybsURLGenerator urlGen = create.getURLGenerator();
		// 4.3.1.0 (2008/08/09) ツールチップス利用フラグ
		// 4.9.9.9 (2009/08/07) メソッド名変更
		boolean useToolTip = create.isUseToolTip();

		List<ChartDataset> datasetList = create.getDatasetList();
		for(int idx=0; idx<datasetList.size(); idx++) {
			ChartDataset chDataset = datasetList.get( idx );

	 		// 5.1.4.0 (2010/03/01) jfreechart-1.0.13 対応。CategoryDataset は、setRangeAxis より先に行う必要がある。
			CategoryDataset dtset = (CategoryDataset)chDataset.getDataset();
			int rowCnt = dtset.getRowCount();	// Series の個数
			cPlot.setDataset( idx,dtset );

			// 個別に、CategoryAxis を作成します。
			CategoryAxis categoryAxis = create.makeCategoryAxis();
			cPlot.setDomainAxis( idx,categoryAxis,false );

			// 横軸ラベルの表示は、グラフ合成時には行いません
			if( idx > 0 ) {
				categoryAxis.setVisible( false );	// ドメインラベルの表示有無
			}

			// 4.1.1.0 (2008/02/04) クリッカブル・マップの設定を移動
			rend = chDataset.getRenderer( serNo,urlGen );
			cPlot.setRenderer( idx,rend,false );

			cPlot.setRangeAxis( idx,daxis,false );

			// 縦軸マーカーの設定(横ライン)
			ValueMarker[] marker = chDataset.getValueMarkers();
			for( int i=0; i<marker.length; i++ ) {
				cPlot.addRangeMarker( idx,marker[i],Layer.FOREGROUND );
			}

			// 4.0.3.0 (2008/01/07) BOXチャートのバー幅(double)を指定
			String barWidth = chDataset.getBarWidth();
			if( barWidth != null ) {
				cPlot.setBarWidth( idx,Double.valueOf( barWidth ) );
			}

			// シリーズの色設定
			Color[] colors = chDataset.getSeriesColors();
			if( serNo < 0 && colors != null && colors.length > 0 ) {
				int size = colors.length ;
				for( int row=0; row<rowCnt; row++ ) {
					rend.setSeriesPaint( row, colors[ row%size ] );
				}
			}

			if( serNo >= 0 && serNo < rowCnt ) {
				for( int row=0; row<rowCnt; row++ ) {
					if( row == serNo ) { continue; }
					rend.setSeriesPaint( row, Color.CYAN );
				}
				rend.setSeriesPaint( serNo, Color.RED );
			}

			// グラデーションの設定
			if( chDataset.isUseGradient() ) {
				for( int row=0; row<rowCnt; row++ ) {
					Color clr  = (Color)rend.getSeriesPaint(row) ;
					if( clr != null ) {
						Color clr1 = clr.brighter().brighter() ;
						Color clr2 = clr.darker().darker() ;

						GradientPaint gp0 = new GradientPaint(0.0f, 0.0f, clr1, 0.0f, 0.0f, clr2);
						rend.setSeriesPaint( row, gp0 );
					}
				}
			}

			// ラインチャートの点線表示
			if( chDataset.isUseDottedLine() ) {
				for( int row=0; row<rowCnt; row++ ) {
					rend.setSeriesStroke( row,DOT_LINE );
				}
			}

			// 4.3.1.0 (2008/08/09) ツールチップスの利用
			if( useToolTip ) {
				rend.setBaseToolTipGenerator( new StandardCategoryToolTipGenerator() );
			}
		}
		return cPlot;
	}
}
