/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.SystemManager;
import org.opengion.fukurou.util.Cleanable;

import java.util.Map;
import java.util.HashMap;
import java.util.Locale ;
import java.util.Collections;

/**
 * DBType オブジェクトを取得する為に使用する，ファクトリクラスです。
 *
 *  DBType オブジェクト の識別ID を元に、DBTypeFactory.newInstance( String id )
 * メソッドで，DBType オブジェクトを取得します。
 * このオブジェクトは、内部的にすべてキャッシュしておき、Webアプリケーション内で
 * 同時アクセスされますが、このオブジェクトは読み取り専用の為，マルチスレッド対応
 * していません。
 * よって、DBTypeFactory.close() メソッドで，オブジェクトを返す必要も
 * ありません。
 *
 * @og.group データ属性
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class DBTypeFactory {
	private static Map<String,DBType> map = Collections.synchronizedMap( new HashMap<String,DBType>() );

	// 4.0.0 (2005/01/31) Cleanable インターフェースによる初期化処理
	static {
		Cleanable clr = new Cleanable() {
			public void clear() {
				DBTypeFactory.clear();
			}
		};

		SystemManager.addCleanable( clr );
	}

	/**
	 *  デフォルトコンストラクターをprivateにして、
	 *  オブジェクトの生成をさせないようにする。
	 *
	 */
	private DBTypeFactory() {}

	/**
	 * 識別id に応じた DBType オブジェクトを取得します。
	 * DBType オブジェクト はすべてのWebアプリケーション中で
	 * 共有して使用されます。
	 *
	 * @og.rev 3.4.0.2 (2003/09/05) DBType のデフォルト値を、'X' から 'XK' に変更します。
	 * @og.rev 3.5.6.0 (2004/06/18) 各種プラグイン関連付け設定を、システムパラメータ に記述します。
	 * @og.rev 4.0.0.0 (2005/01/31) キーの指定を、DBType. から、DBType_ に変更します。
	 * @og.rev 5.1.6.0 (2010/05/01) 初期タイプを DBType.DEF_TYPE を使用するように変更します(設定値は、XK のままです。)
	 *
	 * @param	id DBTypeインターフェースを実装したサブクラスの識別id
	 *
	 * @return	DBTypeオブジェクト
	 */
	public static DBType newInstance( final String id ) {
		String type = ( id == null ) ? DBType.DEF_TYPE : id.toUpperCase(Locale.JAPAN);
		DBType dbType = map.get( type );
		if( dbType == null ) {
			String cls = HybsSystem.sys( "DBType_" + type ) ;	// 4.0.0 (2005/01/31)
			dbType = (DBType)HybsSystem.newInstance( cls );	// 3.5.5.3 (2004/04/09)
			map.put( type,dbType );
		}
		return dbType;
	}

	/**
	 * 内部キャッシュのすべての DBType オブジェクトを削除します。
	 */
	public static void clear() {
		map.clear() ;
	}
}
